/*
 *	Config tool for ADM5120 built in ethernet switch driver
 *
 *	Copyright Jeroen Vreeken (pe1rxq@amsat.org), 2005
 *
 *	This software is released under the GNU General Public License
 *	version 2.
 */

/*
 *	The ADM5120 contains a built in switch that can handle up to 6
 *	seperate vlans and ports.
 *	Each of these ports can be mapped to one or more vlans.
 *	The ethX devices don't receive packets from a vlan unless the
 *	special CPU port is added to it.
 *
 *	Examples:
 *
 *		Each vlan has one port, CPU is part of every vlan.
 *		(i.e. direct mapping of device to port: ethX == portX)
 *		0x41, 0x42, 0x44, 0x48, 0x50, 0x60
 *
 *		One big switch and everything mapped to the first device (eth0):
 *		0x7f, 0x00, 0x00, 0x00, 0x00, 0x00
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <net/if.h>
#include <linux/types.h>
#include <linux/sockios.h>

#define SIOCSMATRIX	SIOCDEVPRIVATE
#define SIOCGMATRIX	SIOCDEVPRIVATE+1
#define SIOCGADMINFO	SIOCDEVPRIVATE+2

struct adm5120_info {
	__u16	magic;
	__u16	ports;	/* Number of ports (depends on package) */
	__u16	vlan;	/* vlan represented by this device */
};

int main(int argc, char **argv)
{
	struct adm5120_info info;
	char *device = "eth0";
	unsigned char matrix[6];
	int i, j, fd;
	struct ifreq ifr;

	if (argc > 1) {
		if (argv[1][0] == '-') {
			printf("admswconfig 0.1 - Configuration tool for ADM5120 ethernet switch\n");
			printf("Copyright Jeroen Vreeken (pe1rxq@amsat.org), 2005\n");
			printf("\n");
			printf("admswconfig [device] [012345c]\n");
			return 0;
		}
		device = argv[1];
	}

	if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		perror("socket()");
		return 1;
	}
	strcpy(ifr.ifr_name, device);
	ifr.ifr_data = (caddr_t)&info;
	if (ioctl(fd, SIOCGADMINFO, &ifr) < 0) {
		perror("SIOCGADMINFO");
		return 1;
	}

	if (info.magic != 0x5120) {
		fprintf(stderr, "%s is not an ADM5120 switch\n", device);
		return 1;
	}
	ifr.ifr_data = matrix;
	if (ioctl(fd, SIOCGMATRIX, &ifr) < 0) {
		perror("SIOCGMATRIX");
		return 1;
	}

	if (argc >= 2) {
		/* Set matrix */
		matrix[info.vlan] = 0;
		if (argc > 2) {
		  int cpu_flag = 0;
		  for (i = 0; i < strlen(argv[argc-1]); i++) {
		    char c = argv[argc-1][i];
		    if (c == 'c'){
		      matrix[info.vlan] |= 0x40;
		      cpu_flag = 1;
		    }else if (c >= '0' && c < '0'+info.ports)
		      matrix[info.vlan] |= 1 << (c - '0');
		    else {
		      printf("Invalid port '%c'\n", c);
		    }
		  }
		  if (!cpu_flag)
		    printf("Warning: CPU port is not part of this VLAN group\n");		  
		}else
		  printf("Removing all ports from %s\n",device);

		if (ioctl(fd, SIOCSMATRIX, &ifr) < 0) {
			perror("SIOCSMATRIX");
			return 1;
		}
	} else {
		/* display matrix */
		printf("ethX\tport0\tport1\tport2\tport3\tport4");
		if (info.ports == 6)
			printf("\tport5");
		printf("\tCPU\n");
		for (i = 0; i < info.ports; i++) {
			printf("%d", i);
			for (j = 0; j < info.ports; j++) {
				printf("\t");
				if (matrix[i] & (1<<j))
					printf("1");
			}
			printf("\t");
			if (matrix[i] & 0x40)
				printf("1");
			printf("\n");
		}
		printf("\n");
	}

	return 0;
}
