/******************************************************************************

                              Copyright (c) 2009
                            Lantiq Deutschland GmbH
                     Am Campeon 3; 85579 Neubiberg, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

*******************************************************************************/

/**
   \file drv_tapi_announcements.c
   Contains the High-level TAPI miscellaneous functions.
*/

/* ============================= */
/* Check if feature is enabled   */
/* ============================= */
#ifdef HAVE_CONFIG_H
#include <drv_config.h>
#endif

/* ============================= */
/* Includes                      */
/* ============================= */

#include "drv_tapi_announcements.h"
#include "drv_tapi_errno.h"
#include "drv_tapi_io.h"

/* ============================= */
/* Global constant definition    */
/* ============================= */

#ifdef TAPI_ANNOUNCEMENTS

/* maximum number of announcements */
#define TAPI_ANNOUNCEMENTS_MAX      256

/* maximum size of one announcement (64 kB) */
#define TAPI_ANNOUNCEMENTS_SIZE_MAX 65536

#endif /* TAPI_ANNOUNCEMENTS */

/* ============================= */
/* Global macro definition       */
/* ============================= */

#ifdef TAPI_ANNOUNCEMENTS

#ifdef LINUX
#define CACHE_INVALIDATE(start, size) \
   dma_cache_inv ((unsigned long)(start), (unsigned long)(size))
#else /* VXWORKS */
#define CACHE_INVALIDATE(start, size) \
   cacheInvalidate (DATA_CACHE, (void *)(start), (size_t)(size))
#endif

#endif /* TAPI_ANNOUNCEMENTS */

/* ============================= */
/* Local variable definition     */
/* ============================= */

#ifdef TAPI_ANNOUNCEMENTS

typedef struct
{
   /* pointer to announcement data */
   IFX_char_t* pAddress;
   /* size of announcemet payload */
   IFX_uint32_t nSize;
} IFX_TAPI_ANN_t;

IFX_TAPI_ANN_t Ann[TAPI_ANNOUNCEMENTS_MAX];

/* announcement ID of currently playing announcement */
static IFX_int32_t nCurrentlyPlaying = -1;

/** Prohibits access to nCurrentlyPlaying while an announcement is played and
 * freeing of currently played announcement. */
static TAPI_OS_mutex_t        semProtectAnn;

#endif /* TAPI_ANNOUNCEMENTS */


/* ============================= */
/* Type declarations             */
/* ============================= */

struct TAPI_ANNOUNCEMENT_HEADER
{
   /* codec type */
   IFX_TAPI_COD_TYPE_t codec;
} __PACKED__ ;

typedef struct TAPI_ANNOUNCEMENT_HEADER TAPI_ANNOUNCEMENT_HEADER_t;

/* ============================= */
/* Local function declaration    */
/* ============================= */

/* ============================= */
/* Global functions declaration  */
/* ============================= */

/* ============================= */
/* Global function definition    */
/* ============================= */

#ifdef TAPI_ANNOUNCEMENTS
/**
   Initialize the announcement service

   \return
   - IFX_SUCCESS  in all cases
*/
IFX_return_t IFX_TAPI_Ann_Init (IFX_void_t)
{
   /* create announcement protection semaphore */
   TAPI_OS_MutexInit (&semProtectAnn);

   memset(&Ann, 0, sizeof(Ann));
   return IFX_SUCCESS;
}

/**
   Clean-up the announcement service

   \return
   - none
*/
IFX_void_t IFX_TAPI_Ann_Cleanup (IFX_void_t)
{
   /* delete announcement protection semaphore */
   TAPI_OS_MutexDelete (&semProtectAnn);
}

/**
   The command adds (or downloads) a new announcement to the TAPI. The TAPI
   allocates a contiguous buffer and copies the data to that buffer.

   \param pChannel        Handle to TAPI_CHANNEL structure
   \param pCfg            Pointer to the IFX_TAPI_COD_ANNOUNCE_CFG_t structure

   \return
   - TAPI_statusOk - if successful
   - TAPI_statusParam - invalid parameter
   - TAPI_statusAnnInUse - announcement ID in use
   - TAPI_statusErrKernCpy - TAPI_OS_CpyUsr2Kern failed
*/
IFX_int32_t IFX_TAPI_Ann_Cfg (TAPI_CHANNEL *pChannel,
                              IFX_TAPI_COD_ANNOUNCE_CFG_t *pCfg)
{
   IFX_TAPI_DRV_CTX_t   *pDrvCtx = pChannel->pTapiDevice->pDevDrvCtx;
   TAPI_DEV             *pTapiDev = pChannel->pTapiDevice;

   /* check announcement resource number */
   if (pCfg->nAnnIdx >= TAPI_ANNOUNCEMENTS_MAX)
   {
      TRACE(TAPI_DRV,DBG_LEVEL_HIGH,
         ("DRV_ERROR: invalid announcement index %d, valid range [0; %d)\n",
            pCfg->nAnnIdx, TAPI_ANNOUNCEMENTS_MAX));
      RETURN_STATUS (TAPI_statusParam, 0);
   }
   /* check announcement message size, should be <= 64 kB */
   if (pCfg->nAnnSize > TAPI_ANNOUNCEMENTS_SIZE_MAX)
   {
      TRACE(TAPI_DRV,DBG_LEVEL_HIGH,
         ("DRV_ERROR: invalid announcement payload size %d, valid range [1; %d]\n",
            pCfg->nAnnSize, TAPI_ANNOUNCEMENTS_SIZE_MAX));
      RETURN_STATUS (TAPI_statusParam, 0);
   }
   /* check if announcement is already in use */
   if (Ann[pCfg->nAnnIdx].pAddress != IFX_NULL)
   {
      TRACE(TAPI_DRV,DBG_LEVEL_HIGH,
         ("DRV_ERROR: announcement %d already in use\n", pCfg->nAnnIdx));
      /* errmsg: Announcement ID already in use */
      RETURN_STATUS (TAPI_statusAnnInUse, 0);
   }

   /* check if specified codec is supported */
   if (ptr_chk(pDrvCtx->CAP_Check, "pDrvCtx->CAP_Check"))
   {
      IFX_TAPI_CAP_t       Capabilities;
      IFX_int32_t          ret;

      memset(&Capabilities, 0, sizeof(Capabilities));
      Capabilities.captype  = IFX_TAPI_CAP_TYPE_CODEC;
      Capabilities.cap = *((IFX_TAPI_COD_TYPE_t*) pCfg->pAnn);
      ret = pDrvCtx->CAP_Check(pTapiDev->pLLDev, &Capabilities);
      if (ret != 1)
      {
         /* errmsg: Encoder not supported by the firmware */
         RETURN_STATUS (TAPI_statusEncoderNotSupported, 0);
      }
   }
   else
      RETURN_DEVSTATUS (TAPI_statusLLNotSupp, 0);

   /* allocate memory in TAPI driver for announcement storage */
   Ann[pCfg->nAnnIdx].pAddress = (IFX_char_t *)
      TAPI_OS_Malloc (pCfg->nAnnSize + sizeof(TAPI_ANNOUNCEMENT_HEADER_t));
   if (Ann[pCfg->nAnnIdx].pAddress == IFX_NULL)
   {
      RETURN_STATUS (TAPI_statusNoMem, 0);
   }

   /* store announcement in TAPI driver */
   if (TAPI_OS_CpyUsr2Kern (Ann[pCfg->nAnnIdx].pAddress,
      (IFX_void_t*)pCfg->pAnn,
      pCfg->nAnnSize + sizeof(TAPI_ANNOUNCEMENT_HEADER_t)) == 0)
   {
      RETURN_STATUS (TAPI_statusErrKernCpy, 0);
   }
   Ann[pCfg->nAnnIdx].nSize = pCfg->nAnnSize;

   return TAPI_statusOk;
}

/**
   This command starts an announcement. An announcement playback may also be
   started, when an announcement playback is ongoing.

   \param pChannel        Handle to TAPI_CHANNEL structure
   \param pStart          Pointer to the IFX_TAPI_COD_ANNOUNCE_START_t structure

   \return
   - TAPI_statusOk - if successful
   - TAPI_statusParam - invalid parameter
   - TAPI_statusAnnNotConfigured - no data for announcement have been downloaded
   - TAPI_statusLLNotSupp - LL driver doesn't support announcement start
*/
IFX_int32_t IFX_TAPI_Ann_Start (TAPI_CHANNEL *pChannel,
   IFX_TAPI_COD_ANNOUNCE_START_t *pStart)
{
   IFX_TAPI_DRV_CTX_t   *pDrvCtx = pChannel->pTapiDevice->pDevDrvCtx;
   IFX_int32_t          ret;

   /* check announcement resource number */
   if (pStart->nAnnIdx >= TAPI_ANNOUNCEMENTS_MAX)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
         ("DRV_ERROR: invalid announcement index %d, valid range [0; %d)\n",
            pStart->nAnnIdx, TAPI_ANNOUNCEMENTS_MAX));
      RETURN_STATUS (TAPI_statusParam, 0);
   }
   /* check if announcement has been configured */
   if (Ann[pStart->nAnnIdx].pAddress == IFX_NULL)
   {
      TRACE(TAPI_DRV,DBG_LEVEL_HIGH,
         ("DRV_ERROR: no data for announcement %d\n", pStart->nAnnIdx));
      /* errmsg: Announcement not configured */
      RETURN_STATUS (TAPI_statusAnnNotConfigured, 0);
   }

   /* check if announcement is already playing */
   if (nCurrentlyPlaying != -1)
   {
      TRACE(TAPI_DRV,DBG_LEVEL_HIGH,
         ("DRV_ERROR: announcement %d already playing\n", nCurrentlyPlaying));
      /* errmsg: Announcement playout ongoing */
      RETURN_STATUS (TAPI_statusAnnActive, 0);
   }

   if (ptr_chk(pDrvCtx->COD.Ann_Start, "pDrvCtx->COD.Ann_Start"))
   {
      /* samaphore will be released by IFX_TAPI_EVENT_COD_ANNOUNCE_END event */
      TAPI_OS_MutexGet (&semProtectAnn);
      nCurrentlyPlaying = pStart->nAnnIdx;
      ret = pDrvCtx->COD.Ann_Start(pChannel->pLLChannel, pStart,
         Ann[pStart->nAnnIdx].pAddress, Ann[pStart->nAnnIdx].nSize);
      if (!TAPI_SUCCESS(ret))
      {
         nCurrentlyPlaying = -1;
         TAPI_OS_MutexRelease (&semProtectAnn);
      }
   }
   else
      RETURN_STATUS (TAPI_statusLLNotSupp, 0);

   RETURN_STATUS (TAPI_statusOk, ret);
}

/**
   This command stops an announcement.

   \param pChannel        Handle to TAPI_CHANNEL structure

   \return
   - TAPI_statusOk - if successful
   - TAPI_statusLLNotSupp - LL driver doesn't support announcement stop
*/
IFX_int32_t IFX_TAPI_Ann_Stop (TAPI_CHANNEL *pChannel)
{
   IFX_TAPI_DRV_CTX_t   *pDrvCtx = pChannel->pTapiDevice->pDevDrvCtx;
   IFX_int32_t          ret;

   /* check if announcement isn't playing */
   if (nCurrentlyPlaying == -1)
   {
      RETURN_STATUS (TAPI_statusOk, 0);
   }

   if (ptr_chk(pDrvCtx->COD.Ann_Stop, "pDrvCtx->COD.Ann_Stop"))
      ret = pDrvCtx->COD.Ann_Stop(pChannel->pLLChannel);
   else
      RETURN_STATUS (TAPI_statusLLNotSupp, 0);

   RETURN_STATUS (TAPI_statusOk, ret);
}

/**
   This command tells the TAPI to release the buffer, which was allocated for
   an announcement. If the announcement with this ID is currently played back
   by the firmware, the TAPI will stop the announcement and wait for the
   "announcement end" event from the firmware. Only after reception of this
   event, the TAPI will actually free the buffer and mark the announcement ID
   as free. (For this buffer handling, the TAPI will have to implement a simple
   state machine).

   \param pChannel        Handle to TAPI_CHANNEL structure
   \param pFree           Pointer to the IFX_TAPI_COD_ANNOUNCE_BUFFER_FREE_t
                          structure

   \return
   - TAPI_statusOk - if successful
   - TAPI_statusParam - invalid parameter
   - TAPI_statusAnnNotConfigured - no data for announcement have been downloaded
*/
IFX_int32_t IFX_TAPI_Ann_Free (TAPI_CHANNEL *pChannel,
   IFX_TAPI_COD_ANNOUNCE_BUFFER_FREE_t *pFree)
{
   /* check announcement resource number */
   if (pFree->nAnnIdx >= TAPI_ANNOUNCEMENTS_MAX)
   {
      TRACE(TAPI_DRV,DBG_LEVEL_HIGH,
         ("DRV_ERROR: invalid announcement index %d, valid range [0; %d)\n",
            pFree->nAnnIdx, TAPI_ANNOUNCEMENTS_MAX));
      RETURN_STATUS (TAPI_statusParam, 0);
   }
   /* check if announcement has been configured */
   if (Ann[pFree->nAnnIdx].pAddress == IFX_NULL)
   {
      TRACE(TAPI_DRV,DBG_LEVEL_HIGH,
         ("DRV_ERROR: no data for announcement %d\n", pFree->nAnnIdx));
      /* errmsg: Announcement not configured */
      RETURN_STATUS (TAPI_statusAnnNotConfigured, 0);
   }

   if (pFree->nAnnIdx == nCurrentlyPlaying)
   {
      /* It is not allowed to free announcement that is currently playing. Stop
       * announcement and wait till it's finished.
       */
      IFX_TAPI_DRV_CTX_t   *pDrvCtx = pChannel->pTapiDevice->pDevDrvCtx;

      if (ptr_chk(pDrvCtx->COD.Ann_Stop, "pDrvCtx->COD.Ann_Stop"))
      {
         pDrvCtx->COD.Ann_Stop(pChannel->pLLChannel);
      }
      else
         RETURN_STATUS (TAPI_statusLLNotSupp, 0);
      /* mutex signals reception of annoucement end event, now it's possible
       * to free the annoucement */
      TAPI_OS_MutexGet (&semProtectAnn);
      /* release the mutex, otherwise annoucement playout won't be possible */
      TAPI_OS_MutexRelease (&semProtectAnn);
   }

   TAPI_OS_Free(Ann[pFree->nAnnIdx].pAddress);
   Ann[pFree->nAnnIdx].pAddress = IFX_NULL;
   Ann[pFree->nAnnIdx].nSize = 0;
   return TAPI_statusOk;
}

/**
   Process announcement end event.
   Releases semProtectAnn semaphore.

   \param   pChannel    Pointer to TAPI_CHANNEL structure.
*/
IFX_void_t IFX_TAPI_AnnEndEventServe(TAPI_CHANNEL* pChannel)
{
   IFX_TAPI_DRV_CTX_t   *pDrvCtx = pChannel->pTapiDevice->pDevDrvCtx;

   if (ptr_chk(pDrvCtx->COD.Ann_Stop, "pDrvCtx->COD.Ann_Stop"))
   {
      /* Even in the case that the annoucement sample has ended the
         Announcement playout needs to be disabled. */
      pDrvCtx->COD.Ann_Stop(pChannel->pLLChannel);
   }
   else
      return;

   nCurrentlyPlaying = -1;
   TAPI_OS_MutexRelease (&semProtectAnn);
   return;
}
#endif /* TAPI_ANNOUNCEMENTS */
