/******************************************************************************

                              Copyright (c) 2009
                            Lantiq Deutschland GmbH
                     Am Campeon 3; 85579 Neubiberg, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

******************************************************************************/

/**
   \file drv_tapi_event.c
   Contains TAPI Event Handling.
*/

/* ============================= */
/* Includes                      */
/* ============================= */

#include "drv_api.h"
#include "drv_tapi.h"
#include "drv_tapi_errno.h"
#include "drv_tapi_ll_interface.h"
#include "drv_tapi_event.h"
#ifdef TAPI_CID
#include "drv_tapi_cid.h"
#endif /* TAPI_CID */
#ifdef TAPI_ANNOUNCEMENTS
#include "drv_tapi_announcements.h"
#endif /* TAPI_ANNOUNCEMENTS */

/* ============================= */
/* Local Macros  Definitions    */
/* ============================= */

/* ============================= */
/* Global variable definition    */
/* ============================= */

/* ============================= */
/* Global functions declaration  */
/* ============================= */

#ifdef TAPI_EXT_KEYPAD
IFX_return_t TAPI_EXT_EVENT_Key_Handler (TAPI_CHANNEL * pChannel,
                                         IFX_TAPI_EVENT_t * pTapiEvent);
#endif /* TAPI_EXT_KEYPAD */


/* ============================= */
/* Type declarations             */
/* ============================= */

/* These structures are used for masking events. */
typedef struct
{
   IFX_uint32_t reserved:32;
} IFX_TAPI_EVENT_IO_GENERAL_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_IO_GENERAL_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_IO_GEN_DIS_t;

typedef struct
{
   IFX_uint32_t reserved:32;
} IFX_TAPI_EVENT_IO_INTERRUPT_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_IO_INTERRUPT_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_IO_INT_DIS_t;

typedef struct
{
   IFX_uint32_t reserved:26;
   IFX_uint32_t flash:1;
   IFX_uint32_t offhook:1;
   IFX_uint32_t onhook:1;
   IFX_uint32_t ringing_end:1;
   IFX_uint32_t ringburst_end:1;
   IFX_uint32_t ring:1;
} IFX_TAPI_EVENT_FXS_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_FXS_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_FXS_DIS_t;

typedef struct
{
   IFX_uint32_t reserved:27;
   IFX_uint32_t bat_feeded:1;
   IFX_uint32_t bat_dropped:1;
   IFX_uint32_t polarity:1;
   IFX_uint32_t ring_start:1;
   IFX_uint32_t ring_stop:1;
} IFX_TAPI_EVENT_FXO_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_FXO_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_FXO_DIS_t;

typedef struct
{
   IFX_uint32_t reserved:29;
   IFX_uint32_t gr909_rdy:1;
} IFX_TAPI_EVENT_LT_BITS_t;


typedef union
{
   IFX_TAPI_EVENT_LT_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_LT_DIS_t;


typedef struct
{
   IFX_uint32_t reserved:30;
   IFX_uint32_t pulseStart:1;
   IFX_uint32_t digit:1;
} IFX_TAPI_EVENT_PULSE_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_PULSE_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_PULSE_DIS_t;

typedef struct
{
   IFX_uint32_t reserved:28;
   IFX_uint32_t end_network:1;
   IFX_uint32_t end_local:1;
   IFX_uint32_t digit_network:1;
   IFX_uint32_t digit_local:1;
} IFX_TAPI_EVENT_DTMF_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_DTMF_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_DTMF_DIS_t;

typedef struct
{
   IFX_uint32_t reserved:19;
   IFX_uint32_t rx_error2:1;
   IFX_uint32_t rx_error1:1;
   IFX_uint32_t rx_err_read:1;
   IFX_uint32_t rx_cd:1;
   IFX_uint32_t rx_end:1;
   IFX_uint32_t tx_error2:1;
   IFX_uint32_t tx_ringcad_err:1;
   IFX_uint32_t tx_noack_err:1;
   IFX_uint32_t tx_info_end:1;
   IFX_uint32_t tx_info_start:1;
   IFX_uint32_t tx_seq_end:1;
   IFX_uint32_t tx_seq_start:1;
} IFX_TAPI_EVENT_CID_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_CID_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_CID_DIS_t;

typedef struct
{
   IFX_uint32_t reserved:29;
   IFX_uint32_t end_network:1;
   IFX_uint32_t end_local:1;
   IFX_uint32_t busy:1;
} IFX_TAPI_EVENT_TONE_GEN_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_TONE_GEN_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_TONE_GEN_DIS_t;

typedef struct
{
   IFX_uint32_t reserved:29;
   IFX_uint32_t receive:1;
   IFX_uint32_t transmit:1;
   IFX_uint32_t cpt:1;
} IFX_TAPI_EVENT_TONE_DET_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_TONE_DET_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_TONE_DET_DIS_t;

typedef struct
{
   IFX_uint32_t reserved:30;
   IFX_uint32_t v21h_network:1;
   IFX_uint32_t v21h_local:1;
   IFX_uint32_t cas_network:1;
   IFX_uint32_t cas_local:1;
   IFX_uint32_t hold_network:1;
   IFX_uint32_t hold_local:1;
   IFX_uint32_t v8bis_network:1;
   IFX_uint32_t v8bis_local:1;
   IFX_uint32_t v32ac_network:1;
   IFX_uint32_t v32ac_local:1;
   IFX_uint32_t v22orbell_network:1;
   IFX_uint32_t v22orbell_local:1;
   IFX_uint32_t v22_network:1;
   IFX_uint32_t v22_local:1;
   IFX_uint32_t bell_network:1;
   IFX_uint32_t bell_local:1;
   IFX_uint32_t v27_network:1;
   IFX_uint32_t v27_local:1;
   IFX_uint32_t v18a_network:1;
   IFX_uint32_t v18a_local:1;
   IFX_uint32_t v21l_network:1;
   IFX_uint32_t v21l_local:1;
   IFX_uint32_t cngmod_network:1;
   IFX_uint32_t cngmod_local:1;
   IFX_uint32_t cngfax_network:1;
   IFX_uint32_t cngfax_local:1;
   IFX_uint32_t am_network:1;
   IFX_uint32_t am_local:1;
   IFX_uint32_t pr_network:1;
   IFX_uint32_t pr_local:1;
   IFX_uint32_t ced_network:1;
   IFX_uint32_t ced_local:1;
   IFX_uint32_t dis_network:1;
   IFX_uint32_t dis_local:1;
} IFX_TAPI_EVENT_FAX_SIG_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_FAX_SIG_BITS_t bits;
   IFX_uint32_t status[2];
} IFX_TAPI_EVENT_FAX_SIG_DIS_t;

typedef struct
{
   IFX_uint32_t reserved:28;
   IFX_uint32_t room_noise:1;
   IFX_uint32_t room_silence:1;
   IFX_uint32_t dec_chg:1;
   IFX_uint32_t announce:1;
} IFX_TAPI_EVENT_CODER_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_CODER_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_CODER_DIS_t;

typedef struct
{
   IFX_uint32_t reserved:32;
} IFX_TAPI_EVENT_RTP_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_RTP_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_RTP_DIS_t;

typedef struct
{
   IFX_uint32_t reserved:32;
} IFX_TAPI_EVENT_AAL_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_AAL_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_AAL_DIS_t;

typedef struct
{
   IFX_uint32_t reserved:30;
   IFX_uint32_t event:1;
} IFX_TAPI_EVENT_RFC2833_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_RFC2833_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_RFC2833_DIS_t;

typedef struct
{
   IFX_uint32_t reserved:31;
   IFX_uint32_t ingress_fifo_full:1;
} IFX_TAPI_EVENT_KPI_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_KPI_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_KPI_DIS_t;

typedef struct
{
   IFX_uint32_t reserved:27;
   IFX_uint32_t error_setup:1;
   IFX_uint32_t error_data:1;
   IFX_uint32_t error_write:1;
   IFX_uint32_t error_read:1;
   IFX_uint32_t error_ovld:1;
   IFX_uint32_t error_gen:1;
} IFX_TAPI_EVENT_T38_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_T38_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_T38_DIS_t;

typedef struct
{
   IFX_uint32_t reserved:32;
} IFX_TAPI_EVENT_JB_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_JB_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_JB_DIS_t;

typedef struct
{
   IFX_uint32_t reserved:32;
} IFX_TAPI_EVENT_DOWNLOAD_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_DOWNLOAD_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_DOWNLOAD_DIS_t;

typedef struct
{
   IFX_uint32_t mbx_congestion:1;
   IFX_uint32_t reserved:31;
} IFX_TAPI_EVENT_INFORMATION_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_INFORMATION_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_INFO_DIS_t;

typedef struct
{
   IFX_uint32_t reserved:32;
} IFX_TAPI_EVENT_DEBUG_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_DEBUG_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_DEBUG_DIS_t;

typedef struct
{
   IFX_uint32_t reserved:31;
   IFX_uint32_t alive:1;
} IFX_TAPI_EVENT_LL_DRIVER_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_LL_DRIVER_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_LL_DRIVER_DIS_t;

typedef struct
{
   IFX_uint32_t reserved:32;
} IFX_TAPI_EVENT_FAULT_GENERAL_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_FAULT_GENERAL_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_FAULT_GEN_DIS_t;

typedef struct
{
   IFX_uint32_t reserved:24;
   IFX_uint32_t overtemp_end:1;
   IFX_uint32_t gk_high_end:1;
   IFX_uint32_t gk_low_end:1;
   IFX_uint32_t overtemp:1;
   IFX_uint32_t gk_high:1;
   IFX_uint32_t gk_low:1;
   IFX_uint32_t gk_neg:1;
   IFX_uint32_t gk_pos:1;
} IFX_TAPI_EVENT_FAULT_LINE_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_FAULT_LINE_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_FAULT_LINE_DIS_t;

typedef struct
{
   IFX_uint32_t reserved:25;
   IFX_uint32_t ssi_err : 1;
   IFX_uint32_t hw_reset:1;
   IFX_uint32_t sync:1;
   IFX_uint32_t spi_access:1;
   IFX_uint32_t clock_fail:1;
   IFX_uint32_t clock_fail_end:1;
   IFX_uint32_t hw_fault:1;
} IFX_TAPI_EVENT_FAULT_HW_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_FAULT_HW_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_FAULT_HW_DIS_t;

typedef struct
{
   IFX_uint32_t reserved:32;
} IFX_TAPI_EVENT_FAULT_FW_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_FAULT_FW_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_FAULT_FW_DIS_t;

typedef struct
{
   IFX_uint32_t reserved:32;
} IFX_TAPI_EVENT_FAULT_SW_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_FAULT_SW_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_FAULT_SW_DIS_t;

typedef struct
{
   IFX_uint32_t reserved:31;
   IFX_uint32_t buf_underrun:1;
} IFX_TAPI_EVENT_LIN_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_LIN_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_LIN_DIS_t;

typedef struct
{
   IFX_uint32_t reserved:31;
   IFX_uint32_t calibration_end:1;
} IFX_TAPI_EVENT_CALIBRATION_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_CALIBRATION_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_CALIBRATION_DIS_t;

typedef struct
{
   IFX_uint32_t reserved:31;
   IFX_uint32_t lcap_disable:1;
} IFX_TAPI_EVENT_LCAP_BITS_t;

typedef union
{
   IFX_TAPI_EVENT_LCAP_BITS_t bits;
   IFX_uint32_t status;
} IFX_TAPI_EVENT_LCAP_DIS_t;

typedef struct
{
   IFX_TAPI_EVENT_IO_GEN_DIS_t io_general;
   IFX_TAPI_EVENT_IO_INT_DIS_t io_interrupt;
   IFX_TAPI_EVENT_FXS_DIS_t fxs;
   IFX_TAPI_EVENT_FXO_DIS_t fxo;
   IFX_TAPI_EVENT_LT_DIS_t lt;
   IFX_TAPI_EVENT_PULSE_DIS_t pulse;
   IFX_TAPI_EVENT_DTMF_DIS_t dtmf;
   IFX_TAPI_EVENT_CID_DIS_t cid;
   IFX_TAPI_EVENT_TONE_GEN_DIS_t tone_gen;
   IFX_TAPI_EVENT_TONE_DET_DIS_t tone_det;
   IFX_TAPI_EVENT_FAX_SIG_DIS_t fax_sig;
   IFX_TAPI_EVENT_CODER_DIS_t coder;
   IFX_TAPI_EVENT_RTP_DIS_t rtp;
   IFX_TAPI_EVENT_AAL_DIS_t aal;
   IFX_TAPI_EVENT_RFC2833_DIS_t rfc2833;
   IFX_TAPI_EVENT_KPI_DIS_t kpi;
   IFX_TAPI_EVENT_T38_DIS_t t38;
   IFX_TAPI_EVENT_JB_DIS_t jb;
   IFX_TAPI_EVENT_DOWNLOAD_DIS_t download;
   IFX_TAPI_EVENT_INFO_DIS_t information;
   IFX_TAPI_EVENT_DEBUG_DIS_t debug;
   IFX_TAPI_EVENT_LL_DRIVER_DIS_t ll_driver;
   IFX_TAPI_EVENT_FAULT_GEN_DIS_t fault_general;
   IFX_TAPI_EVENT_FAULT_LINE_DIS_t fault_line;
   IFX_TAPI_EVENT_FAULT_HW_DIS_t fault_hw;
   IFX_TAPI_EVENT_FAULT_FW_DIS_t fault_fw;
   IFX_TAPI_EVENT_FAULT_SW_DIS_t fault_sw;
   IFX_TAPI_EVENT_LIN_DIS_t lin;
   IFX_TAPI_EVENT_CALIBRATION_DIS_t calibration;
   IFX_TAPI_EVENT_LCAP_DIS_t lcap;
} IFX_TAPI_EVENT_MASK_t;

struct IFX_TAPI_EVENT_HANDLER_DATA
{
   /* Tapi event fifo -- High proity event */
   FIFO_ID              *pTapiEventFifoHi;
   /* Tapi event fifo -- Low proity event */
   FIFO_ID              *pTapiEventFifoLo;
   /* Mask for Tapi events (enable/disable single events) */
   IFX_TAPI_EVENT_MASK_t eventMask;
   /* Fifo concurent access protection mutex. */
   TAPI_OS_mutex_t       fifoAcc;
};


/* ============================= */
/* Local variable definition     */
/* ============================= */

/** Memory pool for event dispatcher (global for all channels) */
static BUFFERPOOL *pIFX_TAPI_BP_Deferred_Event = IFX_NULL;
/* Buffer pool access protection */
static TAPI_OS_mutex_t semBufferPoolAcc;


/* ============================= */
/* Local function declaration    */
/* ============================= */

static IFX_boolean_t Check_FaxModem_Status (TAPI_CHANNEL *pChannel,
                                               IFX_TAPI_EVENT_t *pEvent);

static IFX_void_t TAPI_WakeUp (TAPI_DEV *pTapiDev);


/* ============================= */
/* Local function definition     */
/* ============================= */

/* ============================= */
/* Global function definition    */
/* ============================= */

/*lint -save -esym(530, lock) */
/**
   Enable or disable an event on the specified channel(s).

   \param  pChannel     Pointer to TAPI_CHANNEL structure.
   \param  pEvent       The event which mask is to be set.
   \param  value        -IFX_EVENT_ENABLE to allow the event to the application
                        -IFX_EVENT_DISABLE to discard the event

   \return
   TAPI_statusOk - success
   TAPI_statusEvtNoHandle - event not known
*/
/**\todo Map the signal detection events to the low level signal
enable */
/**\todo group FAXMODE and make use of a common union member:
         if (pEvent->data.fax_sig.local)
            pEventMask->fax_sig.bits.dis_local = value;
         if (pEvent->data.fax_sig.network)
            pEventMask->fax_sig.bits.dis_network = value;
*/
IFX_int32_t IFX_TAPI_Event_SetMask (TAPI_CHANNEL *pChannel,
                                    IFX_TAPI_EVENT_t *pEvent,
                                    IFX_uint32_t value)
{
   IFX_TAPI_DRV_CTX_t *pDrvCtx = pChannel->pTapiDevice->pDevDrvCtx;
   IFX_int32_t ret = TAPI_statusOk;
   IFX_TAPI_EVENT_MASK_t *pEventMask;

   pEventMask = &(pChannel->pEventHandler->eventMask);

   /* set mask for event filtering */
   switch(pEvent->id)
   {
      case IFX_TAPI_EVENT_FXS_RING:
         pEventMask->fxs.bits.ring = value;
         break;
      case IFX_TAPI_EVENT_FXS_RINGBURST_END:
         pEventMask->fxs.bits.ringburst_end = value;
         break;
      case IFX_TAPI_EVENT_FXS_RINGING_END:
         pEventMask->fxs.bits.ringing_end = value;
         break;
      case IFX_TAPI_EVENT_FXS_ONHOOK:
         pEventMask->fxs.bits.onhook = value;
         break;
      case IFX_TAPI_EVENT_FXS_OFFHOOK:
         pEventMask->fxs.bits.offhook = value;
         break;
      case IFX_TAPI_EVENT_FXS_FLASH:
         pEventMask->fxs.bits.flash = value;
         break;
      case IFX_TAPI_EVENT_FXO_RING_START:
         pEventMask->fxo.bits.ring_start = value;
         break;
      case IFX_TAPI_EVENT_FXO_RING_STOP:
         pEventMask->fxo.bits.ring_stop = value;
         break;
      case IFX_TAPI_EVENT_FXO_POLARITY:
         pEventMask->fxo.bits.polarity = value;
         break;
      case IFX_TAPI_EVENT_FXO_BAT_FEEDED:
         pEventMask->fxo.bits.bat_feeded = value;
         break;
      case IFX_TAPI_EVENT_FXO_BAT_DROPPED:
         pEventMask->fxo.bits.bat_dropped = value;
         break;
      case IFX_TAPI_EVENT_LT_GR909_RDY:
         pEventMask->lt.bits.gr909_rdy = value;
         break;
      case IFX_TAPI_EVENT_PULSE_DIGIT:
         pEventMask->pulse.bits.digit = value;
         break;
      case IFX_TAPI_EVENT_PULSE_START:
         pEventMask->pulse.bits.pulseStart = value;
         break;
      case IFX_TAPI_EVENT_DTMF_DIGIT:
         if (pEvent->data.dtmf.local)
            pEventMask->dtmf.bits.digit_local = value;
         if (pEvent->data.dtmf.network)
            pEventMask->dtmf.bits.digit_network = value;
         break;
      case IFX_TAPI_EVENT_DTMF_END:
         if (pEvent->data.dtmf.local)
            pEventMask->dtmf.bits.end_local = value;
         if (pEvent->data.dtmf.network)
            pEventMask->dtmf.bits.end_network = value;
         break;
      case IFX_TAPI_EVENT_CID_TX_SEQ_START:
         pEventMask->cid.bits.tx_seq_start = value;
         break;
      case IFX_TAPI_EVENT_CID_TX_SEQ_END:
         pEventMask->cid.bits.tx_seq_end = value;
         break;
      case IFX_TAPI_EVENT_CID_TX_INFO_START:
         pEventMask->cid.bits.tx_info_start = value;
         break;
      case IFX_TAPI_EVENT_CID_TX_INFO_END:
         pEventMask->cid.bits.tx_info_end = value;
         break;
      case IFX_TAPI_EVENT_CID_RX_END:
         pEventMask->cid.bits.rx_end = value;
         break;
      case IFX_TAPI_EVENT_CID_RX_CD:
         pEventMask->cid.bits.rx_cd = value;
         break;
      case IFX_TAPI_EVENT_TONE_GEN_BUSY:
         pEventMask->tone_gen.bits.busy = value;
         break;
      case IFX_TAPI_EVENT_TONE_GEN_END:
         if (pEvent->data.tone_gen.local)
            pEventMask->tone_gen.bits.end_local = value;
         if (pEvent->data.tone_gen.network)
            pEventMask->tone_gen.bits.end_network = value;
         break;
      case IFX_TAPI_EVENT_TONE_DET_RECEIVE:
         pEventMask->tone_det.bits.receive = value;
         break;
      case IFX_TAPI_EVENT_TONE_DET_TRANSMIT:
         pEventMask->tone_det.bits.transmit = value;
         break;
      case IFX_TAPI_EVENT_TONE_DET_CPT:
         pEventMask->tone_det.bits.cpt = value;
         break;
      case IFX_TAPI_EVENT_FAXMODEM_DIS:
         if (pEvent->data.fax_sig.local)
            pEventMask->fax_sig.bits.dis_local = value;
         if (pEvent->data.fax_sig.network)
            pEventMask->fax_sig.bits.dis_network = value;
         break;
      case IFX_TAPI_EVENT_FAXMODEM_CED:
         if (pEvent->data.fax_sig.local)
            pEventMask->fax_sig.bits.ced_local = value;
         if (pEvent->data.fax_sig.network)
            pEventMask->fax_sig.bits.ced_network = value;
         break;
      case IFX_TAPI_EVENT_FAXMODEM_PR:
         if (pEvent->data.fax_sig.local)
            pEventMask->fax_sig.bits.pr_local = value;
         if (pEvent->data.fax_sig.network)
            pEventMask->fax_sig.bits.pr_network = value;
         break;
      case IFX_TAPI_EVENT_FAXMODEM_AM:
         if (pEvent->data.fax_sig.local)
            pEventMask->fax_sig.bits.am_local = value;
         if (pEvent->data.fax_sig.network)
            pEventMask->fax_sig.bits.am_network = value;
         break;
      case IFX_TAPI_EVENT_FAXMODEM_CNGFAX:
         if (pEvent->data.fax_sig.local)
            pEventMask->fax_sig.bits.cngfax_local = value;
         if (pEvent->data.fax_sig.network)
            pEventMask->fax_sig.bits.cngfax_network = value;
         break;
      case IFX_TAPI_EVENT_FAXMODEM_CNGMOD:
         if (pEvent->data.fax_sig.local)
            pEventMask->fax_sig.bits.cngmod_local = value;
         if (pEvent->data.fax_sig.network)
            pEventMask->fax_sig.bits.cngmod_network = value;
         break;
      case IFX_TAPI_EVENT_FAXMODEM_V21L:
         if (pEvent->data.fax_sig.local)
            pEventMask->fax_sig.bits.v21l_local = value;
         if (pEvent->data.fax_sig.network)
            pEventMask->fax_sig.bits.v21l_network = value;
         break;
      case IFX_TAPI_EVENT_FAXMODEM_V18A:
         if (pEvent->data.fax_sig.local)
            pEventMask->fax_sig.bits.v18a_local = value;
         if (pEvent->data.fax_sig.network)
            pEventMask->fax_sig.bits.v18a_network = value;
         break;
      case IFX_TAPI_EVENT_FAXMODEM_V27:
         if (pEvent->data.fax_sig.local)
            pEventMask->fax_sig.bits.v27_local = value;
         if (pEvent->data.fax_sig.network)
            pEventMask->fax_sig.bits.v27_network = value;
         break;
      case IFX_TAPI_EVENT_FAXMODEM_BELL:
         if (pEvent->data.fax_sig.local)
            pEventMask->fax_sig.bits.bell_local = value;
         if (pEvent->data.fax_sig.network)
            pEventMask->fax_sig.bits.bell_network = value;
         break;
      case IFX_TAPI_EVENT_FAXMODEM_V22:
         if (pEvent->data.fax_sig.local)
            pEventMask->fax_sig.bits.v22_local = value;
         if (pEvent->data.fax_sig.network)
            pEventMask->fax_sig.bits.v22_network = value;
         break;
      case IFX_TAPI_EVENT_FAXMODEM_V22ORBELL:
         if (pEvent->data.fax_sig.local)
            pEventMask->fax_sig.bits.v22orbell_local = value;
         if (pEvent->data.fax_sig.network)
            pEventMask->fax_sig.bits.v22orbell_network = value;
         break;
      case IFX_TAPI_EVENT_FAXMODEM_V32AC:
         if (pEvent->data.fax_sig.local)
            pEventMask->fax_sig.bits.v32ac_local = value;
         if (pEvent->data.fax_sig.network)
            pEventMask->fax_sig.bits.v32ac_network = value;
         break;
      case IFX_TAPI_EVENT_FAXMODEM_CAS_BELL:
         if (pEvent->data.fax_sig.local)
            pEventMask->fax_sig.bits.cas_local = value;
         if (pEvent->data.fax_sig.network)
            pEventMask->fax_sig.bits.cas_network = value;
         break;
      case IFX_TAPI_EVENT_FAXMODEM_V21H:
         if (pEvent->data.fax_sig.local)
            pEventMask->fax_sig.bits.v21h_local = value;
         if (pEvent->data.fax_sig.network)
            pEventMask->fax_sig.bits.v21h_network = value;
         break;
      case IFX_TAPI_EVENT_FAXMODEM_V8BIS:
         if (pEvent->data.fax_sig.local)
            pEventMask->fax_sig.bits.v8bis_local = value;
         if (pEvent->data.fax_sig.network)
            pEventMask->fax_sig.bits.v8bis_network = value;
         break;
      case IFX_TAPI_EVENT_FAXMODEM_HOLDEND:
         if (pEvent->data.fax_sig.local)
            pEventMask->fax_sig.bits.hold_local = value;
         if (pEvent->data.fax_sig.network)
            pEventMask->fax_sig.bits.hold_network = value;
         break;
      case IFX_TAPI_EVENT_RFC2833_EVENT:
         pEventMask->rfc2833.bits.event = value;
         break;
      case IFX_TAPI_EVENT_KPI_INGRESS_FIFO_FULL:
         pEventMask->kpi.bits.ingress_fifo_full = value;
         break;
      case IFX_TAPI_EVENT_LL_DRIVER_WD_FAIL:
         pEventMask->ll_driver.bits.alive = value;
         break;
      case IFX_TAPI_EVENT_FAULT_LINE_GK_POS:
         pEventMask->fault_line.bits.gk_pos = value;
         break;
      case IFX_TAPI_EVENT_FAULT_LINE_GK_NEG:
         pEventMask->fault_line.bits.gk_neg = value;
         break;
      case IFX_TAPI_EVENT_FAULT_LINE_GK_LOW:
         pEventMask->fault_line.bits.gk_low = value;
         break;
      case IFX_TAPI_EVENT_FAULT_LINE_GK_LOW_END:
         pEventMask->fault_line.bits.gk_low_end = value;
         break;
      case IFX_TAPI_EVENT_FAULT_LINE_GK_HIGH:
         pEventMask->fault_line.bits.gk_high = value;
         break;
      case IFX_TAPI_EVENT_FAULT_LINE_GK_HIGH_END:
         pEventMask->fault_line.bits.gk_high_end = value;
         break;
      case IFX_TAPI_EVENT_FAULT_LINE_OVERTEMP:
         pEventMask->fault_line.bits.overtemp = value;
         break;
      case IFX_TAPI_EVENT_FAULT_LINE_OVERTEMP_END:
         pEventMask->fault_line.bits.overtemp_end = value;
         break;
      case IFX_TAPI_EVENT_FAULT_HW_CLOCK_FAIL:
         pEventMask->fault_hw.bits.clock_fail = value;
         break;
      case IFX_TAPI_EVENT_FAULT_HW_CLOCK_FAIL_END:
         pEventMask->fault_hw.bits.clock_fail_end = value;
         break;
      case IFX_TAPI_EVENT_FAULT_HW_FAULT:
         pEventMask->fault_hw.bits.hw_fault = value;
         break;
      case IFX_TAPI_EVENT_FAULT_HW_SYNC:
         pEventMask->fault_hw.bits.sync = value;
         break;
      case IFX_TAPI_EVENT_FAULT_HW_RESET:
         pEventMask->fault_hw.bits.hw_reset = value;
         break;
      case IFX_TAPI_EVENT_FAULT_HW_SSI_ERR:
         pEventMask->fault_hw.bits.ssi_err = value;
         break;
      case IFX_TAPI_EVENT_COD_DEC_CHG:
         pEventMask->coder.bits.dec_chg = value;
         break;
      case IFX_TAPI_EVENT_COD_ROOM_NOISE:
         pEventMask->coder.bits.room_noise = value;
         break;
      case IFX_TAPI_EVENT_COD_ROOM_SILENCE:
         pEventMask->coder.bits.room_silence = value;
         break;
      case IFX_TAPI_EVENT_COD_ANNOUNCE_END:
         pEventMask->coder.bits.announce = value;
         break;
      case IFX_TAPI_EVENT_INFO_MBX_CONGESTION:
         pEventMask->information.bits.mbx_congestion = value;
         break;
      case IFX_TAPI_EVENT_LIN_UNDERFLOW:
         pEventMask->lin.bits.buf_underrun = value;
         break;
      case IFX_TAPI_EVENT_CALIBRATION_END:
         pEventMask->calibration.bits.calibration_end = value;
         break;
      case IFX_TAPI_EVENT_LINE_MEASURE_CAPACITANCE_RDY:
         pEventMask->lcap.bits.lcap_disable = value;
         break;
      default :
         RETURN_STATUS (TAPI_statusEvtNoHandle, 0);
   }

   /* call LL-driver to enable/disable event generation */
   switch(pEvent->id)
   {
#ifdef TAPI_DTMF
      case IFX_TAPI_EVENT_DTMF_DIGIT:
      case IFX_TAPI_EVENT_DTMF_END:
         if (ptr_chk(pDrvCtx->SIG.DTMFD_Start, "") &&
             ptr_chk(pDrvCtx->SIG.DTMFD_Stop, "") &&
             (pChannel->nChannel < pChannel->pTapiDevice->nResource.DTMFRCount))
         {
            IFX_boolean_t bEnableEndEvent =
               (pEventMask->dtmf.bits.end_local == IFX_EVENT_ENABLE ||
                pEventMask->dtmf.bits.end_network == IFX_EVENT_ENABLE) ?
               IFX_TRUE : IFX_FALSE;

            if (pEventMask->dtmf.bits.digit_local == IFX_EVENT_ENABLE)
               ret = pDrvCtx->SIG.DTMFD_Start (pChannel->pLLChannel,
                                               IFX_TAPI_LL_SIG_IN_DIR_LOCAL,
                                               bEnableEndEvent);
            else
            if (pEventMask->dtmf.bits.digit_network == IFX_EVENT_ENABLE)
               ret = pDrvCtx->SIG.DTMFD_Start (pChannel->pLLChannel,
                                               IFX_TAPI_LL_SIG_IN_DIR_REMOTE,
                                               bEnableEndEvent);
            else
               ret = pDrvCtx->SIG.DTMFD_Stop (pChannel->pLLChannel);
         }
         break;
#endif /* TAPI_DTMF */
      default :
         /* nothing to do */
         break;
   }

   RETURN_STATUS (ret, 0);
}


/**
   Reports the event information for a channel

   \param  pTapiDev     Pointer to TAPI device structure.
   \param  pEvent       Pointer to a event structure.

   \return
   TAPI_statusOk - success
   TAPI_statusParam - chnnel out of range
*/
IFX_int32_t TAPI_Phone_GetEvent (TAPI_DEV *pTapiDev,
                                 IFX_TAPI_EVENT_t *pEvent)
{
   static IFX_uint8_t last_ch_nr = 0;  /* remember channel between calls */
   IFX_int16_t found_ch_nr;
   IFX_uint8_t i;
   IFX_int32_t moreEvents = 0;

   if (pEvent->ch == IFX_TAPI_EVENT_ALL_CHANNELS)
   {
      /* Scan every channel to see if any of them has an event waiting.
         If an event is found return it to the application. So each call
         returns one event at one time. */

      /* If this device has less channels than the device in the last call
         where an event was found on a channel not present on this device
         reset the starting value to zero. */
      if (last_ch_nr >= pTapiDev->nMaxChannel)
      {
         last_ch_nr = 0;
      }

      /* We start scanning with the next channel after the last channel
         processed during the last call. We are called at least one time.
         Even if there is only one channel the loop will be aborted because
         of the wrap-around correction. The exit from the loop is done only
         with the break at the end of the loop. */
      for (i = last_ch_nr+1, found_ch_nr = -1 ; /* no condition */;i++)
      {
         i = ((i >= pTapiDev->nMaxChannel) ? 0 : i);

         /* As long as we have not found any event we check all the channels
            for events. As soon as we got an event we still check if there
            are any more events in any of the remaining channels. */
         if (found_ch_nr < 0)
         {
            /* check this channel for events */
            moreEvents = IFX_TAPI_EventFifoGet(pTapiDev->pTapiChanelArray + i,
                                               pEvent);
            if (pEvent->id != IFX_TAPI_EVENT_NONE)
            {
               found_ch_nr = i;
            }
         }
         else
         {
            /* check only if events present but do not get them */
            moreEvents += (IFX_TAPI_EventFifoEmpty(
               pTapiDev->pTapiChanelArray + i) == 0) ? 1 : 0;
         }

         /* abort loop if we either wrapped around or found that there are
            more than just one event which implies that we already got one */
         if ((i == last_ch_nr) || (moreEvents != 0))
            break;
      }

      /* remember the last channel number that was serviced */
      last_ch_nr = found_ch_nr < 0 ? i : found_ch_nr;
   }
   else /* Scanning event by channel index. */
   {
      if (pEvent->ch >= pTapiDev->nMaxChannel)
         RETURN_DEVSTATUS (TAPI_statusParam, 0);
      /* Still get one event at one time. */
      moreEvents = IFX_TAPI_EventFifoGet(pTapiDev->pTapiChanelArray + pEvent->ch,
                                         pEvent);
   }

   /* Set the more events flag inside the event message. The flag indicates
      if there are any more events in the channel that was given as a parameter.
      So for specific channels it returns if there are more events in the
      specific channel. For IFX_TAPI_EVENT_ALL_CHANNELS it returns if there
      are any more events in any of the channels. */
   pEvent->more = moreEvents;

   return TAPI_statusOk;
}


/**
   Returns the enable status of the fax modem event.

   \param pChannel - handle to tapi channel structure
   \param pEvent   - handle to event

   \return
      IFX_TRUE if the event is enabled, otherwise IFX_FALSE

   \remarks
*/
static IFX_boolean_t Check_FaxModem_Status (TAPI_CHANNEL *pChannel,
                                            IFX_TAPI_EVENT_t *pEvent)
{
   IFX_TAPI_EVENT_MASK_t *pEventMask;
   IFX_boolean_t eventDisabled = IFX_FALSE,
                 eventLocal,
                 eventNetwork;

   pEventMask = &(pChannel->pEventHandler->eventMask);

   eventLocal = pEvent->data.fax_sig.local ? IFX_TRUE : IFX_FALSE;
   eventNetwork = pEvent->data.fax_sig.network ? IFX_TRUE : IFX_FALSE;

   if (pEvent->id == IFX_TAPI_EVENT_FAXMODEM_AM)
   {
      if ((eventLocal && pEventMask->fax_sig.bits.am_local) ||
          (eventNetwork && pEventMask->fax_sig.bits.am_network))
      {
         eventDisabled = IFX_TRUE;
      }
   }
   if (pEvent->id == IFX_TAPI_EVENT_FAXMODEM_CED)
   {
      if ((eventLocal && pEventMask->fax_sig.bits.ced_local) ||
          (eventNetwork && pEventMask->fax_sig.bits.ced_network))
      {
         eventDisabled = IFX_TRUE;
      }
   }
   if (pEvent->id == IFX_TAPI_EVENT_FAXMODEM_PR)
   {
      if ((eventLocal && pEventMask->fax_sig.bits.pr_local) ||
          (eventNetwork && pEventMask->fax_sig.bits.pr_network))
      {
         eventDisabled = IFX_TRUE;
      }
   }
   if (pEvent->id == IFX_TAPI_EVENT_FAXMODEM_DIS)
   {
      if ((eventLocal && pEventMask->fax_sig.bits.dis_local) ||
          (eventNetwork && pEventMask->fax_sig.bits.dis_network))
      {
         eventDisabled = IFX_TRUE;
      }
   }
   if (pEvent->id == IFX_TAPI_EVENT_FAXMODEM_CNGFAX)
   {
      if ((eventLocal && pEventMask->fax_sig.bits.cngfax_local) ||
          (eventNetwork && pEventMask->fax_sig.bits.cngfax_network))
      {
         eventDisabled = IFX_TRUE;
      }
   }
   if (pEvent->id == IFX_TAPI_EVENT_FAXMODEM_CNGMOD)
   {
      if ((eventLocal && pEventMask->fax_sig.bits.cngmod_local) ||
          (eventNetwork && pEventMask->fax_sig.bits.cngmod_network))
      {
         eventDisabled = IFX_TRUE;
      }
   }
   if (pEvent->id == IFX_TAPI_EVENT_FAXMODEM_V21L)
   {
      if ((eventLocal && pEventMask->fax_sig.bits.v21l_local) ||
          (eventNetwork && pEventMask->fax_sig.bits.v21l_network))
      {
         eventDisabled = IFX_TRUE;
      }
   }
   if (pEvent->id == IFX_TAPI_EVENT_FAXMODEM_V18A)
   {
      if ((eventLocal && pEventMask->fax_sig.bits.v18a_local) ||
          (eventNetwork && pEventMask->fax_sig.bits.v18a_network))
      {
         eventDisabled = IFX_TRUE;
      }
   }
   if (pEvent->id == IFX_TAPI_EVENT_FAXMODEM_V27)
   {
      if ((eventLocal && pEventMask->fax_sig.bits.v27_local) ||
          (eventNetwork && pEventMask->fax_sig.bits.v27_network))
      {
         eventDisabled = IFX_TRUE;
      }
   }
   if (pEvent->id == IFX_TAPI_EVENT_FAXMODEM_BELL)
   {
      if ((eventLocal && pEventMask->fax_sig.bits.bell_local) ||
          (eventNetwork && pEventMask->fax_sig.bits.bell_network))
      {
         eventDisabled = IFX_TRUE;
      }
   }
   if (pEvent->id == IFX_TAPI_EVENT_FAXMODEM_V22)
   {
      if ((eventLocal && pEventMask->fax_sig.bits.v22_local) ||
          (eventNetwork && pEventMask->fax_sig.bits.v22_network))
      {
         eventDisabled = IFX_TRUE;
      }
   }
   if (pEvent->id == IFX_TAPI_EVENT_FAXMODEM_V22ORBELL)
   {
      if ((eventLocal && pEventMask->fax_sig.bits.v22orbell_local) ||
          (eventNetwork && pEventMask->fax_sig.bits.v22orbell_network))
      {
         eventDisabled = IFX_TRUE;
      }
   }
   if (pEvent->id == IFX_TAPI_EVENT_FAXMODEM_V32AC)
   {
      if ((eventLocal && pEventMask->fax_sig.bits.v32ac_local) ||
          (eventNetwork && pEventMask->fax_sig.bits.v32ac_network))
      {
         eventDisabled = IFX_TRUE;
      }
   }
   if (pEvent->id == IFX_TAPI_EVENT_FAXMODEM_CAS_BELL)
   {
      if ((eventLocal && pEventMask->fax_sig.bits.cas_local) ||
          (eventNetwork && pEventMask->fax_sig.bits.cas_network))
      {
         eventDisabled = IFX_TRUE;
      }
   }
   if (pEvent->id == IFX_TAPI_EVENT_FAXMODEM_V21H)
   {
      if ((eventLocal && pEventMask->fax_sig.bits.v21h_local) ||
          (eventNetwork && pEventMask->fax_sig.bits.v21h_network))
      {
         eventDisabled = IFX_TRUE;
      }
   }
   if (pEvent->id == IFX_TAPI_EVENT_FAXMODEM_V8BIS)
   {
      if ((eventLocal && pEventMask->fax_sig.bits.v8bis_local) ||
          (eventNetwork && pEventMask->fax_sig.bits.v8bis_network))
      {
         eventDisabled = IFX_TRUE;
      }
   }
   if (pEvent->id == IFX_TAPI_EVENT_FAXMODEM_HOLDEND)
   {
      /* holding will usually send from local and network as a sum */
      if ((eventLocal && pEventMask->fax_sig.bits.hold_local) ||
          (eventNetwork && pEventMask->fax_sig.bits.hold_network))
      {
         eventDisabled = IFX_TRUE;
      }
   }

   return eventDisabled;
}


#ifdef TAPI_FAX_T38
/**

   Fax Status update Event handling function. Is called from interrupt routine
   or elsewhere a fax related error occurs.

   \param pChannel    - handle to TAPI_CHANNEL structure
   \param status      - Fax status
   \param error       - Fax error

   \return
      None
*/
IFX_void_t TAPI_FaxT38_Event_Update (TAPI_CHANNEL * pChannel,
                                     IFX_uint8_t status, IFX_uint8_t error)
{
   IFX_TAPI_DRV_CTX_t *pDrvCtx = pChannel->pTapiDevice->pDevDrvCtx;

   if (ptr_chk
       (pDrvCtx->COD.T38_Status_Set, "pDrvCtx->COD.T38_Status_Set"))
      (IFX_void_t)pDrvCtx->COD.T38_Status_Set (pChannel->pLLChannel, status);

   if (ptr_chk (pDrvCtx->COD.T38_Error_Set, "pDrvCtx->COD.T38_Error_Set"))
      (IFX_void_t)pDrvCtx->COD.T38_Error_Set (pChannel->pLLChannel, error);
}
#endif /* TAPI_FAX_T38 */


/**
   WakeUp the TAPI channel task.

   \param  pTapiDev     Pointer to TAPI device structure.
*/
static IFX_void_t TAPI_WakeUp (TAPI_DEV *pTapiDev)
{
   /* In polled mode, there are no sleeping tasks (blocked on select), so no
      wakeup needed */
   /* Polling Not supported
    if ((pTapiDev->bNeedWakeup == IFX_TRUE) && (pDev->IrqPollMode &
    VIN_TAPI_WAKEUP))*/

   if (pTapiDev->bNeedWakeup)
   {
      TAPI_OS_DrvSelectQueueWakeUp (&pTapiDev->wqEvent,
                                    TAPI_OS_DRV_SEL_WAKEUP_TYPE_RD);
   }
}


/**
  Check whether event fifo is empty or not (protected).
  \param pChannel - High level TAPI Channel

  \return
    If empty, return 1,else 0.
*/
IFX_uint8_t IFX_TAPI_EventFifoEmpty (TAPI_CHANNEL * pChannel)
{
   IFX_uint8_t ret = 0;

   /* Lock the fifo access protection. */
   TAPI_OS_MutexGet (&pChannel->pEventHandler->fifoAcc);

   if (fifoEmpty (pChannel->pEventHandler->pTapiEventFifoHi) &&
       fifoEmpty (pChannel->pEventHandler->pTapiEventFifoLo))
   {
      ret = 1;
   }
   else
   {
      ret = 0;
   }

   /* Unlock fifo access protection. */
   TAPI_OS_MutexRelease (&pChannel->pEventHandler->fifoAcc);

   return ret;
}


/**
   Get event stored in the event fifo.

   \param pChannel -Channel context for HL TAPI.
   \param pTapiEvent - Event.

   \return
   If there are more events stored in hi/lo fifo, return 1, otherwise return 0.

   \remarks
   High Level (ioctl) event fifo access function.
   Because this function was called by ioctl, it needs semphore protection
   and ceases interrupt routine.
   It will free the buffer after it returns a event.
*/
IFX_int32_t IFX_TAPI_EventFifoGet (TAPI_CHANNEL *pChannel,
                                   IFX_TAPI_EVENT_t *pTapiEvent)
{
   IFX_TAPI_EXT_EVENT_PARAM_t *pTempEvt = IFX_NULL;
   IFX_int32_t moreEvent = 0;
   IFX_int32_t err;
   TAPI_OS_INTSTAT lock;

   /* Lock the fifo access protection. */
   TAPI_OS_MutexGet (&pChannel->pEventHandler->fifoAcc);

   /* Get event stored in high prio fifo */
   pTempEvt = fifoGet (pChannel->pEventHandler->pTapiEventFifoHi, NULL);

   if (pTempEvt)
   {
      /* copy event data into struct provided by the caller */
      memcpy (pTapiEvent, &pTempEvt->tapiEvent, sizeof (IFX_TAPI_EVENT_t));
      /* release the buffer back to the pool */
      TAPI_OS_MutexGet (&semBufferPoolAcc);
      TAPI_OS_LOCKINT (lock);
      err = bufferPoolPut ((void *) pTempEvt);
      TAPI_OS_UNLOCKINT (lock);
      TAPI_OS_MutexRelease (&semBufferPoolAcc);
      if (err == BUFFERPOOL_ERROR)
      {
         /* \todo if bufferpool handling fail? */
      }

      /* Check whether there are any more events stored in hi or lo fifo. */
      if (fifoEmpty (pChannel->pEventHandler->pTapiEventFifoHi) &&
          fifoEmpty (pChannel->pEventHandler->pTapiEventFifoLo))
      {
         /* no events stored in hi or lo fifo */
         moreEvent = 0;
      }
      else
      {
         moreEvent = 1;
      }

      /* Unlock fifo access protection. */
      TAPI_OS_MutexRelease (&pChannel->pEventHandler->fifoAcc);

      return moreEvent;
   }

   /* If hi prio fifo is empty, get event stored in low prio fifo. */
   pTempEvt = fifoGet (pChannel->pEventHandler->pTapiEventFifoLo, NULL);
   if (pTempEvt)
   {
      /* copy event data into struct provided by the caller */
      memcpy (pTapiEvent, &pTempEvt->tapiEvent, sizeof (IFX_TAPI_EVENT_t));
      /* release the buffer back to the pool */
      TAPI_OS_MutexGet (&semBufferPoolAcc);
      TAPI_OS_LOCKINT (lock);
      err = bufferPoolPut ((void *) pTempEvt);
      TAPI_OS_UNLOCKINT (lock);
      TAPI_OS_MutexRelease (&semBufferPoolAcc);

      if (err == BUFFERPOOL_ERROR)
      {
         /* \todo if bufferpool handling fail? */
      }

      /* Check whether there are any more events stored in lo fifo. */
      if (fifoEmpty (pChannel->pEventHandler->pTapiEventFifoLo))
      {
         /* no events stored in lo fifo */
         moreEvent = 0;
      }
      else
      {
         moreEvent = 1;
      }

      /* Unlock fifo access protection. */
      TAPI_OS_MutexRelease (&pChannel->pEventHandler->fifoAcc);

      return moreEvent;
   }

   /* Unlock fifo access protection. */
   TAPI_OS_MutexRelease (&pChannel->pEventHandler->fifoAcc);

   /* If high and low prio fifo is empty, set moreEvent to 0 and event id
      to IFX_TAPI_EVENT_NONE. Do not change the channel parameter. */
   pTapiEvent->id = IFX_TAPI_EVENT_NONE;
   moreEvent = 0;

   return moreEvent;
}


/********************NOTE *****************************************************

 1.IFX_TAPI_Event_Dispatch : Call this function to dispatch an event. The
 event is stored in an allocated buffer together with additional information
 such as the channel. This is then passed on to task context with the
 TAPI_DeferWork() function.

 2.IFX_TAPI_Event_Dispatch_ProcessCtx : This function is working in task
 context. It receives the events dispatched by IFX_TAPI_Event_Dispatch
 processes and filters them and finally put them into the queue towards the
 application.

*******************************************************************************/

/**
   High Level Event Dispatcher function
   (May be called from task or interrupt context.)

   \param pChannel     - handle to tapi channel structure
   \param pTapiEvent   - handle to event

   \return TAPI_statusOk on ok, otherwise TAPI_statusErr
 */
IFX_return_t IFX_TAPI_Event_Dispatch (TAPI_CHANNEL *pChannel,
                                      IFX_TAPI_EVENT_t *pTapiEvent)
{
   IFX_int32_t ret = TAPI_statusOk;
   IFX_TAPI_EXT_EVENT_PARAM_t *pDeferredEvent;
   TAPI_OS_INTSTAT lock;

   /* filter error message and just copy the error stack */
   if (pTapiEvent->id == IFX_TAPI_EVENT_FAULT_GENERAL_CHINFO ||
       pTapiEvent->id == IFX_TAPI_EVENT_FAULT_GENERAL_DEVINFO)
   {
      TAPI_DEV *pTapiDev = pChannel->pTapiDevice;
      if (pTapiDev->error.nCnt < IFX_TAPI_MAX_ERROR_ENTRIES - 1)
      {
         memcpy (&pTapiDev->error.stack[pTapiDev->error.nCnt++],
                 (IFX_TAPI_ErrorLine_t*)pTapiEvent->data.error,
                 sizeof (IFX_TAPI_ErrorLine_t));
         pTapiDev->error.nCh   = pChannel->nChannel;
         pTapiDev->error.nCode =
            (IFX_uint32_t)pTapiEvent->data.error->nLlCode;
      }
      return TAPI_statusOk;
   }

   /* global irq lock - multiple drivers may be loaded and all share this
      event dispatch function. Lock access to the shared buffer pool. */
   if (!TAPI_OS_IN_INTERRUPT())
      TAPI_OS_MutexGet (&semBufferPoolAcc);
   TAPI_OS_LOCKINT (lock);

   pDeferredEvent = (IFX_TAPI_EXT_EVENT_PARAM_t *)
                               bufferPoolGet (pIFX_TAPI_BP_Deferred_Event);
   if (pDeferredEvent == IFX_NULL)
   {
      /* if no deferred_event buffer is free - discard the event as well :-/ */
      TAPI_OS_UNLOCKINT(lock);
      if (!TAPI_OS_IN_INTERRUPT())
         TAPI_OS_MutexRelease (&semBufferPoolAcc);
      RETURN_STATUS (TAPI_statusErr, 0);
   }

   TAPI_OS_UNLOCKINT (lock);
   if (!TAPI_OS_IN_INTERRUPT())
      TAPI_OS_MutexRelease (&semBufferPoolAcc);

   memcpy (&pDeferredEvent->tapiEvent, pTapiEvent, sizeof (IFX_TAPI_EVENT_t));
   pDeferredEvent->pChannel = pChannel;

   /* decide if the event can be handled immediately or needs to be scheduled
      into the process context first (if called from interrupt context) */
   if (!TAPI_OS_IN_INTERRUPT())
   {
      ret = IFX_TAPI_Event_Dispatch_ProcessCtx(pDeferredEvent);
   }
   else
   {
      /*lint -save -e611 \todo: FIXME
         (ANSI X3.159-1989)
         It is invalid to convert a function pointer to an object pointer
         or a pointer to void, or vice-versa.
      */
      ret = (IFX_int32_t)TAPI_DeferWork(
                           (IFX_void_t *) IFX_TAPI_Event_Dispatch_ProcessCtx,
                           (IFX_void_t *) pDeferredEvent);
      /*lint -restore */

      /* error handling */
      if (ret != TAPI_statusOk)
      {
         /* deferring failed - return the deferred_event data structure
            to the bufferpool */
         if (!TAPI_OS_IN_INTERRUPT())
            TAPI_OS_MutexGet (&semBufferPoolAcc);
         TAPI_OS_LOCKINT (lock);
         bufferPoolPut(pDeferredEvent);
         TAPI_OS_UNLOCKINT (lock);
         if (!TAPI_OS_IN_INTERRUPT())
            TAPI_OS_MutexRelease (&semBufferPoolAcc);
         RETURN_STATUS (ret, 0);
      }
   }

   return (IFX_return_t)ret;
}


/**
   High Level Event Dispatcher task.

   \param pParam - parameters with task structure and detected event.

   \return TAPI_statusOk on ok, otherwise TAPI_statusErr

   \remark
   This function is spawned as a task to dispatch all events from process
   context. It is the single resource of event processing. The buffers it
   uses are allocated in IFX_TAPI_Event_Dispatch above which is called
   from interrupt or task context. So locking needs to be done during
   all buffer operations. Multiple drivers may be loaded which all can
   generate events. Instead of locking each driver separately we lock all
   interrupts globally. Additionally semaphores are needed to lock the
   access to the buffers and fifos.
 */
IFX_int32_t IFX_TAPI_Event_Dispatch_ProcessCtx(IFX_TAPI_EXT_EVENT_PARAM_t*
                                                 pParam)
{
   IFX_int32_t ret = TAPI_statusOk;
   IFX_boolean_t eventDisabled = IFX_TRUE;
   IFX_boolean_t eventDiscard = IFX_TRUE;
   TAPI_CHANNEL *pChannel = pParam->pChannel;
   IFX_TAPI_EVENT_t *pEvent = &pParam->tapiEvent,
                    *pLostEvent;
   IFX_TAPI_EXT_EVENT_PARAM_t *pLostDeferredEvent;
   IFX_TAPI_DRV_CTX_t *pDrvCtx = IFX_NULL;
   IFX_TAPI_EVENT_MASK_t *pEventMask;
   TAPI_OS_INTSTAT lock;

#ifdef TAPI_EXT_KEYPAD
   /* in case of inca2 it will be NULL coz no chnl context when an external
      keypad reports an event. Channel is mapped to 0th channel */
   if (pChannel == IFX_NULL)
   {
      /* STBD : Find channel context */
      pChannel = IFX_TAPI_IncaChannelContextGet ();
   }
#endif /* TAPI_EXT_KEYPAD */

   if (pChannel != IFX_NULL)
   {
      pDrvCtx = pChannel->pTapiDevice->pDevDrvCtx;
      pEvent->ch = (IFX_uint16_t) pChannel->nChannel;
   }
   else
   {
      /* this should not happen - discard event and exit now */
      TAPI_OS_MutexGet (&semBufferPoolAcc);
      TAPI_OS_LOCKINT (lock);
      ret = bufferPoolPut (pParam);
      TAPI_OS_UNLOCKINT (lock);
      TAPI_OS_MutexRelease (&semBufferPoolAcc);
      return ret;
   }
   /* at this point pChannel is now valid */

   pEventMask = &(pChannel->pEventHandler->eventMask);
   /**\todo log event with new log type:
    LOG_STR (pEvent->ch, "Evt ID 0x%X DATA 0x%X", pEvent->id, pEvent->data.value);
    */

   /* Errors are handled first and put into the high-priority fifo.
      Normal events are handled afterwards and go to the low-priority fifo. */
   if ((((IFX_uint32_t)pEvent->id) & IFX_TAPI_EVENT_TYPE_FAULT_MASK) ==
         IFX_TAPI_EVENT_TYPE_FAULT_MASK)
   {
      /*lint -save -e650 */
      /*
       * If any fault event occured, invoke the error state machine.
       *  Then, put this event into the high-priority FIFO.
       */
      switch (((IFX_uint32_t)pEvent->id) & IFX_TAPI_EVENT_TYPE_MASK)
      {
      case IFX_TAPI_EVENT_TYPE_FAULT_GENERAL:
         {
            int src = pEvent->data.value & IFX_TAPI_ERRSRC_MASK;
            /* clear error clasification */
            pEvent->data.value &= ~(IFX_TAPI_ERRSRC_MASK);
            switch (src)
            {
               case  IFX_TAPI_ERRSRC_LL_DEV:
                  /**\todo put in device fifo */
                  pEvent->ch = IFX_TAPI_DEVICE_CH_NUMBER;
                  /*lint -fallthrough */
               case  IFX_TAPI_ERRSRC_LL_CH:
                  pEvent->data.value |= IFX_TAPI_ERRSRC_LL;
                  break;
               case  IFX_TAPI_ERRSRC_TAPI_CH:
               case  IFX_TAPI_ERRSRC_TAPI_DEV:
               default:
               /* do nothing */
                  break;
            }
            eventDisabled = IFX_FALSE;
         }
         break;
      case IFX_TAPI_EVENT_TYPE_FAULT_LINE:
         switch (pEvent->id)
         {
         case IFX_TAPI_EVENT_FAULT_LINE_OVERTEMP:
            pChannel->TapiOpControlData.nLineMode = IFX_TAPI_LINE_FEED_DISABLED;
            eventDisabled = pEventMask->fault_line.bits.overtemp ?
                            IFX_TRUE : IFX_FALSE;
            break;
         case IFX_TAPI_EVENT_FAULT_LINE_OVERTEMP_END:
            eventDisabled = pEventMask->fault_line.bits.overtemp_end ?
                            IFX_TRUE : IFX_FALSE;
            break;
         case IFX_TAPI_EVENT_FAULT_LINE_GK_HIGH:
            /* Set the linefeeding to disabled just in case neither the
               chip FW nor the LL-driver did it already. */
            TAPI_Phone_Set_Linefeed (pChannel, IFX_TAPI_LINE_FEED_DISABLED);
            eventDisabled = pEventMask->fault_line.bits.gk_high ?
                            IFX_TRUE : IFX_FALSE;
            break;
         case IFX_TAPI_EVENT_FAULT_LINE_GK_HIGH_END:
            eventDisabled = pEventMask->fault_line.bits.gk_high_end ?
                            IFX_TRUE : IFX_FALSE;
            break;
         case IFX_TAPI_EVENT_FAULT_LINE_GK_LOW:
            eventDisabled = pEventMask->fault_line.bits.gk_low ?
                            IFX_TRUE : IFX_FALSE;
            break;
         case IFX_TAPI_EVENT_FAULT_LINE_GK_LOW_END:
            eventDisabled = pEventMask->fault_line.bits.gk_low_end ?
                            IFX_TRUE : IFX_FALSE;
            break;
         case IFX_TAPI_EVENT_FAULT_LINE_GK_POS:
            eventDisabled = pEventMask->fault_line.bits.gk_pos ?
                            IFX_TRUE : IFX_FALSE;
            break;
         case IFX_TAPI_EVENT_FAULT_LINE_GK_NEG:
            eventDisabled = pEventMask->fault_line.bits.gk_neg ?
                            IFX_TRUE : IFX_FALSE;
            break;
         default:
         /* do nothing */
            break;
         }
         break;
      case IFX_TAPI_EVENT_TYPE_FAULT_HW:
         switch (pEvent->id)
         {
         case IFX_TAPI_EVENT_FAULT_HW_CLOCK_FAIL:
            eventDisabled = pEventMask->fault_hw.bits.clock_fail ?
                            IFX_TRUE : IFX_FALSE;
            break;
         case IFX_TAPI_EVENT_FAULT_HW_CLOCK_FAIL_END:
            eventDisabled = pEventMask->fault_hw.bits.clock_fail_end ?
                            IFX_TRUE : IFX_FALSE;
            break;
         case IFX_TAPI_EVENT_FAULT_HW_FAULT:
            eventDisabled = pEventMask->fault_hw.bits.hw_fault ?
                            IFX_TRUE : IFX_FALSE;
            break;
         case IFX_TAPI_EVENT_FAULT_HW_SYNC:
            eventDisabled = pEventMask->fault_hw.bits.sync ?
                           IFX_TRUE : IFX_FALSE;
            break;
         case IFX_TAPI_EVENT_FAULT_HW_RESET:
            eventDisabled = pEventMask->fault_hw.bits.hw_reset ?
                           IFX_TRUE : IFX_FALSE;
            break;
         case IFX_TAPI_EVENT_FAULT_HW_SSI_ERR:
            eventDisabled = pEventMask->fault_hw.bits.ssi_err ?
                            IFX_TRUE : IFX_FALSE;
            break;
         default:
         /* do nothing */
            break;
         }
         break;
      case IFX_TAPI_EVENT_TYPE_FAULT_FW:
         switch (pEvent->id)
         {
            case IFX_TAPI_EVENT_FAULT_FW_WATCHDOG:
               /* FW watchdog event cannot be masked */
               eventDisabled = IFX_FALSE;
               break;
            default:
            break;
         }
         break;
      case IFX_TAPI_EVENT_TYPE_FAULT_SW:
         break;
      default:
         break;
      }
      /*lint -restore */

      /* This is cleared when the event was successfully put into the fifo. */
      eventDiscard = IFX_TRUE;

      /* If the event is not masked out put it into the fifo of this channel. */
      if (eventDisabled == IFX_FALSE)
      {
         /* Lock the fifo access protection */
         TAPI_OS_MutexGet (&pChannel->pEventHandler->fifoAcc);
         /* The event is not masked. So put it into the high priority fifo
            and wake up the application waiting on the filedescriptor */
         ret = fifoPut (pChannel->pEventHandler->pTapiEventFifoHi,
                        (void *) pParam, sizeof (IFX_TAPI_EXT_EVENT_PARAM_t));
         if (ret == IFX_SUCCESS)
         {
            TAPI_WakeUp (pChannel->pTapiDevice);
            eventDiscard = IFX_FALSE;
         }
         else
         {
            TRACE (TAPI_DRV, DBG_LEVEL_HIGH,
                   ("TAPI event lost on channel %d.\n", pChannel->nChannel));
            /* Retrieve the event that was written last into the fifo and
               overwrite it with an fifo overflow event. */
            pLostDeferredEvent =
               fifoPeekTail(pChannel->pEventHandler->pTapiEventFifoHi, IFX_NULL);
            if (pLostDeferredEvent != IFX_NULL)
            {
               pLostEvent = &pLostDeferredEvent->tapiEvent;
               if (pLostEvent->id !=
                   IFX_TAPI_EVENT_FAULT_GENERAL_EVT_FIFO_OVERFLOW)
               {
                  /* Overwrite the Event-ID. */
                  pLostEvent->id =
                     IFX_TAPI_EVENT_FAULT_GENERAL_EVT_FIFO_OVERFLOW;
                  /* We just lost the event that we have overwritten. */
                  pLostEvent->data.event_fifo.lost = 1;
               }
               pLostEvent->data.event_fifo.lost++;
            }
         }
         /* Unlock the fifo access protection */
         TAPI_OS_MutexRelease (&pChannel->pEventHandler->fifoAcc);
      }

      if (eventDiscard == IFX_TRUE)
      {
         /* Release the buffer and forget it. */
         TAPI_OS_MutexGet (&semBufferPoolAcc);
         TAPI_OS_LOCKINT (lock);
         ret = bufferPoolPut (pParam);
         TAPI_OS_UNLOCKINT (lock);
         TAPI_OS_MutexRelease (&semBufferPoolAcc);
      }
   }
   else
   {
      /*
       * Normal events, put these into low-priority FIFO
       */
      switch (pEvent->id)
      {
      /* EVENT TYPE: -- IFX_TAPI_EVENT_TYPE_NONE -- */

      /* EVENT TYPE: -- IFX_TAPI_EVENT_TYPE_IO_GENERAL -- */

      /* EVENT TYPE: -- IFX_TAPI_EVENT_TYPE_INTERRUPT -- */

      /* EVENT TYPE: -- IFX_TAPI_EVENT_TYPE_FXS -- */

      case IFX_TAPI_EVENT_FXS_ONHOOK_INT:
      case IFX_TAPI_EVENT_FXS_OFFHOOK_INT:
         {
            IFX_boolean_t bSendHookEvent = IFX_TRUE;
            IFX_boolean_t bDoRingStop = IFX_TRUE;
            IFX_boolean_t bOffhook =
                  (pEvent->id == IFX_TAPI_EVENT_FXS_OFFHOOK_INT) ?
                  IFX_TRUE : IFX_FALSE;

#ifdef TAPI_CID
            IFX_TAPI_CID_OnHookEvent(pChannel, bOffhook,
                                     &bSendHookEvent, &bDoRingStop);
#endif /* TAPI_CID */

            if ((pEvent->id == IFX_TAPI_EVENT_FXS_OFFHOOK_INT) &&
                (bDoRingStop == IFX_TRUE))
            {
               /* stop ringing (including an optional CID transmission) */
               IFX_TAPI_Ring_Stop(pChannel);
            }

            if (bSendHookEvent == IFX_TRUE)
            {
               /* trigger event in the hook state machine */
               IFX_TAPI_Dial_HookEvent (pChannel, bOffhook);
            }

            /* We drop the internal hook events. The hook state machine will
               generate new events from this. Depending on the timing we get
               onhook, offhook, flashhook or pulsedialing digits */
            eventDisabled = IFX_TRUE;
         }
         break;

      case IFX_TAPI_EVENT_FXS_ONHOOK:
         /* IFX_FALSE => nature of hook event is ONhook */
         pChannel->TapiOpControlData.bHookState = IFX_FALSE;
         eventDisabled = pEventMask->fxs.bits.onhook ? IFX_TRUE : IFX_FALSE;
         break;

      case IFX_TAPI_EVENT_FXS_OFFHOOK:
         /* stop ringing (including an optional CID transmission) */
         /* This does nothing if already done in the raw event above. It
            handles the case of the wetting pulse where we do nothing above.*/
         IFX_TAPI_Ring_Stop(pChannel);
         /* IFX_TRUE => nature of hook event is OFFhook */
         pChannel->TapiOpControlData.bHookState = IFX_TRUE;
         eventDisabled = pEventMask->fxs.bits.offhook ? IFX_TRUE : IFX_FALSE;
         break;

      case IFX_TAPI_EVENT_FXS_FLASH:
         eventDisabled = pEventMask->fxs.bits.flash ? IFX_TRUE : IFX_FALSE;
         break;

      case IFX_TAPI_EVENT_FXS_RINGING_END:
         eventDisabled = pEventMask->fxs.bits.ringing_end ?
                         IFX_TRUE : IFX_FALSE;
         break;

      /* EVENT TYPE: -- IFX_TAPI_EVENT_TYPE_FXO -- */
      case IFX_TAPI_EVENT_FXO_RING_START:
         eventDisabled = IFX_FALSE;
         break;

      case IFX_TAPI_EVENT_FXO_RING_STOP:
         eventDisabled = IFX_FALSE;
         break;

      case IFX_TAPI_EVENT_FXO_BAT_FEEDED:
         eventDisabled = IFX_FALSE;
         break;

      case IFX_TAPI_EVENT_FXO_BAT_DROPPED:
         eventDisabled = IFX_FALSE;
         break;

      case IFX_TAPI_EVENT_FXO_OSI:
         eventDisabled = IFX_FALSE;
         break;

      case IFX_TAPI_EVENT_FXO_POLARITY:
         eventDisabled = IFX_FALSE;
         break;

      case IFX_TAPI_EVENT_FXO_APOH:
         eventDisabled = IFX_FALSE;
         break;

      case IFX_TAPI_EVENT_FXO_NOPOH:
         eventDisabled = IFX_FALSE;
         break;


      /* EVENT TYPE: -- IFX_TAPI_EVENT_TYPE_LT -- */

#ifdef TAPI_GR909
      case IFX_TAPI_EVENT_LT_GR909_RDY:
         pChannel->TapiOpControlData.nLineMode = IFX_TAPI_LINE_FEED_DISABLED;
         eventDisabled = pEventMask->lt.bits.gr909_rdy ? IFX_TRUE : IFX_FALSE;
         break;
#endif

      /* EVENT TYPE: -- IFX_TAPI_EVENT_TYPE_PULSE -- */

      case IFX_TAPI_EVENT_PULSE_DIGIT:
         eventDisabled = pEventMask->pulse.bits.digit ? IFX_TRUE : IFX_FALSE;
         break;

      case IFX_TAPI_EVENT_PULSE_START:
         eventDisabled = pEventMask->pulse.bits.pulseStart ?
                         IFX_TRUE : IFX_FALSE;
         break;

      /* EVENT TYPE: -- IFX_TAPI_EVENT_TYPE_DTMF -- */

      case IFX_TAPI_EVENT_DTMF_DIGIT:
         {
#ifdef TAPI_CID
            if (IFX_TAPI_CID_OnDtmfEvent(pChannel, pEvent->data.dtmf.ascii))
            {
               eventDisabled = IFX_TRUE;
            }
            else
#endif /* TAPI_CID */
            {
#ifdef TAPI_DTMF
               if (pEvent->data.dtmf.local)
               {
                  eventDisabled = pEventMask->dtmf.bits.digit_local ?
                                  IFX_TRUE : IFX_FALSE;
               }
               else if (pEvent->data.dtmf.network)
               {
                  eventDisabled = pEventMask->dtmf.bits.digit_network ?
                                  IFX_TRUE : IFX_FALSE;
               }
#endif /* TAPI_DTMF */
            }
         }
         break;

#ifdef TAPI_DTMF
      case IFX_TAPI_EVENT_DTMF_END:
         if (pEvent->data.dtmf.local)
         {
            eventDisabled = pEventMask->dtmf.bits.end_local ?
                            IFX_TRUE : IFX_FALSE;
         }
         else if (pEvent->data.dtmf.network)
         {
            eventDisabled = pEventMask->dtmf.bits.end_network ?
                            IFX_TRUE : IFX_FALSE;
         }
         break;
#endif /* TAPI_DTMF */


      /* EVENT TYPE: -- IFX_TAPI_EVENT_TYPE_CID -- */

#ifdef TAPI_CID
      case IFX_TAPI_EVENT_CID_RX_END:
         IFX_TAPI_CID_OnCidRxEndEvent(pChannel);
         eventDisabled = pEventMask->cid.bits.rx_end ? IFX_TRUE : IFX_FALSE;
         break;

      case IFX_TAPI_EVENT_CID_RX_ERROR_READ:
         eventDisabled = pEventMask->cid.bits.rx_err_read ?
                         IFX_TRUE : IFX_FALSE;
         break;

      case IFX_TAPI_EVENT_CID_TX_END:
         /* This event is sent by the lower-layer after a data transmission
            has ended. This is here translated into the ...TX_INFO_END or
            ...TX_SEQ_END depending on the mode of the CID transmission. */
         if (TAPI_Cid_UseSequence (pChannel) == IFX_TRUE)
         {
            pEvent->id = IFX_TAPI_EVENT_CID_TX_SEQ_END;
            eventDisabled = pEventMask->cid.bits.tx_seq_end ?
                            IFX_TRUE : IFX_FALSE;
         }
         else
         {
            pEvent->id = IFX_TAPI_EVENT_CID_TX_INFO_END;
            eventDisabled = pEventMask->cid.bits.tx_info_end ?
                            IFX_TRUE : IFX_FALSE;
         }
         break;

      case IFX_TAPI_EVENT_CID_TX_RINGCAD_ERR:
         eventDisabled = pEventMask->cid.bits.tx_ringcad_err ?
                         IFX_TRUE : IFX_FALSE;
         break;

      case IFX_TAPI_EVENT_CID_TX_NOACK_ERR:
         eventDisabled = pEventMask->cid.bits.tx_noack_err ?
                         IFX_TRUE : IFX_FALSE;
         break;

      case IFX_TAPI_EVENT_CID_TX_NOACK2_ERR:
         eventDisabled = pEventMask->cid.bits.tx_noack_err ?
                         IFX_TRUE : IFX_FALSE;
         break;
#endif /* TAPI_CID */

      /* EVENT TYPE: -- IFX_TAPI_EVENT_TYPE_TONE_GEN -- */

      case IFX_TAPI_EVENT_TONE_GEN_END:
         {
            eventDisabled = IFX_TRUE;
            /* check event masks */
            if ((pEvent->data.tone_gen.local) &&
               (!pEventMask->tone_gen.bits.end_local))
               eventDisabled = IFX_FALSE;
            if ((pEvent->data.tone_gen.network) &&
               (!pEventMask->tone_gen.bits.end_network))
               eventDisabled = IFX_FALSE;
         }
         break;

      /* EVENT TYPE: -- IFX_TAPI_EVENT_TYPE_TONE_GEN_RAW -- */
      /*                not available to the application    */
      case IFX_TAPI_EVENT_TONE_GEN_END_RAW:
         {
            TAPI_Tone_Step_Completed (pChannel, pEvent->data.value);
            eventDisabled = IFX_TRUE;
         }
         break;

      /* EVENT TYPE: -- IFX_TAPI_EVENT_TYPE_TONE_DET -- */

      case IFX_TAPI_EVENT_TONE_DET_CPT:
         eventDisabled = pEventMask->tone_det.bits.cpt ? IFX_TRUE : IFX_FALSE;
         break;

      case IFX_TAPI_EVENT_TONE_DET_RECEIVE:
         eventDisabled = pEventMask->tone_det.bits.receive ?
                         IFX_TRUE : IFX_FALSE;
         break;

      case IFX_TAPI_EVENT_TONE_DET_TRANSMIT:
         eventDisabled = pEventMask->tone_det.bits.transmit ?
                         IFX_TRUE : IFX_FALSE;
         break;

      /* EVENT TYPE: -- IFX_TAPI_EVENT_TYPE_FAXMODEM_SIGNAL -- */

      case  IFX_TAPI_EVENT_FAXMODEM_AM:
      case  IFX_TAPI_EVENT_FAXMODEM_CED:
      case  IFX_TAPI_EVENT_FAXMODEM_CEDEND:
      case  IFX_TAPI_EVENT_FAXMODEM_DIS:
      case  IFX_TAPI_EVENT_FAXMODEM_PR:
      case  IFX_TAPI_EVENT_FAXMODEM_CNGFAX:
      case  IFX_TAPI_EVENT_FAXMODEM_CNGMOD:
      case  IFX_TAPI_EVENT_FAXMODEM_V21L:
      case  IFX_TAPI_EVENT_FAXMODEM_V18A:
      case  IFX_TAPI_EVENT_FAXMODEM_V27:
      case  IFX_TAPI_EVENT_FAXMODEM_BELL:
      case  IFX_TAPI_EVENT_FAXMODEM_V22:
      case  IFX_TAPI_EVENT_FAXMODEM_V22ORBELL:
      case  IFX_TAPI_EVENT_FAXMODEM_V32AC:
      case  IFX_TAPI_EVENT_FAXMODEM_V8BIS:
      case  IFX_TAPI_EVENT_FAXMODEM_HOLDEND:
      case IFX_TAPI_EVENT_FAXMODEM_CAS_BELL:
      case IFX_TAPI_EVENT_FAXMODEM_V21H:
      case IFX_TAPI_EVENT_FAXMODEM_VMD:
         eventDisabled = Check_FaxModem_Status (pChannel, pEvent);
         break;

      /* EVENT TYPE: -- IFX_TAPI_EVENT_TYPE_CODER -- */

      case IFX_TAPI_EVENT_COD_DEC_CHG:
         eventDisabled = pEventMask->coder.bits.dec_chg ? IFX_TRUE : IFX_FALSE;
         if ( !eventDisabled )
         {
            IFX_TAPI_DEC_DETAILS_t dec_details;
            memset (&dec_details, 0, sizeof(IFX_TAPI_DEC_DETAILS_t));
            if (ptr_chk(pDrvCtx->COD.DEC_Chg_Evt_Detail_Req,
                       "pDrvCtx->COD.DEC_Chg_Evt_Detail_Req"))
            {
               ret = pDrvCtx->COD.DEC_Chg_Evt_Detail_Req(pChannel->pLLChannel,
                                                         &dec_details);
               if (ret == TAPI_statusOk)
               {
                  pEvent->data.dec_chg.dec_type        = dec_details.dec_type;
                  pEvent->data.dec_chg.dec_framelength = dec_details.dec_framelength;
               }
            }
         }
         break;

      case IFX_TAPI_EVENT_COD_ROOM_NOISE:
         eventDisabled = pEventMask->coder.bits.room_noise ?
                         IFX_TRUE : IFX_FALSE;
         break;

      case IFX_TAPI_EVENT_COD_ROOM_SILENCE:
         eventDisabled = pEventMask->coder.bits.room_silence ?
                         IFX_TRUE : IFX_FALSE;
         break;

      case IFX_TAPI_EVENT_COD_ANNOUNCE_END:
#ifdef TAPI_ANNOUNCEMENTS
         IFX_TAPI_AnnEndEventServe(pChannel);
         eventDisabled = pEventMask->coder.bits.announce ?
                         IFX_TRUE : IFX_FALSE;
#endif /* TAPI_ANNOUNCEMENTS */
         break;

      /* EVENT TYPE: -- IFX_TAPI_EVENT_TYPE_RTP -- */

      /* EVENT TYPE: -- IFX_TAPI_EVENT_TYPE_AAL -- */

      /* EVENT TYPE: -- IFX_TAPI_EVENT_TYPE_RFC2833 -- */

      case IFX_TAPI_EVENT_RFC2833_EVENT:
         eventDisabled = pEventMask->rfc2833.bits.event ? IFX_TRUE : IFX_FALSE;
         if ( !eventDisabled )
         {
            IFX_uint16_t nEvent;
            if (ptr_chk(pDrvCtx->SIG.EPOU_Trig_Evt_Detail_Req,
                       "pDrvCtx->SIG.EPOU_Trig_Evt_Detail_Req"))
            {
               ret = pDrvCtx->SIG.EPOU_Trig_Evt_Detail_Req(pChannel->pLLChannel,
                                                           &nEvent);
               if (ret == TAPI_statusOk)
               {
                  pEvent->data.rfc2833.event = nEvent;
               }
            }
         }
         break;

      /* EVENT TYPE: -- IFX_TAPI_EVENT_TYPE_KPI -- */

      case IFX_TAPI_EVENT_KPI_INGRESS_FIFO_FULL:
         eventDisabled = pEventMask->kpi.bits.ingress_fifo_full ?
                         IFX_TRUE : IFX_FALSE;
         break;

      /* EVENT TYPE: -- IFX_TAPI_EVENT_TYPE_T38 -- */

#ifdef TAPI_FAX_T38
      case IFX_TAPI_EVENT_T38_ERROR_OVLD:
         eventDisabled = pEventMask->t38.bits.error_ovld ? IFX_TRUE : IFX_FALSE;
         if ( !eventDisabled)
            TAPI_FaxT38_Event_Update (pChannel, IFX_TAPI_FAX_T38_TX_OFF,
                                      IFX_TAPI_FAX_T38_ERROR_MIPS_OVLD);
         break;

      case IFX_TAPI_EVENT_T38_ERROR_DATA:
         eventDisabled = pEventMask->t38.bits.error_data ? IFX_TRUE : IFX_FALSE;
         if ( !eventDisabled )
         {
            if (pEvent->data.value)
               TAPI_FaxT38_Event_Update (pChannel, IFX_TAPI_FAX_T38_TX_OFF,
                                         IFX_TAPI_FAX_T38_ERROR_DATAPUMP);
            else
               TAPI_FaxT38_Event_Update (pChannel, 0,
                                         IFX_TAPI_FAX_T38_ERROR_DATAPUMP);
         }
         break;

      case IFX_TAPI_EVENT_T38_ERROR_WRITE:
         eventDisabled = pEventMask->t38.bits.error_write ?
                         IFX_TRUE : IFX_FALSE;
         if ( !eventDisabled )
            TAPI_FaxT38_Event_Update (pChannel, 0, IFX_TAPI_FAX_T38_ERROR_WRITE);
         break;

      case IFX_TAPI_EVENT_T38_ERROR_READ:
         eventDisabled = pEventMask->t38.bits.error_read ? IFX_TRUE : IFX_FALSE;
         if ( !eventDisabled )
            TAPI_FaxT38_Event_Update (pChannel, 0, IFX_TAPI_FAX_T38_ERROR_READ);
         break;

      case IFX_TAPI_EVENT_T38_ERROR_SETUP:
         eventDisabled = pEventMask->t38.bits.error_setup ? IFX_TRUE : IFX_FALSE;
         if ( !eventDisabled )
            TAPI_FaxT38_Event_Update (pChannel, 0, IFX_TAPI_FAX_T38_ERROR_SETUP);
         break;

      case IFX_TAPI_EVENT_T38_NONE:
         TAPI_FaxT38_Event_Update (pChannel, IFX_TAPI_FAX_T38_TX_OFF,
                                   IFX_TAPI_FAX_T38_ERROR_NONE);
         eventDisabled = IFX_TRUE;
         break;

      case IFX_TAPI_EVENT_T38_FDP_REQ:
         /* FAX T38 FDP REQ; used for polling mode access only -
            currently not maskable */
         eventDisabled = IFX_FALSE;
         break;
#endif /* TAPI_FAX_T38 */

#ifdef TAPI_FAX_T38_FW
      case IFX_TAPI_EVENT_T38_STATE_CHANGE:
         eventDisabled = IFX_FALSE;
         break;
#endif /* TAPI_FAX_T38_FW */

      /* EVENT TYPE: -- IFX_TAPI_EVENT_TYPE_JB -- */

      /* EVENT TYPE: -- IFX_TAPI_EVENT_TYPE_DOWNLOAD -- */

      /* EVENT TYPE: -- IFX_TAPI_EVENT_TYPE_INFORMATION -- */
      case IFX_TAPI_EVENT_INFO_MBX_CONGESTION:
         eventDisabled = pEventMask->information.bits.mbx_congestion ?
                         IFX_TRUE : IFX_FALSE;
         break;

      /* EVENT TYPE: -- IFX_TAPI_EVENT_TYPE_DEBUG -- */

      case IFX_TAPI_EVENT_DEBUG_CERR:
         if (ptr_chk (pDrvCtx->Dbg_CErr_Handler,
                     "pDrvCtx->Dbg_CErr_Handler"))
         {
            IFX_TAPI_DBG_CERR_t data;
            memset(&data, 0, sizeof(IFX_TAPI_DBG_CERR_t));
            if (TAPI_statusOk ==
                pDrvCtx->Dbg_CErr_Handler(pChannel->pTapiDevice->pLLDev, &data))
            {
               pEvent->data.cerr.fw_id = 0x0001;
               pEvent->data.cerr.reason = data.cause;
               pEvent->data.cerr.command = data.cmd;
            }
         }
         /* print data if any data is available */
         if (pEvent->data.cerr.fw_id != 0x0000)
         {
            TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
                  ("\n!!! cErrHdlr cause 0x%04X, cHdr 0x%08X\n",
                   pEvent->data.cerr.reason, pEvent->data.cerr.command));
         }
         /* report this event to the application. */
         eventDisabled = IFX_FALSE;
         break;

      /* EVENT TYPE: -- IFX_TAPI_EVENT_TYPE_LL_DRIVER -- */

      case IFX_TAPI_EVENT_LL_DRIVER_WD_FAIL:
         eventDisabled =
         pEventMask->ll_driver.bits.alive ? IFX_TRUE : IFX_FALSE;
         break;

      /* EVENT TYPE: -- IFX_TAPI_EVENT_TYPE_EXT -- */

#ifdef TAPI_EXT_KEYPAD
      case IFX_TAPI_EVENT_EXT_KEY_UP:
      case IFX_TAPI_EVENT_EXT_KEY_DOWN:
         /* In case of inca, key event is given to the application by HAPI
            so TAPI does not need to report this */
         ret = TAPI_EXT_EVENT_Key_Handler (pChannel, pEvent);
         eventDisabled = IFX_TRUE;
         break;
#endif /* TAPI_EXT_KEYPAD */

      /* EVENT TYPE: -- IFX_TAPI_EVENT_TYPE_LIN -- */

      case IFX_TAPI_EVENT_LIN_UNDERFLOW:
         eventDisabled = pEventMask->lin.bits.buf_underrun ? IFX_TRUE : IFX_FALSE;
         break;

      /* EVENT TYPE: -- IFX_TAPI_EVENT_TYPE_CALIBRATION -- */

      case IFX_TAPI_EVENT_CALIBRATION_END:
         eventDisabled =
            pEventMask->calibration.bits.calibration_end ? IFX_TRUE : IFX_FALSE;
         break;

      case IFX_TAPI_EVENT_CALIBRATION_END_INT:
         if (ptr_chk(pDrvCtx->ALM.Calibration_Finish,
                    "pDrvCtx->ALM.Calibration_Finish"))
         {
            (IFX_void_t)pDrvCtx->ALM.Calibration_Finish(pChannel->pLLChannel);
         }
         eventDisabled = IFX_TRUE;
         break;

#ifdef TAPI_CONT_MEASUREMENT
      case IFX_TAPI_EVENT_CONTMEASUREMENT:
         eventDisabled = IFX_FALSE;
         break;
#endif /* TAPI_CONT_MEASUREMENT */

#ifdef TAPI_PHONE_DETECTION
      case IFX_TAPI_EVENT_LINE_MEASURE_CAPACITANCE_RDY:
         eventDisabled = pEventMask->lcap.bits.lcap_disable ? IFX_TRUE : IFX_FALSE;
         break;

      /* EVENT TYPE: -- IFX_TAPI_EVENT_LINE_MEASURE_CAPACITANCE_RDY_INT -- */
      case IFX_TAPI_EVENT_LINE_MEASURE_CAPACITANCE_RDY_INT:
         if (ptr_chk(pDrvCtx->ALM.CapMeasResult,
                    "pDrvCtx->ALM.CapMeasResult"))
         {
            (IFX_void_t)pDrvCtx->ALM.CapMeasResult(pChannel->pLLChannel);
         }
         eventDisabled = IFX_TRUE;
         break;

      /* EVENT TYPE: -- IFX_TAPI_EVENT_LINE_MEASURE_CAPACITANCE_START_INT -- */
      case IFX_TAPI_EVENT_LINE_MEASURE_CAPACITANCE_START_INT:
         if (ptr_chk(pDrvCtx->ALM.CapMeasStartInt,
                    "pDrvCtx->ALM.CapMeasStartInt"))
         {
            (IFX_void_t)pDrvCtx->ALM.CapMeasStartInt(pChannel->pLLChannel);
         }
         eventDisabled = IFX_TRUE;
         break;
#endif /* TAPI_PHONE_DETECTION */

      default:
         /* discard all unhandled events */
         eventDisabled = IFX_TRUE;
      }

      /* This is cleared when the event was successfully put into the fifo. */
      eventDiscard = IFX_TRUE;

      /* If the event is not masked out put it into the fifo of this channel. */
      if (eventDisabled == IFX_FALSE)
      {
         /* Lock the fifo access protection */
         TAPI_OS_MutexGet (&pChannel->pEventHandler->fifoAcc);
         /* The event is not masked. So put it into the low priority fifo
            and wake up the application waiting on the filedescriptor */
         ret = fifoPut (pChannel->pEventHandler->pTapiEventFifoLo,
                        (void *) pParam, sizeof (IFX_TAPI_EXT_EVENT_PARAM_t));
         if (ret == IFX_SUCCESS)
         {
            TAPI_WakeUp (pChannel->pTapiDevice);
            eventDiscard = IFX_FALSE;
         }
         else
         {
            TRACE (TAPI_DRV, DBG_LEVEL_HIGH,
                   ("TAPI event lost on channel %d.\n", pChannel->nChannel));
            /* Retrieve the event that was written last into the fifo and
               overwrite it with an fifo overflow event. */
            pLostDeferredEvent =
               fifoPeekTail(pChannel->pEventHandler->pTapiEventFifoLo, IFX_NULL);
            if (pLostDeferredEvent != IFX_NULL)
            {
               pLostEvent = &pLostDeferredEvent->tapiEvent;
               if (pLostEvent->id !=
                   IFX_TAPI_EVENT_FAULT_GENERAL_EVT_FIFO_OVERFLOW)
               {
                  /* Overwrite the Event-ID. */
                  pLostEvent->id =
                     IFX_TAPI_EVENT_FAULT_GENERAL_EVT_FIFO_OVERFLOW;
                  /* We just lost the event that we have overwritten. */
                  pLostEvent->data.event_fifo.lost = 1;
               }
               pLostEvent->data.event_fifo.lost++;
            }
         }
         /* Unlock the fifo access protection */
         TAPI_OS_MutexRelease (&pChannel->pEventHandler->fifoAcc);
      }

      if (eventDiscard == IFX_TRUE)
      {
         /* Release the buffer and forget it. */
         TAPI_OS_MutexGet (&semBufferPoolAcc);
         TAPI_OS_LOCKINT (lock);
         ret = bufferPoolPut (pParam);
         TAPI_OS_UNLOCKINT (lock);
         TAPI_OS_MutexRelease (&semBufferPoolAcc);
      }
   }

   return ret;
}


/**
   Resource allocation and initialisation upon loading the driver.

   This function allocates memory for the buffer pool of events that the
   event dispatcher transports. The buffers contain the event and additional
   fields needed to route and defer the event.
   The pool is shared between all devices of this driver.

   \return
   Returns IFX_ERROR in case of an error, otherwise returns IFX_SUCCESS.
*/
IFX_int32_t IFX_TAPI_Event_On_Driver_Start(IFX_void_t)
{
   /* allocate memory pool for the deferred event structures */
   if (pIFX_TAPI_BP_Deferred_Event == IFX_NULL)
   {
      pIFX_TAPI_BP_Deferred_Event =
         bufferPoolInit (sizeof (IFX_TAPI_EXT_EVENT_PARAM_t),
                         IFX_TAPI_EVENT_POOL_INITIAL_SIZE,
                         IFX_TAPI_EVENT_POOL_GROW_SIZE);
      bufferPoolGrowthLimitSet(pIFX_TAPI_BP_Deferred_Event,
                         IFX_TAPI_EVENT_POOL_GROW_LIMIT);
      bufferPoolIDSet (pIFX_TAPI_BP_Deferred_Event, 10);
   }

   /* initialize buffer pool access protection semaphore */
   if (pIFX_TAPI_BP_Deferred_Event != IFX_NULL)
   {
      TAPI_OS_MutexInit (&semBufferPoolAcc);
   }

   return (pIFX_TAPI_BP_Deferred_Event != IFX_NULL) ? IFX_SUCCESS : IFX_ERROR;
}


/**
   Free resources upon unloading the driver.

   This function frees the memory for the buffer pool of events that the
   event dispatcher transports.
*/
IFX_void_t IFX_TAPI_Event_On_Driver_Stop(IFX_void_t)
{
   TAPI_OS_INTSTAT lock;

   TAPI_OS_LOCKINT (lock);

   /* free memory pool for the deferred event structures */
   if (pIFX_TAPI_BP_Deferred_Event != IFX_NULL)
   {
      if (bufferPoolFree (pIFX_TAPI_BP_Deferred_Event) != IFX_SUCCESS)
      {
         TRACE (TAPI_DRV, DBG_LEVEL_HIGH,
                ("WARN: Free deferred-event buffer error.\n"));
      }
      pIFX_TAPI_BP_Deferred_Event = IFX_NULL;
   }

   /* delete the buffer pool access protection semaphore */
   TAPI_OS_MutexDelete (&semBufferPoolAcc);

   TAPI_OS_UNLOCKINT(lock);
}


/**
   High Level Event Dispatcher function init.

   \param  pChannel     Pointer to tapi channel structure.

   \return
   - TAPI_statusNoMem No memory for data storage
   - TAPI_statusErr   Initialisiation of FIFOs failed
   - TAPI_statusOk    Initialisation successful
*/
IFX_int32_t IFX_TAPI_EventDispatcher_Init (TAPI_CHANNEL * pChannel)
{
   IFX_TAPI_EVENT_HANDLER_DATA_t *pEventHandlerData;
   IFX_int32_t ret = TAPI_statusOk;

   TAPI_ASSERT(pChannel != IFX_NULL);

   TAPI_OS_MutexGet (&pChannel->semTapiChDataLock);

   /* allocate data storage on the channel if not already existing */
   if (pChannel->pEventHandler == IFX_NULL)
   {
      /* allocate data storage */
      if ((pChannel->pEventHandler =
           TAPI_OS_Malloc (sizeof(*pChannel->pEventHandler))) == IFX_NULL)
      {
         TAPI_OS_MutexRelease (&pChannel->semTapiChDataLock);
         /* errmsg: Memory not available */
         RETURN_STATUS (TAPI_statusNoMem, 0);
      }
      /* zero the just allocated data storage */
      memset(pChannel->pEventHandler, 0x00, sizeof(*pChannel->pEventHandler));
   }
   /* from here on pChannel->pEventHandler is valid */

   /* Get pointer to data in the channel. */
   pEventHandlerData = pChannel->pEventHandler;

   /* Initialize high and low priority FIFO struct */
   pEventHandlerData->pTapiEventFifoHi = fifoInit(IFX_TAPI_EVENT_FIFO_SIZE);
   pEventHandlerData->pTapiEventFifoLo = fifoInit(IFX_TAPI_EVENT_FIFO_SIZE);

   if (pEventHandlerData->pTapiEventFifoHi == NULL ||
       pEventHandlerData->pTapiEventFifoLo == NULL)
   {
      ret = TAPI_statusErr;
   }

   /* Clear the mask to enable all events. */
   memset(&pEventHandlerData->eventMask, IFX_EVENT_ENABLE,
          sizeof (pEventHandlerData->eventMask));

   /* disable DTMF-end events */
   pEventHandlerData->eventMask.dtmf.bits.end_local   = IFX_EVENT_DISABLE;
   pEventHandlerData->eventMask.dtmf.bits.end_network = IFX_EVENT_DISABLE;
   /* disable Pulse-start event */
   pEventHandlerData->eventMask.pulse.bits.pulseStart = IFX_EVENT_DISABLE;

   /* initialize fifo access protection semaphore */
   TAPI_OS_MutexInit (&pEventHandlerData->fifoAcc);

   TAPI_OS_MutexRelease (&pChannel->semTapiChDataLock);

   return ret;
}


/**
   High Level Event Dispatcher function exit.

   \param  pChannel     Pointer to tapi channel structure.

   \return IFX_SUCCESS on ok, otherwise IFX_ERROR
*/
IFX_int32_t IFX_TAPI_EventDispatcher_Exit (TAPI_CHANNEL * pChannel)
{
   IFX_TAPI_EVENT_HANDLER_DATA_t *pEventHandlerData;
   IFX_int32_t ret = TAPI_statusOk;
   void *pEvt;

   TAPI_OS_MutexGet (&pChannel->semTapiChDataLock);

   /* Get pointer to data in the channel. */
   pEventHandlerData = pChannel->pEventHandler;

   if (pEventHandlerData != IFX_NULL)
   {
      /* Lock the fifo access protection. */
      TAPI_OS_MutexGet (&pEventHandlerData->fifoAcc);

      /* Silently drop all events stored in high-prio fifo */
      while ((pEvt = fifoGet(pEventHandlerData->pTapiEventFifoHi, NULL)) != NULL)
      {
         bufferPoolPut(pEvt);
      }
      /* Silently drop all events stored in low-prio fifo */
      while ((pEvt = fifoGet(pEventHandlerData->pTapiEventFifoLo, NULL)) != NULL)
      {
         bufferPoolPut(pEvt);
      }

      /* Free the low prio fifo itself */
      if (fifoFree (pEventHandlerData->pTapiEventFifoHi) != IFX_SUCCESS)
      {
         TRACE (TAPI_DRV, DBG_LEVEL_HIGH,
                ("INFO: Free event hi fifo error. ch(%d)\n", pChannel->nChannel));
         ret = IFX_ERROR;
      }
      /* Free the high prio fifo itself */
      if (fifoFree (pEventHandlerData->pTapiEventFifoLo) != IFX_SUCCESS)
      {
         TRACE (TAPI_DRV, DBG_LEVEL_HIGH,
                ("INFO: Free event lo fifo error. ch(%d)\n", pChannel->nChannel));
         ret = IFX_ERROR;
      }

      /* Unlock the protection and delete the semaphore */
      TAPI_OS_MutexRelease (&pEventHandlerData->fifoAcc);
      TAPI_OS_MutexDelete (&pEventHandlerData->fifoAcc);

      /* Free the data storage on the channel */
      TAPI_OS_Free (pChannel->pEventHandler);
      pChannel->pEventHandler = IFX_NULL;
   }

   TAPI_OS_MutexRelease (&pChannel->semTapiChDataLock);

   return ret;
}


/**
   Generate events either as tone events played locally or as RFC2833 events

   \param pChannel -Channel context for HL TAPI.
   \param pPacketEvent - Event description.

   \return
   On success IFX_SUCCESS or else IFX_ERROR

   \remarks
   This functions is generic. This is called when user calls through ioctl to
   handle DTMF and also when other module like hapi reports the events to TAPI
*/
IFX_return_t TAPI_EVENT_PKT_EV_Generate (TAPI_CHANNEL * pChannel,
                                         IFX_TAPI_PKT_EV_GENERATE_t *pPacketEvent)
{
#ifdef TAPI_EXT_KEYPAD
/*   IFX_uint8_t   tmp = 1;  */
   IFX_uint8_t   ch = 0;
   IFX_int32_t dtmfmode;
   TAPI_CHANNEL *pChnl;
   IFX_int32_t   tone_dst = -1;            /* tone destination flag */
   IFX_boolean_t gen_rfc2833 = IFX_FALSE;  /* packet generation flag */
   IFX_int32_t   play_event;
   TAPI_CHANNEL * pDataCh = IFX_NULL;
#endif /* TAPI_EXT_KEYPAD */
   IFX_TAPI_DRV_CTX_t *pDrvCtx;
   IFX_int8_t nVolume;


   if ((pChannel->bInitialized == IFX_FALSE) || (pPacketEvent == IFX_NULL))
   {
      return IFX_ERROR;
   }
#ifdef TAPI_EXT_KEYPAD
   if (pChannel->nChannel > 0)
   {
      TRACE (TAPI_DRV, DBG_LEVEL_HIGH,
             ("TAPI:non supported chnl\n"));
      return IFX_ERROR;
   }
#endif /* TAPI_EXT_KEYPAD */
   pDrvCtx = pChannel->pTapiDevice->pDevDrvCtx;

#ifdef TAPI_EXT_KEYPAD
   if ((IFX_TAPI_Module_Find_Connected_Data_Channel (
           pChannel, IFX_TAPI_MAP_TYPE_AUDIO, &pDataCh) != IFX_SUCCESS) ||
       (pDataCh == IFX_NULL) )
   {
      TRACE (TAPI_DRV, DBG_LEVEL_HIGH,
             ("TAPI: no data chnl added to audio\n"));
      return IFX_ERROR;
   }
#endif /* TAPI_EXT_KEYPAD */

   /* Check if Event packet generation function is available */
   if (!ptr_chk (pDrvCtx->COD.RTP_EV_Generate, "pDrvCtx->COD.RTP_EV_Generate"))
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
         ("%s - COD.RTP_EV_Generate not valid\n", __FUNCTION__));
      return IFX_ERROR;
   }


#ifdef TAPI_EXT_KEYPAD

#if 0
   tmp = pChannel->nAudioDataChannels;

   /* to be clarified how event transmission for conferencing should look like */
   while (tmp)
   {
      if (tmp & 1)
      {
         break;
      }
      tmp = tmp >> 1;
      ch++;
   }
#endif
   /* ch is nothing but the channel number where tone should be played so get
      chnl context */
   pChnl = (pChannel->pTapiDevice->pTapiChanelArray) + ch;
   dtmfmode = (pChannel->nDtmfInfo & ~DTMF_EV_LOCAL_PLAY);


   /*
    * Depending on current event generation mode and local play setting
    * generate tone events to be inserted in data stream (in band) and/or
    * generate RFC2833 events (out of band).
    * If local play flag is set the event is additionally
    * played as tone on the local speaker.
    */
   switch(dtmfmode)
   {
      case IFX_TAPI_PKT_EV_OOB_DEFAULT:
      case IFX_TAPI_PKT_EV_OOB_ALL:
         /* inband and OOB packet generation requested */
         gen_rfc2833 = IFX_TRUE;
         if( pChannel->nDtmfInfo & DTMF_EV_LOCAL_PLAY )
            tone_dst = (IFX_int32_t) TAPI_TONE_DST_NETLOCAL;
         else
            tone_dst = (IFX_int32_t) TAPI_TONE_DST_NET;
         break;

      case IFX_TAPI_PKT_EV_OOB_ONLY:
         /* only OOB packets, no inband tone packets */
         gen_rfc2833 = IFX_TRUE;
         if( pChannel->nDtmfInfo & DTMF_EV_LOCAL_PLAY )
            tone_dst = (IFX_int32_t) TAPI_TONE_DST_LOCAL;
         break;

      case IFX_TAPI_PKT_EV_OOB_NO:
         /* no OOB packets, only inband tone packets */
         gen_rfc2833 = IFX_FALSE;
         if( pChannel->nDtmfInfo & DTMF_EV_LOCAL_PLAY )
            tone_dst = (IFX_int32_t) TAPI_TONE_DST_NETLOCAL;
         else
            tone_dst = (IFX_int32_t) TAPI_TONE_DST_NET;
         break;

      case IFX_TAPI_PKT_EV_OOB_BLOCK:
         /* neither OOB packets nor inband tone packets */
         gen_rfc2833 = IFX_FALSE;
         if( pChannel->nDtmfInfo & DTMF_EV_LOCAL_PLAY )
            tone_dst = (IFX_int32_t) TAPI_TONE_DST_LOCAL;
         break;
      default:
         TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
            ("%s - Invalid event mode %d\n", __FUNCTION__, dtmfmode));
         return IFX_ERROR;
   }

   if (pPacketEvent->action == IFX_TAPI_EV_GEN_START)
   {
      if( tone_dst != -1 )
      {
         /* since we are using Tone table to generate dtmfs,
            if dtmf 0 then we have use the index 11.
            It has to be removed if we are not using Tones for dtmf */
         if(pPacketEvent->event == 11)/* for # dtmfcode=11 but tone index is 12 */
         {
            play_event =12;
         }
         else if(pPacketEvent->event == 0)/*dtmf 0 means index in tone table is 11*/
         {
            play_event = 11;
         }
         else
            play_event = pPacketEvent->event;

         TAPI_Phone_Tone_Play(pChnl, play_event, tone_dst);
      }
   }
   else   /* pPacketEvent->action == IFX_TAPI_EV_GEN_STOP  */
   {
      TAPI_Phone_Tone_Stop(pChnl, 0, TAPI_TONE_DST_DEFAULT);
   }

   if( gen_rfc2833 )
#endif /* TAPI_EXT_KEYPAD */
   {
      /* Transform enum value to value expected by lower layer. */
      nVolume = (IFX_int8_t)pPacketEvent->volume -
                (IFX_int8_t)IFX_TAPI_PKT_EV_GEN_VOLUME_0_dB;

      /* start/stop generation of rfc 2833 pkt */
      (IFX_void_t)pDrvCtx->COD.RTP_EV_Generate(pChannel->pLLChannel,
                                   pPacketEvent->event,
                                   pPacketEvent->action,
                                   pPacketEvent->duration,
                                   nVolume);
   }
#ifdef TAPI_EXT_KEYPAD

#if 0
   /* Incase of conferencing,Check which are the data channel added to audio
      channel,and do for all */
   /* if Local play bit is set then first coder is already played dtmf
      so move to next coderor else repeat from coder 0*/
   if (pChannel->nDtmfInfo & DTMF_EV_LOCAL_PLAY)
   {
      tmp = tmp >> 1;
      ch++;
   }

   while (tmp)
   {
      if (tmp & 1)
      {
         /* ch is nothing but the channel number where tone should be played so
            get chnl context */
         pChnl = (pChannel->pTapiDevice->pTapiChanelArray) + ch;
         /* Vmask the last bit to get the value */
         switch (dtmfmode)
         {
         case IFX_TAPI_PKT_EV_OOB_DEFAULT:
         case IFX_TAPI_PKT_EV_OOB_ALL:
            {
               /* (Transmit in band and out of band). */
               if (pPacketEvent->action == IFX_TAPI_EV_GEN_START)
               {
                  TAPI_Phone_Tone_Play (pChnl,
                                       (IFX_int32_t) pPacketEvent->event,
                                        TAPI_TONE_DST_NET);
                  /* generate rfc 2833 pkt */
                  /* TAPI_Phone_DTMF_OOB(pDrvCtx,pChnl,
                     (IFX_int32_t)pPacketEvent->event,IFX_TAPI_EV_GEN_START); */
               }
               else
               {
                  TAPI_Phone_Tone_Stop (pChnl, 0, TAPI_TONE_DST_DEFAULT);
                  /* generate rfc 2833 pkt */
                  /* TAPI_Phone_DTMF_OOB(pDrvCtx,pChnl,
                     (IFX_int32_t)pPacketEvent->event,IFX_TAPI_EV_GEN_STOP); */
               }
               break;
            }
         case IFX_TAPI_PKT_EV_OOB_ONLY:
            {
               /* Transmit only out of band. */
               if (pPacketEvent->action == IFX_TAPI_EV_GEN_START)
               {
                  /* generate rfc 2833 pkt */
                  /* TAPI_Phone_DTMF_OOB(pDrvCtx,pChnl,
                     (IFX_int32_t)pPacketEvent->event,IFX_TAPI_EV_GEN_START); */
               }
               else
               {
                  /* generate rfc 2833 pkt */
                  /* TAPI_Phone_DTMF_OOB(pDrvCtx,pChnl,
                     (IFX_int32_t)pPacketEvent->event,IFX_TAPI_EV_GEN_STOP); */
               }
               break;
            }
         case IFX_TAPI_PKT_EV_OOB_NO:
            {
               /* Transmit only in band. */
               if (pPacketEvent->action == IFX_TAPI_EV_GEN_START)
               {
                  TAPI_Phone_Tone_Play (pChnl,
                                       (IFX_int32_t) pPacketEvent->event,
                                        TAPI_TONE_DST_NET);
               }
               else
               {
                  TAPI_Phone_Tone_Stop (pChnl, 0, TAPI_TONE_DST_DEFAULT);
               }

               break;
            }
         case IFX_TAPI_PKT_EV_OOB_BLOCK:
            {
               /* neither in-band nor out-of-band */
               /* Do nothing */
               break;
            }
         }
      }                         /* end of if */
      tmp = tmp >> 1;
      ch++;
   }                            /* end of while */
#endif   /* #if 0*/
#endif /* TAPI_EXT_KEYPAD */

   return IFX_SUCCESS;
}


#ifdef TAPI_EXT_KEYPAD
/**

   This function is called by EventDispatchProcesContext.
   This will handle the external keypad DTMF events reported by other
   module like HAPI.

   \param pChannel -Channel context for HL TAPI.
   \param pTapiEvent - TAPIEvent.

   \return
   On success IFX_SUCCESS or else IFX_ERROR

   \remarks
   This functions maps the TAPI event structure to DTMF event structure
   so that a common function is used for both the events reported from
   user or from other kernel module.
*/
IFX_return_t TAPI_EXT_EVENT_Key_Handler (TAPI_CHANNEL * pChannel,
                                         IFX_TAPI_EVENT_t * pTapiEvent)
{
   IFX_return_t ret = IFX_SUCCESS;
   IFX_TAPI_PKT_EV_GENERATE_t dtmf;
   /* IFX_TAPI_EventGen_t event; */
   IFX_TAPI_DRV_CTX_t *pDrvCtx;
   IFX_int32_t state = pTapiEvent->id;
   pDrvCtx = pChannel->pTapiDevice->pDevDrvCtx;
   memset (&dtmf, 0, sizeof (dtmf));
   dtmf.event = pTapiEvent->data.keyinfo.key;
   dtmf.duration = pTapiEvent->data.keyinfo.duration;
   TRACE (TAPI_DRV, DBG_LEVEL_LOW, ("digit = %d\n", dtmf.event));
   if (state == IFX_TAPI_EVENT_EXT_KEY_UP)
   {
      TRACE (TAPI_DRV, DBG_LEVEL_LOW, ("Key is released\n"));
      dtmf.action = IFX_TAPI_EV_GEN_STOP;
   }
   else
   {
      TRACE (TAPI_DRV, DBG_LEVEL_LOW, ("Key is pressed\n"));
      dtmf.action = IFX_TAPI_EV_GEN_START;
   }
   ret = TAPI_EVENT_PKT_EV_Generate (pChannel, &dtmf);
   return ret;
}
#endif /* TAPI_EXT_KEYPAD */


/**
   Retrieve the overall number of elements of the event wrapper bufferpool.

   \return the overall number of elements
*/
IFX_int32_t IFX_TAPI_EventWrpBufferPool_ElementCountGet(IFX_void_t)
{
   TAPI_OS_INTSTAT lock;
   IFX_int32_t  elements;

   if (!TAPI_OS_IN_INTERRUPT())
   {
      TAPI_OS_MutexGet (&semBufferPoolAcc);
   }
   TAPI_OS_LOCKINT (lock);

   elements = bufferPoolSize( pIFX_TAPI_BP_Deferred_Event );

   TAPI_OS_UNLOCKINT(lock);
   if (!TAPI_OS_IN_INTERRUPT())
   {
      TAPI_OS_MutexRelease (&semBufferPoolAcc);
   }

   return elements;
}


/**
   Retrieve the available (free) number of elements of the
   event bufferpool.

   \return the number of available elements
*/
IFX_int32_t IFX_TAPI_EventWrpBufferPool_ElementAvailCountGet(IFX_void_t)
{
   TAPI_OS_INTSTAT lock;
   IFX_int32_t  elements;

   if (!TAPI_OS_IN_INTERRUPT())
   {
      TAPI_OS_MutexGet (&semBufferPoolAcc);
   }
   TAPI_OS_LOCKINT (lock);

   elements = bufferPoolAvail( pIFX_TAPI_BP_Deferred_Event );

   TAPI_OS_UNLOCKINT(lock);
   if (!TAPI_OS_IN_INTERRUPT())
   {
      TAPI_OS_MutexRelease (&semBufferPoolAcc);
   }

   return elements;
}
/*lint -restore*/
