/******************************************************************************

                              Copyright (c) 2009
                            Lantiq Deutschland GmbH
                     Am Campeon 3; 85579 Neubiberg, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

******************************************************************************/

/**
   \file drv_tapi_linux.c
   This file contains the implementation of High-Level TAPI Driver,
   Linux specific part.

   The implementation includes the following parts:
    -Registration part by which the low-level drivers register themselves.
     During the registration appropriate device nodes are created and
     registered with the kernel.
    -Device node operations (open, close, ioctl, read, write, select)
     are done here.
    -Linux module support.
    -Linux /proc fileystem handlers.
    -Timer abstraction layer.
    -Deferring of a function call for later execution.
    -Export of High-Level TAPI function symbols.
*/

/* ============================= */
/* Includes                      */
/* ============================= */
#include "drv_api.h"
#include "drv_tapi_linux.h"

#ifdef __KERNEL__
#include <linux/kernel.h>
#endif
#ifdef MODULE
#include <linux/module.h>
#endif

#include <linux/proc_fs.h>             /*proc-file system*/
#include <linux/timer.h>               /* init_timer() */
#include <linux/init.h>
#include <linux/param.h>               /* HZ */
#include <linux/errno.h>
#include <asm/uaccess.h>               /* copy_from_user(), ... */
#include <asm/byteorder.h>
#include <asm/io.h>

#ifdef LINUX_2_6
   #include <linux/workqueue.h>        /* LINUX 2.6 We need work_struct */
   #include <linux/device.h>
   #include <linux/sched.h>
   #undef   CONFIG_DEVFS_FS
   #ifndef UTS_RELEASE
      #include "linux/utsrelease.h"
   #endif /* UTC_RELEASE */
#else
   #include <linux/tqueue.h>
   #include <linux/sched.h>
   #include <linux/smp_lock.h>         /* lock_kernel() */
#endif /* LINUX_2_6 */

#include "drv_tapi.h"
#include "drv_tapi_api.h"
#include "drv_tapi_ll_interface.h"
#include "drv_tapi_errno.h"
#include "drv_tapi_stream.h"
#include "drv_tapi_fxo_ll_interface.h"
#include "drv_tapi_ioctl.h"

#ifdef KPI_SUPPORT
#include "drv_tapi_kpi.h"
#endif
#ifdef TAPI_CID
#include "drv_tapi_cid.h"
#endif
#ifdef QOS_SUPPORT
#include "drv_tapi_qos_ll_interface.h"
#endif
#ifdef TAPI_ANNOUNCEMENTS
#include "drv_tapi_announcements.h"
#endif /* TAPI_ANNOUNCEMENTS */

#define TAPI_IOCTL_STACKSIZE                 4000 /* allow some overhead 4 k */

/* ================================== */
/* channel specific wrapper structure */
/* ================================== */
typedef struct _TAPI_FD_PRIV_DATA TAPI_FD_PRIV_DATA_t;

struct _TAPI_FD_PRIV_DATA
{
   /* Pointer to the TAPI device struct */
   TAPI_DEV                     *pTapiDev;
   /* file descriptor number (255: device file descriptor <255: channel fds) */
   IFX_uint16_t                 nFds;
   /* channel fifo number */
   IFX_TAPI_STREAM_t            fifo_idx;
};

/* ============================= */
/* Local Functions               */
/* ============================= */
static IFX_void_t TAPI_timer_call_back (IFX_ulong_t arg);
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,20))
static IFX_void_t TAPI_tqueue (IFX_void_t *pWork);
#else /* for Kernel newer or equal 2.6.20 */
static IFX_void_t TAPI_tqueue (struct work_struct *pWork);
#endif
#ifdef LINUX_2_6
/* TAPI's own RT workqueue */
static struct workqueue_struct *pTapiWq;
/* init struct for the RT workqueue to set the scheduling policy
   via TAPI_DeferWork */
static IFX_TAPI_EXT_EVENT_PARAM_t tapi_wq_setscheduler_param;
#endif /* LINUX_2_6 */

static IFX_int32_t TAPI_SelectCh (TAPI_FD_PRIV_DATA_t *pTapiPriv,
                                  TAPI_OS_drvSelectTable_t *,
                                  TAPI_OS_drvSelectOSArg_t *);

static int ifx_tapi_open (struct inode *inode, struct file *filp);
static int ifx_tapi_release (struct inode *inode, struct file *filp);
static ssize_t ifx_tapi_write(struct file *filp, const char *buf,
                              size_t count, loff_t * ppos);
static ssize_t ifx_tapi_read(struct file * filp, char *buf,
                              size_t length, loff_t * ppos);
static int ifx_tapi_ioctl(struct inode *inode, struct file *filp,
                              unsigned int nCmd, unsigned long nArgument);
static unsigned int ifx_tapi_poll (struct file *filp, poll_table *table);

#ifdef CONFIG_PROC_FS
static int proc_read_tapi(char *page, char **start, off_t off,
                          int count, int *eof, void *data);
static IFX_int32_t proc_get_tapi_version(IFX_char_t *buf);
static IFX_int32_t proc_get_tapi_status(IFX_char_t *buf);
static IFX_int32_t proc_get_tapi_registered_drivers(IFX_char_t *buf);

extern IFX_int32_t IFX_TAPI_EventWrpBufferPool_ElementCountGet(IFX_void_t);
extern IFX_int32_t IFX_TAPI_EventWrpBufferPool_ElementAvailCountGet(IFX_void_t);

#ifdef TAPI_STATISTICS
static int proc_read_tapi_stats(char *buf, char **start, off_t offset,
                                int count, int *eof, void *data);
static int proc_write_tapi_stats(struct file *file, const char *buffer,
                                 unsigned long count, void *data);
#endif /* TAPI_STATISTICS */
static IFX_int32_t proc_EntriesInstall(IFX_void_t);
static IFX_void_t  proc_EntriesRemove(IFX_void_t);
#endif /* CONFIG_PROC_FS */

/** install parameter debug_level: LOW (1), NORMAL (2), HIGH (3), OFF (4) */
#ifdef ENABLE_TRACE
extern IFX_uint32_t debug_level;
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,17))
MODULE_PARM(debug_level, "i");
#else
module_param(debug_level, uint, 0);
#endif /* < 2.6.17 */
MODULE_PARM_DESC(debug_level, "set to get more (1) or fewer (4) debug outputs");
#endif /* ENABLE_TRACE */

extern IFX_void_t IFX_TAPI_Update_SlicFxo(TAPI_DEV *pTapiDev, IFX_boolean_t bVal);

IFX_int32_t block_egress_tasklet = 0;
IFX_int32_t block_ingress_tasklet = 0;

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,17))
MODULE_PARM(block_egress_tasklet, "i");
MODULE_PARM(block_ingress_tasklet, "i");
#else
module_param(block_egress_tasklet, int, 0);
module_param(block_ingress_tasklet, int, 0);
#endif /* < 2.6.17 */
MODULE_PARM_DESC(block_egress_tasklet, "block the registration of egress tasklets, i.e. force to use the RT kernel thread");
MODULE_PARM_DESC(block_ingress_tasklet, "block the execution of the ingress tasklet, i.e. force to use the RT kernel thread");

/** The driver callbacks which will be registered with the kernel*/
static struct file_operations tapi_fops = {0};

/* ============================= */
/* Global function definition    */
/* ============================= */

/**
   Register the low level driver with the operating system.

   This function will be called when the low-level driver is added to the
   system. Any OS specific registration or set-up should be done here.

   For Linux kernels with devfs support device nodes will be created here.
   The number of nodes created depend on the "one device node" or "multiple
   device node" configuration.
   For Linux kernels without devfs support the character driver is registered.

   \param  pLLDrvCtx    Pointer to device driver context created by LL-driver.
   \param  pHLDrvCtx    Pointer to high-level driver context.

   \return
   TAPI_statusOk

   \remarks
   For Linux the device nodes need to be registered and then the driver
   itself is registered with the kernel.
*/
IFX_return_t TAPI_OS_RegisterLLDrv (IFX_TAPI_DRV_CTX_t* pLLDrvCtx,
                                    IFX_TAPI_HL_DRV_CTX_t* pHLDrvCtx)
{
   IFX_uint32_t majorNumber;
   IFX_char_t   *pRegDrvName = IFX_NULL;
   IFX_int32_t ret = 0;

   if (tapi_fops.ioctl == IFX_NULL)
   {
#ifdef MODULE
      tapi_fops.owner =    THIS_MODULE;
#endif
      tapi_fops.read =     ifx_tapi_read;
      tapi_fops.write =    ifx_tapi_write;
      tapi_fops.poll =     ifx_tapi_poll;
      tapi_fops.ioctl =    ifx_tapi_ioctl;
      tapi_fops.open =     ifx_tapi_open;
      tapi_fops.release =  ifx_tapi_release;
   }

   /* copy registration info from Low level driver */
   majorNumber = pLLDrvCtx->majorNumber;

   /* pointer to static driver name storage (used for driver registration) */
   pRegDrvName = pHLDrvCtx->registeredDrvName;

   /* limit devNodeName to 8 characters */
   sprintf (pRegDrvName, "ifx_tapi (%.8s)", pLLDrvCtx->devNodeName);

   /* Register the character device */
   ret = register_chrdev (majorNumber, pRegDrvName, &tapi_fops);
   if (ret < 0)
   {
      TRACE( TAPI_DRV, DBG_LEVEL_HIGH,
           ("IFX_TAPI_Register_LL_Drv: unable to register chrdev major number "
            "%d\n", majorNumber));
      return TAPI_statusErr;
   }

#if 0
#ifdef LINUX_2_6
   tapi_driver.name = "VINETIC-CPE";
   tapi_driver.bus  = &platform_bus_type;
   driver_register(&tapi_driver);
   tapi_device.name = "VINETIC-CPE";
   tapi_device.id   = 0;
   tapi_device.dev.driver = &tapi_driver;
   platform_device_register(&tapi_device);
/* device_bind_driver(&tapi_device.dev); */
#endif /* LINUX_2_6 */
#endif /* 0 */

   return TAPI_statusOk;
}


/**
   UnRegister the low level driver from the system.

   This function will be called when the low-level driver is removed from the
   system. Any OS specific deregistration should be done here.

   For Linux kernels with devfs support device nodes will be unregistered here.
   The number of nodes unregistered depend on the "one device node" or "multiple
   device node" configuration.
   For Linux kernels without devfs support the character driver is unregistered.

   \param  pLLDrvCtx    Pointer to device driver context created by LL-driver.
   \param  pHLDrvCtx    Pointer to high-level driver context.

   \return
   TAPI_statusErr on error, otherwise TAPI_statusOk
*/
IFX_return_t TAPI_OS_UnregisterLLDrv (IFX_TAPI_DRV_CTX_t* pLLDrvCtx,
                                      IFX_TAPI_HL_DRV_CTX_t* pHLDrvCtx)
{
   unregister_chrdev (pLLDrvCtx->majorNumber, pHLDrvCtx->registeredDrvName);

   return TAPI_statusOk;
}


/**
   Open a file.

   Both types device and channel file descriptors are handled here.

   \param  inode        Pointer to the inode.
   \param  filp         Pointer to the file descriptor.

   \return
   0 - if no error,
   otherwise error code
*/
static int ifx_tapi_open (struct inode *inode, struct file *filp)
{
   TAPI_FD_PRIV_DATA_t     *pTapiPriv  = IFX_NULL;
   IFX_TAPI_DRV_CTX_t      *pDrvCtx    = IFX_NULL;
   TAPI_DEV                *pTapiDev   = IFX_NULL;
   IFX_uint32_t            dev = 0,
                           ch = 0;
   IFX_uint32_t            majorNum,
                           minorNum;

   majorNum = MAJOR(inode->i_rdev);
   minorNum = MINOR(inode->i_rdev);
   TRACE (TAPI_DRV, DBG_LEVEL_LOW,
         ("ifxTAPI open %d/%d\n", majorNum, minorNum));

   /* Get the pointer to the device driver context based on the major number */
   pDrvCtx = IFX_TAPI_DeviceDriverContextGet (majorNum);

   if (pDrvCtx == IFX_NULL)
   {
      /* This should actually never happen because the file descriptors are
         registered in TAPI_OS_RegisterLLDrv after a driver context is known. */
      printk (KERN_INFO "tapi_open: error finding DrvCtx\n");
      return -ENODEV;
   }

   /* calculate the device number and channel number */
#ifdef TAPI_ONE_DEVNODE
   if (minorNum != 0)
#endif /* TAPI_ONE_DEVNODE */
   {
#ifdef LIN_SUPPORT
      /* decode minor=8,9 for linear channel */
      if (minorNum < pDrvCtx->minorBase)
      {
         dev = ((IFX_uint32_t)(minorNum+3) / pDrvCtx->minorBase) - 1;
         ch  = ((IFX_uint32_t)(minorNum+3) % pDrvCtx->minorBase);
      }
      else
#endif /*LIN_SUPPORT*/
      {
         dev = ((IFX_uint32_t)minorNum / pDrvCtx->minorBase) - 1;
         ch  = ((IFX_uint32_t)minorNum % pDrvCtx->minorBase);
      }
   }

   /* check the device number */
   if (dev >= pDrvCtx->maxDevs)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
           ("TAPI_DRV: max. device number exceed\n"));
      return -ENODEV;
   }

   /* check the channel number */
   if (ch > pDrvCtx->maxChannels)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
          ("TAPI_DRV: max. channel number exceed\n"));
      return -ENODEV;
   }

   pTapiDev = &(pDrvCtx->pTapiDev[dev]);

   /* Increment the Usage counter */
   pTapiDev->nInUse++;

#ifdef TAPI_ONE_DEVNODE
   if (minorNum == 0)
      /* remember it for the poll call */
      pTapiDev->bSingleFd = IFX_TRUE;
   else
      pTapiDev->bSingleFd = IFX_FALSE;
#endif /* TAPI_ONE_DEVNODE */

   /* Allocate memory for a new TAPI context wrapper structure */
   pTapiPriv = (TAPI_FD_PRIV_DATA_t* )
               TAPI_OS_Malloc (sizeof(TAPI_FD_PRIV_DATA_t));
   if (pTapiPriv == IFX_NULL)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
            ("Unable to allocate memory for FD %d private data\n", minorNum));
      return IFX_ERROR;
   }

   /* pointer to the TAPI device struct */
   pTapiPriv->pTapiDev = pTapiDev;
   /* file descriptor number (255: device file descriptor <255: channel fds) */
   pTapiPriv->nFds = (ch == 0 ? IFX_TAPI_DEVICE_CH_NUMBER : ch-1);
   /* index defining which fifo to use */
   pTapiPriv->fifo_idx = (minorNum < pDrvCtx->minorBase) ?
                                     IFX_TAPI_STREAM_LIN : IFX_TAPI_STREAM_COD;
   filp->private_data = pTapiPriv;

   /* Call the Low level Device specific open routine */
   if (ptr_chk(pDrvCtx->Open, "pDrvCtx->Open"))
   {
      IFX_int32_t retLL;

      if (pTapiPriv->nFds == IFX_TAPI_DEVICE_CH_NUMBER)
      {
         retLL = pDrvCtx->Open (pTapiDev->pLLDev);
      }
      else
      {
         TAPI_CHANNEL *pTapiCh = &(pTapiDev->pTapiChanelArray[pTapiPriv->nFds]);
         retLL = pDrvCtx->Open (pTapiCh->pLLChannel);
      }

      if (retLL != IFX_SUCCESS)
      {
         TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
               ("Open LL channel failed for ch: %d\n", pTapiPriv->nFds));
         return IFX_ERROR;
      }
   }

   /* increment module use counter */
#ifdef MODULE
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0))
   MOD_INC_USE_COUNT;
#endif /* (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)) */
#endif

   return IFX_SUCCESS;
}


/**
   Close a file.

   This function gets called when a close is called on the file descriptor.
   Both types device and channel file descriptors are handled here. The
   function decrements the usage count, and calls the LL release routine.

   \param  inode        Pointer to the inode.
   \param  filp         Pointer to the file descriptor.

   \return
      -  0  on success
      - -1  on error
*/
static int ifx_tapi_release (struct inode *inode, struct file *filp)
{
   TAPI_FD_PRIV_DATA_t *pTapiPriv = (TAPI_FD_PRIV_DATA_t* )filp->private_data;
   IFX_TAPI_DRV_CTX_t  *pDrvCtx   = pTapiPriv->pTapiDev->pDevDrvCtx;
   TAPI_DEV            *pTapiDev  = pTapiPriv->pTapiDev;

   TRACE(TAPI_DRV, DBG_LEVEL_LOW, ("ifxTAPI close %d/%d tapi-ch %d\n",
         MAJOR(inode->i_rdev), MINOR(inode->i_rdev), pTapiPriv->nFds));

   if (pTapiDev->bInitialized != IFX_FALSE)
   {
      IFX_uint8_t nChannel = pTapiPriv->nFds;

      if (nChannel != IFX_TAPI_DEVICE_CH_NUMBER)
      {
         /* closing a channel file descriptor */
         TAPI_CHANNEL *pTapiCh = &pTapiDev->pTapiChanelArray[nChannel];

         /* decrement the use counters */
         if (pTapiCh->nInUse > 0)
         {
            pTapiCh->nInUse--;
         }
         if (pTapiDev->nInUse > 0)
         {
            pTapiDev->nInUse--;
         }
      }
      else
      {
         /* closing device file descriptor */

         /* decrement the use counter */
         if (pTapiDev->nInUse > 0)
         {
            pTapiDev->nInUse--;
         }
      }

   }

   /* Call the Low-level Device specific release routine. */
   /* Not having such a function is not an error. */
   if (ptr_chk(pDrvCtx->Release, "pDrvCtx->Release"))
   {
      IFX_int32_t retLL;

      if (pTapiPriv->nFds == IFX_TAPI_DEVICE_CH_NUMBER)
      {
         retLL = pDrvCtx->Release (pTapiDev->pLLDev);
      }
      else
      {
         TAPI_CHANNEL *pTapiCh = &(pTapiDev->pTapiChanelArray[pTapiPriv->nFds]);
         retLL = pDrvCtx->Release (pTapiCh->pLLChannel);
      }

      if (retLL != IFX_SUCCESS)
      {
         TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
               ("Release LL channel failed for ch: %d\n", pTapiPriv->nFds));
         return IFX_ERROR;
      }
   }


   /* decrement use counter */
#ifdef MODULE
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0))
   MOD_DEC_USE_COUNT;
#endif /* (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)) */
#endif

   /* free private data */
   TAPI_OS_Free (pTapiPriv);

   return 0;
}


/**
   Writes data to the device.

   \param  filp         Pointer to the file descriptor.
   \param  buf          Pointer to data to be written.
   \param  count        Data length.
   \param  ppos         unused.

   \return
   length or a negative error code
   \remark
   Currently the low-level write function is called transparently
*/
static ssize_t ifx_tapi_write (struct file *filp,
                               const char *buf, size_t count, loff_t * ppos)
{
#ifdef TAPI_PACKET
   TAPI_FD_PRIV_DATA_t *pTapiPriv = (TAPI_FD_PRIV_DATA_t* )filp->private_data;
   IFX_TAPI_DRV_CTX_t  *pDrvCtx   = pTapiPriv->pTapiDev->pDevDrvCtx;
   TAPI_DEV            *pTapiDev  = pTapiPriv->pTapiDev;
   IFX_TAPI_STREAM_t    fifo_idx  = pTapiPriv->fifo_idx;
   TAPI_CHANNEL        *pTapiCh;
   IFX_uint8_t         *pData;
   IFX_size_t           buf_size;
#endif /* TAPI_PACKET */
   IFX_ssize_t          size = 0;

#ifdef TAPI_PACKET
   if (pTapiDev->bInitialized == IFX_FALSE)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_LOW,
            ("TAPI_DRV: write failed because device is not initialised\n"));
      return -EFAULT;
   }

   pTapiCh = &(pTapiDev->pTapiChanelArray[pTapiPriv->nFds]);

   if (ptr_chk( pDrvCtx->Write, "pDrvCtx->Write"))
   {
      /* Truncate data to the size of our voice buffer. */
      buf_size = IFX_TAPI_VoiceBufferPool_ElementSizeGet();
      if (count > (buf_size - pDrvCtx->pktBufPrependSpace))
      {
         count = (buf_size - pDrvCtx->pktBufPrependSpace);
      }

      /* Get a buffer from the voice buffer pool. */
      if ((pData = IFX_TAPI_VoiceBufferGet()) == IFX_NULL)
      {
         /* voice buffer pool is depleted */
         IFX_TAPI_Stat_Add(pTapiCh, fifo_idx,
                           TAPI_STAT_COUNTER_INGRESS_DISCARDED, 1);
         return -EFAULT;
      }

      /* if you have time and want clean buffers you might like this */
      /* memset(pData, 0, buf_size); */

      /* Copy data from userspace into kernelspace buffer. */
      if (copy_from_user (
            (void *)(pData + pDrvCtx->pktBufPrependSpace),
            (void *)buf, count))
      {
         /* copy into kernel space failed */
         IFX_TAPI_Stat_Add(pTapiCh, fifo_idx,
                           TAPI_STAT_COUNTER_INGRESS_DISCARDED, 1);
         IFX_TAPI_VoiceBufferPut(pData);
         return -EFAULT;
      }

      /* Call the Low level driver's write function. */
      size = pDrvCtx->Write (pTapiCh->pLLChannel, pData, (IFX_int32_t)count,
                             (IFX_int32_t*)(IFX_void_t*)ppos, fifo_idx);

      if (size == IFX_ERROR)
      {
         /* Low level driver's write failed */
         IFX_TAPI_Stat_Add(pTapiCh, fifo_idx,
                           TAPI_STAT_COUNTER_INGRESS_DISCARDED, 1);
         IFX_TAPI_VoiceBufferPut(pData);
         return -EFAULT;
      }
   }
   else
   {
      IFX_TAPI_Stat_Add(pTapiCh, fifo_idx,
                        TAPI_STAT_COUNTER_INGRESS_DISCARDED, 1);
      TRACE(TAPI_DRV, DBG_LEVEL_LOW,
            ("TAPI_DRV: LL-driver does not provide packet write\n"));
      return -EFAULT;
   }
#endif /* TAPI_PACKET */

   return (ssize_t) size;
}


/**
   Reads data from the device.

   \param  filp         Pointer to the file descriptor.
   \param  buf          Pointer to buffer that receives the data.
   \param  count        Max size of data to read in bytes.
   \param  ppos         unused.

   \return
   Number of bytes read successfully into the buffer.
*/
static ssize_t ifx_tapi_read(struct file *filp,
                             char *buf, size_t count, loff_t * ppos)
{
#ifdef TAPI_PACKET
   TAPI_FD_PRIV_DATA_t *pTapiPriv = (TAPI_FD_PRIV_DATA_t* )filp->private_data;
   IFX_TAPI_DRV_CTX_t  *pDrvCtx   = pTapiPriv->pTapiDev->pDevDrvCtx;
   TAPI_DEV            *pTapiDev  = pTapiPriv->pTapiDev;
   IFX_TAPI_STREAM_t    fifo_idx  = pTapiPriv->fifo_idx;
   TAPI_CHANNEL        *pTapiCh;
   IFX_void_t const    *pPacket   = NULL;
   IFX_uint32_t         nOffset;
#endif /* TAPI_PACKET */
   IFX_uint32_t         size = 0;

   IFX_UNUSED (ppos);

#ifdef TAPI_PACKET
   if (pTapiDev->bInitialized == IFX_FALSE)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_LOW,
            ("TAPI_DRV: read failed because device is not initialised\n"));
      return -EFAULT;
   }

   pTapiCh = &(pTapiDev->pTapiChanelArray[pTapiPriv->nFds]);

   if (fifo_idx >= IFX_TAPI_STREAM_MAX)
   {
      /* index out of range - internal error */
      TRACE (TAPI_DRV, DBG_LEVEL_LOW, ("INFO: FIFO index out of range\n"));
      /* return linux error code to the application */
      return -EINVAL;
   }

   /* All packets to be read are already in the channel specific upstream
      fifo of the driver. They are put there by whoever receives packets.
      So all that is done here operates on this fifo and does not need to call
      any ll-driver function any more. */

   /* check for non blocking flag */
   if (filp->f_flags & O_NONBLOCK)
      pTapiCh->nFlags |= CF_NONBLOCK;

   /* Lock access to the fifo. */
   TAPI_OS_MutexGet (&pTapiCh->semTapiChDataLock);
   if (ptr_chk(pDrvCtx->IRQ.LockDevice, "pDrvCtx->IRQ.LockDevice"))
   {
      pDrvCtx->IRQ.LockDevice (pTapiDev->pLLDev);
   }

   /* If FIFO is empty or does not exist we cannot read anything. In this
      case we either wait until data arrives or return if non-blocking.
      If the FIFO does not exist no data will ever arrive and the possible
      wait will last forever. */
   if ((pTapiCh->pUpStreamFifo[fifo_idx] == IFX_NULL) ||
       (fifoEmpty(pTapiCh->pUpStreamFifo[fifo_idx]) == IFX_TRUE))
   {
      /* Unlock access to the fifo. */
      if (pDrvCtx->IRQ.UnlockDevice != IFX_NULL)
      {
         pDrvCtx->IRQ.UnlockDevice (pTapiDev->pLLDev);
      }
      TAPI_OS_MutexRelease (&pTapiCh->semTapiChDataLock);

      /* check the non-blocking flag */
      if (pTapiCh->nFlags & CF_NONBLOCK)
      {
         /* this is a non blocking read call - return immediately */
         return 0;
      }

      /* this is a blocking read call so go to sleep now */
      if (TAPI_OS_EventWait (&pTapiCh->semReadBlock,
                             TAPI_OS_WAIT_FOREVER, IFX_NULL) == IFX_ERROR)
      {
         /* timeout has expired without arrival of data */
         return 0;
      }

      /* we have been woken because data has arrived */

      /* before accessing the mailbox - lock again. */
      TAPI_OS_MutexGet (&pTapiCh->semTapiChDataLock);
      if (ptr_chk(pDrvCtx->IRQ.LockDevice, "pDrvCtx->IRQ.LockDevice"))
      {
         pDrvCtx->IRQ.LockDevice (pTapiDev->pLLDev);
      }
   }

   /* read data from the fifo */
   TAPI_ASSERT (fifoEmpty(pTapiCh->pUpStreamFifo[fifo_idx]) ==
                 IFX_FALSE);
   pPacket = IFX_TAPI_UpStreamFifo_Get(pTapiCh, fifo_idx, &size, &nOffset);

   /* Unlock access to the fifo. */
   if (pDrvCtx->IRQ.UnlockDevice != IFX_NULL)
   {
      pDrvCtx->IRQ.UnlockDevice (pTapiDev->pLLDev);
   }
   TAPI_OS_MutexRelease (&pTapiCh->semTapiChDataLock);

   /* sanity check on the received packet */
   if (pPacket == NULL)
   {
      /* packet may not be NULL - internal error */
      TRACE (TAPI_DRV, DBG_LEVEL_LOW, ("INFO: pPacket Null\n"));
      return 0;
   }
   if (size > (IFX_uint32_t)count)
   {
      /* output buffer not large enough for data */
      /* drop the packet */
      IFX_TAPI_VoiceBufferPut((IFX_void_t *)pPacket);
      /* update statistic */
      IFX_TAPI_Stat_Add(pTapiCh, fifo_idx,
                        TAPI_STAT_COUNTER_EGRESS_DISCARDED, 1);
      /* return linux error code to the application */
      return -EINVAL;
   }

   /* unmap data */
   TAPI_OS_CpyKern2Usr (buf, ((IFX_char_t *)pPacket + nOffset),
                        (IFX_uint32_t)size);

   /* return buffer back to the pool now that is has been copied */
   IFX_TAPI_VoiceBufferPut((IFX_void_t *)pPacket);
   /* update statistic */
   IFX_TAPI_Stat_Add(pTapiCh, fifo_idx,
                     TAPI_STAT_COUNTER_EGRESS_DELIVERED, 1);
#endif /* TAPI_PACKET */

   return (ssize_t) size;
}


/**
   Poll implementation for the device.

   This function does the following functions:
      - Put the event queue in the wait table and sleep if select
        is called on the device itself.
      - Put the read/write queue in the wait table and sleep if select
        is called on the channels.

   \param  filp         Pointer to the file descriptor.
   \param  wait         Pointer to the poll table.

   \return
   Status of the events occured on the device which are:
   0, TAPI_OS_SYSREAD, TAPI_OS_SYSWRITE
*/
static IFX_uint32_t ifx_tapi_poll (struct file *filp, poll_table *wait)
{
   TAPI_FD_PRIV_DATA_t *pTapiPriv = (TAPI_FD_PRIV_DATA_t* )filp->private_data;
   TAPI_DEV            *pTapiDev  = pTapiPriv->pTapiDev;
   IFX_uint16_t         nFds      = pTapiPriv->nFds;
   IFX_uint16_t         i;
   IFX_int32_t          ret = 0;

#ifdef TAPI_ONE_DEVNODE
   if (pTapiDev->nChannel == IFX_TAPI_DEVICE_CH_NUMBER &&
       pTapiDev->bSingleFd == IFX_TRUE)
   {
      IFX_uint16_t      j;

      for (j= 0; j < pDrvCtx->maxDevs; ++j)
      {
         pTapiDev = &pDrvCtx->pTapiDev[j];
         pTapiDev->bNeedWakeup = IFX_TRUE;

         TAPI_OS_DrvSelectQueueAddTask (filp, &pTapiDev->wqEvent, wait);

         if (pTapiDev->pTapiChanelArray == IFX_NULL)
            continue;

         for (i = 0; i < pTapiDev->nMaxChannel; i++)
         {
            TAPI_CHANNEL *pTapiCh = &(pTapiDev->pTapiChanelArray[i]);

            TAPI_OS_DrvSelectQueueAddTask (opt, &pTapiCh->semReadBlock.object,
                                           node);
            TAPI_OS_DrvSelectQueueAddTask (opt, &pTapiCh->wqRead, node);

            if (!IFX_TAPI_EventFifoEmpty(pTapiCh))
            {
               /* exception available so return action */
               ret |= TAPI_OS_SYSREAD;
               pTapiDev->bNeedWakeup = IFX_FALSE;
               break;
            }
         }
      }
   }
   else
#endif /* TAPI_ONE_DEVNODE */
   {
      if (nFds == IFX_TAPI_DEVICE_CH_NUMBER)
      {
         /* Register the TAPI-event waitqueue as wakeup source. */
         TAPI_OS_DrvSelectQueueAddTask (filp, &pTapiDev->wqEvent, wait);

         /* Check if there is any TAPI-event on any of the TAPI channels and
            return if file operations are possible without blocking. */

         TAPI_ASSERT (pTapiDev->pTapiChanelArray != IFX_NULL);

         pTapiDev->bNeedWakeup = IFX_TRUE;
         for (i = 0; i < pTapiDev->nMaxChannel; i++)
         {
            TAPI_CHANNEL *pTapiCh = &(pTapiDev->pTapiChanelArray[i]);
            if ((pTapiCh->bInitialized == IFX_TRUE) &&
                (IFX_TAPI_EventFifoEmpty(pTapiCh) == IFX_FALSE))
            {
               /* TAPI-event available so return action */
               ret |= TAPI_OS_SYSREAD;
               pTapiDev->bNeedWakeup = IFX_FALSE;
               break;
            }
         }
      }
      else if (nFds < pTapiDev->nMaxChannel)
      {
         /* Check if any packet is available in any of the upstream fifos and
            return if file operations are possible without blocking. */
         ret |= TAPI_SelectCh (pTapiPriv,
                               (TAPI_OS_drvSelectTable_t *)wait,
                               (TAPI_OS_drvSelectOSArg_t *)filp);
      }
   }

   return (IFX_uint32_t)ret;
}


/**
   Configuration / Control for the device.

   \param  inode        Pointer to the inode.
   \param  filp         Pointer to the file descriptor.
   \param  nCmd         IOCTL identifier.
   \param  nArg         Optional argument.

   \return
   - 0 and positive values - success
   - negative value - ioctl failed
*/
static int ifx_tapi_ioctl(struct inode *inode, struct file *filp,
                          unsigned int nCmd, unsigned long nArg)
{
   TAPI_FD_PRIV_DATA_t *pTapiPriv;
   IFX_TAPI_ioctlCtx_t  ctx;
   IFX_int32_t          ret;

   pTapiPriv = (TAPI_FD_PRIV_DATA_t* )filp->private_data;

   /* set the ioctl context struct */
   memset (&ctx, 0, sizeof(ctx));
   ctx.pTapiDev = pTapiPriv->pTapiDev;
   ctx.nFds     = pTapiPriv->nFds;
#ifdef TAPI_ONE_DEVNODE
   ctx.bSingleFd = (MINOR(inode->i_rdev) == 0) ? IFX_TRUE : IFX_FALSE;
   /*\todo set fds accoring to the channel, dev according to dev */
#endif /* TAPI_ONE_DEVNODE */
   ctx.bLinChFd = (pTapiPriv->fifo_idx == IFX_TAPI_STREAM_LIN) ? 1 : 0;
   ctx.nParamSize = _IOC_SIZE(nCmd);

   ret = TAPI_Ioctl (&ctx, nCmd, nArg);

   return ret;
}


/**
   Tapi Specific ioctl handling

   \param  pDrvCtx      Pointer to device driver context.
   \param  pCtx         Unused: Pointer to IOCTL context.
   \param  pChannel     Pointer to TAPI_CHANNEL structure.
   \param  iocmd        IOCTL command.
   \param  ioarg        IOCTL argument.
   \param  retLL        Low-level driver return code.

   \return
   - TAPI_statusOk
   - TAPI_statusErr

   \remarks
   New error handling is only partly implemented. Especially the LL calls which
   have a function wrapper are not yet adapted.
*/
IFX_int32_t  TAPI_OS_IoctlCh (IFX_TAPI_DRV_CTX_t* pDrvCtx,
                              IFX_TAPI_ioctlCtx_t* pCtx,
                              TAPI_CHANNEL* pChannel,
                              IFX_uint32_t iocmd,
                              IFX_ulong_t ioarg,
                              IFX_int32_t *retLL)

{
   IFX_void_t   *parg      = (IFX_void_t *)ioarg;
   IFX_void_t   *p_iostack = IFX_NULL;
   IFX_int32_t   ret       = TAPI_statusOk;

   IFX_UNUSED (pCtx);

   /* Get hold of memory to process this ioctl */
   p_iostack = TAPI_OS_Malloc (TAPI_IOCTL_STACKSIZE);
   if (p_iostack == IFX_NULL)
      return TAPI_statusErr;

   /* check channel */
   switch (iocmd)
   {
   /* Ringing Services */
   case IFX_TAPI_RING_CFG_SET:
   {
      IFX_TAPI_RING_CFG_t *p_tmp = (IFX_TAPI_RING_CFG_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_RING_CFG_t) < TAPI_IOCTL_STACKSIZE);
      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_RING_CFG_t)) > 0 )
      {
         /* errmsg: Copy to or from user space not successful */
            ret = TAPI_statusErrKernCpy;
      }
      else
      {
         ret = IFX_TAPI_Ring_SetConfig(pChannel, p_tmp);
      }
      break;
   }

   case IFX_TAPI_RING_CFG_GET:
   {
      IFX_TAPI_RING_CFG_t *p_tmp = (IFX_TAPI_RING_CFG_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_RING_CFG_t) < TAPI_IOCTL_STACKSIZE);
      ret =  IFX_TAPI_Ring_GetConfig(pChannel, p_tmp);
      if ((ret == TAPI_statusOk) &&
          (copy_to_user (parg, p_tmp, sizeof(IFX_TAPI_RING_CFG_t)) > 0))
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      break;
   }

   case IFX_TAPI_RING_CADENCE_HR_SET:
   {
      IFX_TAPI_RING_CADENCE_t *p_tmp = (IFX_TAPI_RING_CADENCE_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_RING_CADENCE_t) < TAPI_IOCTL_STACKSIZE);
      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_RING_CADENCE_t)) > 0 )
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      else
      {
         ret = IFX_TAPI_Ring_SetCadenceHighRes(pChannel, p_tmp);
      }
      break;
   }

   case IFX_TAPI_LINE_TYPE_SET:
   {
      IFX_TAPI_LINE_TYPE_CFG_t *p_tmp = (IFX_TAPI_LINE_TYPE_CFG_t *) p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_LINE_TYPE_CFG_t) < TAPI_IOCTL_STACKSIZE);
      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_LINE_TYPE_CFG_t)) > 0 )
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      else
      {
         ret = TAPI_Phone_Set_LineType(pChannel, p_tmp);
      }
      break;
   }


#ifdef TAPI_VOICE
   /* Connection Services*/
   case IFX_TAPI_MAP_DATA_ADD:
   {
      IFX_TAPI_MAP_DATA_t *p_tmp = (IFX_TAPI_MAP_DATA_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_MAP_DATA_t) < TAPI_IOCTL_STACKSIZE);
      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_MAP_DATA_t)) > 0)
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      else
      {
         ret = TAPI_Data_Channel_Add (pChannel, p_tmp);
      }
      break;
   }

   case IFX_TAPI_MAP_DATA_REMOVE:
   {
      IFX_TAPI_MAP_DATA_t *p_tmp = (IFX_TAPI_MAP_DATA_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_MAP_DATA_t) < TAPI_IOCTL_STACKSIZE);
      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_MAP_DATA_t)) > 0)
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      else
      {
         ret = TAPI_Data_Channel_Remove (pChannel, p_tmp);
      }
      break;
   }

   case IFX_TAPI_MAP_PHONE_ADD:
      if (ptr_chk(pDrvCtx->CON.Module_Connect, ""))
      {
         IFX_TAPI_MAP_PHONE_t  *p_tmp = (IFX_TAPI_MAP_PHONE_t  *)p_iostack;
         TAPI_ASSERT(sizeof(IFX_TAPI_MAP_PHONE_t) < TAPI_IOCTL_STACKSIZE);

         if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_MAP_PHONE_t)) > 0)
         {
            /* errmsg: Copy to or from user space not successful */
            ret = TAPI_statusErrKernCpy;
         }
         else
         {
            *retLL = pDrvCtx->CON.Module_Connect (pChannel->pLLChannel,
                                                  IFX_TAPI_MAP_TYPE_PHONE,
                                                  p_tmp->nPhoneCh,
                                                  p_tmp->nChType);
         }
      }
      else
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      break;

   case IFX_TAPI_MAP_PHONE_REMOVE:
      if (ptr_chk(pDrvCtx->CON.Module_Disconnect, ""))
      {
         IFX_TAPI_MAP_PHONE_t  *p_tmp = (IFX_TAPI_MAP_PHONE_t *)p_iostack;
         TAPI_ASSERT(sizeof(IFX_TAPI_MAP_PHONE_t) < TAPI_IOCTL_STACKSIZE);

         if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_MAP_PHONE_t)) > 0)
         {
            /* errmsg: Copy to or from user space not successful */
            ret = TAPI_statusErrKernCpy;
         }
         else
         {
            *retLL = pDrvCtx->CON.Module_Disconnect (pChannel->pLLChannel,
                                                     IFX_TAPI_MAP_TYPE_PHONE,
                                                     p_tmp->nPhoneCh,
                                                     p_tmp->nChType);
         }
      }
      else
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      break;

   case IFX_TAPI_PCM_IF_CFG_SET:
   {
      IFX_TAPI_PCM_IF_CFG_t *p_tmp = (IFX_TAPI_PCM_IF_CFG_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_PCM_IF_CFG_t) < TAPI_IOCTL_STACKSIZE);
      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_PCM_IF_CFG_t)) > 0)
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      else
      {
         *retLL = TAPI_Phone_PCM_IF_Set_Config(pChannel->pTapiDevice, p_tmp);
      }
      break;
   }

   case IFX_TAPI_MAP_PCM_ADD:
      if (ptr_chk(pDrvCtx->CON.Module_Connect, ""))
      {
         IFX_TAPI_MAP_PCM_t  *p_tmp = (IFX_TAPI_MAP_PCM_t *)p_iostack;
         TAPI_ASSERT(sizeof(IFX_TAPI_MAP_PCM_t) < TAPI_IOCTL_STACKSIZE);

         if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_MAP_PCM_t)) > 0)
         {
            /* errmsg: Copy to or from user space not successful */
            ret = TAPI_statusErrKernCpy;
         }
         else
         {
            *retLL = pDrvCtx->CON.Module_Connect (pChannel->pLLChannel,
                                                  IFX_TAPI_MAP_TYPE_PCM,
                                                  p_tmp->nDstCh,
                                                  p_tmp->nChType);
         }
      }
      else
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      break;

   case IFX_TAPI_MAP_PCM_REMOVE:
      if (ptr_chk(pDrvCtx->CON.Module_Disconnect, ""))
      {
         IFX_TAPI_MAP_PCM_t  *p_tmp = (IFX_TAPI_MAP_PCM_t  *)p_iostack;
         TAPI_ASSERT(sizeof(IFX_TAPI_MAP_PCM_t) < TAPI_IOCTL_STACKSIZE);

         if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_MAP_PCM_t)) > 0)
         {
            /* errmsg: Copy to or from user space not successful */
            ret = TAPI_statusErrKernCpy;
         }
         else
         {
            *retLL = pDrvCtx->CON.Module_Disconnect (pChannel->pLLChannel,
                                                     IFX_TAPI_MAP_TYPE_PCM,
                                                     p_tmp->nDstCh,
                                                     p_tmp->nChType);
         }
      }
      else
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      break;

#ifdef DECT_SUPPORT
   case IFX_TAPI_MAP_DECT_ADD:
      if (ptr_chk(pDrvCtx->CON.Module_Connect, ""))
      {
         IFX_TAPI_MAP_DECT_t  *p_tmp = (IFX_TAPI_MAP_DECT_t  *)p_iostack;
         TAPI_ASSERT(sizeof(IFX_TAPI_MAP_DECT_t) < TAPI_IOCTL_STACKSIZE);

         if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_MAP_DECT_t)) > 0)
         {
            /* errmsg: Copy to or from user space not successful */
            ret = TAPI_statusErrKernCpy;
         }
         else
         {
            *retLL = pDrvCtx->CON.Module_Connect (pChannel->pLLChannel,
                                                  IFX_TAPI_MAP_TYPE_DECT,
                                                  p_tmp->nDstCh,
                                                  p_tmp->nChType);
         }
      }
      else
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      break;


   case IFX_TAPI_MAP_DECT_REMOVE:
      if (ptr_chk(pDrvCtx->CON.Module_Disconnect, ""))
      {
         IFX_TAPI_MAP_DECT_t  *p_tmp = (IFX_TAPI_MAP_DECT_t  *)p_iostack;
         TAPI_ASSERT(sizeof(IFX_TAPI_MAP_DECT_t) < TAPI_IOCTL_STACKSIZE);

         if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_MAP_DECT_t)) > 0)
         {
            /* errmsg: Copy to or from user space not successful */
            ret = TAPI_statusErrKernCpy;
         }
         else
         {
            *retLL = pDrvCtx->CON.Module_Disconnect (pChannel->pLLChannel,
                                                     IFX_TAPI_MAP_TYPE_DECT,
                                                     p_tmp->nDstCh,
                                                     p_tmp->nChType);
         }
      }
      else
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      break;
#endif /* DECT_SUPPORT */

   case IFX_TAPI_PKT_RTP_CFG_SET:
   {
      IFX_TAPI_PKT_RTP_CFG_t *p_tmp = (IFX_TAPI_PKT_RTP_CFG_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_PKT_RTP_CFG_t) < TAPI_IOCTL_STACKSIZE);
      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_PKT_RTP_CFG_t)) > 0)
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      else
      {
         if (pDrvCtx->COD.RTP_Cfg == IFX_NULL)
         {
            /* errmsg: Service is not supported by the low level driver */
            ret = TAPI_statusLLNotSupp;
         }
         else
         {
#ifdef TAPI_EXT_KEYPAD
            /*
             * Clear lower 7 bits of status byte and
             * write passed argument to these bits.
             * Ensure that local play flag remains.
             */
            pChannel->nDtmfInfo = (pChannel->nDtmfInfo & DTMF_EV_LOCAL_PLAY) |
                                  ((IFX_uint8_t)(p_tmp->nEvents) &
                                   ~DTMF_EV_LOCAL_PLAY );
#endif

            *retLL = pDrvCtx->COD.RTP_Cfg (pChannel->pLLChannel, p_tmp);
         }
      }
      break;
   }

   case IFX_TAPI_PKT_RTP_PT_CFG_SET:
   {
      IFX_TAPI_PKT_RTP_PT_CFG_t *p_tmp =
                                     (IFX_TAPI_PKT_RTP_PT_CFG_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_PKT_RTP_PT_CFG_t) < TAPI_IOCTL_STACKSIZE);
      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_PKT_RTP_PT_CFG_t)) > 0)
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      else
      {
         if (pDrvCtx->COD.RTP_PayloadTable_Cfg == IFX_NULL)
         {
            /* errmsg: Service is not supported by the low level driver */
            ret = TAPI_statusLLNotSupp;
         }
         else
            *retLL = pDrvCtx->COD.RTP_PayloadTable_Cfg (pChannel->pLLChannel,
                                                        p_tmp);
      }
      break;
   }

   case IFX_TAPI_PKT_RTCP_STATISTICS_GET:
   {
      IFX_TAPI_PKT_RTCP_STATISTICS_t *p_tmp =
                                 (IFX_TAPI_PKT_RTCP_STATISTICS_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_PKT_RTCP_STATISTICS_t) < TAPI_IOCTL_STACKSIZE);
      if (pDrvCtx->COD.RTCP_Get == IFX_NULL)
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      else
      {
         *retLL = pDrvCtx->COD.RTCP_Get (pChannel->pLLChannel, p_tmp);
      }
      if ((*retLL == TAPI_statusOk) &&
          (copy_to_user (parg, p_tmp, sizeof(IFX_TAPI_PKT_RTCP_STATISTICS_t)) > 0))
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      break;
   }

   case IFX_TAPI_JB_CFG_SET:
   {
      IFX_TAPI_JB_CFG_t *p_tmp = (IFX_TAPI_JB_CFG_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_JB_CFG_t) < TAPI_IOCTL_STACKSIZE);
      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_JB_CFG_t)) > 0)
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      else
      {
         if (pDrvCtx->COD.JB_Cfg == IFX_NULL)
         {
            /* errmsg: Service is not supported by the low level driver */
            ret = TAPI_statusLLNotSupp;
         }
         else
         {
            *retLL = pDrvCtx->COD.JB_Cfg (pChannel->pLLChannel, p_tmp);
         }
      }
      break;
   }

   case IFX_TAPI_JB_STATISTICS_GET:
   {
      IFX_TAPI_JB_STATISTICS_t *p_tmp =
                                   (IFX_TAPI_JB_STATISTICS_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_JB_STATISTICS_t) < TAPI_IOCTL_STACKSIZE);
      if (pDrvCtx->COD.JB_Stat_Get == IFX_NULL)
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      else
         *retLL = pDrvCtx->COD.JB_Stat_Get (pChannel->pLLChannel, p_tmp);
      if ((ret == TAPI_statusOk) && (copy_to_user ( parg, p_tmp,
                                   sizeof(IFX_TAPI_JB_STATISTICS_t)) > 0))
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      break;
   }

   case IFX_TAPI_ENC_CFG_SET:
      if (ptr_chk(pDrvCtx->COD.ENC_Cfg, ""))
      {
         IFX_TAPI_ENC_CFG_t *p_tmp = (IFX_TAPI_ENC_CFG_t *)p_iostack;
         TAPI_ASSERT(sizeof(IFX_TAPI_ENC_CFG_t) < TAPI_IOCTL_STACKSIZE);
         if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_ENC_CFG_t)) > 0 )
         {
            /* errmsg: Copy to or from user space not successful */
            ret = TAPI_statusErrKernCpy;
         }
         else
         {
            *retLL = pDrvCtx->COD.ENC_Cfg(pChannel->pLLChannel,
                                          p_tmp->nEncType, p_tmp->nFrameLen,
                                          p_tmp->AAL2BitPack);
         }
      }
      else
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      break;

   case IFX_TAPI_DEC_CFG_SET:
      if (ptr_chk(pDrvCtx->COD.DEC_Cfg, ""))
      {
         IFX_TAPI_DEC_CFG_t *p_tmp = (IFX_TAPI_DEC_CFG_t *)p_iostack;
         TAPI_ASSERT(sizeof(IFX_TAPI_DEC_CFG_t) < TAPI_IOCTL_STACKSIZE);
         if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_DEC_CFG_t)) > 0 )
         {
            /* errmsg: Copy to or from user space not successful */
            ret = TAPI_statusErrKernCpy;
         }
         else
         {
            *retLL = pDrvCtx->COD.DEC_Cfg(pChannel->pLLChannel,
                                          p_tmp->AAL2BitPack,
                                          p_tmp->Plc);
         }
      }
      else
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      break;
#endif /* TAPI_VOICE */

#ifdef TAPI_FAX_T38
   /* Fax Services */
   case IFX_TAPI_T38_MOD_START:
   {
      IFX_TAPI_T38_MOD_DATA_t *p_tmp = (IFX_TAPI_T38_MOD_DATA_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_T38_MOD_DATA_t) < TAPI_IOCTL_STACKSIZE);
      if (copy_from_user (p_tmp, parg,  sizeof(IFX_TAPI_T38_MOD_DATA_t)) > 0)
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      else
      {
         if (pDrvCtx->COD.T38_Mod_Enable == IFX_NULL)
         {
            /* errmsg: Service is not supported by the low level driver */
            ret = TAPI_statusLLNotSupp;
         }
         else
         {
            *retLL = pDrvCtx->COD.T38_Mod_Enable (pChannel->pLLChannel, p_tmp);
         }
      }
      break;
   }

   case IFX_TAPI_T38_DEMOD_START:
   {
      IFX_TAPI_T38_DEMOD_DATA_t *p_tmp = (IFX_TAPI_T38_DEMOD_DATA_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_T38_DEMOD_DATA_t) < TAPI_IOCTL_STACKSIZE);
      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_T38_DEMOD_DATA_t)) > 0)
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      else
      {
         if (pDrvCtx->COD.T38_DeMod_Enable == IFX_NULL)
         {
            /* errmsg: Service is not supported by the low level driver */
            ret = TAPI_statusLLNotSupp;
         }
         else
            *retLL = pDrvCtx->COD.T38_DeMod_Enable (pChannel->pLLChannel, p_tmp);
      }
      break;
   }


   case IFX_TAPI_T38_STATUS_GET:
   {
      IFX_TAPI_T38_STATUS_t *p_tmp = (IFX_TAPI_T38_STATUS_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_T38_STATUS_t) < TAPI_IOCTL_STACKSIZE);

      if (pDrvCtx->COD.T38_Status_Get == IFX_NULL)
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      else
         *retLL = pDrvCtx->COD.T38_Status_Get (pChannel->pLLChannel, p_tmp);
      if ((ret == TAPI_statusOk) &&
          (copy_to_user (parg, p_tmp, sizeof(IFX_TAPI_T38_STATUS_t)) > 0))
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      break;
   }
#endif /* TAPI_FAX_T38 */

#ifdef TAPI_FAX_T38_FW
   /* Fax Services (T.38 supported by EDSP firmware) */
   case IFX_TAPI_T38_SESS_START:
   {
      if (ptr_chk(pDrvCtx->COD.FAX_Start, ""))
      {
         IFX_TAPI_T38_SESS_CFG_t  *p_tmp = (IFX_TAPI_T38_SESS_CFG_t*)p_iostack;
         TAPI_ASSERT(sizeof(IFX_TAPI_T38_SESS_CFG_t) < TAPI_IOCTL_STACKSIZE);
         if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_T38_SESS_CFG_t)) > 0)
         {
            /* errmsg: Copy to or from user space not successful */
            ret = TAPI_statusErrKernCpy;
         }
         else
         {
            *retLL = pDrvCtx->COD.FAX_Start(pChannel->pLLChannel, p_tmp);
         }
      }
      else
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      break;
   }
   case IFX_TAPI_T38_SESS_STOP:
   {
      if (ptr_chk(pDrvCtx->COD.FAX_Stop, ""))
      {
         *retLL = pDrvCtx->COD.FAX_Stop(pChannel->pLLChannel);
      }
      else
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      break;
   }
   case IFX_TAPI_T38_SESS_STATISTICS_GET:
   {
      if (ptr_chk(pDrvCtx->COD.FAX_Stat_Get, ""))
      {
         IFX_TAPI_T38_SESS_STATISTICS_t *p_tmp =
                                 (IFX_TAPI_T38_SESS_STATISTICS_t *)p_iostack;
         TAPI_ASSERT(sizeof(IFX_TAPI_T38_SESS_STATISTICS_t) <
                                        TAPI_IOCTL_STACKSIZE);
         *retLL = pDrvCtx->COD.FAX_Stat_Get (pChannel->pLLChannel, p_tmp);
         if (TAPI_SUCCESS(*retLL))
         {
            if (copy_to_user (parg, p_tmp,
                  sizeof(IFX_TAPI_T38_SESS_STATISTICS_t)) > 0)
            {
               /* errmsg: Copy to or from user space not successful */
               ret = TAPI_statusErrKernCpy;
            }
         }
      }
      else
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      break;
   }
   case IFX_TAPI_T38_CFG_SET:
   {
      if (ptr_chk(pDrvCtx->COD.FAX_Cfg_Set, ""))
      {
         IFX_TAPI_T38_FAX_CFG_t  *p_tmp = (IFX_TAPI_T38_FAX_CFG_t*)p_iostack;
         TAPI_ASSERT(sizeof(IFX_TAPI_T38_FAX_CFG_t) < TAPI_IOCTL_STACKSIZE);
         if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_T38_FAX_CFG_t)) > 0)
         {
            /* errmsg: Copy to or from user space not successful */
            ret = TAPI_statusErrKernCpy;
         }
         else
         {
            *retLL = pDrvCtx->COD.FAX_Cfg_Set (pChannel->pLLChannel, p_tmp);
         }
      }
      else
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      break;
   }
   case IFX_TAPI_T38_CFG_GET:
   {
      if (ptr_chk(pDrvCtx->COD.FAX_Cfg_Get, ""))
      {
         IFX_TAPI_T38_FAX_CFG_t *p_tmp =
            (IFX_TAPI_T38_FAX_CFG_t *)p_iostack;
         TAPI_ASSERT(sizeof(IFX_TAPI_T38_FAX_CFG_t) < TAPI_IOCTL_STACKSIZE);

         *retLL = pDrvCtx->COD.FAX_Cfg_Get (pChannel->pLLChannel, p_tmp);
         if (TAPI_SUCCESS(*retLL))
         {
            if (copy_to_user (parg, p_tmp,
                  sizeof(IFX_TAPI_T38_FAX_CFG_t)) > 0)
            {
               /* errmsg: Copy to or from user space not successful */
               ret = TAPI_statusErrKernCpy;
            }
         }
      }
      else
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      break;
   }
   case IFX_TAPI_T38_FDP_CFG_SET:
   {
      if (ptr_chk(pDrvCtx->COD.FAX_FDP_Cfg_Set, ""))
      {
         IFX_TAPI_T38_FDP_CFG_t  *p_tmp = (IFX_TAPI_T38_FDP_CFG_t*)p_iostack;
         TAPI_ASSERT(sizeof(IFX_TAPI_T38_FDP_CFG_t) < TAPI_IOCTL_STACKSIZE);

         if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_T38_FDP_CFG_t)) > 0)
         {
            /* errmsg: Copy to or from user space not successful */
            ret = TAPI_statusErrKernCpy;
         }
         else
         {
            *retLL = pDrvCtx->COD.FAX_FDP_Cfg_Set (pChannel->pLLChannel, p_tmp);
         }
      }
      else
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      break;
   }
   case IFX_TAPI_T38_FDP_CFG_GET:
   {
      if (ptr_chk(pDrvCtx->COD.FAX_FDP_Cfg_Get, ""))
      {
         IFX_TAPI_T38_FDP_CFG_t *p_tmp =
                (IFX_TAPI_T38_FDP_CFG_t *)p_iostack;
         TAPI_ASSERT(sizeof(IFX_TAPI_T38_FDP_CFG_t) < TAPI_IOCTL_STACKSIZE);

         *retLL = pDrvCtx->COD.FAX_FDP_Cfg_Get (pChannel->pLLChannel, p_tmp);
         if (TAPI_SUCCESS(*retLL))
         {
            if (copy_to_user (parg, p_tmp,
                  sizeof(IFX_TAPI_T38_FDP_CFG_t)) > 0)
            {
               /* errmsg: Copy to or from user space not successful */
               ret = TAPI_statusErrKernCpy;
            }
         }
      }
      else
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      break;
   }
   case IFX_TAPI_T38_TRACE_SET:
   {
      if (ptr_chk(pDrvCtx->COD.FAX_TraceSet, ""))
      {
         IFX_TAPI_T38_TRACE_CFG_t  *p_tmp = (IFX_TAPI_T38_TRACE_CFG_t*)p_iostack;
         TAPI_ASSERT(sizeof(IFX_TAPI_T38_TRACE_CFG_t) < TAPI_IOCTL_STACKSIZE);

         if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_T38_TRACE_CFG_t)) > 0)
         {
            /* errmsg: Copy to or from user space not successful */
            ret = TAPI_statusErrKernCpy;
         }
         else
         {
            *retLL = pDrvCtx->COD.FAX_TraceSet (pChannel->pLLChannel, p_tmp);
         }
      }
      else
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      break;
   }
#endif /* TAPI_FAX_T38_FW */

#ifdef TAPI_ANNOUNCEMENTS
   case IFX_TAPI_COD_ANNOUNCE_CFG_SET:
   {
      IFX_TAPI_COD_ANNOUNCE_CFG_t *p_tmp =
         (IFX_TAPI_COD_ANNOUNCE_CFG_t *) p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_COD_ANNOUNCE_CFG_t) < TAPI_IOCTL_STACKSIZE);

      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_COD_ANNOUNCE_CFG_t)) > 0)
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      else
      {
         ret = IFX_TAPI_Ann_Cfg (pChannel, p_tmp);
      }
      break;
   }
   case IFX_TAPI_COD_ANNOUNCE_START:
   {
      IFX_TAPI_COD_ANNOUNCE_START_t *p_tmp =
         (IFX_TAPI_COD_ANNOUNCE_START_t *) p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_COD_ANNOUNCE_START_t) < TAPI_IOCTL_STACKSIZE);

      if (copy_from_user (p_tmp, parg,
         sizeof(IFX_TAPI_COD_ANNOUNCE_START_t)) > 0)
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      else
      {
         ret = IFX_TAPI_Ann_Start (pChannel, p_tmp);
      }
      break;
   }
   case IFX_TAPI_COD_ANNOUNCE_BUFFER_FREE:
   {
      IFX_TAPI_COD_ANNOUNCE_BUFFER_FREE_t *p_tmp =
         (IFX_TAPI_COD_ANNOUNCE_BUFFER_FREE_t *) p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_COD_ANNOUNCE_BUFFER_FREE_t) <
         TAPI_IOCTL_STACKSIZE);

      if (copy_from_user (p_tmp, parg,
         sizeof(IFX_TAPI_COD_ANNOUNCE_BUFFER_FREE_t)) > 0)
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      else
      {
         ret = IFX_TAPI_Ann_Free (pChannel, p_tmp);
      }
      break;
   }
#endif /* TAPI_ANNOUNCEMENTS */

   case IFX_TAPI_PCM_CFG_SET:
   {
      IFX_TAPI_PCM_CFG_t *p_tmp = (IFX_TAPI_PCM_CFG_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_PCM_CFG_t) < TAPI_IOCTL_STACKSIZE);
      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_PCM_CFG_t)) > 0)
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      else
      {
         ret = TAPI_Phone_PCM_Set_Config(pChannel, p_tmp);
      }
      break;
   }

   case IFX_TAPI_PCM_CFG_GET:
   {
      IFX_TAPI_PCM_CFG_t *p_tmp = (IFX_TAPI_PCM_CFG_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_PCM_CFG_t) < TAPI_IOCTL_STACKSIZE);
      ret = TAPI_Phone_PCM_Get_Config(pChannel, p_tmp);

      if ((ret == TAPI_statusOk) &&
          (copy_to_user (parg, p_tmp, sizeof(IFX_TAPI_PCM_CFG_t)) > 0))
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      break;
   }

#ifdef TAPI_HDLC
   case IFX_TAPI_PCM_HDLC_CFG_SET:
   {
      IFX_TAPI_PCM_HDLC_CFG_t *p_tmp = (IFX_TAPI_PCM_HDLC_CFG_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_PCM_HDLC_CFG_t) < TAPI_IOCTL_STACKSIZE);
      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_PCM_HDLC_CFG_t)) > 0)
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      else
      {
         ret = TAPI_Phone_PCM_HDLC_Set(pChannel, p_tmp);
      }
      break;
   }
#endif /* TAPI_HDLC */

   case IFX_TAPI_PCM_LOOP_CFG_SET:
   {
      IFX_TAPI_PCM_LOOP_CFG_t *p_tmp = (IFX_TAPI_PCM_LOOP_CFG_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_PCM_LOOP_CFG_t) < TAPI_IOCTL_STACKSIZE);
      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_PCM_LOOP_CFG_t)) > 0)
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      else
      {
         ret = TAPI_Phone_PCM_Loop_Set(pChannel, p_tmp);
      }
      break;
   }

   /* Tone Services */
   case IFX_TAPI_TONE_STOP:
      ret = TAPI_Phone_Tone_Stop (pChannel,
         (IFX_int32_t)(IFX_intptr_t)parg, TAPI_TONE_DST_DEFAULT);
      break;

   case IFX_TAPI_TONE_LEVEL_SET:
   {
      IFX_TAPI_PREDEF_TONE_LEVEL_t  *p_tmp =
                                    (IFX_TAPI_PREDEF_TONE_LEVEL_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_PREDEF_TONE_LEVEL_t) < TAPI_IOCTL_STACKSIZE);
      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_PREDEF_TONE_LEVEL_t)) > 0 )
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      else
      {
         ret = TAPI_Phone_Tone_Set_Level(pChannel, p_tmp);
      }
      break;
   }

   case IFX_TAPI_LINE_HOOK_VT_SET:
   {
      IFX_TAPI_LINE_HOOK_VT_t *p_tmp = (IFX_TAPI_LINE_HOOK_VT_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_LINE_HOOK_VT_t) < TAPI_IOCTL_STACKSIZE);
      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_LINE_HOOK_VT_t)) > 0)
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      else
      {
         ret = IFX_TAPI_Dial_SetValidationTime(pChannel, p_tmp);
      }
      break;
   }

#ifdef TAPI_EXT_KEYPAD
   case IFX_TAPI_PKT_EV_GENERATE_CFG:
   {
      IFX_TAPI_PKT_EV_GENERATE_CFG_t *pEvCfg;

      /* Note : DTMF_EV_LOCAL_PLAY is defined as 0x80 so that we can OR it with
                the other values set by IFX_TAPI_PKT_EV_OOB_DTMF_SET without
                loosing that info */
      if(pChannel->nChannel > 0 )
      {
         TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
              ("TAPI:audio is not supported in this channel \n"));
         ret = TAPI_statusErr;
      }
      else
      {
         pEvCfg = (IFX_TAPI_PKT_EV_GENERATE_CFG_t*)ioarg;
         if(pEvCfg->local) /*set local play*/
         {
            pChannel->nDtmfInfo |= DTMF_EV_LOCAL_PLAY;
         }
         else
         {
            pChannel->nDtmfInfo &= ~DTMF_EV_LOCAL_PLAY;
         }

         ret = TAPI_statusOk;
      }
      break;
   }
#endif /*TAPI_EXT_KEYPAD*/

   case IFX_TAPI_PKT_EV_GENERATE :
   {
      IFX_TAPI_PKT_EV_GENERATE_t   *p_tmp =
                                    (IFX_TAPI_PKT_EV_GENERATE_t  *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_PKT_EV_GENERATE_t ) < TAPI_IOCTL_STACKSIZE);
      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_PKT_EV_GENERATE_t )) > 0 )
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      else
      {
         ret = TAPI_EVENT_PKT_EV_Generate(pChannel,p_tmp);
      }
      break;
   }

#ifdef TAPI_METERING
   case IFX_TAPI_METER_BURST:
   {
      IFX_TAPI_METER_BURST_t *p_tmp = (IFX_TAPI_METER_BURST_t *)p_iostack;
      TAPI_ASSERT (sizeof (IFX_TAPI_METER_BURST_t) < TAPI_IOCTL_STACKSIZE);
      if (copy_from_user (p_tmp, parg, sizeof (IFX_TAPI_METER_BURST_t)) > 0 )
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      else
      {
         ret = TAPI_Phone_Meter_Burst (pChannel, p_tmp);
      }
      break;
   }
#endif /* TAPI_METERING */

   case IFX_TAPI_CALIBRATION_START:
   {
      if (pDrvCtx->ALM.Calibration_Start == IFX_NULL)
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      else
      {
         *retLL = pDrvCtx->ALM.Calibration_Start (pChannel->pLLChannel);
      }
      break;
   }

   case IFX_TAPI_CALIBRATION_STOP:
   {
      if (pDrvCtx->ALM.Calibration_Stop == IFX_NULL)
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      else
      {
         *retLL = pDrvCtx->ALM.Calibration_Stop (pChannel->pLLChannel);
      }
      break;
   }

   case IFX_TAPI_CALIBRATION_CFG_SET:
   {
      IFX_TAPI_CALIBRATION_CFG_t *p_tmp =
         (IFX_TAPI_CALIBRATION_CFG_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_CALIBRATION_CFG_t) < TAPI_IOCTL_STACKSIZE);
      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_CALIBRATION_CFG_t)) > 0)
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      else
      {
         if (pDrvCtx->ALM.Calibration_Set == IFX_NULL)
         {
            /* errmsg: Service is not supported by the low level driver */
            ret = TAPI_statusLLNotSupp;
         }
         else
            *retLL = pDrvCtx->ALM.Calibration_Set (pChannel->pLLChannel,
                                                   p_tmp);
      }
      break;
   }

   case IFX_TAPI_CALIBRATION_CFG_GET:
   {
      IFX_TAPI_CALIBRATION_CFG_t *p_tmp =
         (IFX_TAPI_CALIBRATION_CFG_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_CALIBRATION_CFG_t) < TAPI_IOCTL_STACKSIZE);
      if (pDrvCtx->ALM.Calibration_Get == IFX_NULL)
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      else
      {
         *retLL = pDrvCtx->ALM.Calibration_Get (pChannel->pLLChannel,
                                                p_tmp);
         if ((*retLL == TAPI_statusOk) &&
            (copy_to_user (parg, p_tmp, sizeof(IFX_TAPI_CALIBRATION_CFG_t)) > 0))
         {
            /* errmsg: Copy to or from user space not successful */
            ret = TAPI_statusErrKernCpy;
         }
      }
      break;
   }

   case IFX_TAPI_CALIBRATION_RESULTS_GET:
   {
      IFX_TAPI_CALIBRATION_CFG_t *p_tmp =
         (IFX_TAPI_CALIBRATION_CFG_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_CALIBRATION_CFG_t) < TAPI_IOCTL_STACKSIZE);
      if (pDrvCtx->ALM.Calibration_Results_Get == IFX_NULL)
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      else
      {
         *retLL = pDrvCtx->ALM.Calibration_Results_Get (pChannel->pLLChannel,
                                                        p_tmp);
         if ((*retLL == TAPI_statusOk) &&
            (copy_to_user (parg, p_tmp, sizeof(IFX_TAPI_CALIBRATION_CFG_t)) > 0))
         {
            /* errmsg: Copy to or from user space not successful */
            ret = TAPI_statusErrKernCpy;
         }
      }
      break;
   }

   /* Lec Configuration */
   case IFX_TAPI_WLEC_PHONE_CFG_SET:
   {
      IFX_TAPI_WLEC_CFG_t *p_tmp = (IFX_TAPI_WLEC_CFG_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_WLEC_CFG_t) < TAPI_IOCTL_STACKSIZE);
      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_WLEC_CFG_t)) > 0 )
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      else
      {
         ret = TAPI_Phone_LecMode_Alm_Set (pChannel, p_tmp);
      }
      break;
   }

   case IFX_TAPI_WLEC_PHONE_CFG_GET:
   {
      IFX_TAPI_WLEC_CFG_t *p_tmp = (IFX_TAPI_WLEC_CFG_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_WLEC_CFG_t) < TAPI_IOCTL_STACKSIZE);
      ret = TAPI_Phone_LecMode_Alm_Get (pChannel, p_tmp);
      if ((ret == TAPI_statusOk) &&
          (copy_to_user (parg, p_tmp, sizeof(IFX_TAPI_WLEC_CFG_t)) > 0))
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      break;
   }

   case IFX_TAPI_WLEC_PCM_CFG_SET:
   {
      IFX_TAPI_WLEC_CFG_t *p_tmp = (IFX_TAPI_WLEC_CFG_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_WLEC_CFG_t) < TAPI_IOCTL_STACKSIZE);
      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_WLEC_CFG_t)) > 0 )
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      else
      {
         ret = TAPI_Phone_LecMode_Pcm_Set (pChannel, p_tmp);
      }
      break;
   }

   case IFX_TAPI_WLEC_PCM_CFG_GET:
   {
      IFX_TAPI_WLEC_CFG_t *p_tmp = (IFX_TAPI_WLEC_CFG_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_WLEC_CFG_t) < TAPI_IOCTL_STACKSIZE);
      ret = TAPI_Phone_LecMode_Pcm_Get (pChannel, p_tmp);
      if ((ret == TAPI_statusOk) &&
          (copy_to_user (parg, p_tmp, sizeof(IFX_TAPI_WLEC_CFG_t)) > 0))
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      break;
   }

   case IFX_TAPI_DTMF_RX_CFG_SET:
   {
      IFX_TAPI_DTMF_RX_CFG_t *p_tmp = (IFX_TAPI_DTMF_RX_CFG_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_DTMF_RX_CFG_t) < TAPI_IOCTL_STACKSIZE);

      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_DTMF_RX_CFG_t)) > 0)
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      else
      {
         ret = TAPI_Phone_Dtmf_RxCoeff_Cfg (pChannel, IFX_FALSE, p_tmp);
      }
      break;
   }

   case IFX_TAPI_DTMF_RX_CFG_GET:
   {
      IFX_TAPI_DTMF_RX_CFG_t *p_tmp = (IFX_TAPI_DTMF_RX_CFG_t*)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_DTMF_RX_CFG_t) < TAPI_IOCTL_STACKSIZE);
      memset(p_tmp, 0, sizeof(IFX_TAPI_DTMF_RX_CFG_t));

      ret = TAPI_Phone_Dtmf_RxCoeff_Cfg (pChannel, IFX_TRUE, p_tmp);

      if ((ret == TAPI_statusOk) &&
          (copy_to_user (parg, p_tmp, sizeof(IFX_TAPI_DTMF_RX_CFG_t)) > 0))
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      break;
   }

#ifdef TAPI_CID
   /* Caller ID Transmission service */
   case IFX_TAPI_CID_CFG_SET:
   {
      IFX_TAPI_CID_CFG_t *p_tmp = (IFX_TAPI_CID_CFG_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_CID_CFG_t) < TAPI_IOCTL_STACKSIZE);
      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_CID_CFG_t)) > 0 )
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      else
      {
         ret = TAPI_Phone_CID_SetConfig (pChannel, p_tmp);
      }
      break;
   }

   case IFX_TAPI_CID_TX_INFO_START:
   case IFX_TAPI_CID_TX_SEQ_START:
      if (ptr_chk(pDrvCtx->SIG.CID_TX_Start, ""))
      {
         IFX_TAPI_CID_MSG_t *p_tmp = (IFX_TAPI_CID_MSG_t *)p_iostack;
         TAPI_ASSERT(sizeof(IFX_TAPI_CID_MSG_t) < TAPI_IOCTL_STACKSIZE);
         if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_CID_MSG_t)) > 0 )
         {
            /* errmsg: Copy to or from user space not successful */
            ret = TAPI_statusErrKernCpy;
         }
         else
         {
            if (iocmd == IFX_TAPI_CID_TX_INFO_START)
               ret = TAPI_Phone_CID_Info_Tx (pChannel, p_tmp);
            else
               ret = TAPI_Phone_CID_Seq_Tx (pChannel, p_tmp);
         }
      }
      else
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      break;


   case IFX_TAPI_CID_RX_STATUS_GET:
   {
      IFX_TAPI_CID_RX_STATUS_t *p_tmp = (IFX_TAPI_CID_RX_STATUS_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_CID_RX_STATUS_t) < TAPI_IOCTL_STACKSIZE);
      ret = TAPI_Phone_CidRx_Status (pChannel, p_tmp);
      if ((ret == TAPI_statusOk) &&
          (copy_to_user (parg, p_tmp, sizeof(IFX_TAPI_CID_RX_STATUS_t)) > 0))
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      break;
   }

   case IFX_TAPI_CID_RX_DATA_GET:
   {
      IFX_TAPI_CID_RX_DATA_t *p_tmp = (IFX_TAPI_CID_RX_DATA_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_CID_RX_DATA_t) < TAPI_IOCTL_STACKSIZE);
      ret = TAPI_Phone_Get_CidRxData (pChannel, p_tmp);
      if ((ret == TAPI_statusOk) &&
          (copy_to_user (parg, p_tmp, sizeof(IFX_TAPI_CID_RX_DATA_t)) > 0))
      {
         /* errmsg: Copy to or from user space not successful */
         ret = TAPI_statusErrKernCpy;
      }
      break;
   }
#endif /* TAPI_CID */


   case IFX_TAPI_PHONE_VOLUME_SET:
   {
      IFX_TAPI_LINE_VOLUME_t *p_tmp = (IFX_TAPI_LINE_VOLUME_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_LINE_VOLUME_t) < TAPI_IOCTL_STACKSIZE);
      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_LINE_VOLUME_t)) > 0 )
      {
         ret = TAPI_statusErr;
      }
      else
      {
         if (pDrvCtx->ALM.Volume_Set == IFX_NULL)
         {
            /* errmsg: Service is not supported by the low level driver */
            ret = TAPI_statusLLNotSupp;
         }
         else
            *retLL = pDrvCtx->ALM.Volume_Set (pChannel->pLLChannel, p_tmp);
      }
      break;
   }

   case IFX_TAPI_PCM_VOLUME_SET:
   {
      IFX_TAPI_LINE_VOLUME_t *p_tmp = (IFX_TAPI_LINE_VOLUME_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_LINE_VOLUME_t) < TAPI_IOCTL_STACKSIZE);
      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_LINE_VOLUME_t)) > 0 )
      {
         ret = TAPI_statusErr;
      }
      else
      {
         if (pDrvCtx->PCM.Volume_Set == IFX_NULL)
         {
            /* errmsg: Service is not supported by the low level driver */
            ret = TAPI_statusLLNotSupp;
         }
         else
            *retLL = pDrvCtx->PCM.Volume_Set (pChannel->pLLChannel, p_tmp);
      }
      break;
   }

   case IFX_TAPI_COD_VOLUME_SET:
      if (ptr_chk(pDrvCtx->COD.Volume_Set, ""))
      {
         IFX_TAPI_PKT_VOLUME_t *p_tmp = (IFX_TAPI_PKT_VOLUME_t *)p_iostack;
         TAPI_ASSERT(sizeof(IFX_TAPI_PKT_VOLUME_t) < TAPI_IOCTL_STACKSIZE);

         if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_PKT_VOLUME_t)) > 0 )
         {
            /* errmsg: Service is not supported by the low level driver */
            ret = TAPI_statusLLNotSupp;
         }
         else
         {
            *retLL = pDrvCtx->COD.Volume_Set (pChannel->pLLChannel, p_tmp);
         }
      }
      else
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      break;

   case IFX_TAPI_SIG_DETECT_ENABLE:
   {
      IFX_TAPI_SIG_DETECTION_t *p_tmp = (IFX_TAPI_SIG_DETECTION_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_SIG_DETECTION_t) < TAPI_IOCTL_STACKSIZE);
      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_SIG_DETECTION_t)) > 0 )
      {
         ret = TAPI_statusErr;
      }
      else
      {
         if (pDrvCtx->SIG.DetectEnable == IFX_NULL)
         {
            /* errmsg: Service is not supported by the low level driver */
            ret = TAPI_statusLLNotSupp;
         }
         else
            *retLL = pDrvCtx->SIG.DetectEnable (pChannel->pLLChannel, p_tmp);
      }
      break;
   }

   case IFX_TAPI_SIG_DETECT_DISABLE:
   {
      IFX_TAPI_SIG_DETECTION_t *p_tmp = (IFX_TAPI_SIG_DETECTION_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_SIG_DETECTION_t) < TAPI_IOCTL_STACKSIZE);
      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_SIG_DETECTION_t)) > 0 )
      {
         ret = TAPI_statusErr;
      }
      else
      {
         if (pDrvCtx->SIG.DetectDisable == IFX_NULL)
         {
            /* errmsg: Service is not supported by the low level driver */
            ret = TAPI_statusLLNotSupp;
         }
         else
            *retLL = pDrvCtx->SIG.DetectDisable (pChannel->pLLChannel, p_tmp);
      }
      break;
   }

   case IFX_TAPI_TONE_CPTD_START:
   {
      IFX_TAPI_TONE_CPTD_t *p_tmp = (IFX_TAPI_TONE_CPTD_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_TONE_CPTD_t) < TAPI_IOCTL_STACKSIZE);
      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_TONE_CPTD_t)) > 0 )
      {
         ret = TAPI_statusErr;
      }
      else
      {
         ret = TAPI_Phone_DetectToneStart(pChannel, p_tmp);
      }
      break;
   }

   case IFX_TAPI_ENC_AGC_CFG:
   {
      IFX_TAPI_ENC_AGC_CFG_t *p_tmp = (IFX_TAPI_ENC_AGC_CFG_t *) p_iostack;

      TAPI_ASSERT(sizeof(IFX_TAPI_ENC_AGC_CFG_t) < TAPI_IOCTL_STACKSIZE);

      if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_ENC_AGC_CFG_t)) > 0 )
      {
         ret = TAPI_statusErr;
      }
      else
      {
         if (pDrvCtx->COD.AGC_Cfg == IFX_NULL)
         {
            /* errmsg: Service is not supported by the low level driver */
            ret = TAPI_statusLLNotSupp;
         }
         else
         {
            *retLL = pDrvCtx->COD.AGC_Cfg(pChannel->pLLChannel, p_tmp);
         }
      }
      break;
   }

   case IFX_TAPI_ENC_ROOM_NOISE_DETECT_START:
      if (ptr_chk(pDrvCtx->COD.ENC_RoomNoise, ""))
      {
         IFX_TAPI_ENC_ROOM_NOISE_DETECT_t *p_tmp =
            (IFX_TAPI_ENC_ROOM_NOISE_DETECT_t *) p_iostack;
         TAPI_ASSERT(sizeof(IFX_TAPI_ENC_ROOM_NOISE_DETECT_t) < TAPI_IOCTL_STACKSIZE);

         if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_ENC_ROOM_NOISE_DETECT_t)) > 0 )
         {
            ret = TAPI_statusErr;
         }
         else
         {
            *retLL = pDrvCtx->COD.ENC_RoomNoise(pChannel->pLLChannel, IFX_TRUE,
                                                p_tmp->nThreshold,
                                                p_tmp->nVoicePktCnt,
                                                p_tmp->nSilencePktCnt);
         }
      }
      else
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      break;


#ifdef DECT_SUPPORT
   /* DECT services */
   case IFX_TAPI_DECT_ACTIVATION_SET:
      if (ptr_chk(pDrvCtx->DECT.Enable, ""))
      {
         *retLL = pDrvCtx->DECT.Enable(pChannel->pLLChannel,
               (((IFX_operation_t)ioarg) == IFX_ENABLE) ? 1 : 0);
      }
      else
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      break;

   case IFX_TAPI_DECT_CFG_SET:
      if (ptr_chk(pDrvCtx->DECT.Ch_Cfg, ""))
      {
         IFX_TAPI_DECT_CFG_t *p_tmp = (IFX_TAPI_DECT_CFG_t *)p_iostack;
         TAPI_ASSERT(sizeof(IFX_TAPI_DECT_CFG_t) < TAPI_IOCTL_STACKSIZE);
         if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_DECT_CFG_t)) > 0 )
         {
            /* errmsg: Copy to or from user space not successful */
            ret = TAPI_statusErrKernCpy;
         }
         else
         {
            *retLL = pDrvCtx->DECT.Ch_Cfg(pChannel->pLLChannel,
                                          p_tmp->nEncDelay, p_tmp->nDecDelay);
         }
      }
      else
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      break;

   case IFX_TAPI_DECT_ENC_CFG_SET:
      if (ptr_chk(pDrvCtx->DECT.ENC_Cfg, ""))
      {
         IFX_TAPI_DECT_ENC_CFG_t *p_tmp =
            (IFX_TAPI_DECT_ENC_CFG_t *)p_iostack;
         TAPI_ASSERT(sizeof(IFX_TAPI_DECT_ENC_CFG_t) < TAPI_IOCTL_STACKSIZE);
         if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_DECT_ENC_CFG_t)) > 0 )
         {
            /* errmsg: Copy to or from user space not successful */
            ret = TAPI_statusErrKernCpy;
         }
         else
         {
            *retLL = pDrvCtx->DECT.ENC_Cfg(pChannel->pLLChannel,
                                           p_tmp->nEncType, p_tmp->nFrameLen);
         }
      }
      else
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      break;

   case IFX_TAPI_DECT_VOLUME_SET:
      if (ptr_chk(pDrvCtx->DECT.Gain_Set, ""))
      {
         IFX_TAPI_PKT_VOLUME_t *p_tmp = (IFX_TAPI_PKT_VOLUME_t *)p_iostack;
         TAPI_ASSERT(sizeof(IFX_TAPI_PKT_VOLUME_t) < TAPI_IOCTL_STACKSIZE);

         if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_PKT_VOLUME_t)) > 0 )
         {
            ret = IFX_ERROR;
         }
         else
         {
            *retLL = pDrvCtx->DECT.Gain_Set (pChannel->pLLChannel, p_tmp);
         }
      }
      else
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      break;

   case IFX_TAPI_DECT_STATISTICS_GET:
      if (ptr_chk(pDrvCtx->DECT.Statistic, ""))
      {
         IFX_TAPI_DECT_STATISTICS_t *p_tmp =
            (IFX_TAPI_DECT_STATISTICS_t *)p_iostack;
         TAPI_ASSERT(sizeof(IFX_TAPI_DECT_STATISTICS_t) < TAPI_IOCTL_STACKSIZE);

         if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_DECT_STATISTICS_t)) > 0 )
         {
            /* errmsg: Copy to or from user space not successful */
            ret = TAPI_statusErrKernCpy;
         }
         else
         {
            *retLL = pDrvCtx->DECT.Statistic(pChannel->pLLChannel, p_tmp);
            if ((ret == TAPI_statusOk) &&
                (copy_to_user (parg, p_tmp, sizeof(IFX_TAPI_DECT_STATISTICS_t)) > 0))
            {
               /* errmsg: Copy to or from user space not successful */
               ret = TAPI_statusErrKernCpy;
            }
         }
      }
      else
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      break;

   case IFX_TAPI_DECT_EC_CFG_SET:
      {
         IFX_TAPI_DECT_EC_CFG_t *p_tmp = (IFX_TAPI_DECT_EC_CFG_t *)p_iostack;
         TAPI_ASSERT(sizeof(IFX_TAPI_DECT_EC_CFG_t) < TAPI_IOCTL_STACKSIZE);
         if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_DECT_EC_CFG_t)) > 0 )
         {
            /* errmsg: Copy to or from user space not successful */
            ret = TAPI_statusErrKernCpy;
         }
         else
         {
            if (ptr_chk(pDrvCtx->DECT.EC_Cfg, ""))
            {
               *retLL = pDrvCtx->DECT.EC_Cfg(pChannel->pLLChannel, p_tmp);
            }
            else
            {
               /* errmsg: Service is not supported by the low level driver */
               ret = TAPI_statusLLNotSupp;
            }
         }
      }
      break;
#endif /* DECT_SUPPORT */

#ifdef KPI_SUPPORT
   /* Kernel Packet Interface configuration */
   case IFX_TAPI_KPI_CH_CFG_SET:
      {
         IFX_TAPI_KPI_CH_CFG_t *p_tmp = (IFX_TAPI_KPI_CH_CFG_t *) p_iostack;
         TAPI_ASSERT(sizeof(IFX_TAPI_KPI_CH_CFG_t) < TAPI_IOCTL_STACKSIZE);

         if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_KPI_CH_CFG_t)) > 0 )
         {
            ret = TAPI_statusErr;
         }
         else
         {
            ret = IFX_TAPI_KPI_ChCfgSet (pChannel, p_tmp);
         }
      }
      break;

   case IFX_TAPI_KPI_GRP_CFG_SET:
      {
         IFX_TAPI_KPI_GRP_CFG_t *p_tmp = (IFX_TAPI_KPI_GRP_CFG_t *) p_iostack;
         TAPI_ASSERT(sizeof(IFX_TAPI_KPI_GRP_CFG_t) < TAPI_IOCTL_STACKSIZE);

         if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_KPI_GRP_CFG_t)) > 0 )
         {
            ret = TAPI_statusErr;
         }
         else
         {
            ret = IFX_TAPI_KPI_GrpCfgSet (p_tmp);
         }
      }
      break;
#endif /* KPI_SUPPORT */

#ifdef TAPI_GR909
   case IFX_TAPI_GR909_START:
      {
         if (ptr_chk(pDrvCtx->ALM.GR909_Start, ""))
         {
            IFX_TAPI_GR909_START_t *p_tmp = (IFX_TAPI_GR909_START_t *)p_iostack;
            TAPI_ASSERT(sizeof(IFX_TAPI_GR909_START_t) < TAPI_IOCTL_STACKSIZE);
            if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_GR909_START_t)) > 0)
            {
               /* errmsg: Copy to or from user space not successful */
               ret = TAPI_statusErrKernCpy;
            }
            else
            {
               *retLL = pDrvCtx->ALM.GR909_Start(pChannel->pLLChannel, p_tmp);
            }
         }
         else
         {
            /* errmsg: Service is not supported by the low level driver */
            ret = TAPI_statusLLNotSupp;
         }
      }
      break;
   case IFX_TAPI_GR909_STOP:
      {
         if (ptr_chk(pDrvCtx->ALM.GR909_Stop, ""))
         {
            *retLL = pDrvCtx->ALM.GR909_Stop(pChannel->pLLChannel);
         }
         else
         {
            /* errmsg: Service is not supported by the low level driver */
            ret = TAPI_statusLLNotSupp;
         }
      }
      break;
   case IFX_TAPI_GR909_RESULT:
      {
         if (ptr_chk(pDrvCtx->ALM.GR909_Result, ""))
         {
            IFX_TAPI_GR909_RESULT_t *p_tmp = (IFX_TAPI_GR909_RESULT_t *)p_iostack;
            TAPI_ASSERT(sizeof(IFX_TAPI_GR909_RESULT_t) < TAPI_IOCTL_STACKSIZE);
            *retLL = pDrvCtx->ALM.GR909_Result (pChannel->pLLChannel, p_tmp);
            if (TAPI_SUCCESS(*retLL))
            {
               if (copy_to_user (parg, p_tmp,
                  sizeof(IFX_TAPI_GR909_RESULT_t)) > 0)
               {
                  /* errmsg: Copy to or from user space not successful */
                  ret = TAPI_statusErrKernCpy;
               }
            }
         }
         else
         {
            /* errmsg: Service is not supported by the low level driver */
            ret = TAPI_statusLLNotSupp;
         }
      }
      break;
#endif /* TAPI_GR909 */

#ifdef TAPI_CONT_MEASUREMENT
   case IFX_TAPI_CONTMEASUREMENT_GET:
      {
         if (ptr_chk(pDrvCtx->ALM.ContMeasGet, ""))
         {
            IFX_TAPI_CONTMEASUREMENT_GET_t *p_tmp =
                                    (IFX_TAPI_CONTMEASUREMENT_GET_t *)p_iostack;

            TAPI_ASSERT(sizeof(IFX_TAPI_CONTMEASUREMENT_GET_t) <
                                                          TAPI_IOCTL_STACKSIZE);
            *retLL = pDrvCtx->ALM.ContMeasGet (pChannel->pLLChannel, p_tmp);
            if (TAPI_SUCCESS(*retLL))
            {
               if (copy_to_user (parg, p_tmp,
                  sizeof(IFX_TAPI_CONTMEASUREMENT_GET_t)) > 0)
               {
                  /* errmsg: Copy to or from user space not successful */
                  ret = TAPI_statusErrKernCpy;
               }
            }
         }
         else
         {
            /* errmsg: Service is not supported by the low level driver */
            ret = TAPI_statusLLNotSupp;
         }
      }
      break;
#endif /* TAPI_CONT_MEASUREMENT */

   /* TEST services */
   case IFX_TAPI_TEST_LOOP:
      if (ptr_chk(pDrvCtx->ALM.TestLoop, ""))
      {
         IFX_TAPI_TEST_LOOP_t *p_tmp = (IFX_TAPI_TEST_LOOP_t *)p_iostack;
         TAPI_ASSERT(sizeof(IFX_TAPI_TEST_LOOP_t) < TAPI_IOCTL_STACKSIZE);

         if (copy_from_user (p_tmp, parg, sizeof(IFX_TAPI_TEST_LOOP_t)) > 0)
         {
            /* errmsg: Copy to or from user space not successful */
            ret = TAPI_statusErrKernCpy;
         }
         else
         {
            *retLL = pDrvCtx->ALM.TestLoop (pChannel->pLLChannel, p_tmp);
         }
      }
      else
      {
         /* errmsg: Service is not supported by the low level driver */
         ret = TAPI_statusLLNotSupp;
      }
      break;

   /* to be completed... */
#if 0
   case IFX_TAPI_POLL_CONFIG:
   {
      IFX_TAPI_POLL_CONFIG_t *pPollCfg = (IFX_TAPI_POLL_CONFIG_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_POLL_CONFIG_t) < TAPI_IOCTL_STACKSIZE);

      if (copy_from_user (pPollCfg, parg, sizeof(IFX_TAPI_POLL_CONFIG_t)) > 0 )
      {
         ret = TAPI_statusErr;
      }
      else
      {
         ret = TAPI_IrqPollConf(pPollCfg);
      }
      break;
   }

   case IFX_TAPI_POLL_ADD:
   {
      ret = TAPI_AddDev_PollPkts(pDrvCtx, pTapiDev);
      ret = TAPI_AddDev_PollEvts(pDrvCtx, pTapiDev);
      break;
   }

   case IFX_TAPI_POLL_REM:
   {
      ret = TAPI_RemDev_PollPkts(pDrvCtx, pTapiDev);
      ret = TAPI_RemDev_PollEvts(pDrvCtx, pTapiDev);
      break;
   }

   case IFX_TAPI_POLL_TEST:
   {
      ret = TAPI_Poll_Test();
      break;
   }

   case IFX_TAPI_POLL_WRITE:
   {
      IFX_TAPI_POLL_DATA_t *pPollDown = (IFX_TAPI_POLL_DATA_t *)p_iostack;
      TAPI_ASSERT(sizeof(IFX_TAPI_POLL_DATA_t) < TAPI_IOCTL_STACKSIZE);

      if (copy_from_user (pPollDown, parg, sizeof(IFX_TAPI_POLL_DATA_t)) > 0 )
      {
         ret = TAPI_statusErr;
      }
      else
      {
         IFX_void_t *pPktsArray[IFX_TAPI_POLL_QUEUE];
         IFX_void_t **ppPkts;
         IFX_int32_t pPktsNum = pPollDown->pPktsNum;

         ppPkts = pPollDown->ppPkts;
         for (i = 0; i < pPktsNum;)
         {
            /* skip any NULL pointers in the array */
            while (*ppPkts == NULL)
            {
               *ppPkts++;
            }
            pPkt = *ppPkts++;

            if (copy_from_user (pPollDown, parg,
                                sizeof(IFX_TAPI_POLL_DATA_t)) > 0 )
            {
               ret = TAPI_statusErr;
            }
         }
         ret = TAPI_Poll_Down (pPollDown->ppPkts, &pPollDown->pPktsNum);
      }
      break;
   }

   case IFX_TAPI_POLL_READ:
      /* to be completed... */
      ret = TAPI_Poll_Up (pPollDown->ppPkts, &pPollDown->pPktsNum);
      break;
#endif

   default:
      ret = TAPI_statusInvalidIoctl;
      break;
   }

   TAPI_OS_Free (p_iostack);

   return ret;
}


#ifdef CONFIG_PROC_FS
/**
   Read the version information from the driver.

   \param  buf          Destination buffer.

   \return
   Number of bytes written into the buffer.
*/
static IFX_int32_t proc_get_tapi_version(IFX_char_t *buf)
{
    IFX_int32_t len;

    len = sprintf(buf, "%s\n", &TAPI_WHATVERSION[4]);
    len += sprintf(buf + len, "Compiled on %s, %s for Linux kernel %s\n",
                   __DATE__, __TIME__, UTS_RELEASE);

    return len;
}


#ifdef HAVE_CONFIG_H
/**
   Read the configure parameters of the driver.

   \param  buf    Destination buffer.

   \return
   Length of the string returned in buf.
*/
static IFX_int32_t proc_ConfigureGet (IFX_char_t *buf)
{
   IFX_int32_t len;

   len  = sprintf(buf, "configure %s\n", &DRV_TAPI_WHICHCONFIG[0]);

   return len;
}
#endif /* HAVE_CONFIG_H */


/**
   Read the status information from the driver.

   \param  buf          Destination buffer.

   \return
   Number of bytes written into the buffer.
*/
static IFX_int32_t proc_get_tapi_status(IFX_char_t *buf)
{
   IFX_TAPI_DRV_CTX_t *pDrvCtx  = IFX_NULL;
   TAPI_DEV           *pTapiDev = IFX_NULL;
   IFX_int32_t len=0;
   IFX_int32_t i, j;

   for (i = 0; i < TAPI_MAX_LL_DRIVERS; i++)
   {
      pDrvCtx = gHLDrvCtx[i].pDrvCtx;

      if (pDrvCtx != IFX_NULL)
      {
         len += sprintf(buf+len,
            "-- %s --\n", pDrvCtx->drvName);

         if (pDrvCtx->pTapiDev == IFX_NULL)
         {
            /* driver not initialised yet -> skip it */
            break;
         }

         for (j = 0; j < pDrvCtx->maxDevs; j++)
         {
            pTapiDev = &(pDrvCtx->pTapiDev[j]);
            len += sprintf(buf+len,
                   "  Device #%d: Initialized = %d\n",
                    j,pTapiDev->bInitialized);
         }
      }
   }
   return len;
}


/**
   Read the registered low level driver information

   \param  buf          Destination buffer.

   \return
   Number of bytes written into the buffer.
*/
static IFX_int32_t proc_get_tapi_registered_drivers(IFX_char_t *buf)
{
   IFX_TAPI_DRV_CTX_t *pDrvCtx = IFX_NULL;
   IFX_int32_t i, len = 0;

   len += sprintf(buf+len,
       "Driver \t  version \t major \t devices devname \n");
   len += sprintf(buf+len,
      "==================================================\n");

   for (i = 0; i < TAPI_MAX_LL_DRIVERS; i++)
   {
      pDrvCtx = gHLDrvCtx[i].pDrvCtx;

      if (pDrvCtx != IFX_NULL)
      {
         len += sprintf(buf+len, "%s    %s \t %d \t %d \t /dev/%s\n",
              pDrvCtx->drvName, pDrvCtx->drvVersion,
              pDrvCtx->majorNumber,pDrvCtx->maxDevs, pDrvCtx->devNodeName);
      }
   }

   return len;
}


/**
   Read the current bufferpool status

   \param  buf          Destination buffer.

   \return
   Number of bytes written into the buffer.
*/
static IFX_int32_t proc_read_bufferpool (IFX_char_t *buf)
{
   IFX_int32_t len = 0;

#ifdef KPI_SUPPORT
   len += sprintf(buf+len,
                  "TAPIpool (free/total)  voice (%4d/%4d), "
                                         "kpi (%4d/%4d), "
                                         "evt (%4d/%4d)\n",
                  IFX_TAPI_VoiceBufferPool_ElementAvailCountGet(),
                  IFX_TAPI_VoiceBufferPool_ElementCountGet(),
                  IFX_TAPI_KPI_BufferPool_ElementAvailCountGet(),
                  IFX_TAPI_KPI_VoiceBufferPool_ElementCountGet(),
                  IFX_TAPI_EventWrpBufferPool_ElementAvailCountGet(),
                  IFX_TAPI_EventWrpBufferPool_ElementCountGet());
#else
   len += sprintf(buf+len,
                  "TAPIpool (free/total)  voice (%4d/%4d), "
                                         "evt (%4d/%4d)\n",
                  IFX_TAPI_VoiceBufferPool_ElementAvailCountGet(),
                  IFX_TAPI_VoiceBufferPool_ElementCountGet(),
                  IFX_TAPI_EventWrpBufferPool_ElementAvailCountGet(),
                  IFX_TAPI_EventWrpBufferPool_ElementCountGet());
#endif /* KPI_SUPPORT */

   return len;
}

#ifdef TAPI_PACKET_OWNID
/**
   Print out to the console voice buffer information

   \param  buf          Destination buffer.

   \return
      Always zero.

   \remark
      That function do not using buffer because output too large for
      **non-circled** buffer with **unknown** size.
*/
static IFX_int32_t proc_read_voice_bufferpool (IFX_char_t *buf)
{
   IFX_TAPI_VoiceBufferPoolStatusShow();

   return 0;
}
#endif /* TAPI_PACKET_OWNID */

#ifdef TAPI_STATISTICS
/**
   Read the statistic information from the driver.

   \param   buf     Buffer to fill in the data.
   \param   start   Returns the increment to the offset for the next call.
   \param   offset  Offset where to begin output.
   \param   count   Number of characters in buffer.
   \param   eof     Return if eof was encountered.
   \param   data    Unused.
   \return  count   Number of returned characters.
*/
static int proc_read_tapi_stats(char *buf, char **start, off_t offset,
                                int count, int *eof, void *data)
{
   IFX_TAPI_DRV_CTX_t *pDrvCtx  = IFX_NULL;
   TAPI_DEV           *pTapiDev = IFX_NULL;
   TAPI_CHANNEL       *pChannel = IFX_NULL;
   IFX_uint16_t       i, j, k;
   int                len=0;

   /* Check all slots for registered drivers */
   for (i = 0; i < TAPI_MAX_LL_DRIVERS; i++)
   {
      pDrvCtx = gHLDrvCtx[i].pDrvCtx;

      if (pDrvCtx != IFX_NULL)
      {
         if (pDrvCtx->pTapiDev == IFX_NULL)
         {
            /* driver not initialised yet -> skip it */
            break;
         }

         /* Loop over all devices of the registered driver */
         for (j = 0; j < pDrvCtx->maxDevs; j++)
         {
            pTapiDev = &(pDrvCtx->pTapiDev[j]);

            if (pTapiDev->bInitialized)
            {
               if (offset != 0)
               {
                  offset--;
               }
               else
               {
                  /* Headline */
                  len += sprintf(buf+len,
                     "-- %s #%d --\n", pDrvCtx->drvName, j);
                  len += sprintf(buf+len, "CH STRM "
                     "TX:     ok  discarded  congested "
                     "RX:     ok  discarded  congested\n");

                  /* Loop over all TAPI channels of this device */
                  for (k=0; k < pDrvCtx->maxChannels; k++)
                  {
                     pChannel = &pTapiDev->pTapiChanelArray[k];

                     if (k < pChannel->pTapiDevice->nResource.CodCount)
                     {
                        len += sprintf(buf+len, "%2d COD  "
                           "%10u %10u %10u "
                           "%10u %10u %10u\n", k,
                           IFX_TAPI_Stat_Get(pChannel, IFX_TAPI_STREAM_COD,
                                             TAPI_STAT_COUNTER_EGRESS_DELIVERED),
                           IFX_TAPI_Stat_Get(pChannel, IFX_TAPI_STREAM_COD,
                                             TAPI_STAT_COUNTER_EGRESS_DISCARDED),
                           IFX_TAPI_Stat_Get(pChannel, IFX_TAPI_STREAM_COD,
                                             TAPI_STAT_COUNTER_EGRESS_CONGESTED),
                           IFX_TAPI_Stat_Get(pChannel, IFX_TAPI_STREAM_COD,
                                             TAPI_STAT_COUNTER_INGRESS_DELIVERED),
                           IFX_TAPI_Stat_Get(pChannel, IFX_TAPI_STREAM_COD,
                                             TAPI_STAT_COUNTER_INGRESS_DISCARDED),
                           IFX_TAPI_Stat_Get(pChannel, IFX_TAPI_STREAM_COD,
                                             TAPI_STAT_COUNTER_INGRESS_CONGESTED));
                     }

                     if (k < pChannel->pTapiDevice->nResource.DectCount)
                     {
                        len += sprintf(buf+len, "%2d DECT "
                           "%10u %10u %10u "
                           "%10u %10u %10u\n", k,
                           IFX_TAPI_Stat_Get(pChannel, IFX_TAPI_STREAM_DECT,
                                             TAPI_STAT_COUNTER_EGRESS_DELIVERED),
                           IFX_TAPI_Stat_Get(pChannel, IFX_TAPI_STREAM_DECT,
                                             TAPI_STAT_COUNTER_EGRESS_DISCARDED),
                           IFX_TAPI_Stat_Get(pChannel, IFX_TAPI_STREAM_DECT,
                                             TAPI_STAT_COUNTER_EGRESS_CONGESTED),
                           IFX_TAPI_Stat_Get(pChannel, IFX_TAPI_STREAM_DECT,
                                             TAPI_STAT_COUNTER_INGRESS_DELIVERED),
                           IFX_TAPI_Stat_Get(pChannel, IFX_TAPI_STREAM_DECT,
                                             TAPI_STAT_COUNTER_INGRESS_DISCARDED),
                           IFX_TAPI_Stat_Get(pChannel, IFX_TAPI_STREAM_DECT,
                                             TAPI_STAT_COUNTER_INGRESS_CONGESTED));
                     }

#ifdef TAPI_HDLC
                     if (k < pChannel->pTapiDevice->nResource.HdlcCount)
                     {
                        len += sprintf(buf+len, "%2d HDLC "
                           "%10u %10u %10u "
                           "%10u %10u %10u\n", k,
                           IFX_TAPI_Stat_Get(pChannel, IFX_TAPI_STREAM_HDLC,
                                             TAPI_STAT_COUNTER_EGRESS_DELIVERED),
                           IFX_TAPI_Stat_Get(pChannel, IFX_TAPI_STREAM_HDLC,
                                             TAPI_STAT_COUNTER_EGRESS_DISCARDED),
                           IFX_TAPI_Stat_Get(pChannel, IFX_TAPI_STREAM_HDLC,
                                             TAPI_STAT_COUNTER_EGRESS_CONGESTED),
                           IFX_TAPI_Stat_Get(pChannel, IFX_TAPI_STREAM_HDLC,
                                             TAPI_STAT_COUNTER_INGRESS_DELIVERED),
                           IFX_TAPI_Stat_Get(pChannel, IFX_TAPI_STREAM_HDLC,
                                             TAPI_STAT_COUNTER_INGRESS_DISCARDED),
                           IFX_TAPI_Stat_Get(pChannel, IFX_TAPI_STREAM_HDLC,
                                             TAPI_STAT_COUNTER_INGRESS_CONGESTED));
                     }
#endif /* TAPI_HDLC */
                  }

                  *start = (char *)1;
                  *eof = 1;
                  return len;
               }
            }
         }
      }
   }

   *eof = 0;

   return len;
}


/**
   Reset all the packet path statistic information of the driver.

   \param   file    file structure for proc file
   \param   buffer  buffer holding the data
   \param   count   number of characters in buffer
   \param   data    unused
   \return  count   Number of processed characters
*/
static int proc_write_tapi_stats(struct file *file, const char *buffer,
                                 unsigned long count, void *data)
{
   IFX_TAPI_DRV_CTX_t *pDrvCtx  = IFX_NULL;
   TAPI_DEV           *pTapiDev = IFX_NULL;
   TAPI_CHANNEL       *pChannel = IFX_NULL;
   IFX_uint16_t       i, j, k;

   for (i = 0; i < TAPI_MAX_LL_DRIVERS; i++)
   {
      pDrvCtx = gHLDrvCtx[i].pDrvCtx;

      if (pDrvCtx != IFX_NULL)
      {
         /* Loop over all devices of the registered driver */
         for (j = 0; j < pDrvCtx->maxDevs; j++)
         {
            if (pDrvCtx->pTapiDev == IFX_NULL)
            {
               /* driver not initialised yet -> skip it */
               break;
            }

            pTapiDev = &(pDrvCtx->pTapiDev[j]);

            if (pTapiDev->bInitialized)
            {
               /* Loop over all TAPI channels of this device */
               for (k=0; k < pDrvCtx->maxChannels; k++)
               {
                  pChannel = &pTapiDev->pTapiChanelArray[k];
                  IFX_TAPI_Stat_Reset(pChannel);
               }
            }
         }
      }
   }

   return count;
}
#endif /* TAPI_STATISTICS */


/**
   The proc filesystem: function to read an entry.
   This function provides information of proc files to the user space

   \param   page    Buffer to fill in the data.
   \param   start   Returns the increment to the offset for the next call.
   \param   off     Offset where to begin output.
   \param   count   Number of characters in buffer.
   \param   eof     Return if eof was encountered.
   \param   data    Unused.
   \return  Number of returned characters.
*/
static int proc_read_tapi(char *page, char **start,
                          off_t off, int count, int *eof, void *data)
{
    int len;

    IFX_int32_t (*fn)(IFX_char_t *buf);

    /* write data into the page */
    if (data != NULL)
    {
        fn = data;
        /* FIXME: buffer have to be circled!!! */
        len = (int)fn((IFX_char_t *)page);
    }
    else
        return 0;

    if (len <= off+count)
        *eof = 1;
    *start = page + off;
    len -= off;
    if (len>count)
        len = count;
    if (len<0)
        len = 0;
    /* return the data length  */
    return len;
}


/**
   Initialize and install the proc entry

   \return
   -1 or 0 on success
*/
/*lint -save -e611
\todo: FIXME
   (ANSI X3.159-1989)
   It is invalid to convert a function pointer to an object pointer
   or a pointer to void, or vice-versa.
*/
static IFX_int32_t proc_EntriesInstall(IFX_void_t)
{
   struct proc_dir_entry *driver_proc_node,
                         *proc_file;

   /* install the proc entry */
   TRACE(TAPI_DRV, DBG_LEVEL_LOW, ("TAPI_DRV: using proc fs\n"));
   driver_proc_node = proc_mkdir( "driver/" DRV_TAPI_NAME, IFX_NULL);
   if (driver_proc_node != IFX_NULL)
   {
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,30))
      driver_proc_node->owner = THIS_MODULE;
#endif /* < Linux 2.6.30 */

      proc_file = create_proc_read_entry( "version" , S_IFREG|S_IRUGO,
                                         driver_proc_node, proc_read_tapi,
                                         (void *)proc_get_tapi_version);
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,30))
      if (proc_file != NULL)
         proc_file->owner = THIS_MODULE;
#endif /* < Linux 2.6.30 */

      proc_file = create_proc_read_entry( "status" , S_IFREG|S_IRUGO,
                                         driver_proc_node, proc_read_tapi,
                                         (void *)proc_get_tapi_status);
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,30))
      if (proc_file != NULL)
         proc_file->owner = THIS_MODULE;
#endif /* < Linux 2.6.30 */

      proc_file = create_proc_read_entry( "registered_drivers" , S_IFREG|S_IRUGO,
                                         driver_proc_node, proc_read_tapi,
                                         (void *)proc_get_tapi_registered_drivers);
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,30))
      if (proc_file != NULL)
         proc_file->owner = THIS_MODULE;
#endif /* < Linux 2.6.30 */

      proc_file = create_proc_read_entry( "bufferpool", S_IFREG|S_IRUGO,
                                         driver_proc_node, proc_read_tapi,
                                         (void *) proc_read_bufferpool);
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,30))
      if (proc_file != NULL)
         proc_file->owner = THIS_MODULE;
#endif /* < Linux 2.6.30 */

#ifdef TAPI_PACKET_OWNID
      proc_file = create_proc_read_entry( "voice_buffers", S_IFREG|S_IRUGO,
                                         driver_proc_node, proc_read_tapi,
                                         (void *) proc_read_voice_bufferpool);
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,30))
      if (proc_file != NULL)
         proc_file->owner = THIS_MODULE;
#endif /* < Linux 2.6.30 */
#endif /* TAPI_PACKET_OWNID */

#ifdef TAPI_STATISTICS
      {
         struct proc_dir_entry *proc_stat_node;

         if ((proc_stat_node =
                 create_proc_entry ("statistic", 0, driver_proc_node)))
         {
            proc_stat_node->read_proc  = proc_read_tapi_stats;
            proc_stat_node->write_proc = proc_write_tapi_stats;
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,30))
            proc_stat_node->owner = THIS_MODULE;
#endif /* < Linux 2.6.30 */
         }
      }
#endif /* TAPI_STATISTICS */
#ifdef HAVE_CONFIG_H
      proc_file = create_proc_read_entry("configure", S_IFREG|S_IRUGO,
                                         driver_proc_node, proc_read_tapi,
                                         (void *)proc_ConfigureGet);
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,30))
      if (proc_file != NULL)
         proc_file->owner = THIS_MODULE;
#endif /* < Linux 2.6.30 */
#endif /* HAVE_CONFIG_H */
   }
   else
   {
      TRACE(TAPI_DRV,DBG_LEVEL_HIGH,("TAPI_DRV: cannot create proc entry\n"));
      return -1;
   }

   return 0;
}
/*lint -restore */

/**
   Remove proc filesystem entries.

   \return
   None.

   \remark
   Called by the kernel.
*/
static IFX_void_t proc_EntriesRemove(IFX_void_t)
{
#ifdef HAVE_CONFIG_H
   remove_proc_entry("driver/" DRV_TAPI_NAME "/configure",0);
#endif /* HAVE_CONFIG_H */
#ifdef TAPI_STATISTICS
   remove_proc_entry("driver/" DRV_TAPI_NAME "/statistic" ,0);
#endif /* TAPI_STATISTICS */
   remove_proc_entry("driver/" DRV_TAPI_NAME "/version" ,0);
   remove_proc_entry("driver/" DRV_TAPI_NAME "/status",0);
   remove_proc_entry("driver/" DRV_TAPI_NAME "/registered_drivers",0);
   remove_proc_entry("driver/" DRV_TAPI_NAME "/bufferpool",0);
#ifdef TAPI_PACKET_OWNID
   remove_proc_entry("driver/" DRV_TAPI_NAME "/voice_buffers",0);
#endif /* TAPI_PACKET_OWNID */
   remove_proc_entry("driver/" DRV_TAPI_NAME ,0);

   return;
}
#endif /* CONFIG_PROC_FS */


#ifdef LINUX_2_6
/**
   Set the scheduling policy for the TAPIevent workqueue to RT scheduling

   \param  foo          unused.
*/
static IFX_void_t tapi_wq_setscheduler (IFX_int32_t foo)
{
   struct sched_param sched_params;

   sched_params.sched_priority = TAPI_OS_THREAD_PRIO_HIGH;
   sched_setscheduler(current, SCHED_FIFO, &sched_params);
}
#endif /* LINUX_2_6 */


/**
   Initialize the module.

   \return
   Error code or 0 on success

   \remark
   Called by the kernel.
*/

static int __init ifx_tapi_module_init(void)
{
   /* copyrigth trace shall not be prefixed with KERN_INFO */
   printk("%s, (c) 2001-2010 Lantiq Deutschland GmbH\n", &TAPI_WHATVERSION[4]);

   if (IFX_TAPI_Driver_Start() != IFX_SUCCESS)
   {
      printk (KERN_ERR "Driver start failed\n");
      return -1;
   }

#ifdef CONFIG_PROC_FS
   proc_EntriesInstall();
#endif

#ifdef LINUX_2_6
   pTapiWq = create_workqueue("TAPIevents");
   TAPI_DeferWork((IFX_void_t *) tapi_wq_setscheduler,
                  (IFX_void_t *)&tapi_wq_setscheduler_param);
#endif /* LINUX_2_6 */

   return 0;
}


/**
   Clean up the module.

   \remark
   Called by the kernel.
*/
static void __exit ifx_tapi_module_exit(void)
{
   printk (KERN_INFO "removing TAPI\n");

#ifdef CONFIG_PROC_FS
   proc_EntriesRemove();
#endif /* CONFIG_PROC_FS */

#ifdef LINUX_2_6
   /* as we are using work queues to schedule events from the interrupt
      context to the process context, we use work queues in case of
      Linux 2.6. they must be flushed on driver unload... */
   flush_scheduled_work();
   destroy_workqueue(pTapiWq);
#endif /* LINUX_2_6 */

   IFX_TAPI_Driver_Stop();

   TRACE(TAPI_DRV, DBG_LEVEL_NORMAL, ("TAPI_DRV: cleanup successful\n"));
}


/**
   Function create a timer.

   \param  pTimerEntry  Functionpointer to the call back function.
   \param  nArgument    Pointer to TAPI channel structure.

   \return
   Timer_ID    - pointer to internal timer structure

   \remark
   Initialize a task queue which will be scheduled once a timer interrupt occurs
   to execute the appropriate operation in a process context, process in which
   semaphores ... are allowed.
   Please notice that this task has to run under the keventd process, in which it
   can be executed thousands of times within a single timer tick.
*/
Timer_ID TAPI_Create_Timer(TIMER_ENTRY pTimerEntry, IFX_ulong_t nArgument)
{
   Timer_ID pTimerData;

   /* allocate memory for the channel */
   pTimerData = kmalloc(sizeof(*pTimerData), GFP_KERNEL);
   if (pTimerData == IFX_NULL)
      return IFX_NULL;

   /* set function to be called after timer expires */
   pTimerData->pTimerEntry = pTimerEntry;
   pTimerData->nArgument = nArgument;
   pTimerData->bStopped = IFX_FALSE;

   init_timer(&(pTimerData->Timer_List));

   /* set timer call back function */
   pTimerData->Timer_List.function = TAPI_timer_call_back;
   pTimerData->Timer_List.data = (IFX_ulong_t) pTimerData;

   /* Initialize Timer Task */
#ifdef LINUX_2_6
   #if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,20))
   INIT_WORK(&(pTimerData->timerTask), TAPI_tqueue, (IFX_void_t *) pTimerData);
   #else
   INIT_WORK(&(pTimerData->timerTask), TAPI_tqueue);
   #endif
#else
   /* initialize tq_struct member of Timer_ID structure */
   memset(&(pTimerData->timerTask), 0, sizeof(struct tq_struct));
   pTimerData->timerTask.routine = TAPI_tqueue;
   pTimerData->timerTask.data = (IFX_void_t *) pTimerData;
#endif /* LINUX_2_6 */
   return pTimerData;
}


/**
   Function set and starts a timer with a specific time. It can be choose if the
   timer starts periodically.

   \param  Timer_ID     Pointer to internal timer structure.
   \param  nTime        Time in ms.
   \param  bPeriodically Starts the timer periodically or not.
   \param  bRestart     Restart the timer or normal start.

   \return
   Returns an error code: IFX_TRUE / IFX_FALSE
*/
IFX_boolean_t TAPI_SetTime_Timer(Timer_ID Timer, IFX_uint32_t nTime,
                                 IFX_boolean_t bPeriodically,
                                 IFX_boolean_t bRestart)
{
   IFX_UNUSED (bRestart);

   if (Timer == 0)
   {
     TRACE(TAPI_DRV,DBG_LEVEL_HIGH,
          ("\nDRV_ERROR: TAPI_SetTime_Timer (Timer == 0)\n"));
     return IFX_FALSE;
   }
   Timer->Periodical_Time = HZ*nTime/1000;
   /* prevent restart of driver */
   Timer->bPeriodical = IFX_FALSE;
   /* remove driver from list */
   del_timer_sync(&(Timer->Timer_List));

   Timer->bPeriodical = bPeriodically;

   Timer->Timer_List.expires = jiffies + Timer->Periodical_Time;
   Timer->bStopped = IFX_FALSE;
   add_timer(&(Timer->Timer_List));

   return IFX_TRUE;
}


/**
   Function stop a timer.

   \param  Timer_ID     Pointer to internal timer structure.

   \return
   Returns an error code: IFX_TRUE / IFX_FALSE
*/
IFX_boolean_t TAPI_Stop_Timer(Timer_ID Timer)
{
   /* stop timer */
   Timer->bStopped = IFX_TRUE;
   /* prevent restart of driver */
   Timer->bPeriodical = IFX_FALSE;
   /* remove driver from list */
   del_timer_sync(&(Timer->Timer_List));
   return (IFX_TRUE);
}


/**
   Function delete a timer.

   \param  Timer_ID     Pointer to internal timer structure.

   \return
   Returns an error code: IFX_TRUE / IFX_FALSE
*/
IFX_boolean_t TAPI_Delete_Timer(Timer_ID Timer)
{
   if (Timer)
   {
      TAPI_Stop_Timer(Timer);
      /* free memory */
      kfree(Timer);
      return IFX_TRUE;
   }
   return IFX_FALSE;
}


/**
   Helper function to get a periodical timer.

   \param  pWork        Pointer to corresponding timer ID.

   \remark
   This function will be executed in  the process context, so to avoid
   scheduling in Interrupt Mode while working with semaphores etc...
   The task is always running under the keventd process and is also running
   very quickly. Even on a very heavily loaded system, the latency in the
   scheduler queue is quite small
*/
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,20))
static IFX_void_t TAPI_tqueue (IFX_void_t *pWork)
#else /* for Kernel newer or equal 2.6.20 */
static IFX_void_t TAPI_tqueue (struct work_struct *pWork)
#endif
{
   Timer_ID Timer = (Timer_ID) pWork;

   if (!Timer->bStopped)
   {
      /* Call TAPI Timer function */
      Timer->pTimerEntry(Timer, Timer->nArgument);
      if (Timer->bPeriodical)
      {
         /* remove driver from list */
         del_timer_sync(&(Timer->Timer_List));
         /* start new timer, then call function to gain precision */
         Timer->Timer_List.expires = jiffies + Timer->Periodical_Time;
         add_timer(&(Timer->Timer_List));
      }
   }
}


/**
   Helper function to get a periodical timer.

   \param  arg          Pointer to corresponding timer ID.
*/
static IFX_void_t TAPI_timer_call_back (IFX_ulong_t arg)
{
   Timer_ID Timer = (Timer_ID) arg;
   /* do the operation in process context,
      not in interrupt context */
#ifdef LINUX_2_6
   queue_work (pTapiWq, &(Timer->timerTask));
#else
   schedule_task (&(Timer->timerTask));
#endif /* LINUX_2_6 */
}

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,20))
/* High Level Event Dispatcher function. */
static IFX_void_t Deferred_Worker (IFX_void_t *pWork)
{
   IFX_TAPI_EXT_EVENT_PARAM_t *pEvParam = (IFX_TAPI_EXT_EVENT_PARAM_t *) pWork;
   pEvParam->pFunc(pEvParam);
}
#else /* for Kernel newer or equal 2.6.20 */
static IFX_void_t Deferred_Worker (struct work_struct *pWork)
{
   IFX_TAPI_EXT_EVENT_PARAM_t *pEvParam = (IFX_TAPI_EXT_EVENT_PARAM_t *) pWork;
   pEvParam->pFunc(pEvParam);
}
#endif


/**
   Defer work to process context

   \param  pFunc        Pointer to function to be called.
   \param  pParam       Parameter passed to the function.
                        Attention: this must be a valid structure of
                        type IFX_TAPI_EXT_EVENT_PARAM_t as we need to
                        do some casting for Linux.

   \return TAPI_statusOk or TAPI_statusErr in case of an error.
*/
IFX_return_t TAPI_DeferWork (IFX_void_t *pFunc, IFX_void_t *pParam)
{
   IFX_return_t ret = TAPI_statusOk;
   IFX_TAPI_EXT_EVENT_PARAM_t *pEvParam = (IFX_TAPI_EXT_EVENT_PARAM_t *) pParam;
#ifdef LINUX_2_6
   struct work_struct         *pTapiWs;

   pTapiWs = (struct work_struct *) &pEvParam->tapiWs;
   pEvParam->pFunc   = (IFX_void_t *)pFunc;

   #if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,20))
   INIT_WORK(pTapiWs, Deferred_Worker, (IFX_void_t *) pEvParam);
   #else
   INIT_WORK(pTapiWs, Deferred_Worker);
   #endif

   if (queue_work (pTapiWq, pTapiWs) == 0)
   {
      ret = TAPI_statusWorkFail;
   }
#else
   struct tq_struct           *pTapiTq;

   pTapiTq           = (struct tq_struct *)&pEvParam->tapiTq;
   pTapiTq->routine  = Deferred_Worker;
   pEvParam->pFunc   = (IFX_void_t *)pFunc;
   pTapiTq->data     = (IFX_void_t *)pEvParam;
   pTapiTq->sync     = 0;
   if (schedule_task (pTapiTq) == 0)
   {
      ret = TAPI_statusWorkFail;
   }
#endif /* LINUX_2_6 */
   return ret;
}


/**
   Executes the select for the channel fd

   \param  pTapiPriv    Pointer fd private data carrying the context.
   \param  pNode        node list.
   \param  pOpt         Optional argument, which contains needed information for
                        IFXOS_SleepQueue.

   \return
   System event qualifier. Either 0 or TAPI_OS_SYSREAD

   \remarks
   This function needs operating system services, that are hidden by
   IFXOS macros.
*/
static IFX_int32_t TAPI_SelectCh (TAPI_FD_PRIV_DATA_t *pTapiPriv,
                                  TAPI_OS_drvSelectTable_t *pNode,
                                  TAPI_OS_drvSelectOSArg_t *pOpt)
{
   IFX_TAPI_DRV_CTX_t  *pDrvCtx  = pTapiPriv->pTapiDev->pDevDrvCtx;
   TAPI_DEV            *pTapiDev = pTapiPriv->pTapiDev;
   IFX_TAPI_STREAM_t    fifo_idx = pTapiPriv->fifo_idx;
   TAPI_CHANNEL        *pTapiCh;
   IFX_TAPI_LL_DEV_t   *pLLDev;
   IFX_uint32_t  flags = 0;
   IFX_int32_t   ret = 0;
#if (defined(TAPI_FAX_T38) || defined(LIN_SUPPORT))
   IFX_TAPI_T38_STATUS_t TapiFaxStatus;

   memset (&TapiFaxStatus, 0, sizeof (IFX_TAPI_T38_STATUS_t));
#endif /* TAPI_FAX_T38 || LIN_SUPPORT */

   pTapiCh = &(pTapiDev->pTapiChanelArray[pTapiPriv->nFds]);
   pLLDev = pTapiDev->pLLDev;

#ifdef TAPI_FAX_T38
   /* Get the Status from the low level driver */
   if (ptr_chk(pDrvCtx->COD.T38_Status_Get,
              "pDrvCtx->COD.T38_Status_Get"))
      (IFX_void_t)pDrvCtx->COD.T38_Status_Get (pTapiCh->pLLChannel, &TapiFaxStatus);
#endif /* TAPI_FAX_T38 */

   /* Register the voice channel waitqueues as wakeup source. */
   TAPI_OS_DrvSelectQueueAddTask (pOpt, &pTapiCh->semReadBlock.object, pNode);
   TAPI_OS_DrvSelectQueueAddTask (pOpt, &pTapiCh->wqRead, pNode);

#if (defined(TAPI_FAX_T38) || defined(LIN_SUPPORT))
   TAPI_OS_DrvSelectQueueAddTask (pOpt, &pTapiCh->wqWrite, pNode);

   if ((TapiFaxStatus.nStatus & IFX_TAPI_FAX_T38_TX_ON) &&
       (pTapiCh->bFaxDataRequest == IFX_TRUE))
   {
      /* task should write a new packet now */
      ret |= TAPI_OS_SYSWRITE;
   }

   if ((pTapiCh->bLinChannelActive == IFX_TRUE) &&
       (pTapiCh->bLinDataRequest == IFX_TRUE))
   {
      /* task should write a new packet now */
      ret |= TAPI_OS_SYSWRITE;
   }
#endif /* TAPI_FAX_T38 || LIN_SUPPORT */

   /* select on a voice channel -- only implemented for TAPI */
   flags |= CF_NEED_WAKEUP;
   /* clear flags first, then apply new flags */

   if (pDrvCtx->IRQ.LockDevice != IFX_NULL)
   {
      pDrvCtx->IRQ.LockDevice (pLLDev);
   }

#ifdef TAPI_PACKET
   if ( (pTapiCh->pUpStreamFifo[fifo_idx] != IFX_NULL) &&
        !fifoEmpty(pTapiCh->pUpStreamFifo[fifo_idx]) )
   {
      flags |= CF_WAKEUPSRC_STREAM;
      flags &= ~CF_NEED_WAKEUP;
      ret |= TAPI_OS_SYSREAD;
   }
#endif /* TAPI_PACKET */
   pTapiCh->nFlags &= ~(CF_WAKEUPSRC_GR909 | CF_WAKEUPSRC_STREAM |
                        CF_WAKEUPSRC_TAPI | CF_NEED_WAKEUP);
   pTapiCh->nFlags |= flags;

   if (pDrvCtx->IRQ.UnlockDevice != IFX_NULL)
   {
      pDrvCtx->IRQ.UnlockDevice (pLLDev);
   }

   return ret;
}


/**
   Modify own thread priority.

   \param  newPriority  New thread priority.

   \return
   - IFX_SUCCESS priority changed.
   - IFX_ERROR priority not changed.
*/
IFX_int32_t TAPI_OS_ThreadPriorityModify(IFX_uint32_t newPriority)
{
   struct sched_param sched_params;
   IFX_int32_t ret;

   sched_params.sched_priority = newPriority;
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0))
   ret = setscheduler(current->pid, SCHED_FIFO, &sched_params);
#else
   ret = sched_setscheduler(current, SCHED_FIFO, &sched_params);
#endif

   if (ret < 0)
   {
      printk(KERN_ERR "Failed to set the thread priority to %d"
             " ret=%d\n", newPriority, ret);
   }

   return (ret < 0) ? IFX_ERROR : IFX_SUCCESS;
}


/**
   stop a kernel thread. Called by the removing instance

   \param  pThrCntrl    Pointer to a thread control struct.
*/
IFX_void_t TAPI_OS_ThreadKill(IFXOS_ThreadCtrl_t *pThrCntrl)
{
   if ((pThrCntrl) &&
       (IFXOS_THREAD_INIT_VALID(pThrCntrl) == IFX_TRUE) &&
       (pThrCntrl->thrParams.bRunning == 1))
   {
         /* trigger user thread routine to shutdown */ \
         pThrCntrl->thrParams.bShutDown = IFX_TRUE;

         /* this function needs to be protected with the big
            kernel lock (lock_kernel()). The lock must be
            grabbed before changing the terminate
            flag and released after the down() call. */
         lock_kernel();
         mb();
         kill_proc(pThrCntrl->tid, SIGKILL, 1);
         /* release the big kernel lock */
         unlock_kernel();
         wait_for_completion (&pThrCntrl->thrCompletion);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,23)
         /* Now we are sure the thread is in zombie state.
            We notify keventd to clean the process up. */
         kill_proc(2, SIGCHLD, 1);
#endif /* kernel 2.6.23 */
   }
}


/*lint -save
-e611 -e546 -e19
-esym(752, init_module -e528
*/
module_init (ifx_tapi_module_init);
module_exit (ifx_tapi_module_exit);

/****************************************************************************/

MODULE_AUTHOR           ("Lantiq Deutschland GmbH");
MODULE_DESCRIPTION      ("TAPI Driver - www.lantiq.com");
MODULE_SUPPORTED_DEVICE ("TAPI DEVICE");
MODULE_LICENSE          ("Dual BSD/GPL");

EXPORT_SYMBOL (IFX_TAPI_Register_LL_Drv);
EXPORT_SYMBOL (IFX_TAPI_Unregister_LL_Drv);
EXPORT_SYMBOL (IFX_TAPI_DeviceReset);
EXPORT_SYMBOL (IFX_TAPI_ReportResources);

EXPORT_SYMBOL (IFX_TAPI_Event_Dispatch);

#ifdef TAPI_CID
EXPORT_SYMBOL (TAPI_Phone_GetCidRxBuf);
EXPORT_SYMBOL (TAPI_Cid_Abort);
EXPORT_SYMBOL (TAPI_Cid_IsActive);
#endif /* TAPI_CID */

EXPORT_SYMBOL (TAPI_Create_Timer);
EXPORT_SYMBOL (TAPI_SetTime_Timer);
EXPORT_SYMBOL (TAPI_Stop_Timer);
EXPORT_SYMBOL (TAPI_Delete_Timer);

EXPORT_SYMBOL (TAPI_Tone_Set_Source);
EXPORT_SYMBOL (TAPI_ToneState);

EXPORT_SYMBOL(fifoInit);
EXPORT_SYMBOL(fifoReset);
EXPORT_SYMBOL(fifoPut);
EXPORT_SYMBOL(fifoGet);
EXPORT_SYMBOL(fifoPeek);
EXPORT_SYMBOL(fifoEmpty);
EXPORT_SYMBOL(fifoFree);
EXPORT_SYMBOL(fifoElements);
EXPORT_SYMBOL(fifoSize);

#ifdef KPI_SUPPORT
EXPORT_SYMBOL(IFX_TAPI_KPI_WaitForData);
EXPORT_SYMBOL(IFX_TAPI_KPI_ReadData);
EXPORT_SYMBOL(IFX_TAPI_KPI_WriteData);
EXPORT_SYMBOL(IFX_TAPI_KPI_ChGet);
EXPORT_SYMBOL(IFX_TAPI_KPI_EgressTaskletRegister);
EXPORT_SYMBOL(irq_IFX_TAPI_KPI_PutToEgress);
EXPORT_SYMBOL(IFX_TAPI_KPI_ScheduleIngressHandling);
#endif /* KPI_SUPPORT */

#ifdef TAPI_PACKET
EXPORT_SYMBOL(IFX_TAPI_VoiceBufferGet);
EXPORT_SYMBOL(IFX_TAPI_VoiceBufferPut);
EXPORT_SYMBOL(IFX_TAPI_UpStreamFifo_Put);
EXPORT_SYMBOL(IFX_TAPI_UpStreamFifo_Reset);
EXPORT_SYMBOL(IFX_TAPI_VoiceBufferGetWithOwnerId);
EXPORT_SYMBOL(IFX_TAPI_VoiceBufferFreeAllOwnerId);
#ifdef TAPI_PACKET_OWNID
EXPORT_SYMBOL(IFX_TAPI_VoiceBufferChOwn);
#endif /* TAPI_PACKET_OWNID */
#ifdef TAPI_POLL
EXPORT_SYMBOL(IFX_TAPI_DownStreamFifo_Handle_Get);
EXPORT_SYMBOL(bufferPoolInit);
EXPORT_SYMBOL(bufferPoolGet);
EXPORT_SYMBOL(bufferPoolPut);
#endif /* TAPI_POLL */
#endif /* TAPI_PACKET */

/* packet statistic related exports */
EXPORT_SYMBOL(IFX_TAPI_Stat_Add);

/* FXO related exports */
EXPORT_SYMBOL(IFX_TAPI_Register_DAA_Drv);
EXPORT_SYMBOL(IFX_TAPI_FXO_Event_Dispatch);

#ifdef QOS_SUPPORT
EXPORT_SYMBOL(IFX_TAPI_QOS_DrvRegister);
#endif /* QOS_SUPPORT */

/*lint -restore*/
EXPORT_SYMBOL(IFX_TAPI_Update_SlicFxo);
