/******************************************************************************

                              Copyright (c) 2009
                            Lantiq Deutschland GmbH
                     Am Campeon 3; 85579 Neubiberg, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

******************************************************************************/

/**
   \file  drv_tapi_voice.c
   Contains TAPI Voice Services : Play, Recording, Conferencing.
*/

/* ============================= */
/* Includes                      */
/* ============================= */

#include "drv_tapi.h"
#include "drv_tapi_ll_interface.h"


/* ============================= */
/* Local variable definition     */
/* ============================= */

/* ============================= */
/* Local function definition     */
/* ============================= */

/* ============================= */
/* Global function definition    */
/* ============================= */

#ifdef TAPI_VOICE
/**
   This interface adds a data channel to an analog phone device

   \param pChannel      Pointer to TAPI_CHANNEL structure.
   \param pMap          Pointer to IFX_TAPI_MAP_DATA_t structure.

   \return
   \ref IFX_SUCCESS or \ref IFX_ERROR
*/
IFX_int32_t TAPI_Data_Channel_Add (TAPI_CHANNEL *pChannel,
                                   IFX_TAPI_MAP_DATA_t const *pMap)
{
   IFX_TAPI_DRV_CTX_t *pDrvCtx = pChannel->pTapiDevice->pDevDrvCtx;
   IFX_int32_t ret = IFX_SUCCESS;

   switch (pMap->nPlayStart)
   {
      case IFX_TAPI_MAP_DATA_UNCHANGED:
         /* do nothing */
         break;
      case IFX_TAPI_MAP_DATA_START:
         if (ptr_chk(pDrvCtx->COD.DEC_Start, "pDrvCtx->COD.DEC_Start"))
            ret = pDrvCtx->COD.DEC_Start (pChannel->pLLChannel);
         break;
      case IFX_TAPI_MAP_DATA_STOP:
         if (ptr_chk(pDrvCtx->COD.DEC_Stop, "pDrvCtx->COD.DEC_Stop"))
            ret = pDrvCtx->COD.DEC_Stop (pChannel->pLLChannel);
         break;
   }
   if (ret == IFX_SUCCESS)
   {
      switch (pMap->nRecStart)
      {
         case IFX_TAPI_MAP_DATA_UNCHANGED:
            /* do nothing */
            break;
         case IFX_TAPI_MAP_DATA_START:
            if (ptr_chk(pDrvCtx->COD.ENC_Start, "pDrvCtx->COD.ENC_Start"))
               ret = pDrvCtx->COD.ENC_Start (pChannel->pLLChannel);
            break;
         case IFX_TAPI_MAP_DATA_STOP:
            if (ptr_chk(pDrvCtx->COD.ENC_Stop, "pDrvCtx->COD.ENC_Stop"))
               ret = pDrvCtx->COD.ENC_Stop (pChannel->pLLChannel);
            break;
      }
   }

   /* call low level function here */
   if (ret == IFX_SUCCESS)
   {
      if (ptr_chk(pDrvCtx->CON.Data_Channel_Add,
                 "pDrvCtx->CON.Data_Channel_Add"))
      {
         ret = pDrvCtx->CON.Data_Channel_Add (pChannel->pLLChannel, pMap);
      }
   }

   return ret;
}
#endif /* TAPI_VOICE */


#ifdef TAPI_VOICE
/**
   This interface removes a data channel from an analog phone device

   \param pChannel      Pointer to TAPI_CHANNEL structure.
   \param pMap          Pointer to IFX_TAPI_MAP_DATA_t structure.

   \return
   \ref IFX_SUCCESS or \ref IFX_ERROR
*/
IFX_int32_t TAPI_Data_Channel_Remove (TAPI_CHANNEL *pChannel,
                                      IFX_TAPI_MAP_DATA_t const *pMap)
{
   IFX_TAPI_DRV_CTX_t *pDrvCtx = pChannel->pTapiDevice->pDevDrvCtx;
   IFX_int32_t ret = IFX_SUCCESS;

   switch (pMap->nPlayStart)
   {
      case IFX_TAPI_MAP_DATA_UNCHANGED:
         /* do nothing */
         break;
      case IFX_TAPI_MAP_DATA_START:
         if (ptr_chk(pDrvCtx->COD.DEC_Start, "pDrvCtx->COD.DEC_Start"))
            ret = pDrvCtx->COD.DEC_Start (pChannel->pLLChannel);
         break;
      case IFX_TAPI_MAP_DATA_STOP:
         if (ptr_chk(pDrvCtx->COD.DEC_Stop, "pDrvCtx->COD.DEC_Stop"))
            ret = pDrvCtx->COD.DEC_Stop (pChannel->pLLChannel);
         break;
   }
   if (ret == IFX_SUCCESS)
   {
      switch (pMap->nRecStart)
      {
         case IFX_TAPI_MAP_DATA_UNCHANGED:
            /* do nothing */
            break;
         case IFX_TAPI_MAP_DATA_START:
            if (ptr_chk(pDrvCtx->COD.ENC_Start, "pDrvCtx->COD.ENC_Start"))
               ret = pDrvCtx->COD.ENC_Start (pChannel->pLLChannel);
            break;
         case IFX_TAPI_MAP_DATA_STOP:
            if (ptr_chk(pDrvCtx->COD.ENC_Stop, "pDrvCtx->COD.ENC_Stop"))
               ret = pDrvCtx->COD.ENC_Stop (pChannel->pLLChannel);
            break;
      }
   }

   /* call low level function here */
   if (ret == IFX_SUCCESS)
   {
      if (ptr_chk(pDrvCtx->CON.Data_Channel_Remove,
                 "pDrvCtx->CON.Data_Channel_Remove"))
         ret = pDrvCtx->CON.Data_Channel_Remove (pChannel->pLLChannel, pMap);
   }

   return ret;
}
#endif /* TAPI_VOICE */


/**
   Find the data channel which gets it's main input from a given module
   and which provides it's input to the given module.

    There can be more than one data channel connected to a module. So the
    function acts as an iterator with pTapiCh as input and output parameter.

    If no LL function is implemented this function reports a single straight
    connection between module and data channel.

   \param  pChannel     Pointer to TAPI_CHANNEL structure.
   \param  nModType     Type of the module where to start search.
   \param  pTapiCh      On calling specifies which was the last channel found
                        by this function. For the first call use IFX_NULL.
                        For iteration calls use the last output.
                        Returns pointer to the TAPI channel the found module
                        belongs to or IFX_NULL if no module is connected.

   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured
*/
IFX_int32_t IFX_TAPI_Module_Find_Connected_Data_Channel (
                                          TAPI_CHANNEL *pChannel,
                                          IFX_TAPI_MAP_TYPE_t nModType,
                                          TAPI_CHANNEL **pTapiCh)
{
   IFX_TAPI_DRV_CTX_t *pDrvCtx = pChannel->pTapiDevice->pDevDrvCtx;
   IFX_int32_t ret = IFX_SUCCESS;

   if (ptr_chk(pDrvCtx->CON.Module_Find_Connected_Data_Channel, ""))
   {
      /* Pass parameters to LL implementation. */
      ret = pDrvCtx->CON.Module_Find_Connected_Data_Channel(
                                       pChannel->pLLChannel, nModType, pTapiCh);
   }
   else
   {
      /* Report a single straight connection between module and data channel. */
      if (*pTapiCh == IFX_NULL)
      {
         *pTapiCh = pChannel;
      }
      else
      {
         *pTapiCh = IFX_NULL;
      }
   }

   return ret;
}
