/******************************************************************************

                              Copyright (c) 2009
                            Lantiq Deutschland GmbH
                     Am Campeon 3; 85579 Neubiberg, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

******************************************************************************/

/**
   \file drv_tapi_vxworks.c
   This file contains the implementation of High-Level TAPI Driver,
   VxWorks specific part.

   The implementation includes the following parts:
    -Registration part by which the low-level drivers register themselves.
     During the registration appropriate device nodes are created and
     registered with the IOS.
    -Device node operations (open, close, ioctl, read, write, select)
     are done here.
    -Timer abstraction layer.
    -Deferring of a function call for later execution.
*/

/* ============================= */
/* Includes                      */
/* ============================= */

#include "drv_api.h"
#include "drv_tapi_vxworks.h"

#include "drv_tapi.h"
#include "drv_tapi_api.h"
#include "drv_tapi_ll_interface.h"
#include "drv_tapi_event.h"
#include "drv_tapi_errno.h"
#include "drv_tapi_ioctl.h"
#include "drv_tapi_stream.h"

#ifdef TAPI_CID
#include "drv_tapi_cid.h"
#endif /* TAPI_CID */

#ifdef TAPI_POLL
#include "drv_tapi_polling.h"
#endif /* TAPI_POLL */

/* ============================= */
/* Defines                       */
/* ============================= */

/* VxWorks task priority of the task that executes the functions upon
   expiration of a timer. */
#define  TSK_PRIO_TIMER_HANDLER   20
/* VxWorks task priority of the task that processes all events */
#define  TSK_PRIO_EVENT_HANDLER   21

#ifdef VXWORKS

/*
    Device header passed to the IOS at the device driver creation during
    the system startup.
*/
typedef struct
{
   /* IOS DEV_HDR struct. This must be kept at the first position of this stuct
      so that the address of the struct and this member are identical. */
   DEV_HDR           DevHdr;
    /* Major number (used to identify the driver) */
   IFX_int32_t       nMajorNum;
   /* Device number (used to identify the device within the driver)*/
   IFX_uint32_t      nDevNum;
   /* In Use counter */
   IFX_int32_t       nInUse;
   /* Pointer to the TAPI device struct (valid only from open to close) */
   IFX_void_t        *pTapiDev;
   /* file descriptor number (255: device file descriptor <255: channel fds) */
   IFX_uint16_t      nFds;
} Y_S_dev_header;

/*
    Global device driver structure.
    IOS Device driver Context.
*/
typedef struct
{
   /* Device Driver Number     */
   IFX_int32_t         nDrvNum;
   /* Major Driver Number     */
   IFX_int32_t         nMajorNum;
   /* Max device               */
   IFX_int32_t         nMaxDev;
   /* Max device channel       */
   IFX_int32_t         nMaxChn;
   /* Interface Semaphore      */
   TAPI_OS_mutex_t     oSemDrvIF;
   /* device header pointer    */
   Y_S_dev_header**     pDevHdr;
} Y_S_dev_ctx;


/* ============================= */
/* Local variable definition     */
/* ============================= */

/* The Global device driver structure */
Y_S_dev_ctx TAPI_devCtx;

/** Protection for array of timer structs. */
static TAPI_OS_mutex_t semTimerArrDataLock;


/* ============================= */
/* Global variable definition    */
/* ============================= */

/** Message queue ID, holding events to be handled. */
static MSG_Q_ID nMsgQueue_Events = 0;

/** Task ID of task that processes the event queue. */
static IFX_int32_t nTaskForEvents_ID = -1;

/** Max. number of messages in event queue. */
enum { MAX_MSG_CNT_EVENT_QUEUE = 32 };

/** Size of one message in the event queue. (Size of one pointer) */
const IFX_int32_t MSG_SIZE_EVENT_QUEUE = sizeof(IFX_int32_t);


/* -------------------------------------------------------------------------- */

extern IFX_int32_t IFX_TAPI_Event_Dispatch_ProcessCtx(IFX_TAPI_EXT_EVENT_PARAM_t *pParam);

/* ============================= */
/* Local function declaration    */
/* ============================= */

/* Functions handling the select/unselect driver system io calls */
static IFX_int32_t OS_OnSelect(IFX_TAPI_ioctlCtx_t* pCtx,
                               IFX_int32_t nArgument);
static IFX_int32_t OS_OnUnselect(IFX_TAPI_ioctlCtx_t* pCtx,
                                 IFX_int32_t nArgument);
static IFX_int32_t OS_DevNodeRegister (Y_S_dev_ctx* pDevCtx, IFX_uint8_t nDev,
                                       int nMinor, IFX_char_t* buf);

static IFX_int32_t TAPI_DevCreate(Y_S_dev_ctx* pDevCtx,
                                  IFX_char_t* pzsDeviceName);
static IFX_int32_t TAPI_DevDelete(Y_S_dev_ctx* pDevCtx);

static IFX_return_t ifx_tapi_Event_StartMsgQueue(IFX_void_t);

/* ============================= */
/* Local function definition     */
/* ============================= */

/* ============================= */
/* Global function definition    */
/* ============================= */


/* --------------------------------------------------------------------------
                          REGISTRATION mechanisms  -->  BEGIN
   --------------------------------------------------------------------------
*/

/**
   Register the low level driver with the operating system

   This function will be called when the low-level driver is added to the
   system. Any OS specific registration or set-up should be done here.

   \param  pLLDrvCtx    Pointer to device driver context created by LL-driver.
   \param  pHLDrvCtx    Pointer to high-level driver context.

   \return
   IFX_SUCCESS
*/
IFX_return_t TAPI_OS_RegisterLLDrv (IFX_TAPI_DRV_CTX_t* pLLDrvCtx,
                                    IFX_TAPI_HL_DRV_CTX_t* pHLDrvCtx)
{
   /* The "TAPI_devCtx" global area contains all the         */
   /* data related to the VINETIC device driver installation */
   /* The area is used during all the life of the VINETIC    */
   /* device driver. The area is updated during the creation */
   /* and the access to the device driver.                   */
   /* Reset the memory area.                                 */
   memset(&TAPI_devCtx, 0x00, sizeof(Y_S_dev_ctx));

   /* Copy registration info from Low level driver. */
   TAPI_devCtx.nMaxDev = pLLDrvCtx->maxDevs;
   TAPI_devCtx.nMaxChn = pLLDrvCtx->maxChannels;
   TAPI_devCtx.nMajorNum = pLLDrvCtx->majorNumber;

   if (TAPI_DevCreate(&TAPI_devCtx, pLLDrvCtx->devNodeName) == IFX_ERROR)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Err creating device. "
            "(File: %s, line: %d)\n", __FILE__, __LINE__));

      return IFX_ERROR;
   }

   TAPI_OS_MutexInit (&TAPI_devCtx.oSemDrvIF);
   /* Initialize mutex for timer struct array access */
   TAPI_OS_MutexInit (&semTimerArrDataLock);

   return IFX_SUCCESS;
} /* IFX_TAPI_Register_LL_Drv() */


/**
   UnRegister the low level driver from the operating system

   This function will be called when the low-level driver is removed from the
   system. Any OS specific deregistration should be done here.

   \param  pLLDrvCtx    Pointer to device driver context created by LL-driver.
   \param  pHLDrvCtx    Pointer to high-level driver context.

   \return
   IFX_ERROR on error, otherwise IFX_SUCCESS
*/
IFX_return_t TAPI_OS_UnregisterLLDrv (IFX_TAPI_DRV_CTX_t* pLLDrvCtx,
                                      IFX_TAPI_HL_DRV_CTX_t* pHLDrvCtx)
{
   TAPI_OS_MutexDelete (&semTimerArrDataLock);
   TAPI_OS_MutexDelete (&TAPI_devCtx.oSemDrvIF);

   return IFX_SUCCESS;
}


/* --------------------------------------------------------------------------
                          REGISTRATION mechanisms  -->  END
   --------------------------------------------------------------------------
*/


/* --------------------------------------------------------------------------
         DRIVER funcs (read/write/ioctl/release/open/poll)  -->  BEGIN
   --------------------------------------------------------------------------
*/


/**
   This function open a TAPI device previously registered to the OS during the
   device driver creation. For each Tapi channel, a device have been added in
   the device list with the device string "/dev/<device>DC":
      *<device> - vin, vmmc, ...
      - D: device number
      - C: channel number in the device

   The OS passes the Y_S_dev_header structure associated to the
   device "/dev/<device>DC":
      DevHdr: VxWorks specific header
      ddrvn : TAPI Device Driver Number (allocated by IOS)
      devn  : device number  (1 to TAPI_MAX_DEVICES)
      chnn  : channel number (0 to TAPI_MAX_CH_NR)
      pctx  : pointer to the channel context (will be allocated at the device open)

   The function will update the pctx field in the device header with the TAPI
   device channel (control/voice) selected structure.

   \param pDevHeader - device header pointer
   \param pAnnex - tail of device name / not used
   \param flags - open flags / not used

  \return pDevHeader - device header pointer
                <> IFX_NULL, the device is open without error
                == IFX_NULL, error
*/
static IFX_int32_t ifx_tapi_open(Y_S_dev_header* pDevHeader,
                                 INT8* pAnnex, IFX_int32_t flags)
{
   Y_S_dev_ctx             *pDevCtx    = IFX_NULL; /* device context pointer */
   IFX_TAPI_DRV_CTX_t      *pDrvCtx    = IFX_NULL;
   TAPI_DEV                *pTapiDev   = IFX_NULL;
   IFX_uint32_t            devNum;

   if (pDevHeader == IFX_NULL)
   {
      /* The device driver is not installed */
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
            ("pDevHeader is NULL. (File: %s, line: %d)\n", __FILE__, __LINE__));
      errno = -ENODEV;
      return (IFX_int32_t) IFX_NULL;
   }

   /* Get the Global device driver context and check that the device driver
      is  already installed.  */
   pDevCtx = &TAPI_devCtx;

   /* Check device range. */
   if ((pDevHeader->nDevNum > pDevCtx->nMaxDev) || (0 > pDevHeader->nDevNum))
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Max. device number exceed. "
            "(File: %s, line: %d)\n", __FILE__, __LINE__));
      errno = -ENODEV;
      return (IFX_int32_t) IFX_NULL;
   }
   devNum = pDevHeader->nDevNum - 1;

   /* Get the pointer to the device driver context based on the major number */
   pDrvCtx = IFX_TAPI_DeviceDriverContextGet(pDevHeader->nMajorNum);

   if (pDrvCtx == IFX_NULL)
   {
      /* This should actually never happen because the file descriptors are
         registered in TAPI_OS_RegisterLLDrv after a driver context is known. */
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("tapi_open: error finding DrvCtx\n"));
      errno = -ENODEV;
      return (IFX_int32_t) IFX_NULL;
   }

   /* protect against concurrent access */
   if (&pDevCtx->oSemDrvIF != NULL)
   {
      TAPI_OS_MutexGet (&pDevCtx->oSemDrvIF);
   }

   /* pointer to the TAPI device struct */
   pTapiDev = &(pDrvCtx->pTapiDev[devNum]);
   pDevHeader->pTapiDev = pTapiDev;

   /* Increment the Usage counter */
   pDevHeader->nInUse++;

   /* Call the Low level Device specific open routine */
   if (ptr_chk(pDrvCtx->Open, "pDrvCtx->Open"))
   {
      IFX_int32_t retLL;

      if (pDevHeader->nFds == IFX_TAPI_DEVICE_CH_NUMBER)
      {
         retLL = pDrvCtx->Open (pTapiDev->pLLDev);
      }
      else
      {
         TAPI_CHANNEL *pTapiCh = &(pTapiDev->pTapiChanelArray[pDevHeader->nFds]);
         retLL = pDrvCtx->Open (pTapiCh->pLLChannel);
      }

      if (retLL != IFX_SUCCESS)
      {
         TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
               ("Open LL channel failed for ch: %d\n", pDevHeader->nFds));
         return IFX_ERROR;
      }
   }

   /* release lock */
   if (&pDevCtx->oSemDrvIF != NULL)
   {
      TAPI_OS_MutexRelease (&pDevCtx->oSemDrvIF);
   }

   return (IFX_int32_t) pDevHeader;
} /* ifx_tapi_open() */


/**
   Configuration / Control for the device.

   \param pDevHeader - device header pointer
   \param nCmd - Configuration/Control command
   \param nArg - Configuration/Control arguments, optional

   \return IFX_SUCCESS    - no problems
           <> IFX_SUCCESS - Function is not implemented or other error
*/
static STATUS ifx_tapi_ioctl(Y_S_dev_header* pDevHeader,
                             IFX_int32_t nCmd,
                             IFX_int32_t nArg)
{
   IFX_TAPI_ioctlCtx_t  ctx;
   IFX_int32_t          ret;

   if (pDevHeader == IFX_NULL)
   {
      /* The device driver is not installed */
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("pDevHeader is NULL. "
            "(File: %s, line: %d)\n", __FILE__, __LINE__));
      errno = -ENODEV;
      return IFX_ERROR;
   }

   /* set the ioctl context struct */
   memset (&ctx, 0, sizeof(ctx));
   ctx.pTapiDev = pDevHeader->pTapiDev;
   ctx.nFds     = pDevHeader->nFds;
#ifdef TAPI_ONE_DEVNODE
   ctx.bSingleFd = IFX_TRUE;
   /*\todo set fds accoring to the channel, dev according to dev */
#endif /* TAPI_ONE_DEVNODE */
   /* In linux is like that, but actualy VxWorks does not have this command.
      Content of IOC_SIZE is used here */
   /*ctx.nParamSize = _IOC_SIZE(nCmd);*/
   ctx.nParamSize = (nCmd >> 16) & ((1 << 13) - 1);

   switch (nCmd)
   {
   case FIOSELECT:
      ret = OS_OnSelect(&ctx, nArg);
      break;
   case FIOUNSELECT:
      ret = OS_OnUnselect(&ctx, nArg);
      break;
   default:
      ret = TAPI_Ioctl (&ctx, nCmd, nArg);
      break;
   } /* switch */

   if ((ret == -1) && (ctx.pTapiDev->error.nCode == 0))
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Error code not set %d\n", nCmd));
   }

   return ret;
} /* ifx_tapi_ioctl() */


/**
    Release the device. The in-use counter will be decreased

   \param pDevHeader - device header pointer

   \return OK - no error
           -ENODEV - MAX_SOC_CHANNELS is exceeded
           ERROR on error

   \remark
      This function gets called when a close is called on the device.
      It decrements the usage count, free the FIFOs
*/
static STATUS ifx_tapi_release(Y_S_dev_header *pDevHeader)
{
   /* device context pointer */
   Y_S_dev_ctx* pDevCtx = IFX_NULL;
   /* Current Tapi device structure */
   TAPI_DEV* pTapiDev = IFX_NULL;
   IFX_TAPI_DRV_CTX_t* pDrvCtx = IFX_NULL;

   if (pDevHeader == IFX_NULL)
   {
      /* The device driver is not installed */
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("pDevHeader is NULL. "
            "(File: %s, line: %d)\n", __FILE__, __LINE__));
      errno = -ENODEV;
      return IFX_ERROR;
   }

   /* pDevHeader->pTapiDev is only valid in between open and close */
   if (pDevHeader->pTapiDev == IFX_NULL)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("File descriptor is already closed. "
            "(File: %s, line: %d)\n", __FILE__, __LINE__));
      errno = -ENODEV;
      return IFX_ERROR;
   }

   /* Get the Global device driver context */
   pDevCtx = &TAPI_devCtx;  /* Get the device context */

   TAPI_OS_MutexGet (&pDevCtx->oSemDrvIF);

   pTapiDev = (TAPI_DEV *)(pDevHeader->pTapiDev);
   pDrvCtx = pTapiDev->pDevDrvCtx;

   /* Decrement the Usage counter */
   pDevHeader->nInUse--;

   if (pDevHeader->nInUse == 0)
   {
      /* We erase the reference to the Tapi structure in the header. */
      pDevHeader->pTapiDev = IFX_NULL;
   }

   /* Call the Low-level Device specific release routine. */
   /* Not having such a function is not an error. */
   if (ptr_chk(pDrvCtx->Release, ""))
   {
      IFX_int32_t retLL;

      if (pDevHeader->nFds == IFX_TAPI_DEVICE_CH_NUMBER)
      {
         retLL = pDrvCtx->Release (pTapiDev->pLLDev);
      }
      else
      {
         TAPI_CHANNEL *pTapiCh = &(pTapiDev->pTapiChanelArray[pDevHeader->nFds]);
         retLL = pDrvCtx->Release (pTapiCh->pLLChannel);
      }

      if (retLL != IFX_SUCCESS)
      {
         TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
               ("Release LL channel failed for ch: %d\n", pDevHeader->nFds));
         return IFX_ERROR;
      }
   }

   TAPI_OS_MutexRelease (&pDevCtx->oSemDrvIF);

   TRACE(TAPI_DRV, DBG_LEVEL_LOW,
        ("closing device %d, Minor %d...\n",
         pDevHeader->nDevNum, pDevHeader->nDevNum));

   return (OK);
} /* ifx_tapi_release() */


/**

   Read data from the Tapi.

   \param pDevHeader - device header pointer
   \param pDest - data destination pointer
   \param nLength - data length to read

   \return len - data length
*/
static IFX_int32_t ifx_tapi_read(Y_S_dev_header* pDevHeader,
                                 IFX_uint8_t* pDest,
                                 IFX_int32_t nLength)
{
#ifdef TAPI_PACKET
   IFX_TAPI_DRV_CTX_t  *pDrvCtx;
   TAPI_DEV            *pTapiDev;
   TAPI_CHANNEL        *pTapiCh;
   IFX_void_t const    *pPacket   = NULL;
   IFX_uint32_t         nOffset;
#endif /* TAPI_PACKET */
   IFX_uint32_t         size = 0;

#ifdef TAPI_PACKET
   if (IFX_NULL == pDevHeader)
   {
      /* The device driver is not installed */
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("pDevHeader is NULL. "
            "(File: %s, line: %d)\n", __FILE__, __LINE__));
      errno = -ENODEV;
      return 0;
   }

   pTapiDev = (TAPI_DEV *)(pDevHeader->pTapiDev);

   if (pTapiDev->bInitialized == IFX_FALSE)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_LOW,
            ("TAPI_DRV: read failed because device is not initialised\n"));
      return -EFAULT;
   }

   pDrvCtx  = pTapiDev->pDevDrvCtx;
   pTapiCh  = &(pTapiDev->pTapiChanelArray[pDevHeader->nFds]);

   /* All packets to be read are already in the channel specific upstream
      fifo of the driver. They are put there by whoever receives packets.
      So all that is done here operates on this fifo and does not need to call
      any ll-driver function any more. */

   /* Lock access to the fifo. */
   TAPI_OS_MutexGet (&pTapiCh->semTapiChDataLock);
   if (ptr_chk(pDrvCtx->IRQ.LockDevice, "pDrvCtx->IRQ.LockDevice"))
   {
      pDrvCtx->IRQ.LockDevice (pTapiDev->pLLDev);
   }

   /* If FIFO is empty or does not exist we cannot read anything. In this
      case we either wait until data arrives or return if non-blocking.
      If the FIFO does not exist no data will ever arrive and the possible
      wait will last forever. */
   if ((pTapiCh->pUpStreamFifo[IFX_TAPI_STREAM_COD] == IFX_NULL) ||
       (fifoEmpty(pTapiCh->pUpStreamFifo[IFX_TAPI_STREAM_COD]) == IFX_TRUE))
   {
      /* Unlock access to the fifo. */
      if (pDrvCtx->IRQ.UnlockDevice != IFX_NULL)
      {
         pDrvCtx->IRQ.UnlockDevice (pTapiDev->pLLDev);
      }
      TAPI_OS_MutexRelease (&pTapiCh->semTapiChDataLock);

      /* check the non-blocking flag */
      if (pTapiCh->nFlags & CF_NONBLOCK)
      {
         /* this is a non blocking read call - return immediately */
         return 0;
      }

      /* this is a blocking read call so go to sleep now */
      if (TAPI_OS_EventWait (&pTapiCh->semReadBlock,
                             TAPI_OS_WAIT_FOREVER, IFX_NULL) == IFX_ERROR)
      {
         /* timeout has expired without arrival of data */
         return 0;
      }

      /* we have been woken because data has arrived */

      /* before accessing the mailbox - lock again. */
      TAPI_OS_MutexGet (&pTapiCh->semTapiChDataLock);
      if (ptr_chk(pDrvCtx->IRQ.LockDevice, "pDrvCtx->IRQ.LockDevice"))
      {
         pDrvCtx->IRQ.LockDevice (pTapiDev->pLLDev);
      }
   }

   /* read data from the fifo */
   TAPI_ASSERT (
      fifoEmpty(pTapiCh->pUpStreamFifo[IFX_TAPI_STREAM_COD]) == IFX_FALSE);
   pPacket = IFX_TAPI_UpStreamFifo_Get(pTapiCh,
                                       IFX_TAPI_STREAM_COD, &size, &nOffset);

   /* Unlock access to the fifo. */
   if (pDrvCtx->IRQ.UnlockDevice != IFX_NULL)
   {
      pDrvCtx->IRQ.UnlockDevice (pTapiDev->pLLDev);
   }
   TAPI_OS_MutexRelease (&pTapiCh->semTapiChDataLock);

   /* sanity check on the received packet */
   if (pPacket == NULL)
   {
      /* packet may not be NULL - internal error */
      TRACE (TAPI_DRV, DBG_LEVEL_LOW, ("INFO: pPacket Null\n"));
      return 0;
   }
   if (size > (IFX_uint32_t)nLength)
   {
      /* output buffer not large enough for data */
      /* drop the packet */
      IFX_TAPI_VoiceBufferPut((IFX_void_t *)pPacket);
      /* return linux error code to the application */
      return -EINVAL;
   }

   /* unmap data */
   TAPI_OS_CpyKern2Usr ((void *)pDest,
                        (void *)((char *)pPacket + nOffset),
                        (IFX_uint32_t)size);

   /* return buffer back to the pool now that is has been copied */
   IFX_TAPI_VoiceBufferPut((IFX_void_t *)pPacket);
#endif /* TAPI_PACKET */

   return (IFX_int32_t)size;
}


/**
   Writes data to the device.

   \param pDevHeader - device header pointer
   \param pSrc - data source pointer
   \param nLength - data length to write

   \return nLength - 0 if failure else the length
*/
static IFX_int32_t ifx_tapi_write(Y_S_dev_header* pDevHeader,
                                  IFX_uint8_t* pSrc,
                                  IFX_int32_t nLength)
{
#ifdef TAPI_PACKET
   IFX_TAPI_DRV_CTX_t  *pDrvCtx;
   TAPI_DEV            *pTapiDev;
   TAPI_CHANNEL        *pTapiCh;
   IFX_uint8_t         *pData;
   IFX_size_t           buf_size;
#endif /* TAPI_PACKET */
   IFX_ssize_t          size = 0;

#ifdef TAPI_PACKET
   if (IFX_NULL == pDevHeader)
   {
      /* The device driver is not installed */
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("pDevHeader is NULL. "
            "(File: %s, line: %d)\n", __FILE__, __LINE__));
      errno = -ENODEV;
      return 0;
   }

   pTapiDev = (TAPI_DEV *)(pDevHeader->pTapiDev);

   if (pTapiDev->bInitialized == IFX_FALSE)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_LOW,
            ("TAPI_DRV: read failed because device is not initialised\n"));
      return -EFAULT;
   }

   pDrvCtx  = pTapiDev->pDevDrvCtx;
   pTapiCh  = &(pTapiDev->pTapiChanelArray[pDevHeader->nFds]);

   if (ptr_chk( pDrvCtx->Write, "pDrvCtx->Write"))
   {
      /* Truncate data to the size of our voice buffer. */
      buf_size = IFX_TAPI_VoiceBufferPool_ElementSizeGet();
      if (nLength > (buf_size - pDrvCtx->pktBufPrependSpace))
      {
         nLength = (buf_size - pDrvCtx->pktBufPrependSpace);
      }

      /* Get a buffer from the voice buffer pool. */
      /* The vinetic ll-write function requires a TAPI voice buffer
         to have some space in which it can add the mailbox header. */
      if ((pData = IFX_TAPI_VoiceBufferGet()) == IFX_NULL)
      {
         /* voice buffer pool is depleted */
         return -EFAULT;
      }

      /* if you have time and want clean buffers you might like this */
      /* memset(pData, 0, buf_size); */

      /* Copy data into buffer. */
      TAPI_OS_CpyUsr2Kern ((void *)(pData + pDrvCtx->pktBufPrependSpace),
                           (void *)pSrc, (IFX_uint32_t)nLength);

      /* Call the low-level driver's write function. */
      size = pDrvCtx->Write (pTapiCh->pLLChannel, pData, nLength,
                             (IFX_int32_t*)IFX_NULL, IFX_TAPI_STREAM_COD);
   }
   else
   {
      TRACE(TAPI_DRV, DBG_LEVEL_LOW,
            ("TAPI_DRV: LL-driver does not provide packet write\n"));
      return -EFAULT;
   }
#endif /* TAPI_PACKET */

   return size;
}


/* --------------------------------------------------------------------------
             DRIVER funcs (read/write/ioctl/release/open)  -->  END
   --------------------------------------------------------------------------
*/


/* --------------------------------------------------------------------------
                          CREATE/DELETE driver  -->  BEGIN
   --------------------------------------------------------------------------
*/


/**

   TAPI device driver initialization.
   This is the device driver initialization function to call at the system
   startup prior any access to the TAPI device driver.
   After the initialization the device driver is ready to be accessed by
   the application. The global structure "TAPI_devCtx" contains all the data
   handling the interface (open, close, ioctl,...).

   \arguments None

   \return OK or ERROR
*/
IFX_int32_t Tapi_DeviceDriverInit(IFX_void_t)
{
   printf("%s, (c) 2001-2010 Lantiq Deutschland GmbH\n", &TAPI_WHATVERSION[4]);

   ifx_tapi_Event_StartMsgQueue();

   IFX_TAPI_Driver_Start();

   /* Very first call used for calibration  of the self-calibrating hard
      delay routines library "DelayLib.c" For IDES3300 this calibration
      is done before, so this leads only to an additonal affordable
      wait of 1us. */
   TAPI_OS_USecSleep(1);

   return IFX_SUCCESS;
} /* Tapi_DeviceDriverInit() */


/**

   TAPI device driver shutdown.
   This is the device driver shutdown function. This function is called by the
   system when the TAPI device driver is no longer needed. Prior to shutdown
   the device driver all the device channels should be closed.
   This function releases all the resources granted by the device driver.

   \param None

   \return OK or ERROR

   \remarks
*/
IFX_int32_t Tapi_DeviceDriverStop(VOID)
{
   IFX_int32_t i = 0;

   printf("Removing Highlevel TAPI module\n");

   if (TAPI_DevDelete(&TAPI_devCtx) == IFX_ERROR)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Err deleting device. "
            "(File: %s, line: %d)\n", __FILE__, __LINE__));

      return IFX_ERROR;
   }

   TAPI_OS_MutexDelete (&TAPI_devCtx.oSemDrvIF);

   /* Delete mutex for timer struct array */
   TAPI_OS_MutexDelete (&semTimerArrDataLock);

   /* Actually all LL drivers should have unregistered here. Being careful
      we force unregister of any drivers which may still be registered. */
   for (i = 0; i < TAPI_MAX_LL_DRIVERS; i++)
   {
      if (gHLDrvCtx [i].pDrvCtx != IFX_NULL)
      {
         IFX_TAPI_Unregister_LL_Drv (gHLDrvCtx [i].pDrvCtx->majorNumber);
      }
   }

   IFX_TAPI_Driver_Stop();

#if 0
   IFX_TAPI_Event_StopMsgQueue();
#endif

   TRACE(TAPI_DRV,DBG_LEVEL_NORMAL,("TAPI_DRV: cleanup successful\n"));

   return IFX_SUCCESS;
} /* Tapi_DeviceDriverStop() */


/**

   TAPI device driver creation.
   This function is called at the system startup (Tapi_DeviceDriverInit)
   in order to create and install the TAPI device driver in the target OS (VxWorks).

   Device driver Creation: the device driver is declared to the IOS system with
   "iosDrvInstall" OS function. We pass to the IOS the device driver interface
   functions (open, close, ioctl, ...), the IOS returns a Device Driver Number which
   will be used for adding the devices to the device driver.

   Device addition: Once the Device driver have been declared to the IOS system,
   we gone add the devices to the device driver list. Each device is identified by
   a unique string which will be used by the application for opening a device
   (the application will get in return a unique file descriptor (fd) allocated by
   the system and will be used for further access to the selected device).

   Device string selection: "/dev/vinDC" with,

        - "/dev/vin", this is a VINETIC device
        - "D",        designate the device number
                        "1",    VINETIC device 1
                        "2",    VINETIC device 2
                        ...
                        "VINETIC_MAX_DEVICES" VINETIC device VINETIC_MAX_DEVICES
        - "C",        designate a channel (C) in the designated device (D)
                        "0",    Control channel (device wise)
                        "1",    voice channel 1 (channel wise)
                        ...,
                        "VINETIC_MAX_CH_NR", voice channel VINETIC_MAX_CH_NR (channel wise)


   VINETIC_MAX_DEVICES, number maximum of VINETIC device
   VINETIC_MAX_CH_NR,   number maximum of voice channels

   The TAPI device driver is a multi-device driver, for each device, a device is
   accessed through channels. We distinghish two types of channels:
    - Control channel, used for control and configuration purpose. It's device wise and
    is always the channel number ZERO (0).
    - Voice channel, used to access a device voice channel. It's channel wise and the
    channel is <> to ZERO (0).

    Each device channel (Control/Voice) is represented by a device in the IOS system.
    The device is added to the device list for the VINETIC device driver, the number
    of devices per VINETIC device is:

          VINETIC_MAX_CH_NR (number of Voice channels) + 1 (Control Channel).


    The "Y_S_dev_header" structure is passed to the IOS sytem:
        DevHdr: VxWorks specific header
        ddrvn : VINETIC Device Driver Number (allocated by IOS)
        devn  : device number  (1 to VINETIC_MAX_DEVICES)
        chnn  : channel number (0 to VINETIC_MAX_CH_NR)
        pctx  : pointer to the channel context (will be allocated at the device open)


   \param pDevCtx - The Global device driver structure address

   \return OK or ERROR
*/
static IFX_int32_t TAPI_DevCreate(Y_S_dev_ctx* pDevCtx,
                                  IFX_char_t* pzsDeviceName)
{
   IFX_uint8_t    nDev, nCh, nMinor;
   IFX_char_t     buf[64];

   if ((IFX_NULL == pDevCtx) || (IFX_NULL == pzsDeviceName))
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Wrong input arguments. "
            "(File: %s, line: %d)\n", __FILE__, __LINE__));
      return IFX_ERROR;
   }
   /* IOS device driver creation. We add the TAPI device driver to the
      IOS device list providing the device driver function interface.
      The IOS returns the Device Driver Number which will be used later on. */
   pDevCtx->nDrvNum = iosDrvInstall(IFX_NULL, IFX_NULL,
                                   (FUNCPTR) ifx_tapi_open,
                                   (FUNCPTR) ifx_tapi_release,
                                   (FUNCPTR) ifx_tapi_read,
                                   (FUNCPTR) ifx_tapi_write,
                                   (FUNCPTR) ifx_tapi_ioctl);

   if (ERROR == pDevCtx->nDrvNum)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Unable to install the driver. "
            "(File: %s, line: %d)\n", __FILE__, __LINE__));
      return IFX_ERROR;
   }

   /* The device driver is declared to the IOS: Now we add to the device
      driver IOS list the devices for each TAPI device channel. */
   nMinor = 0;
#ifdef TAPI_ONE_DEVNODE
   sprintf(buf, "/dev/%s", pzsDeviceName);
   if (OS_DevNodeRegister (pDevCtx, 1, nMinor, buf) == IFX_ERROR)
      return IFX_ERROR;
   pDevCtx->pDevHdr[nMinor]->nDevNum = nDev;
#else /* TAPI_ONE_DEVNODE */
   for (nDev = 1; nDev < pDevCtx->nMaxDev + 1; nDev++)
   {
      /* Allocate memory for channels and control device. */
      for (nCh = 0; nCh < pDevCtx->nMaxChn + 1; nCh++)
      {
         sprintf(buf, "/dev/%s%d%d", pzsDeviceName, nDev, nCh);
         if (OS_DevNodeRegister (pDevCtx, nDev, nMinor, buf) == IFX_ERROR)
         {
            return IFX_ERROR;
         }
         nMinor++;
      }
   }
#endif /* TAPI_ONE_DEVNODE */

   return IFX_SUCCESS;
} /* TAPI_DevCreate() */


static IFX_int32_t OS_DevNodeRegister (Y_S_dev_ctx* pDevCtx, IFX_uint8_t nDev,
                                       int nMinor, IFX_char_t* buf)
{
   Y_S_dev_header* pDevHeader = IFX_NULL;


   /* Allocate memory for the device header. Reset the memory. */
   pDevHeader = (Y_S_dev_header*) TAPI_OS_Malloc (sizeof(Y_S_dev_header));
   if (IFX_NULL == pDevHeader)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Err get mem pDevHeader. "
            "(File: %s, line: %d)\n", __FILE__, __LINE__));
      return IFX_ERROR;
   }
   memset(pDevHeader, 0x00, sizeof(Y_S_dev_header));

   /* Build the device string "/dev/vinij" and add the device to the
      IOS device list. */
   if (iosDevAdd(&pDevHeader->DevHdr, buf, pDevCtx->nDrvNum) == ERROR)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Unable to add the device %s "
            "(File: %s, line: %d)\n", buf, __FILE__, __LINE__));
      TAPI_OS_Free (pDevHeader);
      return (IFX_ERROR);
   }

   /* Fill the device header with intial values. */
   pDevHeader->nMajorNum = pDevCtx->nMajorNum;
   pDevHeader->nDevNum = nDev;
   pDevHeader->nInUse = 0;
   pDevHeader->nFds = (nMinor == 0 ? IFX_TAPI_DEVICE_CH_NUMBER : nMinor-1);

   /* Add crossreference to the device context. */
   pDevCtx->pDevHdr[nMinor] = pDevHeader;

   return IFX_SUCCESS;
}

/**

   TAPI device driver deletion.
   This function is called at the device driver shutdown
   (Tapi_DeviceDriverStop) in order to release and uninstall the OS-target
   resources (VxWorks) granted by the TAPI device driver.

   \param pDevCtx - The Global device driver structure address

   \return
   OK or ERROR
*/
static IFX_int32_t TAPI_DevDelete(Y_S_dev_ctx* pDevCtx)
{
   IFX_uint8_t nDev = 0;


   /* Check that we have a valid Global structure address      */
   /* and that a device is installed (OS-target point of view) */

   if ((pDevCtx == IFX_NULL) || (pDevCtx->nDrvNum == ERROR))
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Wrong input arguments. "
            "(File: %s, line: %d)\n", __FILE__, __LINE__));

      return (ERROR);
   }

   for (nDev = 0; nDev < pDevCtx->nMaxDev; nDev++)
   {

      /* Delete the device from the IOS link list */
      /* free the device header                   */

      if (pDevCtx->pDevHdr[nDev])
      {
         iosDevDelete((DEV_HDR *)pDevCtx->pDevHdr[nDev]);
         TAPI_OS_Free (pDevCtx->pDevHdr[nDev]);
      }
   }

   /* We remove the TAPI device driver from the IOS. */
   /* We free the Tapi device structures.            */

   iosDrvRemove(pDevCtx->nDrvNum, IFX_TRUE);

   /* Reset to 0 the Global device driver structure */
   /* in order to reset all the pointers to NULL    */
   /* marking that the device driver is no longer   */
   /* installed.                                    */

   memset(pDevCtx, 0x00, sizeof(Y_S_dev_ctx));
   return (OK);
} /* TAPI_DevDelete() */


/* --------------------------------------------------------------------------
                          CREATE/DELETE driver  -->  END
   --------------------------------------------------------------------------
*/


/* --------------------------------------------------------------------------
                             SELECT mechanism  -->  BEGIN
   --------------------------------------------------------------------------
*/

/**
   Executes the select for the channel fd.

   \param pTapiCh  - handle to channel control structure
   \param node - node list
   \param opt - optional argument, which contains needed information for
                IFXOS_SleepQueue

   \return System event qualifier. Either 0 or TAPI_OS_SYSREAD.
           IFX_ERROR fo error.

   \remarks
   This function needs operating system services, that are hidden by
   IFXOS macros.
*/
IFX_int32_t TAPI_SelectCh(TAPI_CHANNEL* pTapiCh,
                          IFX_int32_t node,
                          IFX_int32_t opt)
{
   IFX_TAPI_DRV_CTX_t* pDrvCtx = IFX_NULL;
   IFX_TAPI_LL_DEV_t* pLLDev = IFX_NULL;
   IFX_uint32_t  flags = 0;
   IFX_int32_t   ret = 0;
   IFX_TAPI_T38_STATUS_t TapiFaxStatus;


   if (IFX_NULL == pTapiCh)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Wrong input arguments. "
            "(File: %s, line: %d)\n", __FILE__, __LINE__));
      return IFX_ERROR;
   }

   pDrvCtx = (IFX_TAPI_DRV_CTX_t*) pTapiCh->pTapiDevice->pDevDrvCtx;
   pLLDev = pTapiCh->pTapiDevice->pLLDev;

   /* Get the Status from the low level driver. */
   if (ptr_chk(pDrvCtx->COD.T38_Status_Get,
              "pDrvCtx->COD.T38_Status_Get"))
   {
      pDrvCtx->COD.T38_Status_Get(pTapiCh->pLLChannel, &TapiFaxStatus);
   }

   selNodeAdd ((SEL_WAKEUP_LIST *)&pTapiCh->wqRead,  (SEL_WAKEUP_NODE *)node);

#ifdef TAPI_FAX_T38
   selNodeAdd ((SEL_WAKEUP_LIST *)&pTapiCh->wqWrite,  (SEL_WAKEUP_NODE *)node);

   if ((TapiFaxStatus.nStatus & IFX_TAPI_FAX_T38_TX_ON)
      && (IFX_TRUE == pTapiCh->bFaxDataRequest))
   {
      /* Task should write a new packet now. */
      ret |= TAPI_OS_SYSWRITE;
   }
#endif /* TAPI_FAX_T38 */
   /* Select on a voice channel -- only implemented for TAPI. */
   flags |= CF_NEED_WAKEUP;
   /* Clear flags first, then apply new flags. */

   if ((IFX_NULL == pDrvCtx->IRQ.LockDevice)
       || (IFX_NULL == pDrvCtx->IRQ.UnlockDevice))
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("IRQ missing for locking/unlocking"
            " the device. (File: %s, line: %d)\n", __FILE__, __LINE__));
      return IFX_ERROR;
   }

   pDrvCtx->IRQ.LockDevice (pLLDev);

#ifdef TAPI_PACKET
   if (!fifoEmpty(pTapiCh->pUpStreamFifo[IFX_TAPI_STREAM_COD]))
   {
      flags |= CF_WAKEUPSRC_STREAM;
      flags &= ~CF_NEED_WAKEUP;
      ret |= TAPI_OS_SYSREAD;
   }
#endif /* TAPI_PACKET */

   pTapiCh->nFlags &= ~(CF_WAKEUPSRC_GR909
                        | CF_WAKEUPSRC_STREAM
                        | CF_WAKEUPSRC_TAPI
                        | CF_NEED_WAKEUP);
   pTapiCh->nFlags |= flags;

   pDrvCtx->IRQ.UnlockDevice(pLLDev);

   return ret;
} /* TAPI_SelectCh() */


/**
   Handles the FIOSELECT system call.

   \param  pCtx         Pointer to IOCTL context struct.
   \param  nArgument    Node argument of FIOSELECT call.

   \return IFX_SUCCESS / IFX_ERROR

   \remark
   The set of wake up queues are different for the VINETIC configuration/control
   channel and for the voice channel.
*/
static IFX_int32_t OS_OnSelect(IFX_TAPI_ioctlCtx_t* pCtx,
                               IFX_int32_t nArgument)
{
   TAPI_DEV       *pTapiDev = pCtx->pTapiDev;
   IFX_int32_t    ret = 0;
   IFX_uint8_t    i,
                  needWakeup;

   if (pCtx->nFds == IFX_TAPI_DEVICE_CH_NUMBER)
   {
      /* handle device file descriptor */

      /* add node to the wakup list */
      selNodeAdd(&(pTapiDev->wqEvent), (SEL_WAKEUP_NODE *) nArgument);
      pTapiDev->bNeedWakeup = IFX_TRUE;

      /* check if any data channels have events available */
      for (i=0, needWakeup=0;
           (i < pTapiDev->nMaxChannel) && (needWakeup == 0); i++)
      {
         TAPI_CHANNEL *pTapiCh = &pTapiDev->pTapiChanelArray[i];

         if ((pTapiCh->bInitialized == IFX_TRUE) &&
                     !IFX_TAPI_EventFifoEmpty(pTapiCh))
            needWakeup ++;
      }

      if (needWakeup != 0)
      {
         /* events are pending - wake up the task */
         selWakeup((SEL_WAKEUP_NODE *) nArgument);
         pTapiDev->bNeedWakeup = IFX_FALSE;
      }

#ifdef EVALUATION
      {
         IFX_int32_t    ch;
         VINETIC_EVAL   *pEval = pTapiDev->pEval;

         /* No LEC tests are supported. */
         /* Select on a configuration/control channel. */
         if (IFX_TRUE == pEval->bPollRead [POLLDEV_INTINFO])
         {
            pEval->bNeedWakeup = IFX_TRUE;
            selNodeAdd(&(pEval->PollQueue[POLLDEV_INTINFO]),
                       (SEL_WAKEUP_NODE *) nArgument);
            if (pEval->bPollEvt[POLLDEV_INTINFO] == IFX_TRUE)
            {
               pEval->bNeedWakeup = IFX_FALSE;
               selWakeup((SEL_WAKEUP_NODE *) nArgument);
            }
            /* Reset for futher use. */
            pEval->bPollEvt[POLLDEV_INTINFO] = IFX_FALSE;
         }

         for (ch = 0; ch < pTapiDev->nMaxChannel; ch ++)
         {
            if (IFX_TRUE ==
                ((VINCH_EVAL *) pTapiDev->pChannel[ch].pEval)->bDialDetection)
            {
               pEval->bNeedWakeup = IFX_TRUE;
               selNodeAdd(&(pEval->PollQueue[POLLDEV_INTINFO]),
                          (SEL_WAKEUP_NODE *) nArgument);
               if (IFX_TRUE ==
                   ((VINCH_EVAL *) pTapiDev->pChannel[ch].pEval)->bDialDetected)
               {
                  pEval->bNeedWakeup = IFX_FALSE;
                  selWakeup((SEL_WAKEUP_NODE *) nArgument);
               }
            }
         }
      }
#endif /* EVALUATION */
   }
   else
   {
      /* handle channel file descriptors */

      TAPI_CHANNEL *pTapiCh = &(pTapiDev->pTapiChanelArray[pCtx->nFds]);

      if (pTapiCh->bInitialized == IFX_TRUE)
      {
         ret = TAPI_SelectCh(pTapiCh, nArgument, 0);

         if ((ret & TAPI_OS_SYSREAD) &&
             (selWakeupType((SEL_WAKEUP_NODE*) nArgument) == SELREAD))
         {
            selWakeup((SEL_WAKEUP_NODE*) nArgument);
         }

         if ((ret & TAPI_OS_SYSWRITE) &&
             (selWakeupType((SEL_WAKEUP_NODE*) nArgument) == SELWRITE))
         {
            selWakeup((SEL_WAKEUP_NODE*) nArgument);
         }
      }

      /* Select() requires a return value of zero (OK, IFX_SUCCESS). */
   }

   return IFX_SUCCESS;
} /* OS_OnSelect() */


/**
   Handles the FIOUNSELECT system call.

   \param  pCtx         Pointer to IOCTL context struct.
   \param  nArgument    Node argument of FIOUNSELECT call.

   \return IFX_SUCCESS / IFX_ERROR

   \remarks
   The set of wake up queues are different for the VINETIC configuration/control
   channel and for the voice channel.
*/
static IFX_int32_t OS_OnUnselect(IFX_TAPI_ioctlCtx_t* pCtx,
                                 IFX_int32_t nArgument)
{
   TAPI_DEV       *pTapiDev = pCtx->pTapiDev;

   if (pCtx->nFds == IFX_TAPI_DEVICE_CH_NUMBER)
   {
      /* handle device file descriptor */

      selNodeDelete(&(pTapiDev->wqEvent), (SEL_WAKEUP_NODE *) nArgument);
#ifdef EVALUATION
      {
         IFX_int32_t    ch;
         VINETIC_EVAL   *pEval = pTapiDev->pEval;

         /* unselect on a configuration/control  channel */
         if (IFX_TRUE == pEval->bPollRead[POLLDEV_INTINFO])
         {
            selNodeDelete(&(pEval->PollQueue[POLLDEV_INTINFO]),
                          (SEL_WAKEUP_NODE *) nArgument);
         }
         for (ch = 0; ch < pTapiDev->nMaxChannel; ch++)
         {
            if (IFX_TRUE == ((VINCH_EVAL *) pTapiDev->pChannel[ch].pEval)->bDialDetection)
            {
               selNodeDelete(&(pEval->PollQueue[POLLDEV_INTINFO]),
                             (SEL_WAKEUP_NODE *) nArgument);
            }
         }
      }
#endif /* EVALUATION */
   }
   else
   {
      /* handle channel file descriptors */

      TAPI_CHANNEL *pTapiCh = &(pTapiDev->pTapiChanelArray[pCtx->nFds]);

      /* Voice Streaming */
      selNodeDelete(&(pTapiCh->wqRead), (SEL_WAKEUP_NODE *) nArgument);
#ifdef EXCHFD_SUPPORT
      selNodeDelete(&(pTapiCh->WakeupList), (SEL_WAKEUP_NODE *) nArgument);
#endif /* EXCHFD_SUPPORT */
#ifdef TAPI_FAX_T38
      selNodeDelete(&(pTapiCh->wqWrite), (SEL_WAKEUP_NODE *) nArgument);
#endif /* (TAPI_FAX_T38) */
      /* clear */
      pTapiCh->nFlags &= ~CF_NEED_WAKEUP;
   }

   return IFX_SUCCESS;
}


/* --------------------------------------------------------------------------
                             SELECT mechanism  -->  END
   --------------------------------------------------------------------------
*/


/* --------------------------------------------------------------------------
                             TIMER mechanisms  -->  BEGIN
   --------------------------------------------------------------------------
*/


/* New usage of timers. We have high priority task which waits for message
   pointer to timer struct or timer struct. When he gets it it starts func with
   parameters (both arguments located in timer struct).
   Timer are used as before (create it, start it, stop it, ..) BUT when timer
   elapses message with usefull data is send.
   When timer is created also array of timer struct is created.
 */


/** Message queue ID, holding funcs with arguments to be handled
    after timer has elapsed. */
static MSG_Q_ID nTimerMsgQueue = 0;

/** Task ID for high priority task handling timer messages */
static IFX_int32_t nTimerMsgHandler_ID = -1;

/** Max. messages in queue. */
enum { MSG_CNT_TIMER_QUEUE = 32 };

/** Message size in queue. */
const IFX_int32_t MSG_SIZE_TIMER_QUEUE = sizeof(IFX_int32_t);

/** Initial size of array in element count when created. */
static IFX_int32_t START_ELEM_CNT = 10;

/** Increase number of elements for array if full. */
enum { INCREASE_ELEM_CNT = 5 };

/** Holding timer information. When timer will elapse function
    with following prototype will be called : func(timer, arguments) */
typedef struct _TIMER_STRUCT_t
{
   /** Timer ID, needed when calling function */
   Timer_ID Timer;
   /** Arguments to func when timer will elapse. */
   IFX_int32_t nArg;
   /** Func called when timer will elapse. */
   TIMER_ENTRY pFunc;
   /** Flag if used by timer, IFX_TRUE used, IFX_FALSE not used */
/*   IFX_boolean_t fUsed;*/
} TIMER_STRUCT_t;

/** Array of timer struct. */
static TIMER_STRUCT_t* rgTimers = IFX_NULL;

/** Number of timer struct in array. */
static IFX_int32_t nTimersCnt = 0;

/** Number of used timer struct in array. */
static IFX_int32_t nUsedTimers = 0;



/**
   When timer elapses this function is called and will send message
   to message queue.

   \param Timer - Timer ID (timer structure)
   \param nArg  - array index of timers
 */
IFX_void_t TAPI_Timer_SendMsg(Timer_ID Timer, IFX_int32_t nArg)
{
   IFX_return_t ret = IFX_SUCCESS;


   /* Check if message queue exists */
   if (nTimerMsgQueue == IFX_NULL)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Message queue is missing.\n"));
      return;
   }

   /* Just make copy of ptr and send it. */
   ret = msgQSend(nTimerMsgQueue, /* Message queue ID */
                  (IFX_char_t *) &nArg, /* Message, just pointer to it */
                   MSG_SIZE_TIMER_QUEUE, /* Message len */
                   NO_WAIT,
                   MSG_PRI_NORMAL);

   if (ret != IFX_SUCCESS)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
            ("TAPI_EVENT: Error sending messsage, errno %d.\n", errno));
      return;
   }
}


/**
   This function will read message queue and start func.

   \return
      On success IFX_SUCCESS or else IFX_ERROR

   \remarks
    Aftert timer has elapsed message was send with msgQSend()
    this handler is waiting for it and handle it.
 */
static IFX_void_t TAPI_HandleTimerMsg(IFX_void_t)
{
   IFX_int32_t ret = IFX_SUCCESS;
   TIMER_STRUCT_t* pParam = IFX_NULL;
   IFX_int32_t timer_idx = 0;


   /* Check if message queue exists */
   if (nTimerMsgQueue == IFX_NULL)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Message queue is missing.\n"));
      return;
   }

   for(;;)
   {
      /* Wait for message, got number of bytes read or error (-1). */
      ret = msgQReceive(nTimerMsgQueue,
                        (IFX_char_t *) &timer_idx,
                        MSG_SIZE_TIMER_QUEUE,
                        WAIT_FOREVER);

      if ((ret == ERROR) || (ret < MSG_SIZE_TIMER_QUEUE))
      {
         TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Error receiving message %d.\n", ret));
      }
      else
      {
         if ((0 > timer_idx) || (nTimersCnt < timer_idx))
         {
            TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Timer message with wrong value received %d\n", timer_idx));
            return;
         }

         /* Call function to handle this event. This one will also
            free buffer, put them back to bufferpool. */
         pParam = &rgTimers[timer_idx - 1];

         pParam->pFunc(pParam->Timer, (IFX_int32_t) pParam->nArg);
      }
   }
}


/**
   Initialize mesage queue and starts message queue handler.

   \param none

   \return IFX_SUCESS on ok, otherwise IFX_ERROR.
*/
IFX_return_t TAPI_Timer_StartMsgQueue(IFX_void_t)
{
   IFX_return_t ret = IFX_SUCCESS;


   TRACE(TAPI_DRV, DBG_LEVEL_NORMAL, ("Create message queue.\n"));

   /* Create message queue for dispatching events */
   nTimerMsgQueue = msgQCreate(MSG_CNT_TIMER_QUEUE,
                               MSG_SIZE_TIMER_QUEUE,
                               0 /* MSG_Q_FIFO */);

   if (nTimerMsgQueue == IFX_NULL)
   {
      /* Error creating mesage queue. */
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Error creating message queue.\n"));

      ret = IFX_ERROR;
   }

   if ((ret != IFX_ERROR) && (nTimerMsgHandler_ID == -1))
   {
      TRACE(TAPI_DRV, DBG_LEVEL_NORMAL,
            ("Start task which will handle timer messages.\n"));

      /* Create task which will read events from message queue and
         call func to handle them */
      nTimerMsgHandler_ID = taskSpawn("tTimerMsgHandler",
                                      TSK_PRIO_TIMER_HANDLER,
                                      0, /* Options */
                                      8192, /* Stack size */
                                      (FUNCPTR) TAPI_HandleTimerMsg,
                                      /* 10 arguments */
                                      0, 0, 0, 0, 0, 0, 0, 0, 0, 0);

      if (nTimerMsgHandler_ID == IFX_ERROR)
      {
         TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Error creating task.\n"));

         ret = IFX_ERROR;
      }
   }

   return ret;
}


/**
   Search for free timer struct in array.

   \param nTimerIdx - array index of free element in array

   \return pointer to timer structure, otherwise IFX_NULL.
 */
TIMER_STRUCT_t* TAPI_Timer_GetFreeElem(IFX_int32_t* nTimerIdx)
{
   TIMER_STRUCT_t* free_struct = IFX_NULL;
   IFX_int32_t i = 0;
   IFX_void_t* increased_part = IFX_NULL;


   if (nTimerIdx == IFX_NULL)
   {
      /* Wrong input arguments */
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
           ("Invalid input argument(s). (File: %s, line: %d)\n",
            __FILE__, __LINE__));
      return IFX_NULL;
   }

   if (rgTimers == IFX_NULL)
   {
      nTimersCnt = START_ELEM_CNT;

      TRACE(TAPI_DRV, DBG_LEVEL_NORMAL,
            ("Timer array is created with num %d of timers\n",
            (int) nTimersCnt));

      /* Allocate memory for timers. */
      /*          dev * ch * max_timers elements, SIZE - size of struct */
      rgTimers = calloc(nTimersCnt, sizeof(TIMER_STRUCT_t));
      if (rgTimers == IFX_NULL)
      {
         TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
               ("Error creating array of timer structs."
                "(File: %s, line: %d)\n", __FILE__, __LINE__));
         return IFX_NULL;
      }
   }

   free_struct = IFX_NULL;

   /* Search for free struct */
   for (i = 0; i < nTimersCnt; i++)
   {
      if (rgTimers[i].pFunc == IFX_NULL)
      {
         /* Got free struct */
         free_struct = &rgTimers[i];
         *nTimerIdx = i + 1;
         break;
      }
   }

   if (free_struct == IFX_NULL)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_NORMAL,
            ("Array is full, increase its size for %d timers\n",
             INCREASE_ELEM_CNT));
      /* Array of timer struct is full, try to increase it */
      increased_part = realloc(rgTimers, (nTimersCnt + INCREASE_ELEM_CNT)
                                          * sizeof(TIMER_STRUCT_t));
      if (increased_part == IFX_NULL)
      {
         TRACE(TAPI_DRV, DBG_LEVEL_NORMAL,
               ("Error increasing array of timer structs."
                " (File: %s, line: %d)\n", __FILE__, __LINE__));
         return IFX_NULL;
      }
      else
      {
         /* \todo should we free old structure */
         /*if (IFX_NULL != rgTimers)
         {
            free(rgTimers);
            rgTimers = IFX_NULL;
         }*/
         rgTimers = increased_part;
      }

      /* Set to zero increased memory. */
      memset(&rgTimers[nTimersCnt], 0,
             INCREASE_ELEM_CNT * sizeof(TIMER_STRUCT_t));

      /* Set free_struct to first timer in mem which was increased */
      free_struct = &rgTimers[nTimersCnt];
      *nTimerIdx = nTimersCnt + 1;

      /* Increase count of timers */
      nTimersCnt += INCREASE_ELEM_CNT;
   }

   /* Also increase number of used timer structs. */
   nUsedTimers++;

   return free_struct;
}


/**
   Function create a timer.

   \param pTimerEntry - Function pointer to the call back function
   \param nArgument   - Argument including the Dev structure
                       (as integer pointer)
   \return Timer - Timer ID handle for vxWorks
*/
Timer_ID TAPI_Create_Timer(TIMER_ENTRY pTimerEntry, IFX_ulong_t nArgument)
{
   /* Use common timer functions if sys_timerlib_vxworks.c is included
      (done in BSPProj.c) */
   TIMER_STRUCT_t* free_struct = IFX_NULL;
   IFX_int32_t timer_idx = 0;


   TAPI_OS_MutexGet (&semTimerArrDataLock);

   if (nTimerMsgQueue == 0)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_NORMAL, ("Create timer msg queue and start handler\n"));
      TAPI_Timer_StartMsgQueue();
   }

   free_struct = TAPI_Timer_GetFreeElem(&timer_idx);
   if (free_struct == IFX_NULL)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Err getting free timer struct elem"
            " from array. (File: %s, line: %d)\n", __FILE__, __LINE__));

      TAPI_OS_MutexRelease (&semTimerArrDataLock);

      return (0);
   }

   TAPI_OS_MutexRelease (&semTimerArrDataLock);

   free_struct->nArg = nArgument;
   free_struct->pFunc = pTimerEntry;

#ifndef SYS_TIMERLIB_VXWORKS_H

   /* Derive timer from CLK realtimer, do not use signal handler. */
   if (timer_create(CLOCK_REALTIME, NULL, &free_struct->Timer) == IFX_ERROR)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Err creating timer. "
            "(File: %s, line: %d)\n", __FILE__, __LINE__));

      /* Set also that this timer struct is free, because could not create timer. */
      free_struct->pFunc = IFX_NULL;

      return (0);
   }

   /* Connect timer to function, which will send message with arguments. */
   if (timer_connect(free_struct->Timer, TAPI_Timer_SendMsg,
                     (IFX_int32_t) timer_idx) == IFX_ERROR)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Err connecting to timer. "
            "(File: %s, line: %d)\n", __FILE__, __LINE__));

      /* Set also that this timer struct is free, because could not create timer. */
      free_struct->pFunc = IFX_NULL;

      return (0);
   }


   return (free_struct->Timer);
#else
   free_struct->Timer = InstallTimer(TAPI_Timer_SendMsg, 0, timer_idx,
                                                 IFX_FALSE);
   return free_struct->Timer;
#endif
}


/**
   Function set and starts a timer with a specific time. It can be choose if the
   timer starts periodically.

   \param Timer - Timer ID handle for vxWorks
   \param nTime - Time in ms
   \param bPeriodically - Starts the timer periodically or not
   \param bRestart - Restart the timer or normal start

   \return Returns an error code: IFX_TRUE / IFX_FALSE
*/
IFX_boolean_t TAPI_SetTime_Timer(Timer_ID Timer,
                                 IFX_uint32_t nTime,
                                 IFX_boolean_t bPeriodically,
                                 IFX_boolean_t bRestart)
{
   /* Use common timer functions if sys_timerlib_vxworks.c is included (done in BSPProj.c). */
#ifndef SYS_TIMERLIB_VXWORKS_H
   struct itimerspec   timeToSet;        /* time to be set */
   struct timespec     timeValue;        /* timer expiration value */
   struct timespec     timeInterval;     /* timer period */


   /* Stop timer. */
   if (bRestart == IFX_TRUE)
   {
      if (timer_cancel(Timer) == IFX_ERROR)
      {
         TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Err cancelling timer. "
               "(File: %s, line: %d)\n", __FILE__, __LINE__));

         return (IFX_FALSE);
      }
   }

   /* Initialize timer expiration value. */
   timeValue.tv_sec = (nTime / 1000);
   timeValue.tv_nsec = (nTime % 1000) * 1000 * 1000;

   /* Initialize timer period */
   if (bPeriodically == IFX_TRUE)
   {
      timeInterval.tv_sec = (nTime / 1000);
      timeInterval.tv_nsec = (nTime % 1000) * 1000 * 1000;
   }
   else
   {
      timeInterval.tv_sec = 0;
      timeInterval.tv_nsec = 0;
   }

   /* Reset timer structure. */
   memset((IFX_char_t *) &timeToSet, 0, sizeof (struct itimerspec));

   /* Set the time to be set value. */
   /* NOTICE: Copy all parameter in simple steps. This is a workaround to avoid
      crashes on the CheckBoard, because of incompatiple compiler versions. */
   timeToSet.it_value.tv_sec = timeValue.tv_sec;
   timeToSet.it_value.tv_nsec = timeValue.tv_nsec;
   timeToSet.it_interval.tv_sec = timeInterval.tv_sec;
   timeToSet.it_interval.tv_nsec = timeInterval.tv_nsec;

   /* Pass timer value & reload value. */
   /* NOTICE: Do not use 'TIMER_ABSTIME' flag because timer will then expire
      immediatly. */
   if (timer_settime(Timer, 0, &timeToSet, IFX_NULL) == IFX_ERROR)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Err setting time for timer. "
            "(File: %s, line: %d)\n", __FILE__, __LINE__));

      return (IFX_FALSE);
   }

   return (IFX_TRUE);
#else
   return ( SetTimeTimer(Timer, nTime, bPeriodically, bRestart) );
#endif
}


/**

   Function stop a timer.

   \param Timer - Timer ID handle for vxWorks

   \return Returns an error code: IFX_TRUE / IFX_FALSE
*/
IFX_boolean_t TAPI_Stop_Timer(Timer_ID Timer)
{
#ifndef SYS_TIMERLIB_VXWORKS_H
   /* Stop timer. */
   if (timer_cancel(Timer) == IFX_ERROR)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Err cancelling a timer. "
            "(File: %s, line: %d)\n", __FILE__, __LINE__));

      return (IFX_FALSE);
   }

   return (IFX_TRUE);
#else
   return ( StopTimer(Timer) );
#endif
}


/**
   Search for used struct and free it (its not used by this timer anymore).

   \param Timer - Pointer to timer struct
*/
IFX_void_t TAPI_Timer_RemoveStruct(Timer_ID* Timer)
{
   IFX_int32_t i = 0;


   if (Timer == IFX_NULL)
   {
      /* Wrong input arguments */
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
           ("Invalid input argument(s). (File: %s, line: %d)\n",
            __FILE__, __LINE__));
      return;
   }

   TAPI_OS_MutexGet (&semTimerArrDataLock);

   /* Search for free struct */
   for (i = 0; i < nTimersCnt; i++)
   {
      if (Timer == &rgTimers[i].Timer)
      {
         /* Set to unused */
         memset(&rgTimers[i], 0, sizeof(TIMER_STRUCT_t));

         /* Decrease number of used timer structs. */
         nUsedTimers--;

         if (nUsedTimers == 0)
         {
            TRACE(TAPI_DRV, DBG_LEVEL_NORMAL, ("Remove array with timer structs\n"));
            /* Release array of timer structs, because its empty. */
            free(rgTimers);
            rgTimers = IFX_NULL;
         }

         break;
      }
   }

   TAPI_OS_MutexRelease (&semTimerArrDataLock);
}


/**

   Function delete a timer.

   \param Timer - Timer ID handle for vxWorks

   \return Returns an error code: IFX_TRUE / IFX_FALSE
*/
IFX_boolean_t TAPI_Delete_Timer(Timer_ID Timer)
{
#ifndef SYS_TIMERLIB_VXWORKS_H
   /* Stop timer. */
   if (timer_cancel(Timer) == IFX_ERROR)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Err cancelling a timer. "
            "(File: %s, line: %d)\n", __FILE__, __LINE__));
      return (IFX_FALSE);
   }

   /* Delete timer. */
   if (timer_delete(Timer) == IFX_ERROR)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Err deleting timer. "
            "(File: %s, line: %d)\n", __FILE__, __LINE__));
      return (IFX_FALSE);
   }

   TAPI_Timer_RemoveStruct(&Timer);

   return (IFX_TRUE);
#else
   TAPI_Timer_RemoveStruct(&Timer);

   return ( DeleteTimer(Timer) );
#endif
}


/* --------------------------------------------------------------------------
                             TIMER mechanisms  -->  END
   --------------------------------------------------------------------------
*/


/* --------------------------------------------------------------------------
                             EVENT handling  -->  BEGIN
   --------------------------------------------------------------------------
*/

/**
   Defer work to process context

   Puts the pointer into a msgQ from where it will be processed by a task.

   \param pFunc - pointer to function to be called (not needed in VxWorks)
   \param pParam - parameter passed to the function

   \return IFX_SUCCESS or IFX_ERROR in case of an error

   \remarks
    In VxWorks taskSpawn() is not working when we are in interrupt context,
    so message queue will be used.
*/
IFX_return_t TAPI_DeferWork(IFX_void_t* pFunc, IFX_void_t* pParam)
{
   IFX_return_t ret = IFX_ERROR;

   if (pParam == IFX_NULL)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Wrong input arguments."
            "(File: %s, line: %d)\n", __FILE__, __LINE__));
      return IFX_ERROR;
   }

   if (nMsgQueue_Events != IFX_NULL)
   {
      /* We need a variable that we can take the address of. */
      IFX_void_t *tmp_param = pParam;

      /* Copy the pointer value into the queue. */
      ret = msgQSend(nMsgQueue_Events, /* Message queue ID */
                     (IFX_char_t *) &tmp_param, /* Message */
                     MSG_SIZE_EVENT_QUEUE, /* Message len */
                     NO_WAIT,
                     MSG_PRI_NORMAL);
      if (ret != IFX_SUCCESS)
      {
         TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
               ("TAPI_EVENT: Error sending messsage, errno %d.\n", errno));
      }
   }
   else
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
            ("Defer TAPI event: No message queue, so won't send message.\n"));

      ret = IFX_ERROR;
   }

   return ret;
}


/**
   This function will read message queue for events.

   \remarks
   When message is send with msgQSend() this handler is waiting for it
   and dispatch it to event handler.
*/
static IFX_void_t IFX_TAPI_Event_Dispatch_Queue(IFX_void_t)
{
   IFX_return_t ret = IFX_SUCCESS;
   IFX_TAPI_EXT_EVENT_PARAM_t* pParam = IFX_NULL;

   /* Check if message queue exists */
   if (nMsgQueue_Events == IFX_NULL)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Message queue is missing.\n"));
      return;
   }

   for (;;)
   {
      /* Wait for message */
      ret = msgQReceive(nMsgQueue_Events,
                        (IFX_char_t *) &pParam,
                        MSG_SIZE_EVENT_QUEUE,
                        WAIT_FOREVER);

      if ((ret == IFX_ERROR) || (ret < MSG_SIZE_EVENT_QUEUE))
      {
         TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
               ("Error receiving message %d.\n", ret));
      }
      else
      {
         if (IFX_TAPI_Event_Dispatch_ProcessCtx(pParam) != IFX_SUCCESS)
         {
            TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Error when processing event.\n"));
         }
      }
   }
}


/**
   Initialize mesage queue and starts message queue handler.

   \param none

   \return IFX_SUCESS on ok, otherwise IFX_ERROR.
*/
IFX_return_t ifx_tapi_Event_StartMsgQueue(IFX_void_t)
{
   IFX_return_t ret = IFX_SUCCESS;

   /* Create message queue for dispatching events */
   nMsgQueue_Events = msgQCreate(MAX_MSG_CNT_EVENT_QUEUE,
                                 MSG_SIZE_EVENT_QUEUE,
                                 0 /* MSG_Q_FIFO */);

   if (IFX_NULL == nMsgQueue_Events)
   {
      /* Error creating mesage queue. */
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Error creating message queue.\n"));
      ret = IFX_ERROR;
   }

   if ((ret != IFX_ERROR) && (nTaskForEvents_ID == -1))
   {
      TRACE(TAPI_DRV, DBG_LEVEL_NORMAL,
            ("Start task for handling msg queue with events.\n"));

      /* Create task which will read events from message queue and
         calls functions to handle them before putting it into the
         fifo towards the application */
      nTaskForEvents_ID = taskSpawn("tEventHandler",
                                    TSK_PRIO_EVENT_HANDLER,
                                    0, /* Options */
                                    8192, /* Stack size */
                                    (FUNCPTR) IFX_TAPI_Event_Dispatch_Queue,
                                    /* 10 arguments */
                                    0, 0, 0, 0, 0, 0, 0, 0, 0, 0);

      if (nTaskForEvents_ID == IFX_ERROR)
      {
         TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Error creating task.\n"));
         ret = IFX_ERROR;
      }
   }

   return ret;
}


#if 0 /* For future use */
/**
   Deletes mesage queue and stops message queue handler.

   \pararm none

   \return IFX_SUCCESS on ok, otherwise IFX_ERROR.
*/
IFX_int32_t IFX_TAPI_Event_StopMsgQueue(IFX_void_t)
{
   IFX_int32_t ret = IFX_SUCCESS;

   /* Delete task */
   if (taskDelete(nTaskForEvents_ID) != IFX_SUCCESS)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Error deleting task.\n"));
      ret = IFX_ERROR;
   }
   /* set to inital state even if kill failed */
   nTaskForEvents_ID = -1;

   /* Delete message queue (try even if task delete failed) */
   if (msgQDelete(nMsgQueue_Events) != IFX_SUCCESS)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Error deleting message queue.\n"));
      ret = IFX_ERROR;
   }
   /* set to inital state even if delete failed */
   nMsgQueue_Events = 0;

   return ret;
}
#endif /* if 0 */

#endif /* VXWORKS */
