#ifndef _LIB_TAPI_LT_GR909_H
#define _LIB_TAPI_LT_GR909_H
/****************************************************************************

                            Copyright (c) 2009
                            Infineon Technologies AG
                     Am Campeon 1-12; 81726 Munich, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

 ****************************************************************************
   Module      : lib_tapi_lt_gr909.h
   Description :
*******************************************************************************/

/**
   \file lib_tapi_lt_gr909.h
   This file contains the interface declaration.
*/

/** \defgroup TAPI_INTERFACE_LT_GR909 Line Testing Interfaces
   This section describes line testing interfaces.*/

/** \addtogroup TAPI_INTERFACE_LT_GR909 */
/*@{*/

/* ============================= */
/* Global Defines & enums        */
/* ============================= */

/** GR-909 tests masks.*/
typedef enum
{
   /** Mask to select HPT. */
   IFX_LT_GR909_HPT_MASK  = (1 << 0),
   /** Mask to select FEMF. */
   IFX_LT_GR909_FEMF_MASK = (1 << 1),
   /** Mask to select RFT. */
   IFX_LT_GR909_RFT_MASK  = (1 << 2),
   /** Mask to select ROH. */
   IFX_LT_GR909_ROH_MASK  = (1 << 3),
   /** Mask to select RIT. */
   IFX_LT_GR909_RIT_MASK  = (1 << 4)
} IFX_LT_GR909_MASK_t;

/* ============================= */
/* Global Structures             */
/* ============================= */

/** Hazardous potential test results. */
typedef struct
{
   /** HPT result, passed or failed. */
   IFX_boolean_t b_result;
   /** HPT AC ring wire to gnd value, [Vrms]. */
   IFX_float_t   f_hpt_ac_r2g;
   /** HPT AC TIP wire to GND value, [Vrms]. */
   IFX_float_t   f_hpt_ac_t2g;
   /** HPT AC tip wire to ring wire value, [Vrms]. */
   IFX_float_t   f_hpt_ac_t2r;
   /** HPT DC ring wire to gnd value, [V]. */
   IFX_float_t   f_hpt_dc_r2g;
   /** HPT DC tip wire to gnd value, [V]. */
   IFX_float_t   f_hpt_dc_t2g;
   /** HPT DC tip wire to ring wire value, [V].*/
   IFX_float_t   f_hpt_dc_t2r;
} IFX_LT_GR909_HPT_t;

/** Foreign electromotive forces test results. */
typedef struct
{
   /** FEMF result, passed or failed. */
   IFX_boolean_t b_result;
   /** FEMF ac ring wire to gnd value, [Vrms]. */
   IFX_float_t   f_femf_ac_r2g;
   /** fFEMF AC tip wire to gnd value, [Vrms.] */
   IFX_float_t   f_femf_ac_t2g;
   /** FEMF AC tip wire to ring wire value, [Vrms]. */
   IFX_float_t   f_femf_ac_t2r;
   /** FEMF DC ring wire to gnd value, [V]. */
   IFX_float_t   f_femf_dc_r2g;
   /** FEMF DC tip wire to gnd value, [V]. */
   IFX_float_t   f_femf_dc_t2g;
   /** FEMF DC tip wire to ring wire value, [V]. */
   IFX_float_t   f_femf_dc_t2r;
} IFX_LT_GR909_FEMF_t;

/** Resistive faults test results. */
typedef struct
{
   /** RFT result, passed or failed. */
   IFX_boolean_t b_result;
   /** RFT ring wire to gnd value, [Ohm]. */
   IFX_float_t   f_rft_r2g;
   /** RFT tip wire to gnd value, [Ohm]. */
   IFX_float_t   f_rft_t2g;
   /** RFT tip wire to ring wire value, [Ohm]. */
   IFX_float_t   f_rft_t2r;
} IFX_LT_GR909_RFT_t;

/** Receiver off-hook test results. */
typedef struct
{
   /** ROH result, passed or failed. */
   IFX_boolean_t b_result;
   /** ROH tip wire to ring wire value for low voltage, [Ohm]. */
   IFX_float_t   f_roh_t2r_l;
   /** ROH tip wire to ring wire value for high voltage, [Ohm]. */
   IFX_float_t   f_roh_t2r_h;
} IFX_LT_GR909_ROH_t;

/** Ringer impedance test results. */
typedef struct
{
   /** RIT result, passed or failed. */
   IFX_boolean_t b_result;
   /** RIT value, , [Ohm]. */
   IFX_float_t   f_rit_res;
} IFX_LT_GR909_RIT_t;

/** GR-909 results structure. */
typedef struct
{
   /** Valid results mask, set with \ref IFX_LT_GR909_MASK_t. */
   IFX_uint32_t         valid_mask;
   /** Hazardous potential test results. */
   IFX_LT_GR909_HPT_t   hpt;
   /** Foreign electromotive forces test results. */
   IFX_LT_GR909_FEMF_t  femf;
   /** Resistive faults test results. */
   IFX_LT_GR909_RFT_t   rft;
   /** Receiver off-hook test results. */
   IFX_LT_GR909_ROH_t   roh;
   /** Ringer impedance test results. */
   IFX_LT_GR909_RIT_t   rit;
} IFX_LT_GR909_RESULT_t;

/** GR-909 parameter configuration structure.
The values f_R1, f_R2 and f_R3 are chip-set dependent, 
see also the product hardware design guidelines for reference.
R1 and R2 are the resistors used for the voltage measurements
via IO pins. They are connected between the tip/ring wires and
the chip set to divide the external voltage; R1 is the the high-
ohmic resistor, while R2 is the low-ohmic external resistor of
that voltage divider.
f_R1=R1.
For DUSLIC-XT: f_R2=f_R3=2065.
For products using the SLIC-DC, the f_R2 and f_R3 parameters
correspond to the parallel between the R2 resistor and two different
internal resistors. For more details, please refer to the product
hardware documentation. */
typedef struct
{
   /** High-ohmic resistor of the voltage divider connected to
    the line. The suggested value for DUSLIC-XT is f_R1=Rm=1000000.
    For SLIC-DC based products (VINETIC-CPE/PLUS, DANUBE, TWINPASS,
    VINAX-VE), the recommended value is f_R1=R1=1500000. */
   IFX_float_t f_R1;
   /** For DUSLIC-XT: f_R2=f_R3=2065.
    For SLIC-DC based products (VINETIC-CPE/PLUS, DANUBE, TWINPASS,
    VINAX-VE): low-ohmic resistor of the voltage
    divider in parallel to 1 MOhm (internal resistor).
    For example, if R2=3.3 kOhm, f_R2 = 3.3k || 1M = 3289 */
   IFX_float_t f_R2;
   /** For DUSLIC-XT: f_R2=f_R3=2065.
    For SLIC-DC based products, (VINETIC-CPE/PLUS, DANUBE, TWINPASS,
    VINAX-VE): low-ohmic resistor of the voltage
    divider in parallel to 750 kOhm (internal resistor).
    For example, if R2=3.3 kOhm, f_R3 = 3.3k || 750k = 3286 */
   IFX_float_t f_R3;
} IFX_LT_GR909_CFG_t;

/* ============================= */
/* Global function declaration   */
/* ============================= */

/** Configures SLIC system parameters to be used for GR-909 calculation.
  This function shall be used only if the voltage divider resistors
  connected to the SLIC do not follow the Lantiq hardware design
  guidelines document.
  This interface is not supported for ARX188/GRX188.

\param dev - device type of \ref IFX_TAPI_GR909_DEV_t.
\param p_cfg Handles to \ref IFX_LT_GR909_CFG_t structure.

 \return Returns value as follows:
   - \ref IFX_SUCCESS: if successful
   - \ref IFX_ERROR: in case of an error

\remark Use of this function is optional; default parameters are used
   otherwise.
*/
IFX_return_t Ifxphone_LT_GR909_Config (IFX_TAPI_GR909_DEV_t dev,
                                                    IFX_LT_GR909_CFG_t *p_cfg);

/** Starts a GR909 test sequence according to measurement mask set with
 \ref IFX_LT_GR909_MASK_t. All tests in the mask must be selected.

\param fd_line   Line file descriptor.
\param b_euLike  - IFX_TRUE: EU-like powerline frequency (50 Hz).
                 - FX_FALSE: US-like power line frequency (60 Hz).
\param meas_mask Measurement mask set with values out of \ref IFX_LT_GR909_MASK_t.

\return Returns value as follows:
   - \ref IFX_SUCCESS: if successful
   - \ref IFX_ERROR: in case of an error
*/
IFX_return_t Ifxphone_LT_GR909_Start (IFX_int32_t   fd_line,
                                      IFX_boolean_t b_euLike,
                                      IFX_uint32_t  meas_mask);


IFX_return_t Ifxphone_LT_GR909_Stop (IFX_int32_t fd_line);

/** Receives Gr909 measurement results.

\param fd_line Line file descriptor.
\param p_res   Handles to result structure.

\return Returns value as follows:
   - \ref IFX_SUCCESS: if successful
   - \ref IFX_ERROR: in case of an error

\remark  Only evaluates results which are marked as valid.
*/
IFX_return_t Ifxphone_LT_GR909_GetResults (IFX_int32_t  fd_line,
                                           IFX_LT_GR909_RESULT_t *p_res);

/*@}*/ /* TAPI_INTERFACE_LT_GR909 */
#endif /* _LIB_TAPI_LT_GR909_H */


