/*
     This file is part of GNUnet.
     Copyright (C) 2001-2013 GNUnet e.V.

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 3, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
     Boston, MA 02110-1301, USA.
*/

/**
 * @author Christian Grothoff
 *
 * @file
 * Functions related to accessing services

 * @defgroup client  Client library
 * Generic client-side communication with services
 *
 * @see [Documentation](https://gnunet.org/ipc)
 *
 * @{
 */

#ifndef GNUNET_CLIENT_LIB_H
#define GNUNET_CLIENT_LIB_H

#ifdef __cplusplus
extern "C"
{
#if 0                           /* keep Emacsens' auto-indent happy */
}
#endif
#endif

#include "gnunet_mq_lib.h"

/**
 * Opaque handle for a connection to a service.
 */
struct GNUNET_CLIENT_Connection;


/**
 * Create a message queue to connect to a GNUnet service.
 * If handlers are specfied, receive messages from the connection.
 *
 * @param connection the client connection
 * @param handlers handlers for receiving messages, can be NULL
 * @param error_handler error handler
 * @param error_handler_cls closure for the @a error_handler
 * @return the message queue, NULL on error
 */
struct GNUNET_MQ_Handle *
GNUNET_CLIENT_connecT (const struct GNUNET_CONFIGURATION_Handle *cfg,
                       const char *service_name,
                       const struct GNUNET_MQ_MessageHandler *handlers,
                       GNUNET_MQ_ErrorHandler error_handler,
                       void *error_handler_cls);


/**
 * Create a message queue for a GNUNET_CLIENT_Connection.
 * If handlers are specfied, receive messages from the connection.
 *
 * @param connection the client connection, taken over and freed by the MQ
 * @param handlers handlers for receiving messages
 * @param error_handler error handler
 * @param error_handler_cls closure for the @a error_handler
 * @return the message queue
 * @deprecated use #GNUNET_CLIENT_connecT
 */
struct GNUNET_MQ_Handle *
GNUNET_MQ_queue_for_connection_client (struct GNUNET_CLIENT_Connection *connection,
                                       const struct GNUNET_MQ_MessageHandler *handlers,
                                       GNUNET_MQ_ErrorHandler error_handler,
                                       void *error_handler_cls);


/**
 * Get a connection with a service.
 *
 * @param service_name name of the service
 * @param cfg configuration to use
 * @return NULL on error (service unknown to configuration)
 * @deprecated use #GNUNET_CLIENT_connect2
 */
struct GNUNET_CLIENT_Connection *
GNUNET_CLIENT_connect (const char *service_name,
                       const struct GNUNET_CONFIGURATION_Handle *cfg);


/**
 * Destroy connection with the service.  This will automatically
 * cancel any pending "receive" request (however, the handler will
 * *NOT* be called, not even with a NULL message).  Any pending
 * transmission request will also be cancelled UNLESS the callback for
 * the transmission request has already been called, in which case the
 * transmission 'finish_pending_write' argument determines whether or
 * not the write is guaranteed to complete before the socket is fully
 * destroyed (unless, of course, there is an error with the server in
 * which case the message may still be lost).
 *
 * @param client handle to the service connection
 * @deprecated
 */
void
GNUNET_CLIENT_disconnect (struct GNUNET_CLIENT_Connection *client);


/**
 * Type of a function to call when we receive a message
 * from the service.
 *
 * @param cls closure
 * @param msg message received, NULL on timeout or fatal error
 */
typedef void
(*GNUNET_CLIENT_MessageHandler) (void *cls,
				 const struct GNUNET_MessageHeader *msg);


/**
 * Read from the service.
 *
 * @param client connection to the service
 * @param handler function to call with the message
 * @param handler_cls closure for @a handler
 * @param timeout how long to wait until timing out
 * @deprecated
 */
void
GNUNET_CLIENT_receive (struct GNUNET_CLIENT_Connection *client,
                       GNUNET_CLIENT_MessageHandler handler,
		       void *handler_cls,
                       struct GNUNET_TIME_Relative timeout);


/**
 * Transmit handle for client connections.
 */
struct GNUNET_CLIENT_TransmitHandle;


/**
 * Ask the client to call us once the specified number of bytes
 * are free in the transmission buffer.  Will never call the @a notify
 * callback in this task, but always first go into the scheduler.
 *
 * @param client connection to the service
 * @param size number of bytes to send
 * @param timeout after how long should we give up (and call
 *        @a notify with buf NULL and size 0)?
 * @param auto_retry if the connection to the service dies, should we
 *        automatically re-connect and retry (within the timeout period)
 *        or should we immediately fail in this case?  Pass #GNUNET_YES
 *        if the caller does not care about temporary connection errors,
 *        for example because the protocol is stateless
 * @param notify function to call
 * @param notify_cls closure for @a notify
 * @return NULL if someone else is already waiting to be notified
 *         non-NULL if the notify callback was queued (can be used to cancel
 *         using #GNUNET_CONNECTION_notify_transmit_ready_cancel)
 * @deprecated
 */
struct GNUNET_CLIENT_TransmitHandle *
GNUNET_CLIENT_notify_transmit_ready (struct GNUNET_CLIENT_Connection *client,
                                     size_t size,
                                     struct GNUNET_TIME_Relative timeout,
                                     int auto_retry,
                                     GNUNET_CONNECTION_TransmitReadyNotify notify,
                                     void *notify_cls);


/**
 * Cancel a request for notification.
 *
 * @param th handle from the original request.
 * @deprecated
 */
void
GNUNET_CLIENT_notify_transmit_ready_cancel (struct GNUNET_CLIENT_TransmitHandle *th);


/**
 * Convenience API that combines sending a request
 * to the service and waiting for a response.
 * If either operation times out, the callback
 * will be called with a "NULL" response (in which
 * case the connection should probably be destroyed).
 *
 * @param client connection to use
 * @param hdr message to transmit
 * @param timeout when to give up (for both transmission
 *         and for waiting for a response)
 * @param auto_retry if the connection to the service dies, should we
 *        automatically re-connect and retry (within the timeout period)
 *        or should we immediately fail in this case?  Pass #GNUNET_YES
 *        if the caller does not care about temporary connection errors,
 *        for example because the protocol is stateless
 * @param rn function to call with the response
 * @param rn_cls closure for @a rn
 * @return #GNUNET_OK on success, #GNUNET_SYSERR if a request
 *         is already pending
 * @deprecated
 */
int
GNUNET_CLIENT_transmit_and_get_response (struct GNUNET_CLIENT_Connection *client,
                                         const struct GNUNET_MessageHeader *hdr,
                                         struct GNUNET_TIME_Relative timeout,
                                         int auto_retry,
                                         GNUNET_CLIENT_MessageHandler rn,
                                         void *rn_cls);


/**
 * Handle for a test to check if a service is running.
 */
struct GNUNET_CLIENT_TestHandle;

/**
 * Function called with the result on the service test.
 *
 * @param cls closure
 * @param result #GNUNET_YES if the service is running,
 *               #GNUNET_NO if the service is not running
 *               #GNUNET_SYSERR if the configuration is invalid
 */
typedef void
(*GNUNET_CLIENT_TestResultCallback)(void *cls,
				    int result);


/**
 * Test if the service is running.  If we are given a UNIXPATH or a
 * local address, we do this NOT by trying to connect to the service,
 * but by trying to BIND to the same port.  If the BIND fails, we know
 * the service is running.
 *
 * @param service name of the service to wait for
 * @param cfg configuration to use
 * @param timeout how long to wait at most
 * @param cb function to call with the result
 * @param cb_cls closure for @a cb
 * @return handle to cancel the test
 * @deprecated
 */
struct GNUNET_CLIENT_TestHandle *
GNUNET_CLIENT_service_test (const char *service,
                            const struct GNUNET_CONFIGURATION_Handle *cfg,
                            struct GNUNET_TIME_Relative timeout,
                            GNUNET_CLIENT_TestResultCallback cb, void *cb_cls);


/**
 * Abort testing for service.
 *
 * @param th test handle
 * @deprecated
 */
void
GNUNET_CLIENT_service_test_cancel (struct GNUNET_CLIENT_TestHandle *th);


#if 0                           /* keep Emacsens' auto-indent happy */
{
#endif
#ifdef __cplusplus
}
#endif

/* ifndef GNUNET_CLIENT_LIB_H */
#endif

/** @} */ /* end of group client */

/* end of gnunet_client_lib.h */
