/*   FILE: lcdringer.vala -- Receives XMPP messages and rings
 * AUTHOR: W. Michael Petullo <mike@flyn.org>
 *   DATE: 20 September 2016
 *
 * Copyright (c) 2016 W. Michael Petullo <new@flyn.org>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using Lm;
using Gst;
using Pifacecad;
using Posix;

const uint LIGHTS_OUT_SECONDS = 60;
const uint SWITCH_POLL_MILLISECONDS = 1;

bool no_pifacecad;
string jid;
string password;
string ring_path;

const OptionEntry[] options = {
	{ "jid", 'j', 0, OptionArg.STRING, ref jid, "JID", null },
	{ "password", 'p', 0, OptionArg.STRING, ref password, "Password", null },
	{ "ring-path", 'r', 0, OptionArg.STRING, ref ring_path, "Ring path", null },
	{ "no-pifacecad", 'n', 0, OptionArg.NONE, ref no_pifacecad, "Do not try to use PiFace CAD for I/O", null },
	{ null }
};

enum RingerResetType {
	TIMEOUT,
	ACKNOWLEDGED,
}

private class LCDRinger {
	private MainLoop loop;
	private Lm.Connection connection;
	private Lm.MessageHandler message_handler;
	private dynamic Element playbin;
	private string sender;
        private uint8[] lastSwitchState = { 1, 1, 1, 1, 1, 1, 1, 1 };
	private uint lights_out_timeout_id = 0;

	static uint output_shift = 0;
	static string output_str = "";

	public LCDRinger (string jid, string password) {
		loop    = new MainLoop ();

		// FIXME: Fix this throughout: write wrapper functions.
		if (! no_pifacecad) {
			Pifacecad.open ();
			Pifacecad.lcd_backlight_on ();

			Timeout.add_seconds (LIGHTS_OUT_SECONDS, () => {
				Pifacecad.lcd_backlight_off (); return false;
			});
		}

		output("Starting");
		register_handlers ();

		output("Setting up ringer");
		setup_ringer ();

		output("Connecting to XMPP server");
		connect_to_xmpp ();

		output("Waiting for messages");
		loop.run ();

		if (! no_pifacecad) {
			Pifacecad.close ();
		}
	}

	private void setup_ringer () {
		playbin = ElementFactory.make ("playbin", "play");
		playbin.uri = "file://" + ring_path;

		Bus bus = playbin.get_bus ();
		bus.add_watch (0, bus_callback);

		playbin.set_state (State.READY);
	}

	private void connect_to_xmpp () {
		connection = new Lm.Connection("www.flyn.org");
		connection.set_port(5222);
		connection.set_jid(jid);

		message_handler = new Lm.MessageHandler(handle_message, null);
		connection.register_message_handler(message_handler,
		                                    Lm.MessageType.MESSAGE,
		                                    Lm.HandlerPriority.NORMAL);

		connection.set_disconnect_function(disconnect_cb, null);

		try {
			if (!connection.open_and_block()) {
				output("Failure to open connection");
				return;
			}
			output("Connected");
			if (!connection.authenticate_and_block(jid.split("@")[0], password, "lcdringer")) {
				output("Failure to authenticate");
				return;
			}

			output("Authenticated");

			var msg = new Lm.Message.with_sub_type(null,
			                                    Lm.MessageType.PRESENCE,
			                                    Lm.MessageSubType.AVAILABLE);
			msg.node.add_child ("show", "chat");
			connection.send(msg);
		} catch (GLib.Error e) {
			GLib.error ("%s", e.message);
		}
	}

	private Lm.HandlerResult handle_message(Lm.MessageHandler handler,
	                       Lm.Connection connection,
	                       Lm.Message message) {
		if (null == sender) {
			if (Lm.MessageType.MESSAGE == message.get_type()) {
				output(message.node.get_child("body").get_value());
			} else {
				output("Invalid message type rec'd");
			}

			sender = message.node.get_attribute("from");

			ring();
		} else {
			var msg    = new Lm.Message (sender, Lm.MessageType.MESSAGE);
			msg.node.add_child ("body", "Already ringing.");

			try {
				connection.send (msg);
			} catch (GLib.Error e) {
				GLib.stderr.printf ("Error sending message: %s\n", e.message);
			}
		}

		return Lm.HandlerResult.REMOVE_MESSAGE;
	}

	private void disconnect_cb(Lm.Connection connection, Lm.DisconnectReason reason) {
		output("Disconnected");
	}

	private bool render_output () {
                Pifacecad.lcd_clear ();

		Pifacecad.lcd_backlight_on ();
		if (0 != lights_out_timeout_id) {
			Source.remove (lights_out_timeout_id);
		}
		lights_out_timeout_id = Timeout.add_seconds (LIGHTS_OUT_SECONDS, lights_out);

                if (output_str.length <= 15) {
                        Pifacecad.lcd_write (output_str);
                } else {
                        var builder = new StringBuilder ();
                        uint rest = output_str.length > 30 ? 15 : output_str.length - 15;
                        uint cutoff = output_str.length > 30 ? output_str.length - 30 : 0;

                        // FIXME: Should be able to replace 15 with 16,
                        // but libpifacecad seems to require a "cell" for '\n'.
                        // See https://github.com/piface/libpifacecad/issues/2.
                        builder.append (output_str.substring (output_shift, 15));
                        builder.append ("\n ");
                        builder.append (output_str.substring (15 + output_shift, rest));

                        Pifacecad.lcd_write (builder.str);

                        output_shift = output_shift < cutoff ? output_shift + 1 : 0;
                }

                return true;
        }

	protected bool lights_out () {
                GLib.debug ("Lights out.\n");
                if (! no_pifacecad) {
                        Pifacecad.lcd_backlight_off ();
                }
                this.lights_out_timeout_id = 0;
                return false;
        }

	public void output (string str) {
                GLib.stdout.printf ("%s\n", str);
                output_shift = 0;
                output_str = str;
                if (! no_pifacecad) {
                        render_output();
                }
        }

	private bool bus_callback (Gst.Bus bus, Gst.Message message) {
		switch (message.type) {
		case Gst.MessageType.ERROR:
			GLib.Error err;
			string debug;
			message.parse_error (out err, out debug);
			output ("Error: " + err.message);
			break;
		case Gst.MessageType.EOS:
			reset_ringer (RingerResetType.TIMEOUT);

			break;
		default:
			break;
		}

		return true;
	}

	private void ring () {
		var msg    = new Lm.Message (sender, Lm.MessageType.MESSAGE);
		msg.node.add_child ("body", "Ringing ...");

		try {
			connection.send (msg);
		} catch (GLib.Error e) {
			GLib.stderr.printf ("Error sending message: %s\n", e.message);
		}

		playbin.set_state (State.PLAYING);
	}

	protected void register_handlers () {
                try {
                        IOChannel channel = new IOChannel.unix_new (Posix.STDIN_FILENO);
                        channel.set_encoding (null);
                        channel.add_watch (IOCondition.IN, check_keyboard);
                        Timeout.add (SWITCH_POLL_MILLISECONDS, check_switches);
                } catch (IOChannelError e) {
                        GLib.error ("Channel error initializing state");
                }
        }

	protected bool check_switches () {
                if (! no_pifacecad) {
			for (uint8 switch = 0; switch < 8; switch++) {
				uint8 current = Pifacecad.read_switch (switch);
				
				if (0 == current && 1 == lastSwitchState[switch]) {
					reset_ringer (RingerResetType.ACKNOWLEDGED);
                                }

                                lastSwitchState[switch] = current;
			}
                }

                return true;
        }

	private void reset_ringer (RingerResetType reset_type) {
		if (null != sender) {
			var msg    = new Lm.Message (sender, Lm.MessageType.MESSAGE);

			switch (reset_type) {
			case RingerResetType.ACKNOWLEDGED:
				msg.node.add_child ("body", "Ringing acknowledged.");
				break;
			case RingerResetType.TIMEOUT:
			default:
				msg.node.add_child ("body", "Ringing timed out.");
				break;
			}

			try {
				connection.send (msg);
			} catch (GLib.Error e) {
				GLib.stderr.printf ("Error sending message: %s\n", e.message);
			}

			sender = null;
		}

		playbin.set_state (State.READY);
	}

	public bool check_keyboard (IOChannel channel, IOCondition condition) {
                char []c = new char[1];

                if (condition == IOCondition.HUP) {
                        GLib.stderr.printf ("Standard in closed?\n");
                        return false;
                }

                try {
                        size_t length = -1;
                        IOStatus status = channel.read_chars (c, out length);
                        if (status == IOStatus.EOF || length != 1) {
                                GLib.stderr.printf ("Standard in closed?\n");
                                return false;
                        }
                } catch (IOChannelError e) {
                        GLib.stderr.printf ("IOChannelError: %s\n", e.message);
                        return false;
                } catch (ConvertError e) {
                        GLib.stderr.printf ("ConvertError: %s\n", e.message);
                        return false;
                }

                switch (c[0]) {
                case '\n':
			if (null != sender) {
				reset_ringer (RingerResetType.ACKNOWLEDGED);
			}
                        break;
		default:
                        if (! no_pifacecad) {
                                GLib.warning ("Invalid key pressed");
                        } else {
                                GLib.stderr.printf ("Valid keys are:\n");
                                GLib.stderr.printf ("  'enter' (answer)\n");
                        }
                        break;
                }

                return true;
        }	
}

int main (string[] args) {     
	var key_file = new GLib.KeyFile();
	try {
		var ok = key_file.load_from_file("/etc/lcdringer.conf",
						  GLib.KeyFileFlags.NONE);

		if (ok) {
			jid       = key_file.get_string("account", "jid");
			password  = key_file.get_string("account", "password");
			ring_path = key_file.get_string("account", "ring_path");
		}
	} catch (GLib.Error e) {
		GLib.error ("%s", e.message);
	}

	try {
		var opt_context = new OptionContext ("- Ringer");
		opt_context.set_help_enabled (true);
		opt_context.add_main_entries (options, null);
		opt_context.parse (ref args);
	} catch (OptionError e) {
		GLib.error ("%s", e.message);
	}

	if (null == jid) {
		GLib.error("Requires a JID");
	}

	if (null == password) {
		GLib.error("Requires a password");
	}

	if (null == ring_path) {
		GLib.error("Requires a ring path");
	}

	Posix.termios old_tio, new_tio;
	Posix.tcgetattr(Posix.STDIN_FILENO, out old_tio);
	new_tio = old_tio;
	new_tio.c_lflag &=(~ICANON & ~ECHO);
	Posix.tcsetattr(STDIN_FILENO, TCSANOW, new_tio);

	Gst.init (ref args);

	new LCDRinger (jid, password);

	Posix.tcsetattr(STDIN_FILENO, TCSANOW, old_tio);

	return 0;
}
