/**
 * \file libnetconf.h
 * \author Radek Krejci <rkrejci@cesnet.cz>
 * \author Michal Vasko <mvasko@cesnet.cz>
 * \brief libnetconf2 main internal header.
 *
 * Copyright (c) 2015 CESNET, z.s.p.o.
 *
 * This source code is licensed under BSD 3-Clause License (the "License").
 * You may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://opensource.org/licenses/BSD-3-Clause
 */

#ifndef NC_LIBNETCONF_H_
#define NC_LIBNETCONF_H_

#include "config.h"
#include "netconf.h"
#include "log_p.h"
#include "session_p.h"
#include "messages_p.h"

/* Tests whether string is empty or non-empty. */
#define strisempty(str) ((str)[0] == '\0')
#define strnonempty(str) ((str)[0] != '\0')

/**
 * @mainpage About
 *
 * libnetconf2 is a NETCONF library in C handling NETCONF authentication and all NETCONF
 * RPC communication both server and client-side. NETCONF datastore and session management is not a part of this library,
 * but it helps a lot with the sessions.
 *
 * @section about-features Main Features
 *
 * - Creating SSH (using libssh) or TLS (using OpenSSL) authenticated NETCONF sessions.
 * - Creating NETCONF sessions with a pre-established transport protocol
 *   (using this mechanism the communication can be tunneled through sshd(8), for instance).
 * - Creating NETCONF Call Home sessions.
 * - Creating, sending, receiving, and replying to RPCs.
 * - Receiving notifications.
 *
 * - \todo Creating and sending notifications.
 *
 * @section about-license License
 *
 * Copyright (c) 2015-2016 CESNET, z.s.p.o.
 *
 * (The BSD 3-Clause License)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 */

/**
 * @page howto How To ...
 *
 * - @subpage howtoinit
 * - @subpage howtoclient
 * - @subpage howtoserver
 * - @subpage howtoclientcomm
 * - @subpage howtoservercomm
 */

/**
 * @page howtoinit Init and Thread-safety Information
 *
 * Before working with the library, it must be initialized using nc_client_init()
 * or nc_server_init(). Optionally, a client can use nc_client_set_schema_searchpath()
 * to set the path to a directory with modules that will be loaded from there if they
 * could not be downloaded from the server (it does not support \<get-schema\>).
 * However, to be able to create at least the \<get-schema\> RPC, this directory must
 * contain the module _ietf-netconf-monitoring_. If this directory is not set,
 * the default _libnetconf2_ schema directory is used that includes this module
 * and a few others.
 *
 * Based on how the library was compiled, also _libssh_ and/or
 * _libssh_/_libcrypto_ are initialized (for multi-threaded use) too. It is advised
 * to compile _libnetconf2_, for instance, with TLS support even if you do not want
 * to use _lnc2_ TLS functions, but only use _libssl/libcrypto_ functions in your
 * application. You can then use _libnetconf2_ cleanup function and do not
 * trouble yourself with the cleanup.
 *
 * To prevent any reachable memory at the end of your application, there
 * are complementary destroy functions available. If your application is
 * multi-threaded, call the destroy functions in the last thread, after all
 * the other threads have ended. In every other thread you should call
 * nc_thread_destroy() just before it exits.
 *
 * If _libnetconf2_ is used in accordance with this information, there should
 * not be memory leaks of any kind at program exit. For thread-safety details
 * of _libssh_, _libssl_, and _libcrypto_, please refer to the corresponding project
 * documentation. _libnetconf2_ thread-safety information is below.
 *
 * Client is __NOT__ thread-safe and there is no access control in the client
 * functions at all. Server is __MOSTLY__ thread-safe meaning you can set all the
 * options simultaneously while listening for or accepting new sessions or
 * polling the existing ones. It should even be safe to poll one session in
 * several threads, but it is definitely discouraged. Generally, servers can
 * use more threads without any problems as long as they keep their workflow sane
 * (behavior such as freeing sessions only after no thread uses them or similar).
 *
 * Functions List
 * --------------
 *
 * Available in __nc_client.h__.
 *
 * - nc_client_init()
 * - nc_client_destroy()
 *
 * - nc_client_set_schema_searchpath()
 * - nc_client_get_schema_searchpath()
 *
 * Available in __nc_server.h__.
 *
 * - nc_server_init()
 * - nc_server_destroy()
 *
 * Available in both __nc_client.h__ and __nc_server.h__.
 *
 * - nc_thread_destroy()
 */

/**
 * @page howtoclient Client sessions
 *
 * To connect to a NETCONF server, a NETCONF session must be established,
 * which requires a working transport session. It is possible to create
 * NETCONF sessions with SSH (using _libssh_) or TLS (using _libssl/libcrypto_)
 * as the underlying transport protocol. It is also possible to establish
 * the transport protocol outside _libnetconf2_ and then provide these file
 * descriptors (FD) for full NETCONF session creation.
 *
 * There are a lot of options for both an SSH and a TLS client. All of them
 * have setters and getters so that there is no need to duplicate them in
 * a client.
 *
 * SSH
 * ===
 *
 * Connecting to a server using SSH does not strictly require to set any
 * options, there are sensible default values for all the basic ones.
 * Except all the SSH options, optionally some authetication callbacks can be set,
 * which are particulary useful in automated clients (passwords cannot be
 * asked a user) or simply if any additional information is retrieved some
 * other way than from standard terminal input.
 *
 * Having the default options or changing any unsuitable ones, there are 2 functions
 * to use for a new server connection. nc_connect_ssh() is the standard function
 * that creates sessions using the set options. If there are some options, which
 * cannot be changed with the provided API, there is nc_connect_libssh() available.
 * It requires a _libssh_ session, in which all the SSH options can be modified
 * and even the connection established. This allows for full customization and
 * should fit any specific situation.
 *
 * New NETCONF sessions can also be created on existing authenticated SSH sessions.
 * There is a new SSH channel needed, on which the NETCONF session is then created.
 * Use nc_connect_ssh_channel() for this purpose.
 *
 * Functions List
 * --------------
 *
 * Available in __nc_client.h__.
 *
 * - nc_client_ssh_set_auth_hostkey_check_clb()
 * - nc_client_ssh_set_auth_password_clb()
 * - nc_client_ssh_set_auth_interactive_clb()
 * - nc_client_ssh_set_auth_privkey_passphrase_clb()
 * - nc_client_ssh_add_keypair()
 * - nc_client_ssh_del_keypair()
 * - nc_client_ssh_get_keypair_count()
 * - nc_client_ssh_get_keypair()
 * - nc_client_ssh_set_auth_pref()
 * - nc_client_ssh_get_auth_pref()
 * - nc_client_ssh_set_username()
 * - nc_client_ssh_get_username()
 *
 * - nc_connect_ssh()
 * - nc_connect_libssh()
 * - nc_connect_ssh_channel()
 *
 *
 * TLS
 * ===
 *
 * To connect to a server using TLS, there must be some client identification
 * options set. Client must specify its certificate with a private key using
 * nc_client_tls_set_cert_key_paths(). Also, the Certificate Authority of
 * a server certificate must be considered trusted. Paths to all the trusted
 * CA certificates can be set by nc_client_tls_set_trusted_ca_paths().
 *
 * Then there are again 2 functions for connecting, nc_connect_tls() being
 * the standard way of connecting. nc_connect_libssl() again enables
 * to customize the TLS session in every way _libssl_ allows.
 *
 * Functions List
 * --------------
 *
 * Available in __nc_client.h__.
 *
 * - nc_client_tls_set_cert_key_paths()
 * - nc_client_tls_get_cert_key_paths()
 * - nc_client_tls_set_trusted_ca_paths()
 * - nc_client_tls_get_trusted_ca_paths()
 * - nc_client_tls_set_crl_paths()
 * - nc_client_tls_get_crl_paths()
 *
 * - nc_connect_tls()
 * - nc_connect_libssl()
 *
 *
 * FD
 * ==
 *
 * If you authenticated the connection using some tunneling software, you
 * can pass its file descriptors to _libnetconf2_ using nc_connect_inout(),
 * which will continue to establish a full NETCONF session.
 *
 * Funtions List
 * -------------
 *
 * Available in __nc_client.h__.
 *
 * - nc_connect_inout()
 *
 *
 * Call Home
 * =========
 *
 * Call Home needs the same options set as standard SSH or TLS and the functions
 * reflect it exactly. However, to accept a connection, the client must first
 * specify addresses and ports, which to listen on by nc_client_ssh_ch_add_bind_listen()
 * and nc_client_tls_ch_add_bind_listen(). Then connections can be
 * accepted using nc_accept_callhome().
 *
 * Functions List
 * --------------
 *
 * Available in __nc_client.h__.
 *
 * - nc_client_ssh_ch_set_auth_hostkey_check_clb()
 * - nc_client_ssh_ch_set_auth_password_clb()
 * - nc_client_ssh_ch_set_auth_interactive_clb()
 * - nc_client_ssh_ch_set_auth_privkey_passphrase_clb()
 * - nc_client_ssh_ch_add_bind_listen()
 * - nc_client_ssh_ch_del_bind()
 * - nc_client_ssh_ch_add_keypair()
 * - nc_client_ssh_ch_del_keypair()
 * - nc_client_ssh_ch_get_keypair_count()
 * - nc_client_ssh_ch_get_keypair()
 * - nc_client_ssh_ch_set_auth_pref()
 * - nc_client_ssh_ch_get_auth_pref()
 * - nc_client_ssh_ch_set_username()
 * - nc_client_ssh_ch_get_username()
 *
 * - nc_client_tls_ch_add_bind_listen()
 * - nc_client_tls_ch_del_bind()
 * - nc_client_tls_ch_set_cert_key_paths()
 * - nc_client_tls_ch_get_cert_key_paths()
 * - nc_client_tls_ch_set_trusted_ca_paths()
 * - nc_client_tls_ch_get_trusted_ca_paths()
 * - nc_client_tls_ch_set_crl_paths()
 * - nc_client_tls_ch_get_crl_paths()
 *
 * - nc_accept_callhome()
 *
 *
 * Cleanup
 * =======
 *
 * These options and the schema searchpath are stored in dynamically
 * allocated memory. They are freed as a part of [destroying the client](@ref howtoinit).
 */

/**
 * @page howtoserver Server sessions
 *
 * Init
 * ====
 *
 * Server takes an argument for its [initialization function](@ref howtoinit).
 * In it, you set the server context, which determines what modules it
 * supports and what capabilities to advertise. Few capabilities that
 * cannot be learnt from the context are set with separate functions
 * nc_server_set_capab_withdefaults() and nc_server_set_capab_interleave().
 * Timeout for receiving the _hello_ message on a new session can be set
 * by nc_server_set_hello_timeout() and the timeout for disconnecting
 * an inactive session by nc_server_set_idle_timeout().
 *
 * Context does not only determine server modules, but its overall
 * functionality as well. For every RPC the server should support,
 * an nc_rpc_clb callback should be set on that node in the context.
 * Server then calls these as appropriate [during poll](@ref howtoservercomm).
 *
 * Just like in the [client](@ref howtoclient), you can let _libnetconf2_
 * establish SSH or TLS transport or do it yourself and only provide the file
 * descriptors of the connection.
 *
 * Server options can be only set, there are no getters.
 *
 * Functions List
 * --------------
 *
 * Available in __nc_server.h__.
 *
 * - nc_server_set_capab_withdefaults()
 * - nc_server_set_capab_interleave()
 * - nc_server_set_hello_timeout()
 * - nc_server_set_idle_timeout()
 *
 *
 * SSH
 * ===
 *
 * To be able to accept SSH connections, individual endpoints must be added
 * with nc_server_ssh_add_endpt_listen() and their options set. The only
 * mandatory SSH option before sessions can be established on an endpoint
 * is the host key, set it using nc_server_ssh_endpt_set_hostkey().
 *
 * Functions List
 * --------------
 *
 * Available in __nc_server.h__.
 *
 * - nc_server_ssh_add_endpt_listen()
 * - nc_server_ssh_endpt_set_address()
 * - nc_server_ssh_endpt_set_port()
 * - nc_server_ssh_del_endpt()
 *
 * - nc_server_ssh_endpt_set_hostkey()
 * - nc_server_ssh_endpt_set_banner()
 * - nc_server_ssh_endpt_set_auth_methods()
 * - nc_server_ssh_endpt_set_auth_attempts()
 * - nc_server_ssh_endpt_set_auth_timeout()
 * - nc_server_ssh_endpt_add_authkey()
 * - nc_server_ssh_endpt_del_authkey()
 *
 *
 * TLS
 * ===
 *
 * TLS works with endpoints too, but its options differ
 * significantly from the SSH ones, especially in the _cert-to-name_
 * options that TLS uses to derive usernames from client certificates.
 * So, after starting listening on an endpoint with nc_server_tls_add_endpt_listen(),
 * you need to set the server certificate (nc_server_tls_endpt_set_cert()
 * or nc_server_tls_endpt_set_cert_path()) and private key (nc_server_tls_endpt_set_key()
 * or nc_server_tls_endpt_set_key_path()).
 *
 * To accept client certificates, they must first be considered trusted,
 * which you have three ways of achieving. You can add each of their Certificate Authority
 * certificates to the trusted ones or mark a specific client certificate
 * as trusted using nc_server_tls_endpt_add_trusted_cert(). Lastly, you can
 * set paths with all the trusted CA certificates with nc_server_tls_endpt_set_trusted_ca_paths().
 *
 * Then, from each trusted client certificate a username must be derived
 * for the NETCONF session. This is accomplished by finding a matching
 * _cert-to-name_ entry. They are added using nc_server_tls_endpt_add_ctn().
 *
 * If you need to remove trusted certificates or Certificate Revocation
 * Lists, you must first clear them all with nc_server_tls_endpt_clear_certs()
 * and nc_server_tls_endpt_clear_crls(), respectively. Then add all
 * the certificates again.
 *
 * Functions List
 * --------------
 *
 * Available in __nc_server.h__.
 *
 * - nc_server_tls_add_endpt_listen()
 * - nc_server_tls_endpt_set_address()
 * - nc_server_tls_endpt_set_port()
 * - nc_server_tls_del_endpt()
 *
 * - nc_server_tls_endpt_set_cert()
 * - nc_server_tls_endpt_set_cert_path()
 * - nc_server_tls_endpt_set_key()
 * - nc_server_tls_endpt_set_key_path()
 * - nc_server_tls_endpt_add_trusted_cert()
 * - nc_server_tls_endpt_add_trusted_cert_path()
 * - nc_server_tls_endpt_set_trusted_ca_paths()
 * - nc_server_tls_endpt_clear_certs()
 * - nc_server_tls_endpt_set_crl_paths()
 * - nc_server_tls_endpt_clear_crls()
 * - nc_server_tls_endpt_add_ctn()
 * - nc_server_tls_endpt_del_ctn()
 *
 * FD
 * ==
 *
 * If you used a tunneling software, which does its own authentication,
 * you can accept a NETCONF session on its file descriptors with
 * nc_accept_inout().
 *
 * Functions List
 * --------------
 *
 * Available in __nc_server.h__.
 *
 * - nc_accept_inout()
 *
 *
 * Call Home
 * =========
 *
 * Call Home does not work with endpoints like standard sessions.
 * Connecting is similar to the [client](@ref howtoclient), just call
 * nc_connect_callhome_ssh() or nc_connect_callhome_tls(). Any options
 * must be reset manually by nc_server_ssh_ch_clear_opts()
 * or nc_server_tls_ch_clear_crls() after another Call Home session
 * (with different options than the previous one) is to be established.
 * Also, monitoring of these sessions is up to the application.
 *
 * Functions List
 * --------------
 *
 * Available in __nc_server.h__.
 *
 * - nc_connect_callhome_ssh()
 * - nc_connect_callhome_tls()
 *
 * - nc_server_ssh_ch_set_hostkey()
 * - nc_server_ssh_ch_set_banner()
 * - nc_server_ssh_ch_set_auth_methods()
 * - nc_server_ssh_ch_set_auth_attempts()
 * - nc_server_ssh_ch_set_auth_timeout()
 * - nc_server_ssh_ch_add_authkey()
 * - nc_server_ssh_ch_del_authkey()
 * - nc_server_ssh_ch_clear_opts()
 *
 * - nc_server_tls_ch_set_cert()
 * - nc_server_tls_ch_set_cert_path()
 * - nc_server_tls_ch_set_key()
 * - nc_server_tls_ch_set_key_path()
 * - nc_server_tls_ch_add_trusted_cert()
 * - nc_server_tls_ch_add_trusted_cert_path()
 * - nc_server_tls_ch_set_trusted_ca_paths()
 * - nc_server_tls_ch_clear_certs()
 * - nc_server_tls_ch_set_crl_paths()
 * - nc_server_tls_ch_clear_crls()
 * - nc_server_tls_ch_add_ctn()
 * - nc_server_tls_ch_del_ctn()
 * - nc_server_tls_ch_clear_opts()
 *
 *
 * Connecting And Cleanup
 * ======================
 *
 * When accepting connections with nc_accept(), all the endpoints are examined
 * and the first with a pending connection is used. To remove all
 * the endpoints and free any used dynamic memory, [destroy](@ref howtoinit) the server.
 *
 * Functions List
 * --------------
 *
 * Available in __nc_server.h__.
 *
 * - nc_accept()
 */

/**
 * @page howtoclientcomm Client communication
 *
 * To send RPCs on a session, you simply create an RPC, send it using nc_send_rpc(),
 * and then wait for a reply using nc_recv_reply(). If you are subscribed, there are 2 ways
 * of receiving notifications. Either you wait for them the same way
 * as for standard replies with nc_recv_notif() or you create a dispatcher
 * with nc_recv_notif_dispatch() that asynchronously (in a separate thread)
 * reads notifications and passes them to your callback.
 *
 * Functions List
 * --------------
 *
 * Available in __nc_client.h__.
 *
 * - nc_rpc_generic()
 * - nc_rpc_generic_xml()
 * - nc_rpc_getconfig()
 * - nc_rpc_edit()
 * - nc_rpc_copy()
 * - nc_rpc_delete()
 * - nc_rpc_lock()
 * - nc_rpc_unlock()
 * - nc_rpc_get()
 * - nc_rpc_kill()
 * - nc_rpc_commit()
 * - nc_rpc_discard()
 * - nc_rpc_cancel()
 * - nc_rpc_validate()
 * - nc_rpc_getschema()
 * - nc_rpc_subscribe()
 *
 * - nc_send_rpc()
 * - nc_recv_reply()
 * - nc_recv_notif()
 * - nc_recv_notif_dispatch()
 */

/**
 * @page howtoservercomm Server communication
 *
 * Once at least one session is established, an nc_pollsession structure
 * should be created with nc_ps_new(), filled with the session using
 * nc_ps_add_session() and finally polled with nc_ps_poll(). Based on
 * the return value from the poll, further actions can be taken. More
 * sessions can be polled at the same time and any requests received on
 * the sessions are [handled internally](@ref howtoserver).
 *
 * If an SSH NETCONF session asks for a new channel, you can accept
 * this request with nc_ps_accept_ssh_channel().
 *
 * Functions List
 * --------------
 *
 * Available in __nc_server.h__.
 *
 * - nc_ps_new()
 * - nc_ps_add_session()
 * - nc_ps_del_session()
 * - nc_ps_session_count()
 * - nc_ps_free()
 *
 * - nc_ps_poll()
 * - nc_ps_clear()
 * - nc_ps_accept_ssh_channel()
 */

#endif /* NC_LIBNETCONF_H_ */
