/******************************************************************************
 *
 * icmpping.c - part of smap
 *
 * ICMP Echo Request to check if hosts are up before fingerprinting
 *
 * $Id: icmpping.c,v 1.6 2007-10-07 10:15:11 hscholz Exp $
 *****************************************************************************/

#ifndef _ICMPPING_C
#define _ICMPPING_C

#include "config.h"

/******************************************************************************
 *
 * icmp_in_cksum()
 *
 * we use the standard checksum from Stevens et al.
 *
 *****************************************************************************/

unsigned short icmp_in_cksum(unsigned short *data, int size) {
	unsigned long sum = 0;

	while(size > 1) {
		sum += *(data++);
		size -= 2;
	}
	if (size > 0) sum += (*data) & 0xff00;
	sum = (sum & 0xffff) + (sum >> 16);

	return ((~(short)((sum >> 16) + (sum & 0xffff))));
}

/******************************************************************************
 *
 * icmp_message_str()
 *
 * map a received icmp packet to an error string and store a copy of
 * the string in the pointer given.
 * Don't forget to free the pointer afterwards!
 *
 ******************************************************************************/

char *icmp_message_str(struct icmp *icmp) {

	char *p;

	if (!icmp) return NULL;

	switch(icmp->icmp_type) {
	case ICMP_ECHOREPLY:
		p = strdup("Echo Reply");
		break;
	case ICMP_UNREACH:
		switch(icmp->icmp_code) {
		case ICMP_UNREACH_NET:
			p = strdup("Destination Net Unreachable");
			break;
		case ICMP_UNREACH_HOST:
			p = strdup("Destination Host Unreachable");
			break;
		case ICMP_UNREACH_PROTOCOL:
			p = strdup("Destination Protocol Unreachable");
			break;
		case ICMP_UNREACH_PORT:
			p = strdup("Destination Port Unreachable");
			break;
		case ICMP_UNREACH_SRCFAIL:
			p = strdup("Source Route Failed");
			break;
		case ICMP_UNREACH_NET_UNKNOWN:
			p = strdup("Unknown network");
			break;
		case ICMP_UNREACH_HOST_UNKNOWN:
			p = strdup("Unknown host");
			break;
		case ICMP_UNREACH_NEEDFRAG:
			p = strdup("Fragmentation needed");
			break;
		case ICMP_UNREACH_HOST_PROHIB:
			p = strdup("Communication prohibited by host");
			break;
		case ICMP_UNREACH_NET_PROHIB:
			p = strdup("Communication prohibited by network");
			break;
		case ICMP_UNREACH_FILTER_PROHIB:
			p = strdup("Communication prohibited by filter");
			break;
		default:
			p = strdup("[Destination Unreachable]");
			break;
		}
		break;
	case ICMP_SOURCEQUENCH:
		p = strdup("Source Quench");
		break;
	case ICMP_REDIRECT:
		switch(icmp->icmp_code) {
		case ICMP_REDIRECT_NET:
			p = strdup("Redirect Network");
			break;
		case ICMP_REDIRECT_HOST:
			p = strdup("Redirect Host");
			break;
		case ICMP_REDIRECT_TOSNET:
			p = strdup("Redirect Type of Service and Network");
			break;
		case ICMP_REDIRECT_TOSHOST:
			p = strdup("Redirect Type of Service and Host");
			break;
		default:
			p = strdup("[Redirect]");
			break;
		}
		break;
	case ICMP_ECHO:
		p = strdup("Echo Request");
		break;
	case ICMP_TIMXCEED:
		switch(icmp->icmp_code) {
		case ICMP_TIMXCEED_INTRANS:
			p = strdup("Time to live exceeded");
			break;
		case ICMP_TIMXCEED_REASS:
			p = strdup("Frag reassembly time exceeded");
			break;
		default:
			p = strdup("[Time exceeded]");
			break;
		}
		break;
	case ICMP_PARAMPROB:
		p = strdup("Parameter problem");
		break;
	case ICMP_TSTAMP:
		p = strdup("Timestamp");
		break;
	case ICMP_TSTAMPREPLY:
		p = strdup("Timestamp Reply");
		break;
	case ICMP_IREQ:
		p = strdup("Information Request");
		break;
	case ICMP_IREQREPLY:
		p = strdup("Information Request Reply");
		break;
	case ICMP_MASKREPLY:
		p = strdup("Address Mask Reply");
		break;
	case ICMP_MASKREQ:
		p = strdup("Address Mask Request");
		break;
	case ICMP_ROUTERADVERT:
		p = strdup("Router Advertisement");
		break;
	case ICMP_ROUTERSOLICIT:
		p = strdup("Router Solicitation");
		break;
	default:
		p = strdup("[Bad ICMP type]");
		break;
	}

	return p;
}


/******************************************************************************
 *
 * icmpping()
 *
 * perform ping and return 1 if host is up (received any kind of data)
 * and 0 in case of timeout as well as -1 in case of error
 *
 *****************************************************************************/
int icmpping(struct in_addr dst) {

	int 				sock, len, one;
	struct icmp		*icmp;
	struct sockaddr_in	sin;
#define ICMP_PKT_LEN	(sizeof(struct icmp) + ICMP_PAYLOAD_LEN)
	char				icmpbuf[ICMP_PKT_LEN];

	/* prepare ICMP packet */
	memset(&icmpbuf, 'A', ICMP_PKT_LEN);
	icmp = (struct icmp *) &icmpbuf;
	icmp->icmp_type = ICMP_ECHO;
	icmp->icmp_code = 0;
	icmp->icmp_cksum = 0;
	/* IP layer */
	len = sizeof(sin);
	sin.sin_family = AF_INET;
	memcpy(&(sin.sin_addr), (const void*)&dst, sizeof(struct in_addr));

	if ((sock = socket(AF_INET, SOCK_RAW, IPPROTO_ICMP)) < 0) {
		perror("ICMP PING:socket");
		return -1;
	}

	one = 1;
	if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &one, sizeof(one)) != 0)
		perror("setsockopt");
/* Linux doesn't have SO_REUSEPORT but SO_REUSEADDR should be fine */
#if defined (SO_REUSEPORT)
	if (setsockopt(sock, SOL_SOCKET, SO_REUSEPORT, &one, sizeof(one)) != 0)
		perror("setsockopt");
#endif
	icmp->icmp_cksum = icmp_in_cksum((unsigned short *) &icmpbuf, ICMP_PKT_LEN);

	/* send data */
	if ((sendto(sock, &icmpbuf, ICMP_PKT_LEN, 0,
			(struct sockaddr *) &sin, len)) < 0) {
		perror("ICMP PING:sendto");
		close(sock);
		return -1;
	}
	close(sock);
	return 1;
}

#endif /* _ICMPPING_C */
