/******************************************************************************
 *
 * smap.c - part of smap
 *
 * smap main program is in here
 *
 * Copyright 2006-2007 by Hendrik Scholz <hs@123.org>
 *
 * $Id: smap.c,v 1.29 2007-11-03 18:14:52 hscholz Exp $
 *****************************************************************************/
#ifndef _SMAP_C
#define _SMAP_C

#include "config.h"

int udp_sendsock;
struct sockaddr_in udp_sendsock_sockaddr;
#ifdef RAW_SOCKET
int udp_rawsock;
#endif

pthread_mutex_t  scrlock = PTHREAD_MUTEX_INITIALIZER; /* screen mutex */
pthread_mutex_t tasklock = PTHREAD_MUTEX_INITIALIZER; /* task list mutex */

/******************************************************************************
 *
 * usage()
 *
 * print usage information
 *
 *****************************************************************************/

void usage(int rc, char **argv) {
    fprintf(stderr,
        "usage: smap [ Options ] <ip | ip:port | ip/mask | host>\n\n"
        "            -h: this help\n"
        "            -d: increase debugging\n"
        "            -o: enable fingerprinting\n"
        "            -O: enable more verbose fingerprinting\n"
                        /* shameless copy from nmap ;) */
        "            -l: fingerprint learning mode\n"
        "            -t: TCP transport\n"
        "            -u: UDP transport (default\n"
        "           -P0: Treat all hosts as online - skip host discovery\n"
        "     -p <port>: destination port\n"
		"     -r <rate>: messages per second rate limit\n"
        "   -D <domain>: SIP domain to use without leading sip:\n"
        "  -w <timeout>: timeout in msec\n\n");
    exit ((rc > 0) ? rc : 0);
}

/******************************************************************************
 *
 * targets()
 *
 * we support a single string representing
 *  - an IP address
 *  - a IP:port tuple (i.e. 81.169.153.132:5060)
 *  - a network in CIDR (i.e. 192.168.123.0/24) notation
 *  - a hostname w/ IPv4 A record
 *
 *****************************************************************************/

int targets(char *arg) {

    char *p;
    char *str;

    int mask;
    struct in_addr net, host;

    FUNCTION(__FILE__, __FUNCTION__);

    str = strdup(arg);
    if (!str) goto trouble;

    if (strstr(str, "/") != NULL) {
        /* we have a slash so assume a.b.c.d/mask notation */
        p = strstr(str, "/");
        if (!p) goto trouble;
        *p = '\0';
        host.s_addr = inet_addr(str);
        if (host.s_addr == INADDR_NONE) {
            fprintf(stderr, "invalid IP address\n");
            goto trouble;
        }
        mask = atoi(p+1);
        if ((mask > 32) || (mask < 1)) {
            fprintf(stderr, "netmask out of bounds\n");
            goto trouble;
        }
        /* determine network address */
        net.s_addr = (host.s_addr) & ~htonl((uint32_t)(pow(2, 32-mask)-1));
        config.net_mask = (unsigned int) mask;
        config.net_base = net;
        config.mode = MODE_NET;
    } else if ((strstr(str, ".")) && (strstr(str, ":"))) {
        /* IP address:port */
        char *div;
        int port = 0;
        div = strstr(str, ":");
        if (!div) goto trouble;
        *div='\0'; /* *yuck* */
        div += 1;
        port = atoi(div);
        if (port <= 0) goto trouble;
        config.sip_port = port;
        host.s_addr = 0;
        host.s_addr = inet_addr(str);
        if ((host.s_addr = inet_addr(str)) == INADDR_NONE) goto trouble;
        config.net_ip = host;
        config.net_ip = host;
        config.mode = MODE_IP;
    } else if ((host.s_addr = inet_addr(str)) != INADDR_NONE) {
        /* IP address */
        config.net_ip = host;
        config.mode = MODE_IP;
    } else {
        /* check for hostname */
        struct hostent *hp;

        hp = gethostbyname(str);
        if (hp == NULL) {
            herror("cannot resolve host");
            goto trouble;
        }
        if (hp->h_addrtype != AF_INET) {
            printf("need AF_INET type address\n");
            goto trouble;
        }
        memcpy(&(host.s_addr), hp->h_addr, hp->h_length);
        config.target_host = (char *) arg;
        config.net_ip = host;
        config.mode = MODE_HOST;
    }
    free(str);
    return 1;
trouble:
    if (str) free(str);
    return 0;
}

/******************************************************************************
 *
 * main_sighandler()
 *
 * main process signal handler
 *
 *****************************************************************************/

void main_sighandler(int sig) {

	switch(sig) {
	case SIGTERM:	/* fall through */
	case SIGINT:	/* fall through */
	case SIGKILL:
		stats_dump();
		pthread_kill(pth_worker, SIGINT);
		pthread_kill(pth_listener, SIGINT);
		pthread_mutex_destroy(&scrlock);
		pthread_mutex_destroy(&tasklock);
		fingerprint_destroydb();
		exit(0);
		break;
	default:
		error(ERR_NOTICE, "received undefined signal %d", sig);
		break;
	}
}


/******************************************************************************
 *
 * smap main
 *
 * what do you expect here?
 *
 *****************************************************************************/

int main (int argc, char **argv) {
	extern struct config_t config;
	extern int last_task_id;
	last_task_id = 0;
	active_tasks = 0;
	int i, opt;
	struct in_addr target_ip;

    /* defaults */
    config.debug = 0;
    config.flags = FLAG_EMPTY | FLAG_UDP_TRANSPORT;
    config.mode = MODE_NONE;
    config.target_host = NULL;
    config.localip = NULL;
    config.net_ip.s_addr = INADDR_NONE;
    config.net_base.s_addr = INADDR_NONE;
    config.net_mask = 0;
    config.sip_port = DEFAULT_SIP_PORT;
    config.sip_domain = NULL;
    config.timeout = DEFAULT_TIMEOUT;
	config.ratelimit = 0;

	printf("\nsmap %s <hs@123.org> "
        "http://www.wormulon.net/\n\n", SMAP_VERSION);
    while ((opt = getopt(argc, argv, "hdloOp:w:D:f:P:tur:")) != EOF) {
        switch (opt) {
            case 'h': /* help */
                usage(0, argv);
                break;
            case 'd': /* increase debugging */
                config.debug++;
                break;
            case 'o': /* fingerprinting with less output */
                config.flags |= FLAG_FINGERPRINT;
                config.flags |= FLAG_FINGERPRINT_SMALL;
                break;
            case 'O': /* fingerprinting */
                config.flags |= FLAG_FINGERPRINT;
                config.flags &= ~FLAG_FINGERPRINT_SMALL;
                break;
            case 'p': /* SIP destination port */
                config.sip_port = atoi(optarg);
                if (config.sip_port < 0)
                    usage(1, argv);
                break;
            case 'w': /* select() timeout in msec */
                config.timeout = atoi(optarg);
                if ((config.timeout < 10) || (config.timeout > 60*1000)) {
                    fprintf(stderr, "timeout out of bounds\n");
                    usage(1, argv);
                }
                break;
            case 'D': /* SIP domain */
                config.sip_domain = optarg;
                break;
            case 'P': /* nmap-like -P0 */
                if (optarg[0] == '0') {
                    config.flags |= FLAG_NOICMPPING;
                }
                break;
            case 'l': /* fingerprinting learning mode */
                /* learning for fingerprinting somehow only makes
                 * sense with fingerprinting also enabled */
                config.flags |= (FLAG_FPLEARNING | FLAG_FINGERPRINT);
                break;
            case 't': /* TCP transport */
                /* UDP is default so TCP would override it */
                config.flags ^= FLAG_UDP_TRANSPORT;
                config.flags |= FLAG_TCP_TRANSPORT;
                break;
            case 'u': /* UDP transport */
                config.flags ^= FLAG_TCP_TRANSPORT;
                config.flags |= FLAG_UDP_TRANSPORT;
                break;
			case 'r': /* ratelimit messages/sec */
				config.ratelimit = (unsigned int) atoi(optarg);
				break;
            default:
                usage(1, argv);
                break;
        }
    }
    if (argc >= 2) {
        if (!targets(argv[argc-1])) {
            return 1;
        }
    } else {
        usage(1, argv);
    }

	/* don't try ICMP or raw sockets if we don't have root privileges */
	if (geteuid() != 0) {
		config.flags |= FLAG_NOROOT;
		config.flags |= FLAG_NOICMPPING;
	}

    /* set SIP domain to 'localhost' */
    if (config.sip_domain == NULL)
        config.sip_domain = strdup("localhost");

	/* register signal handler */
	signal(SIGINT, main_sighandler);
	signal(SIGTERM, main_sighandler);
	signal(SIGKILL, main_sighandler);

	/* initialize statistics */
	stats_init();

	if (!stun_getip(STUN_SERVER_IP)) {
		error(ERR_NOTICE,
			"STUN based IP discovery failed, falling back to ioctl()");
		if (!get_localip()) {
			error(ERR_FATAL, "could not obtain local IP address");
		}
	}
	error(ERR_DEBUG, "local IP: %s", config.localip);

	fingerprint_loaddb();
	randomizer_init();

	if (config.flags & FLAG_TCP_TRANSPORT) {
		/* use TCP transport */
		if (!tcp_setup_sender()) {
			error(ERR_FATAL, "setting up TCP sender failed");
		}
	} else {
		/* default to UDP transport */
		if (!udp_setup_sender()) {
			error(ERR_FATAL, "setting up UDP sender failed");
		}
	}
#ifdef RAW_SOCKET
	udp_setup_rawsock();
#endif

	if (pthread_create(&pth_listener, NULL, listener_start, NULL) != 0) {
		fprintf(stderr, "pthread listener creation failed\n");
	}
	if (pthread_create(&pth_worker, NULL, worker_start, NULL) != 0) {
		fprintf(stderr, "pthread worker creation failed\n");
	}

	/* add tasks */
    switch (config.mode) {
        case MODE_IP:
        case MODE_HOST:
			task_add((config.flags & FLAG_FINGERPRINT) ? SCAN_FINGERPRINT : SCAN_ENABLED, config.net_ip);
            break;
        case MODE_NET:
            for (i = 0; i < pow(2, 32-config.net_mask); i++) {
                target_ip.s_addr = config.net_base.s_addr + htonl(i);
				task_add((config.flags & FLAG_FINGERPRINT) ? SCAN_FINGERPRINT : SCAN_ENABLED, target_ip);
            }
            break;
        case MODE_NONE:
        default:
            usage(1, argv);
            goto error_while_scanning;
            break; /* make gcc happy */
    }
error_while_scanning:

	pthread_join(pth_worker, NULL);
	pthread_kill(pth_worker, SIGINT);
	pthread_kill(pth_listener, SIGINT);
	stats_dump();

	/* clean up */
	pthread_mutex_destroy(&scrlock);
	pthread_mutex_destroy(&tasklock);
	fingerprint_destroydb();

	return 1;
}
#endif /* _SMAP_C */
