/*
 * Copyright (c) 2010-2014 Todd C. Miller <Todd.Miller@courtesan.com>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <config.h>

#include <sys/types.h>
#include <sys/stat.h>

#include <stdio.h>
#ifdef STDC_HEADERS
# include <stdlib.h>
# include <stddef.h>
#else
# ifdef HAVE_STDLIB_H
#  include <stdlib.h>
# endif
#endif /* STDC_HEADERS */
#ifdef HAVE_STDBOOL_H
# include <stdbool.h>
#else
# include "compat/stdbool.h"
#endif /* HAVE_STDBOOL_H */
#ifdef HAVE_STRING_H
# if defined(HAVE_MEMORY_H) && !defined(STDC_HEADERS)
#  include <memory.h>
# endif
# include <string.h>
#endif /* HAVE_STRING_H */
#ifdef HAVE_STRINGS_H
# include <strings.h>
#endif /* HAVE_STRINGS_H */
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif /* HAVE_UNISTD_H */
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <grp.h>
#include <pwd.h>

#include "sudo_plugin.h"
#include "sudo_compat.h"

/*
 * Sample sudoers group plugin that uses an extra group file with the
 * same format as /etc/group.
 */

static sudo_printf_t sudo_log;

extern void mysetgrfile(const char *);
extern void mysetgrent(void);
extern void myendgrent(void);
extern struct group *mygetgrnam(const char *);

static int
sample_init(int version, sudo_printf_t sudo_printf, char *const argv[])
{
    struct stat sb;

    sudo_log = sudo_printf;

    if (GROUP_API_VERSION_GET_MAJOR(version) != GROUP_API_VERSION_MAJOR) {
	sudo_log(SUDO_CONV_ERROR_MSG,
	    "group_file: incompatible major version %d, expected %d\n",
	    GROUP_API_VERSION_GET_MAJOR(version),
	    GROUP_API_VERSION_MAJOR);
	return -1;
    }

    /* Sanity check the specified group file. */
    if (argv == NULL || argv[0] == NULL) {
	sudo_log(SUDO_CONV_ERROR_MSG,
	    "group_file: path to group file not specified\n");
	return -1;
    }
    if (stat(argv[0], &sb) != 0) {
	sudo_log(SUDO_CONV_ERROR_MSG,
	    "group_file: %s: %s\n", argv[0], strerror(errno));
	return -1;
    }
    if ((sb.st_mode & (S_IWGRP|S_IWOTH)) != 0) {
	sudo_log(SUDO_CONV_ERROR_MSG,
	    "%s must be only be writable by owner\n", argv[0]);
	return -1;
    }

    mysetgrfile(argv[0]);
    mysetgrent();

    return true;
}

static void
sample_cleanup(void)
{
    myendgrent();
}

/*
 * Returns true if "user" is a member of "group", else false.
 */
static int
sample_query(const char *user, const char *group, const struct passwd *pwd)
{
    struct group *grp;
    char **member;

    grp = mygetgrnam(group);
    if (grp != NULL && grp->gr_mem != NULL) {
	for (member = grp->gr_mem; *member != NULL; member++) {
	    if (strcasecmp(user, *member) == 0)
		return true;
	}
    }

    return false;
}

__dso_public struct sudoers_group_plugin group_plugin = {
    GROUP_API_VERSION,
    sample_init,
    sample_cleanup,
    sample_query
};
