# Copyright (c) 2001-2008 Twisted Matrix Laboratories.
# See LICENSE for details.

"""
Tests for implementations of L{IReactorUTP}.
"""

import socket, random, errno

from zope.interface import implements

from twisted.trial import unittest

from twisted.python.log import msg
from twisted.internet import protocol, reactor, defer, interfaces
from twisted.internet import error
from twisted.internet.address import IPv4Address
from twisted.internet.interfaces import IPullProducer
from twisted.protocols import policies
from twisted.test.proto_helpers import AccumulatingProtocol

import utp.utp_twisted

def loopUntil(predicate, interval=0):
    """
    Poor excuse for an event notification helper.  This polls a condition and
    calls back a Deferred when it is seen to be true.

    Do not use this function.
    """
    from twisted.internet import task
    d = defer.Deferred()
    def check():
        res = predicate()
        if res:
            d.callback(res)
    call = task.LoopingCall(check)
    def stop(result):
        call.stop()
        return result
    d.addCallback(stop)
    d2 = call.start(interval)
    d2.addErrback(d.errback)
    return d


class ClosingProtocol(protocol.Protocol):

    def connectionMade(self):
        self.transport.loseConnection()

    def connectionLost(self, reason):
        reason.trap(error.ConnectionDone)

class ClosingFactory(protocol.ServerFactory):
    """Factory that closes port immediatley."""

    def buildProtocol(self, conn):
        self.port.stopListening()
        return ClosingProtocol()


def disconnectAndWait(connector):
    if connector.factory.protocol is None:
        return
    assert isinstance(connector.factory.protocol, AccumulatingProtocol)
    if connector.factory.protocol.closed:
        return
    if connector.factory.protocol.closedDeferred is None:
        connector.factory.protocol.closedDeferred = defer.Deferred()
    connector.disconnect()
    return connector.factory.protocol.closedDeferred


def loseConnectionAndWait(protocol):
    assert isinstance(protocol, AccumulatingProtocol), (protocol, )
    if protocol.closed:
        return
    if protocol.closedDeferred is None:
        protocol.closedDeferred = defer.Deferred()
    protocol.transport.loseConnection()
    return protocol.closedDeferred


class MyProtocolFactoryMixin(object):
    """
    Mixin for factories which create L{AccumulatingProtocol} instances.

    @type protocolFactory: no-argument callable
    @ivar protocolFactory: Factory for protocols - takes the place of the
        typical C{protocol} attribute of factories (but that name is used by
        this class for something else).

    @type protocolConnectionMade: L{NoneType} or L{defer.Deferred}
    @ivar protocolConnectionMade: When an instance of L{AccumulatingProtocol}
        is connected, if this is not C{None}, the L{Deferred} will be called
        back with the protocol instance and the attribute set to C{None}.

    @type protocolConnectionLost: L{NoneType} or L{defer.Deferred}
    @ivar protocolConnectionLost: When an instance of L{AccumulatingProtocol}
        is created, this will be set as its C{closedDeferred} attribute and
        then this attribute will be set to C{None} so the L{defer.Deferred} is
        not used by more than one protocol.

    @ivar protocol: The most recently created L{AccumulatingProtocol} instance
        which was returned from C{buildProtocol}.

    @type called: C{int}
    @ivar called: A counter which is incremented each time C{buildProtocol}
        is called.

    @ivar peerAddresses: A C{list} of the addresses passed to C{buildProtocol}.
    """
    protocolFactory = AccumulatingProtocol

    protocolConnectionMade = None
    protocolConnectionLost = None
    protocol = None
    called = 0

    def __init__(self):
        self.peerAddresses = []


    def buildProtocol(self, addr):
        """
        Create a L{AccumulatingProtocol} and set it up to be able to perform
        callbacks.
        """
        self.peerAddresses.append(addr)
        self.called += 1
        p = self.protocolFactory()
        p.factory = self
        p.closedDeferred = self.protocolConnectionLost
        self.protocolConnectionLost = None
        self.protocol = p
        return p



class MyServerFactory(MyProtocolFactoryMixin, protocol.ServerFactory):
    """
    Server factory which creates L{AccumulatingProtocol} instances.
    """



class MyClientFactory(MyProtocolFactoryMixin, protocol.ClientFactory):
    """
    Client factory which creates L{AccumulatingProtocol} instances.
    """
    failed = 0
    stopped = 0

    def __init__(self):
        MyProtocolFactoryMixin.__init__(self)
        self.deferred = defer.Deferred()
        self.failDeferred = defer.Deferred()

    def clientConnectionFailed(self, connector, reason):
        self.failed = 1
        self.reason = reason
        self.failDeferred.callback(None)

    def clientConnectionLost(self, connector, reason):
        self.lostReason = reason
        self.deferred.callback(None)

    def stopFactory(self):
        self.stopped = 1



class ListeningTestCase(unittest.TestCase):

    def test_listen(self):
        """
        L{IReactorUTP.listenUTP} returns an object which provides
        L{IListeningPort}.
        """
        f = MyServerFactory()
        p1 = reactor.listenUTP(0, f, interface="127.0.0.1")
        self.addCleanup(p1.stopListening)
        self.failUnless(interfaces.IListeningPort.providedBy(p1))


    def testStopListening(self):
        """
        The L{IListeningPort} returned by L{IReactorUTP.listenUTP} can be
        stopped with its C{stopListening} method.  After the L{Deferred} it
        (optionally) returns has been called back, the port number can be bound
        to a new server.
        """
        f = MyServerFactory()
        port = reactor.listenUTP(0, f, interface="127.0.0.1")
        n = port.getHost().port

        def cbStopListening(ignored):
            # Make sure we can rebind the port right away
            port = reactor.listenUTP(n, f, interface="127.0.0.1")
            return port.stopListening()

        d = defer.maybeDeferred(port.stopListening)
        d.addCallback(cbStopListening)
        return d


    def testNumberedInterface(self):
        f = MyServerFactory()
        # listen only on the loopback interface
        p1 = reactor.listenUTP(0, f, interface='127.0.0.1')
        return p1.stopListening()

    def testPortRepr(self):
        f = MyServerFactory()
        p = reactor.listenUTP(0, f)
        portNo = str(p.getHost().port)
        self.failIf(repr(p).find(portNo) == -1)
        def stoppedListening(ign):
            self.failIf(repr(p).find(portNo) != -1)
        d = defer.maybeDeferred(p.stopListening)
        return d.addCallback(stoppedListening)


    def test_serverRepr(self):
        """
        Check that the repr string of the server transport get the good port
        number if the server listens on 0.
        """
        server = MyServerFactory()
        serverConnMade = server.protocolConnectionMade = defer.Deferred()
        port = reactor.listenUTP(0, server)
        self.addCleanup(port.stopListening)

        client = MyClientFactory()
        clientConnMade = client.protocolConnectionMade = defer.Deferred()
        connector = reactor.connectUTP("127.0.0.1",
                                       port.getHost().port, client)
        self.addCleanup(disconnectAndWait, connector)
        def check(tup):
            (serverProto, clientProto) = tup
            portNumber = port.getHost().port
            self.assertEquals(
                repr(serverProto.transport),
                "<AccumulatingProtocol #0 on %s>" % (portNumber,))
            self.addCleanup(loseConnectionAndWait, serverProto)
            self.addCleanup(loseConnectionAndWait, clientProto)
        return defer.gatherResults([serverConnMade, clientConnMade]
            ).addCallback(check)


    def test_restartListening(self):
        """
        Stop and then try to restart a L{utp.Port}: after a restart, the
        server should be able to handle client connections.
        """
        serverFactory = MyServerFactory()
        port = reactor.listenUTP(0, serverFactory, interface="127.0.0.1")
        self.addCleanup(port.stopListening)

        def cbStopListening(ignored):
            port.startListening()

            client = MyClientFactory()
            serverFactory.protocolConnectionMade = defer.Deferred()
            client.protocolConnectionMade = defer.Deferred()
            connector = reactor.connectUTP("127.0.0.1",
                                           port.getHost().port, client)
            self.addCleanup(disconnectAndWait, connector)
            return defer.gatherResults([serverFactory.protocolConnectionMade,
                                        client.protocolConnectionMade]
                ).addCallback(close)

        def close(tup):
            (serverProto, clientProto) = tup
            self.addCleanup(loseConnectionAndWait, clientProto)
            self.addCleanup(loseConnectionAndWait, serverProto)

        d = defer.maybeDeferred(port.stopListening)
        d.addCallback(cbStopListening)
        return d


    def test_exceptInStop(self):
        """
        If the server factory raises an exception in C{stopFactory}, the
        deferred returned by L{utp.Port.stopListening} should fail with the
        corresponding error.
        """
        serverFactory = MyServerFactory()
        def raiseException():
            raise RuntimeError("An error")
        serverFactory.stopFactory = raiseException
        port = reactor.listenUTP(0, serverFactory, interface="127.0.0.1")

        return self.assertFailure(port.stopListening(), RuntimeError)


    def test_restartAfterExcept(self):
        """
        Even if the server factory raise an exception in C{stopFactory}, the
        corresponding C{utp.Port} instance should be in a sane state and can
        be restarted.
        """
        serverFactory = MyServerFactory()
        def raiseException():
            raise RuntimeError("An error")
        serverFactory.stopFactory = raiseException
        port = reactor.listenUTP(0, serverFactory, interface="127.0.0.1")
        self.addCleanup(port.stopListening)

        def cbStopListening(ignored):
            del serverFactory.stopFactory
            port.startListening()

            client = MyClientFactory()
            serverFactory.protocolConnectionMade = defer.Deferred()
            client.protocolConnectionMade = defer.Deferred()
            connector = reactor.connectUTP("127.0.0.1",
                                           port.getHost().port, client)
            self.addCleanup(disconnectAndWait, connector)
            return defer.gatherResults([serverFactory.protocolConnectionMade,
                                        client.protocolConnectionMade]
                ).addCallback(close)

        def close(tup):
            (serverProto, clientProto) = tup
            self.addCleanup(loseConnectionAndWait, clientProto)
            self.addCleanup(loseConnectionAndWait, serverProto)

        return self.assertFailure(port.stopListening(), RuntimeError
            ).addCallback(cbStopListening)


    def test_directConnectionLostCall(self):
        """
        If C{connectionLost} is called directly on a port object, it succeeds
        (and doesn't expect the presence of a C{deferred} attribute).

        C{connectionLost} is called by L{reactor.disconnectAll} at shutdown.
        """
        serverFactory = MyServerFactory()
        port = reactor.listenUTP(0, serverFactory, interface="127.0.0.1")
        portNumber = port.getHost().port
        port.connectionLost(None)

        client = MyClientFactory()
        serverFactory.protocolConnectionMade = defer.Deferred()
        client.protocolConnectionMade = defer.Deferred()
        connector = reactor.connectUTP("127.0.0.1", portNumber, client, timeout=1)
        def check(ign):
            # we can't get ConnectionRefusedErrors without ICMP
            #client.reason.trap(error.ConnectionRefusedError)
            client.reason.trap(error.TimeoutError)
        return client.failDeferred.addCallback(check)


    def test_exceptInConnectionLostCall(self):
        """
        If C{connectionLost} is called directory on a port object and that the
        server factory raises an exception in C{stopFactory}, the exception is
        passed through to the caller.

        C{connectionLost} is called by L{reactor.disconnectAll} at shutdown.
        """
        serverFactory = MyServerFactory()
        def raiseException():
            raise RuntimeError("An error")
        serverFactory.stopFactory = raiseException
        port = reactor.listenUTP(0, serverFactory, interface="127.0.0.1")
        self.assertRaises(RuntimeError, port.connectionLost, None)



def callWithSpew(f):
    from twisted.python.util import spewerWithLinenums as spewer
    import sys
    sys.settrace(spewer)
    try:
        f()
    finally:
        sys.settrace(None)

class LoopbackTestCase(unittest.TestCase):
    """
    Test loopback connections.
    """
    def test_closePortInProtocolFactory(self):
        """
        A port created with L{IReactorUTP.listenUTP} can be connected to with
        L{IReactorUTP.connectUTP}.
        """
        f = ClosingFactory()
        port = reactor.listenUTP(0, f, interface="127.0.0.1")
        self.addCleanup(port.stopListening)
        portNumber = port.getHost().port
        f.port = port
        clientF = MyClientFactory()
        reactor.connectUTP("127.0.0.1", portNumber, clientF)
        def check(x):
            self.assertTrue(clientF.protocol.made)
            self.assertTrue(port.disconnected)
            clientF.lostReason.trap(error.ConnectionDone)
        return clientF.deferred.addCallback(check)

    def _trapCnxDone(self, obj):
        getattr(obj, 'trap', lambda x: None)(error.ConnectionDone)


    def _connectedClientAndServerTest(self, callback):
        """
        Invoke the given callback with a client protocol and a server protocol
        which have been connected to each other.
        """
        serverFactory = MyServerFactory()
        serverConnMade = defer.Deferred()
        serverFactory.protocolConnectionMade = serverConnMade
        port = reactor.listenUTP(0, serverFactory, interface="127.0.0.1")
        self.addCleanup(port.stopListening)

        portNumber = port.getHost().port
        clientF = MyClientFactory()
        clientConnMade = defer.Deferred()
        clientF.protocolConnectionMade = clientConnMade
        reactor.connectUTP("127.0.0.1", portNumber, clientF)

        connsMade = defer.gatherResults([serverConnMade, clientConnMade])
        def connected(tup):
            (serverProtocol, clientProtocol) = tup
            callback(serverProtocol, clientProtocol)
            serverProtocol.transport.loseConnection()
            clientProtocol.transport.loseConnection()
        connsMade.addCallback(connected)
        return connsMade


    def testFailing(self):
        clientF = MyClientFactory()
        # XXX we assume no one is listening on UDP port 69
        reactor.connectUTP("127.0.0.1", 69, clientF, timeout=1)
        def check(ignored):
            # we can't get ConnectionRefusedErrors without ICMP
            #clientF.reason.trap(error.ConnectionRefusedError)
            clientF.reason.trap(error.TimeoutError)
        return clientF.failDeferred.addCallback(check)


    def test_connectByServiceFail(self):
        """
        Connecting to a named service which does not exist raises
        L{error.ServiceNameUnknownError}.
        """
        self.assertRaises(
            error.ServiceNameUnknownError,
            reactor.connectUTP,
            "127.0.0.1", "thisbetternotexist", MyClientFactory())


    def test_connectByService(self):
        """
        L{IReactorUTP.connectUTP} accepts the name of a service instead of a
        port number and connects to the port number associated with that
        service, as defined by L{socket.getservbyname}.
        """
        serverFactory = MyServerFactory()
        serverConnMade = defer.Deferred()
        serverFactory.protocolConnectionMade = serverConnMade
        port = reactor.listenUTP(0, serverFactory, interface="127.0.0.1")
        self.addCleanup(port.stopListening)
        portNumber = port.getHost().port
        clientFactory = MyClientFactory()
        clientConnMade = defer.Deferred()
        clientFactory.protocolConnectionMade = clientConnMade

        def fakeGetServicePortByName(serviceName, protocolName):
            if serviceName == 'http' and protocolName == 'tcp':
                return portNumber
            return 10
        self.patch(socket, 'getservbyname', fakeGetServicePortByName)

        c = reactor.connectUTP('127.0.0.1', 'http', clientFactory)

        connMade = defer.gatherResults([serverConnMade, clientConnMade])
        def connected(tup):
            (serverProtocol, clientProtocol) = tup
            self.assertTrue(
                serverFactory.called,
                "Server factory was not called upon to build a protocol.")
            self.addCleanup(loseConnectionAndWait, serverProtocol)
            self.addCleanup(loseConnectionAndWait, clientProtocol)
        connMade.addCallback(connected)
        return connMade


class StartStopFactory(protocol.Factory):

    started = 0
    stopped = 0

    def startFactory(self):
        if self.started or self.stopped:
            raise RuntimeError
        self.started = 1

    def stopFactory(self):
        if not self.started or self.stopped:
            raise RuntimeError
        self.stopped = 1


class ClientStartStopFactory(MyClientFactory):

    started = 0
    stopped = 0

    def startFactory(self):
        if self.started or self.stopped:
            raise RuntimeError
        self.started = 1

    def stopFactory(self):
        if not self.started or self.stopped:
            raise RuntimeError
        self.stopped = 1


class FactoryTestCase(unittest.TestCase):
    """Tests for factories."""

    def test_serverStartStop(self):
        """
        The factory passed to L{IReactorUTP.listenUTP} should be started only
        when it transitions from being used on no ports to being used on one
        port and should be stopped only when it transitions from being used on
        one port to being used on no ports.
        """
        # Note - this test doesn't need to use listenUTP.  It is exercising
        # logic implemented in Factory.doStart and Factory.doStop, so it could
        # just call that directly.  Some other test can make sure that
        # listenUTP and stopListening correctly call doStart and
        # doStop. -exarkun

        f = StartStopFactory()

        # listen on port
        p1 = reactor.listenUTP(0, f, interface='127.0.0.1')
        self.addCleanup(p1.stopListening)

        self.assertEqual((f.started, f.stopped), (1, 0))

        # listen on two more ports
        p2 = reactor.listenUTP(0, f, interface='127.0.0.1')
        p3 = reactor.listenUTP(0, f, interface='127.0.0.1')

        self.assertEqual((f.started, f.stopped), (1, 0))

        # close two ports
        d1 = defer.maybeDeferred(p1.stopListening)
        d2 = defer.maybeDeferred(p2.stopListening)
        closedDeferred = defer.gatherResults([d1, d2])
        def cbClosed(ignored):
            self.assertEqual((f.started, f.stopped), (1, 0))
            # Close the last port
            return p3.stopListening()
        closedDeferred.addCallback(cbClosed)

        def cbClosedAll(ignored):
            self.assertEquals((f.started, f.stopped), (1, 1))
        closedDeferred.addCallback(cbClosedAll)
        return closedDeferred


    def test_clientStartStop(self):
        """
        The factory passed to L{IReactorUTP.connectUTP} should be started when
        the connection attempt starts and stopped when it is over.
        """
        f = ClosingFactory()
        p = reactor.listenUTP(0, f, interface="127.0.0.1")
        self.addCleanup(p.stopListening)
        portNumber = p.getHost().port
        f.port = p

        factory = ClientStartStopFactory()
        reactor.connectUTP("127.0.0.1", portNumber, factory)
        self.assertTrue(factory.started)
        return loopUntil(lambda: factory.stopped)



class ConnectorTestCase(unittest.TestCase):

    def test_connectorIdentity(self):
        """
        L{IReactorUTP.connectUTP} returns an object which provides
        L{IConnector}.  The destination of the connector is the address which
        was passed to C{connectUTP}.  The same connector object is passed to
        the factory's C{startedConnecting} method as to the factory's
        C{clientConnectionLost} method.
        """
        serverFactory = ClosingFactory()
        utpPort = reactor.listenUTP(0, serverFactory, interface="127.0.0.1")
        self.addCleanup(utpPort.stopListening)
        portNumber = utpPort.getHost().port
        serverFactory.port = utpPort

        seenConnectors = []
        seenFailures = []

        clientFactory = ClientStartStopFactory()
        clientFactory.clientConnectionLost = (
            lambda connector, reason: (seenConnectors.append(connector),
                                       seenFailures.append(reason)))
        clientFactory.startedConnecting = seenConnectors.append

        connector = reactor.connectUTP("127.0.0.1", portNumber, clientFactory)
        self.assertTrue(interfaces.IConnector.providedBy(connector))
        dest = connector.getDestination()
        self.assertEquals(dest.type, "UDP")
        self.assertEquals(dest.host, "127.0.0.1")
        self.assertEquals(dest.port, portNumber)

        d = loopUntil(lambda: clientFactory.stopped)
        def clientFactoryStopped(ignored):
            seenFailures[0].trap(error.ConnectionDone)
            self.assertEqual(seenConnectors, [connector, connector])
        d.addCallback(clientFactoryStopped)
        return d


    def test_userFail(self):
        """
        Calling L{IConnector.stopConnecting} in C{Factory.startedConnecting}
        results in C{Factory.clientConnectionFailed} being called with
        L{error.UserError} as the reason.
        """
        serverFactory = MyServerFactory()
        utpPort = reactor.listenUTP(0, serverFactory, interface="127.0.0.1")
        self.addCleanup(utpPort.stopListening)
        portNumber = utpPort.getHost().port

        def startedConnecting(connector):
            connector.stopConnecting()
 
        clientFactory = ClientStartStopFactory()
        clientFactory.startedConnecting = startedConnecting
        def waitForFail():
            return clientFactory.failDeferred
        self.addCleanup(waitForFail)
        reactor.connectUTP("127.0.0.1", portNumber, clientFactory)

        d = loopUntil(lambda: clientFactory.stopped)
        def check(ignored):
            self.assertEquals(clientFactory.failed, 1)
            clientFactory.reason.trap(error.UserError)
        return d.addCallback(check)


    def test_reconnect(self):
        """
        Calling L{IConnector.connect} in C{Factory.clientConnectionLost} causes
        a new connection attempt to be made.
        """
        serverFactory = ClosingFactory()
        utpPort = reactor.listenUTP(0, serverFactory, interface="127.0.0.1")
        self.addCleanup(utpPort.stopListening)
        portNumber = utpPort.getHost().port
        serverFactory.port = utpPort

        clientFactory = MyClientFactory()

        def clientConnectionLost(connector, reason):
            connector.timeout = 1
            connector.connect()
        clientFactory.clientConnectionLost = clientConnectionLost
        reactor.connectUTP("127.0.0.1", portNumber, clientFactory)

        d = loopUntil(lambda: clientFactory.failed)
        def reconnectFailed(ignored):
            p = clientFactory.protocol
            self.assertEqual((p.made, p.closed), (1, 1))
            #clientFactory.reason.trap(error.ConnectionRefusedError)
            clientFactory.reason.trap(error.TimeoutError)
            self.assertEqual(clientFactory.stopped, 1)
        return d.addCallback(reconnectFailed)


class CannotBindTestCase(unittest.TestCase):
    """
    Tests for correct behavior when a reactor cannot bind to the required UTP
    port.
    """

    def test_cannotBind(self):
        """
        L{IReactorUTP.listenUTP} raises L{error.CannotListenError} if the
        address to listen on is already in use.
        """
        f = MyServerFactory()

        p1 = reactor.listenUTP(0, f, interface='127.0.0.1')
        self.addCleanup(p1.stopListening)
        n = p1.getHost().port
        dest = p1.getHost()
        self.assertEquals(dest.type, "UDP")
        self.assertEquals(dest.host, "127.0.0.1")
        self.assertEquals(dest.port, n)

        # make sure new listen raises error
        self.assertRaises(error.CannotListenError,
                          reactor.listenUTP, n, f, interface='127.0.0.1')



    def _fireWhenDoneFunc(self, d, f):
        """Returns closure that when called calls f and then callbacks d.
        """
        from twisted.python import util as tputil
        def newf(*args, **kw):
            rtn = f(*args, **kw)
            d.callback('')
            return rtn
        return tputil.mergeFunctionMetadata(f, newf)


    def test_clientBind(self):
        """
        L{IReactorUTP.connectUTP} calls C{Factory.clientConnectionFailed} with
        L{error.ConnectBindError} if the bind address specified is already in
        use.
        """
        theDeferred = defer.Deferred()
        sf = MyServerFactory()
        sf.startFactory = self._fireWhenDoneFunc(theDeferred, sf.startFactory)
        p = reactor.listenUTP(0, sf, interface="127.0.0.1")
        self.addCleanup(p.stopListening)

        def _connect1(results):
            d = defer.Deferred()
            cf1 = MyClientFactory()
            cf1.buildProtocol = self._fireWhenDoneFunc(d, cf1.buildProtocol)
            reactor.connectUTP("127.0.0.1", p.getHost().port, cf1,
                               bindAddress=("127.0.0.1", 0))
            d.addCallback(_conmade, cf1)
            return d

        def _conmade(results, cf1):
            d = defer.Deferred()
            cf1.protocol.connectionMade = self._fireWhenDoneFunc(
                d, cf1.protocol.connectionMade)
            d.addCallback(_check1connect2, cf1)
            return d

        def _check1connect2(results, cf1):
            self.assertEquals(cf1.protocol.made, 1)

            d1 = defer.Deferred()
            d2 = defer.Deferred()
            port = cf1.protocol.transport.getHost().port
            cf2 = MyClientFactory()
            cf2.clientConnectionFailed = self._fireWhenDoneFunc(
                d1, cf2.clientConnectionFailed)
            cf2.stopFactory = self._fireWhenDoneFunc(d2, cf2.stopFactory)
            reactor.connectUTP("127.0.0.1", p.getHost().port, cf2,
                               bindAddress=("127.0.0.1", port))
            d1.addCallback(_check2failed, cf1, cf2)
            d2.addCallback(_check2stopped, cf1, cf2)
            dl = defer.DeferredList([d1, d2])
            dl.addCallback(_stop, cf1, cf2)
            return dl

        def _check2failed(results, cf1, cf2):
            self.assertEquals(cf2.failed, 1)
            cf2.reason.trap(error.ConnectBindError)
            self.assertTrue(cf2.reason.check(error.ConnectBindError))
            return results

        def _check2stopped(results, cf1, cf2):
            self.assertEquals(cf2.stopped, 1)
            return results

        def _stop(results, cf1, cf2):
            d = defer.Deferred()
            d.addCallback(_check1cleanup, cf1)
            cf1.stopFactory = self._fireWhenDoneFunc(d, cf1.stopFactory)
            df = loseConnectionAndWait(cf1.protocol)
            return defer.gatherResults([d, df])

        def _check1cleanup(results, cf1):
            self.assertEquals(cf1.stopped, 1)
            self.addCleanup(loseConnectionAndWait, sf.protocol)

        theDeferred.addCallback(_connect1)
        return theDeferred



class MyOtherClientFactory(protocol.ClientFactory):
    def buildProtocol(self, address):
        self.address = address
        self.protocol = AccumulatingProtocol()
        return self.protocol



class LocalRemoteAddressTestCase(unittest.TestCase):
    """
    Tests for correct getHost/getPeer values and that the correct address is
    passed to buildProtocol.
    """
    def test_hostAddress(self):
        """
        L{IListeningPort.getHost} returns the same address as a client
        connection's L{IUTPTransport.getPeer}.
        """
        serverFactory = MyServerFactory()
        serverFactory.protocolConnectionLost = defer.Deferred()
        serverConnectionLost = serverFactory.protocolConnectionLost
        port = reactor.listenUTP(0, serverFactory, interface='127.0.0.1')
        self.addCleanup(port.stopListening)
        n = port.getHost().port

        clientFactory = MyClientFactory()
        onConnection = clientFactory.protocolConnectionMade = defer.Deferred()
        connector = reactor.connectUTP('127.0.0.1', n, clientFactory)

        def check(ignored):
            self.assertEquals([port.getHost()], clientFactory.peerAddresses)
            self.assertEquals(
                port.getHost(), clientFactory.protocol.transport.getPeer())
        onConnection.addCallback(check)

        def cleanup(ignored):
            # Clean up the client explicitly here so that tear down of
            # the server side of the connection begins, then wait for
            # the server side to actually disconnect.
            connector.disconnect()
            return serverConnectionLost
        onConnection.addCallback(cleanup)

        return onConnection



class WriterProtocol(protocol.Protocol):
    def connectionMade(self):
        # use everything ITransport claims to provide. If something here
        # fails, the exception will be written to the log, but it will not
        # directly flunk the test. The test will fail when maximum number of
        # iterations have passed and the writer's factory.done has not yet
        # been set.
        self.transport.write("Hello Cleveland!\n")
        seq = ["Goodbye", " cruel", " world", "\n"]
        self.transport.writeSequence(seq)
        peer = self.transport.getPeer()
        if peer.type != "UDP":
            print("getPeer returned non-UDP socket:", peer)
            self.factory.problem = 1
        us = self.transport.getHost()
        if us.type != "UDP":
            print("getHost returned non-UDP socket:", us)
            self.factory.problem = 1
        self.factory.done = 1

        self.transport.loseConnection()

class ReaderProtocol(protocol.Protocol):
    def dataReceived(self, data):
        self.factory.data += data
    def connectionLost(self, reason):
        self.factory.done = 1

class WriterClientFactory(protocol.ClientFactory):
    def __init__(self):
        self.done = 0
        self.data = ""
    def buildProtocol(self, addr):
        p = ReaderProtocol()
        p.factory = self
        self.protocol = p
        return p

class WriteDataTestCase(unittest.TestCase):
    """
    Test that connected UTP sockets can actually write data. Try to exercise
    the entire ITransport interface.
    """

    def test_writer(self):
        """
        L{IUTPTransport.write} and L{IUTPTransport.writeSequence} send bytes to
        the other end of the connection.
        """
        f = protocol.Factory()
        f.protocol = WriterProtocol
        f.done = 0
        f.problem = 0
        wrappedF = WiredFactory(f)
        p = reactor.listenUTP(0, wrappedF, interface="127.0.0.1")
        self.addCleanup(p.stopListening)
        n = p.getHost().port
        clientF = WriterClientFactory()
        wrappedClientF = WiredFactory(clientF)
        reactor.connectUTP("127.0.0.1", n, wrappedClientF)

        def check(ignored):
            self.failUnless(f.done, "writer didn't finish, it probably died")
            self.failUnless(f.problem == 0, "writer indicated an error")
            self.failUnless(clientF.done,
                            "client didn't see connection dropped")
            expected = "".join(["Hello Cleveland!\n",
                                "Goodbye", " cruel", " world", "\n"])
            self.failUnless(clientF.data == expected,
                            "client didn't receive all the data it expected")
        d = defer.gatherResults([wrappedF.onDisconnect,
                                 wrappedClientF.onDisconnect])
        return d.addCallback(check)


    def test_writeAfterShutdownWithoutReading(self):
        """
        A UTP transport which is written to after the connection has been shut
        down should notify its protocol that the connection has been lost, even
        if the UTP transport is not actively being monitored for read events
        (ie, pauseProducing was called on it).
        """
        # This is an unpleasant thing.  Generally tests shouldn't skip or
        # run based on the name of the reactor being used (most tests
        # shouldn't care _at all_ what reactor is being used, in fact).  The
        # Gtk reactor cannot pass this test, though, because it fails to
        # implement IReactorUTP entirely correctly.  Gtk is quite old at
        # this point, so it's more likely that gtkreactor will be deprecated
        # and removed rather than fixed to handle this case correctly.
        # Since this is a pre-existing (and very long-standing) issue with
        # the Gtk reactor, there's no reason for it to prevent this test
        # being added to exercise the other reactors, for which the behavior
        # was also untested but at least works correctly (now).  See #2833
        # for information on the status of gtkreactor.
        if reactor.__class__.__name__ == 'IOCPReactor':
            raise unittest.SkipTest(
                "iocpreactor does not, in fact, stop reading immediately after "
                "pauseProducing is called. This results in a bonus disconnection "
                "notification. Under some circumstances, it might be possible to "
                "not receive this notifications (specifically, pauseProducing, "
                "deliver some data, proceed with this test).")
        if reactor.__class__.__name__ == 'GtkReactor':
            raise unittest.SkipTest(
                "gtkreactor does not implement unclean disconnection "
                "notification correctly.  This might more properly be "
                "a todo, but due to technical limitations it cannot be.")

        # Called back after the protocol for the client side of the connection
        # has paused its transport, preventing it from reading, therefore
        # preventing it from noticing the disconnection before the rest of the
        # actions which are necessary to trigger the case this test is for have
        # been taken.
        clientPaused = defer.Deferred()

        # Called back when the protocol for the server side of the connection
        # has received connection lost notification.
        serverLost = defer.Deferred()

        class Disconnecter(protocol.Protocol):
            """
            Protocol for the server side of the connection which disconnects
            itself in a callback on clientPaused and publishes notification
            when its connection is actually lost.
            """
            def connectionMade(self):
                """
                Set up a callback on clientPaused to lose the connection.
                """
                msg('Disconnector.connectionMade')
                def disconnect(ignored):
                    msg('Disconnector.connectionMade disconnect')
                    self.transport.loseConnection()
                    msg('loseConnection called')
                clientPaused.addCallback(disconnect)

            def connectionLost(self, reason):
                """
                Notify observers that the server side of the connection has
                ended.
                """
                msg('Disconnecter.connectionLost')
                serverLost.callback(None)
                msg('serverLost called back')

        # Create the server port to which a connection will be made.
        server = protocol.ServerFactory()
        server.protocol = Disconnecter
        port = reactor.listenUTP(0, server, interface='127.0.0.1')
        self.addCleanup(port.stopListening)
        addr = port.getHost()

        class Infinite(object):
            """
            A producer which will write to its consumer as long as
            resumeProducing is called.

            @ivar consumer: The L{IConsumer} which will be written to.
            """
            implements(IPullProducer)

            def __init__(self, consumer):
                self.consumer = consumer

            def resumeProducing(self):
                msg('Infinite.resumeProducing')
                self.consumer.write('x')
                msg('Infinite.resumeProducing wrote to consumer')

            def stopProducing(self):
                msg('Infinite.stopProducing')


        class UnreadingWriter(protocol.Protocol):
            """
            Trivial protocol which pauses its transport immediately and then
            writes some bytes to it.
            """
            def connectionMade(self):
                msg('UnreadingWriter.connectionMade')
                self.transport.pauseProducing()
                clientPaused.callback(None)
                msg('clientPaused called back')
                def write(ignored):
                    msg('UnreadingWriter.connectionMade write')
                    # This needs to be enough bytes to spill over into the
                    # userspace Twisted send buffer - if it all fits into
                    # the kernel, Twisted won't even poll for OUT events,
                    # which means it won't poll for any events at all, so
                    # the disconnection is never noticed.  This is due to
                    # #1662.  When #1662 is fixed, this test will likely
                    # need to be adjusted, otherwise connection lost
                    # notification will happen too soon and the test will
                    # probably begin to fail with ConnectionDone instead of
                    # ConnectionLost (in any case, it will no longer be
                    # entirely correct).
                    producer = Infinite(self.transport)
                    msg('UnreadingWriter.connectionMade write created producer')
                    self.transport.registerProducer(producer, False)
                    msg('UnreadingWriter.connectionMade write registered producer')
                #serverLost.addCallback(write)
                write(None)

        # Create the client and initiate the connection
        client = MyClientFactory()
        client.protocolFactory = UnreadingWriter
        clientConnectionLost = client.deferred
        def cbClientLost(ignored):
            msg('cbClientLost', client.lostReason)
            return client.lostReason
        clientConnectionLost.addCallback(cbClientLost)
        msg('Connecting to %s:%s' % (addr.host, addr.port))
        connector = reactor.connectUTP(addr.host, addr.port, client)

        # By the end of the test, the client should have received notification
        # of unclean disconnection.
        msg('Returning Deferred')
        return self.assertFailure(clientConnectionLost, error.ConnectionLost)



class ConnectionLosingProtocol(protocol.Protocol):
    def connectionMade(self):
        self.transport.write("1")
        self.transport.loseConnection()
        self.master._connectionMade()
        self.master.ports.append(self.transport)



class NoopProtocol(protocol.Protocol):
    def connectionMade(self):
        self.d = defer.Deferred()
        self.master.serverConns.append(self.d)

    def connectionLost(self, reason):
        self.d.callback(True)



class ConnectionLostNotifyingProtocol(protocol.Protocol):
    """
    Protocol which fires a Deferred which was previously passed to
    its initializer when the connection is lost.

    @ivar onConnectionLost: The L{Deferred} which will be fired in
        C{connectionLost}.

    @ivar lostConnectionReason: C{None} until the connection is lost, then a
        reference to the reason passed to C{connectionLost}.
    """
    def __init__(self, onConnectionLost):
        self.lostConnectionReason = None
        self.onConnectionLost = onConnectionLost


    def connectionLost(self, reason):
        self.lostConnectionReason = reason
        self.onConnectionLost.callback(self)



class WiredForDeferreds(policies.ProtocolWrapper):
    def __init__(self, factory, wrappedProtocol):
        policies.ProtocolWrapper.__init__(self, factory, wrappedProtocol)

    def connectionMade(self):
        policies.ProtocolWrapper.connectionMade(self)
        self.factory.onConnect.callback(None)

    def connectionLost(self, reason):
        policies.ProtocolWrapper.connectionLost(self, reason)
        self.factory.onDisconnect.callback(None)



class WiredFactory(policies.WrappingFactory):
    protocol = WiredForDeferreds

    def __init__(self, wrappedFactory):
        policies.WrappingFactory.__init__(self, wrappedFactory)
        self.onConnect = defer.Deferred()
        self.onDisconnect = defer.Deferred()



class AddressTestCase(unittest.TestCase):
    """
    Tests for address-related interactions with client and server protocols.
    """
    def setUp(self):
        """
        Create a port and connected client/server pair which can be used
        to test factory behavior related to addresses.

        @return: A L{defer.Deferred} which will be called back when both the
            client and server protocols have received their connection made
            callback.
        """
        class RememberingWrapper(protocol.ClientFactory):
            """
            Simple wrapper factory which records the addresses which are
            passed to its L{buildProtocol} method and delegates actual
            protocol creation to another factory.

            @ivar addresses: A list of the objects passed to buildProtocol.
            @ivar factory: The wrapped factory to which protocol creation is
                delegated.
            """
            def __init__(self, factory):
                self.addresses = []
                self.factory = factory

            # Only bother to pass on buildProtocol calls to the wrapped
            # factory - doStart, doStop, etc aren't necessary for this test
            # to pass.
            def buildProtocol(self, addr):
                """
                Append the given address to C{self.addresses} and forward
                the call to C{self.factory}.
                """
                self.addresses.append(addr)
                return self.factory.buildProtocol(addr)

        # Make a server which we can receive connection and disconnection
        # notification for, and which will record the address passed to its
        # buildProtocol.
        self.server = MyServerFactory()
        self.serverConnMade = self.server.protocolConnectionMade = defer.Deferred()
        self.serverConnLost = self.server.protocolConnectionLost = defer.Deferred()
        # RememberingWrapper is a ClientFactory, but ClientFactory is-a
        # ServerFactory, so this is okay.
        self.serverWrapper = RememberingWrapper(self.server)

        # Do something similar for a client.
        self.client = MyClientFactory()
        self.clientConnMade = self.client.protocolConnectionMade = defer.Deferred()
        self.clientConnLost = self.client.protocolConnectionLost = defer.Deferred()
        self.clientWrapper = RememberingWrapper(self.client)

        self.port = reactor.listenUTP(0, self.serverWrapper, interface='127.0.0.1')
        self.connector = reactor.connectUTP(
            self.port.getHost().host, self.port.getHost().port, self.clientWrapper)

        return defer.gatherResults([self.serverConnMade, self.clientConnMade])


    def tearDown(self):
        """
        Disconnect the client/server pair and shutdown the port created in
        L{setUp}.
        """
        self.connector.disconnect()
        return defer.gatherResults([
            self.serverConnLost, self.clientConnLost,
            defer.maybeDeferred(self.port.stopListening)])


    def test_buildProtocolClient(self):
        """
        L{ClientFactory.buildProtocol} should be invoked with the address of
        the server to which a connection has been established, which should
        be the same as the address reported by the C{getHost} method of the
        transport of the server protocol and as the C{getPeer} method of the
        transport of the client protocol.
        """
        serverHost = self.server.protocol.transport.getHost()
        clientPeer = self.client.protocol.transport.getPeer()

        self.assertEqual(
            self.clientWrapper.addresses,
            [IPv4Address('UDP', serverHost.host, serverHost.port)])
        self.assertEqual(
            self.clientWrapper.addresses,
            [IPv4Address('UDP', clientPeer.host, clientPeer.port)])


    def test_buildProtocolServer(self):
        """
        L{ServerFactory.buildProtocol} should be invoked with the address of
        the client which has connected to the port the factory is listening on,
        which should be the same as the address reported by the C{getPeer}
        method of the transport of the server protocol and as the C{getHost}
        method of the transport of the client protocol.
        """
        clientHost = self.client.protocol.transport.getHost()
        serverPeer = self.server.protocol.transport.getPeer()

        self.assertEqual(
            self.serverWrapper.addresses,
            [IPv4Address('UDP', serverPeer.host, serverPeer.port)])
        # we can't actually know this, because UDP sucks
##        self.assertEqual(
##            self.serverWrapper.addresses,
##            [IPv4Address('UDP', clientHost.host, clientHost.port)])
        self.assertEqual(
            self.serverWrapper.addresses[0].port,
            clientHost.port)



class LargeBufferWriterProtocol(protocol.Protocol):

    # Win32 sockets cannot handle single huge chunks of bytes.  Write one
    # massive string to make sure Twisted deals with this fact.

    def connectionMade(self):
        # write 60MB
        self.transport.write('X'*self.factory.len)
        self.factory.done = 1
        self.transport.loseConnection()

class LargeBufferReaderProtocol(protocol.Protocol):
    def dataReceived(self, data):
        self.factory.len += len(data)
    def connectionLost(self, reason):
        self.factory.done = 1

class LargeBufferReaderClientFactory(protocol.ClientFactory):
    def __init__(self):
        self.done = 0
        self.len = 0
    def buildProtocol(self, addr):
        p = LargeBufferReaderProtocol()
        p.factory = self
        self.protocol = p
        return p


class FireOnClose(policies.ProtocolWrapper):
    """A wrapper around a protocol that makes it fire a deferred when
    connectionLost is called.
    """
    def connectionLost(self, reason):
        policies.ProtocolWrapper.connectionLost(self, reason)
        self.factory.deferred.callback(None)


class FireOnCloseFactory(policies.WrappingFactory):
    protocol = FireOnClose

    def __init__(self, wrappedFactory):
        policies.WrappingFactory.__init__(self, wrappedFactory)
        self.deferred = defer.Deferred()


class LargeBufferTestCase(unittest.TestCase):
    """Test that buffering large amounts of data works.
    """

    datalen = 60*1024*1024
    def testWriter(self):
        f = protocol.Factory()
        f.protocol = LargeBufferWriterProtocol
        f.done = 0
        f.problem = 0
        f.len = self.datalen
        wrappedF = FireOnCloseFactory(f)
        p = reactor.listenUTP(0, wrappedF, interface="127.0.0.1")
        self.addCleanup(p.stopListening)
        n = p.getHost().port
        clientF = LargeBufferReaderClientFactory()
        wrappedClientF = FireOnCloseFactory(clientF)
        reactor.connectUTP("127.0.0.1", n, wrappedClientF)

        d = defer.gatherResults([wrappedF.deferred, wrappedClientF.deferred])
        def check(ignored):
            self.failUnless(f.done, "writer didn't finish, it probably died")
            self.failUnless(clientF.len == self.datalen,
                            "client didn't receive all the data it expected "
                            "(%d != %d)" % (clientF.len, self.datalen))
            self.failUnless(clientF.done,
                            "client didn't see connection dropped")
        return d.addCallback(check)


class LogTestCase(unittest.TestCase):
    """
    Test logging facility of UTP base classes.
    """

    def test_logstrClientSetup(self):
        """
        Check that the log customization of the client transport happens
        once the client is connected.
        """
        server = MyServerFactory()
        serverLost = server.protocolConnectionLost = defer.Deferred()

        client = MyClientFactory()
        client.protocolConnectionMade = defer.Deferred()

        port = reactor.listenUTP(0, server, interface='127.0.0.1')
        self.addCleanup(port.stopListening)

        connector = reactor.connectUTP(
            port.getHost().host, port.getHost().port, client)
        self.addCleanup(disconnectAndWait, connector)

        # It should still have the default value
        self.assertEquals(connector.transport.logstr,
                          "Uninitialized")

        def cb(ign):
            self.assertEquals(connector.transport.logstr,
                              "AccumulatingProtocol,client")
            def cleanupServer():
                server.protocol.transport.loseConnection()
                return serverLost
            self.addCleanup(cleanupServer)            

        client.protocolConnectionMade.addCallback(cb)
        return client.protocolConnectionMade



class PauseProducingTestCase(unittest.TestCase):
    """
    Test some behaviors of pausing the production of a transport.
    """

    def test_pauseProducingInConnectionMade(self):
        """
        In C{connectionMade} of a client protocol, C{pauseProducing} used to be
        ignored: this test is here to ensure it's not ignored.
        """
        server = MyServerFactory()

        server.protocolConnectionMade = defer.Deferred()
        def _sCM(res):
            self.addCleanup(loseConnectionAndWait, server.protocol)
        server.protocolConnectionMade.addCallback(_sCM)

        client = MyClientFactory()
        client.protocolConnectionMade = defer.Deferred()

        port = reactor.listenUTP(0, server, interface='127.0.0.1')
        self.addCleanup(port.stopListening)

        connector = reactor.connectUTP(
            port.getHost().host, port.getHost().port, client)
        self.addCleanup(disconnectAndWait, connector)

        def checkInConnectionMade(proto):
            tr = proto.transport
            # The transport should already be monitored
            self.assertTrue(tr.reading or tr.writing)
            proto.transport.pauseProducing()
            self.assertFalse(tr.reading or tr.writing)
            d = defer.Deferred()
            d.addCallback(checkAfterConnectionMade)
            reactor.callLater(0, d.callback, proto)
            return d
        def checkAfterConnectionMade(proto):
            tr = proto.transport
            # The transport should still not be monitored
            self.assertFalse(tr.reading or tr.writing)
        client.protocolConnectionMade.addCallback(checkInConnectionMade)
        return client.protocolConnectionMade

    if not interfaces.IReactorFDSet.providedBy(reactor):
        test_pauseProducingInConnectionMade.skip = "Reactor not providing IReactorFDSet"



class CallBackOrderTestCase(unittest.TestCase):
    """
    Test the order of reactor callbacks
    """

    def test_loseOrder(self):
        """
        Check that Protocol.connectionLost is called before factory's
        clientConnectionLost
        """
        server = MyServerFactory()

        server.protocolConnectionMade = defer.Deferred()
        def _sCM(res):
            self.addCleanup(loseConnectionAndWait, server.protocol)
        server.protocolConnectionMade.addCallback(_sCM)

        client = MyClientFactory()
        client.protocolConnectionLost = defer.Deferred()
        client.protocolConnectionMade = defer.Deferred()

        def _cbCM(res):
            """
            protocol.connectionMade callback
            """
            reactor.callLater(0, client.protocol.transport.loseConnection)

        client.protocolConnectionMade.addCallback(_cbCM)

        port = reactor.listenUTP(0, server, interface='127.0.0.1')
        self.addCleanup(port.stopListening)

        connector = reactor.connectUTP(
            port.getHost().host, port.getHost().port, client)
        self.addCleanup(connector.disconnect)

        def _cbCCL(res):
            """
            factory.clientConnectionLost callback
            """
            return 'CCL'

        def _cbCL(res):
            """
            protocol.connectionLost callback
            """
            return 'CL'

        def _cbGather(res):
            self.assertEquals(res, ['CL', 'CCL'])

        d = defer.gatherResults([
                client.protocolConnectionLost.addCallback(_cbCL),
                client.deferred.addCallback(_cbCCL)])
        return d.addCallback(_cbGather)

