/******************************************************************************

                              Copyright (c) 2009
                            Lantiq Deutschland GmbH
                     Am Campeon 3; 85579 Neubiberg, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

******************************************************************************/

/**
   \file drv_vmmc_gr909.c
   This file contains the implementation of functions for GR909 linetesting.
*/

/** +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   \todo
   - protect running GR909, keep status information
   - check how to set different thresholds for pass/fail decision
*/


/* ========================================================================== */
/*                                 Includes                                   */
/* ========================================================================== */
#include "drv_vmmc_api.h"
#if (VMMC_CFG_FEATURES & VMMC_FEAT_GR909)
#include "drv_vmmc_alm_priv.h"
#include "drv_vmmc_errno.h"

/* ========================================================================== */
/*                             Macro definitions                              */
/* ========================================================================== */
#define HPT_W2G_AC_LIM_DEFAULT  9553  /* 0x2551 */
#define HPT_W2W_AC_LIM_DEFAULT  9542  /* 0x2546 */
#define HPT_W2G_DC_LIM_DEFAULT  18144 /* 0x46e0 */
#define HPT_W2W_DC_LIM_DEFAULT  18218 /* 0x472a */
#define FEMF_W2G_AC_LIM_DEFAULT 1910  /* 0x0776 */
#define FEMF_W2W_AC_LIM_DEFAULT 1908  /* 0x0774 */
#define FEMF_W2G_DC_LIM_DEFAULT 716   /* 0x02cc */
#define FEMF_W2W_DC_LIM_DEFAULT 809   /* 0x0329 */
#define RFT_RES_LIM_DEFAULT     8573  /* 0x217d */
#define ROH_LIN_LIM_DEFAULT     15    /* 0x000f */
#define RIT_LOW_LIM_DEFAULT     333   /* 0x014d */
#define RIT_HIGH_LIM_DEFAULT    9538  /* 0x2542 */

/* ========================================================================== */
/*                             Type definitions                               */
/* ========================================================================== */

/* ========================================================================== */
/*                             Global variables                               */
/* ========================================================================== */

/* ========================================================================== */
/*                           Function prototypes                              */
/* ========================================================================== */

/* ========================================================================== */
/*                         Function implementation                            */
/* ========================================================================== */

/**
   Start selected subset (or all) GR909 tests
\param
   pLLChannel - pointer to LL channel context
\param
   p_start - pointer to IFX_TAPI_GR909_START_t structure
\return
   IFX_SUCCESS or IFX_ERROR
\remarks
   GR909 tests can be started only on disabled lines.
*/
IFX_int32_t VMMC_TAPI_LL_ALM_GR909_Start (IFX_TAPI_LL_CH_t *pLLChannel,
                                        IFX_TAPI_GR909_START_t *p_start)
{
   VMMC_CHANNEL *pCh  = (VMMC_CHANNEL *) pLLChannel;
   IFX_int32_t  ret;
   VMMC_DEVICE  *pDev = pCh->pParent;
   GR909_Linetesting_Control_t *pGR909ctl = &pCh->pALM->gr909_ctrl;
   OPMODE_CMD_t *pOpmod = &pCh->pALM->dc_opmode;
   IFX_uint8_t nCh = pCh->nChannel - 1;

   if (nCh >= pDev->caps.nALI)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   /* Nothing to test ? */
   if (p_start->test_mask == 0)
   {
      /* errmsg: Parameter is out of range */
      RETURN_STATUS (VMMC_statusFuncParm);
   }

   /* protect channel against concurrent tasks */
   VMMC_OS_MutexGet (&pCh->chAcc);

   /* report error if line is not in "disabled" state */
   if (pOpmod->OP_MODE != VMMC_OPMOD_PDH)
   {
      VMMC_OS_MutexRelease (&pCh->chAcc);
      RETURN_STATUS (VMMC_statusGR909LineNotDisabled);
   }

   pGR909ctl->RW      = CMDWRITE;
   pGR909ctl->CMD     = CMD_ALI;
   pGR909ctl->CHAN    = nCh;
   pGR909ctl->MOD     = MOD_DCCTL;
   pGR909ctl->ECMD    = ECMD_GR909_CTRL_CMD;
   pGR909ctl->LENGTH  = 28;
   pGR909ctl->COUNTRY = p_start->pl_freq;
   pGR909ctl->HPT     = ((p_start->test_mask & IFX_TAPI_GR909_HPT) ? 1 : 0);
   pGR909ctl->FEMF    = ((p_start->test_mask & IFX_TAPI_GR909_FEMF)? 1 : 0);
   pGR909ctl->RFT     = ((p_start->test_mask & IFX_TAPI_GR909_RFT) ? 1 : 0);
   pGR909ctl->ROH     = ((p_start->test_mask & IFX_TAPI_GR909_ROH) ? 1 : 0);
   pGR909ctl->RIT     = ((p_start->test_mask & IFX_TAPI_GR909_RIT) ? 1 : 0);
   pGR909ctl->HPT_W2G_AC_LIM  = HPT_W2G_AC_LIM_DEFAULT;
   pGR909ctl->HPT_W2W_AC_LIM  = HPT_W2W_AC_LIM_DEFAULT;
   pGR909ctl->HPT_W2G_DC_LIM  = HPT_W2G_DC_LIM_DEFAULT;
   pGR909ctl->HPT_W2W_DC_LIM  = HPT_W2W_DC_LIM_DEFAULT;
   pGR909ctl->FEMF_W2G_AC_LIM = FEMF_W2G_AC_LIM_DEFAULT;
   pGR909ctl->FEMF_W2W_AC_LIM = FEMF_W2W_AC_LIM_DEFAULT;
   pGR909ctl->FEMF_W2G_DC_LIM = FEMF_W2G_DC_LIM_DEFAULT;
   pGR909ctl->FEMF_W2W_DC_LIM = FEMF_W2W_DC_LIM_DEFAULT;
   pGR909ctl->RFT_RES_LIM     = RFT_RES_LIM_DEFAULT;
   pGR909ctl->ROH_LIN_LIM     = ROH_LIN_LIM_DEFAULT;
   pGR909ctl->RIT_LOW_LIM     = RIT_LOW_LIM_DEFAULT;
   pGR909ctl->RIT_HIGH_LIM    = RIT_HIGH_LIM_DEFAULT;

   /* If one of these measurements
      is started, both bits should be set because the DC firmware can't handle
      the ROH measurement separately at the moment */
   if ((pGR909ctl->RFT == 1) || (pGR909ctl->ROH == 1))
   {
     pGR909ctl->RFT = 1;
     pGR909ctl->ROH = 1;
   }

   ret = CmdWrite (pDev, (IFX_uint32_t *)((IFX_void_t *)pGR909ctl),
                                                            pGR909ctl->LENGTH);

   if (!VMMC_SUCCESS(ret))
   {
      VMMC_OS_MutexRelease (&pCh->chAcc);
      RETURN_STATUS (ret);
   }

   /* start GR909 measurement */
   pOpmod->RW      = CMDWRITE;
   pOpmod->CMD     = CMD_ALI;
   pOpmod->CHAN    = nCh;
   pOpmod->MOD     = MOD_DCCTL;
   pOpmod->ECMD    = ECMD_Opmode_CMD;
   pOpmod->LENGTH  = 4;
   pOpmod->OP_MODE = VMMC_OPMOD_GR909;

   ret = CmdWrite (pDev, (IFX_uint32_t *)((IFX_void_t *)pOpmod),
                                                               pOpmod->LENGTH);

   VMMC_OS_MutexRelease (&pCh->chAcc);

   RETURN_STATUS (ret);
}


/**
   Stop GR909 tests
\param
   pLLChannel - pointer to LL channel context
\return
   IFX_SUCCESS or IFX_ERROR
*/
IFX_int32_t VMMC_TAPI_LL_ALM_GR909_Stop (IFX_TAPI_LL_CH_t *pLLChannel)
{
   VMMC_CHANNEL *pCh  = (VMMC_CHANNEL *) pLLChannel;
   IFX_int32_t  ret;
   VMMC_DEVICE  *pDev = pCh->pParent;
   OPMODE_CMD_t *pOpmod = &pCh->pALM->dc_opmode;
   IFX_uint8_t nCh = pCh->nChannel - 1;

   if (nCh >= pDev->caps.nALI)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   /* \todo check if GR909 measurement is running */

   /* protect channel against concurrent tasks */
   VMMC_OS_MutexGet (&pCh->chAcc);

   pOpmod->RW      = CMDWRITE;
   pOpmod->CMD     = CMD_ALI;
   pOpmod->CHAN    = nCh;
   pOpmod->MOD     = MOD_DCCTL;
   pOpmod->ECMD    = ECMD_Opmode_CMD;
   pOpmod->LENGTH  = 4;
   pOpmod->OP_MODE = VMMC_OPMOD_PDH;

   ret = CmdWrite (pDev, (IFX_uint32_t *)((IFX_void_t *)pOpmod),
                                                               pOpmod->LENGTH);

   VMMC_OS_MutexRelease (&pCh->chAcc);

   RETURN_STATUS (ret);
}


/**
   Read GR909 results
\param
   pLLChannel - pointer to LL channel context
   pResults   - pointer to TAPI result structure
\return
   IFX_SUCCESS or IFX_ERROR
\remarks
   GR909 results are polled by the application after reception of the
   IFX_TAPI_GR909_RDY event
*/
IFX_int32_t VMMC_TAPI_LL_ALM_GR909_Result (IFX_TAPI_LL_CH_t *pLLChannel,
                                        IFX_TAPI_GR909_RESULT_t *pResults)
{
   VMMC_CHANNEL *pCh  = (VMMC_CHANNEL *) pLLChannel;
   IFX_int32_t  ret;
   VMMC_DEVICE  *pDev = pCh->pParent;
   GR909_Result_Pass_Fail_t *pPassFail = &pCh->pALM->gr909_result;
   IFX_uint8_t nCh = pCh->nChannel - 1;

   if (nCh >= pDev->caps.nALI)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   pPassFail->RW   = CMDREAD;
   pPassFail->CMD  = CMD_ALI;
   pPassFail->CHAN = nCh;
   pPassFail->MOD  = MOD_DCCTL;
   pPassFail->ECMD = ECMD_GR909_RES_PASS_FAIL;
   pPassFail->LENGTH = 4;

   ret = CmdRead (pDev, (IFX_uint32_t *)((IFX_void_t *)pPassFail),
                           (IFX_uint32_t *)((IFX_void_t *)pPassFail),
                           pPassFail->LENGTH);

   if (!VMMC_SUCCESS(ret))
      RETURN_STATUS (ret);

   /* \todo check that tests are really finished (status registers?) */

   /* fill in validity information in the Results structure */
   pResults->valid  = pPassFail->HPT_VALID  ? IFX_TAPI_GR909_HPT : 0;
   pResults->valid |= pPassFail->FEMF_VALID ? IFX_TAPI_GR909_FEMF : 0;
   pResults->valid |= pPassFail->RFT_VALID  ? IFX_TAPI_GR909_RFT : 0;
   pResults->valid |= pPassFail->ROH_VALID  ? IFX_TAPI_GR909_ROH : 0;
   pResults->valid |= pPassFail->RIT_VALID  ? IFX_TAPI_GR909_RIT : 0;
   /* fill in pass/fail information in the Results structure */
   pResults->passed  = pPassFail->HPT_PASS  ? IFX_TAPI_GR909_HPT : 0;
   pResults->passed |= pPassFail->FEMF_PASS ? IFX_TAPI_GR909_FEMF : 0;
   pResults->passed |= pPassFail->RFT_PASS  ? IFX_TAPI_GR909_RFT : 0;
   pResults->passed |= pPassFail->ROH_PASS  ? IFX_TAPI_GR909_ROH : 0;
   pResults->passed |= pPassFail->RIT_PASS  ? IFX_TAPI_GR909_RIT : 0;

   /* read the test results */
   if (pPassFail->HPT_VALID)
   {
      GR909_Result_HPT_t *pHPT = &pCh->pALM->gr909_hpt;

      pHPT->RW    = CMDREAD;
      pHPT->CMD   = CMD_ALI;
      pHPT->CHAN  = nCh;
      pHPT->MOD   = MOD_DCCTL;
      pHPT->ECMD  = ECMD_GR909_RES_HPT;
      pHPT->LENGTH  = 12;

      ret = CmdRead (pDev, (IFX_uint32_t *)((IFX_void_t *)pHPT),
                           (IFX_uint32_t *)((IFX_void_t *)pHPT),
                           pHPT->LENGTH);

      if (!VMMC_SUCCESS(ret))
         RETURN_STATUS (ret);

      pResults->HPT_AC_R2G = pHPT->HPT_AC_R2G;
      pResults->HPT_AC_T2G = pHPT->HPT_AC_T2G;
      pResults->HPT_AC_T2R = pHPT->HPT_AC_T2R;
      pResults->HPT_DC_R2G = pHPT->HPT_DC_R2G;
      pResults->HPT_DC_T2G = pHPT->HPT_DC_T2G;
      pResults->HPT_DC_T2R = pHPT->HPT_DC_T2R;
   }
   if (pPassFail->FEMF_VALID)
   {
      GR909_Result_FEMF_t *pFEMF = &pCh->pALM->gr909_femf;

      pFEMF->RW    = CMDREAD;
      pFEMF->CMD   = CMD_ALI;
      pFEMF->CHAN  = nCh;
      pFEMF->MOD   = MOD_DCCTL;
      pFEMF->ECMD  = ECMD_GR909_RES_FEMF;
      pFEMF->LENGTH  = 12;

      ret = CmdRead (pDev, (IFX_uint32_t *)((IFX_void_t *)pFEMF),
                           (IFX_uint32_t *)((IFX_void_t *)pFEMF),
                           pFEMF->LENGTH);

      if (!VMMC_SUCCESS(ret))
         RETURN_STATUS (ret);

      pResults->FEMF_AC_R2G = pFEMF->FEMF_AC_R2G;
      pResults->FEMF_AC_T2G = pFEMF->FEMF_AC_T2G;
      pResults->FEMF_AC_T2R = pFEMF->FEMF_AC_T2R;
      pResults->FEMF_DC_R2G = pFEMF->FEMF_DC_R2G;
      pResults->FEMF_DC_T2G = pFEMF->FEMF_DC_T2G;
      pResults->FEMF_DC_T2R = pFEMF->FEMF_DC_T2R;
   }
   if (pPassFail->RFT_VALID)
   {
      GR909_Result_RFT_t  *pRFT = &pCh->pALM->gr909_rft;

      pRFT->RW    = CMDREAD;
      pRFT->CMD   = CMD_ALI;
      pRFT->CHAN  = nCh;
      pRFT->MOD   = MOD_DCCTL;
      pRFT->ECMD  = ECMD_GR909_RES_RFT;
      pRFT->LENGTH  = 8;

      ret = CmdRead (pDev, (IFX_uint32_t *)((IFX_void_t *)pRFT),
                           (IFX_uint32_t *)((IFX_void_t *)pRFT),
                           pRFT->LENGTH);

      if (!VMMC_SUCCESS(ret))
         RETURN_STATUS (ret);

      pResults->RFT_R2G = pRFT->RFT_R2G;
      pResults->RFT_T2G = pRFT->RFT_T2G;
      pResults->RFT_T2R = pRFT->RFT_T2R;
   }
   if (pPassFail->ROH_VALID)
   {
      GR909_Result_ROH_t    *pROH = &pCh->pALM->gr909_roh;

      pROH->RW    = CMDREAD;
      pROH->CMD   = CMD_ALI;
      pROH->CHAN  = nCh;
      pROH->MOD   = MOD_DCCTL;
      pROH->ECMD  = ECMD_GR909_RES_ROH;
      pROH->LENGTH  = 4;

      ret = CmdRead (pDev, (IFX_uint32_t *)((IFX_void_t *)pROH),
                           (IFX_uint32_t *)((IFX_void_t *)pROH),
                           pROH->LENGTH);

      if (!VMMC_SUCCESS(ret))
         RETURN_STATUS (ret);

      pResults->ROH_T2R_L = pROH->ROH_T2R_L;
      pResults->ROH_T2R_H = pROH->ROH_T2R_H;
   }
   if (pPassFail->RIT_VALID)
   {
      GR909_Result_RIT_t *pRIT = &pCh->pALM->gr909_rit;

      pRIT->RW    = CMDREAD;
      pRIT->CMD   = CMD_ALI;
      pRIT->CHAN  = nCh;
      pRIT->MOD   = MOD_DCCTL;
      pRIT->ECMD  = ECMD_GR909_RES_RIT;
      pRIT->LENGTH  = 4;

      ret = CmdRead (pDev, (IFX_uint32_t *)((IFX_void_t *)pRIT),
                           (IFX_uint32_t *)((IFX_void_t *)pRIT),
                           pRIT->LENGTH);

      if (!VMMC_SUCCESS(ret))
         RETURN_STATUS (ret);

      pResults->RIT_RES = pRIT->RIT_RES;
   }

   pResults->dev = IFX_TAPI_GR909_DEV_VMMC;

   RETURN_STATUS (ret);
}
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_GR909) */
