/*
     This file is part of GNUnet.
     Copyright (C) 2009-2014 GNUnet e.V.

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 3, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
     Boston, MA 02110-1301, USA.
*/

/**
 * @file core/gnunet-service-core_neighbours.h
 * @brief code for managing low-level 'plaintext' connections with transport (key exchange may or may not be done yet)
 * @author Christian Grothoff
 */
#ifndef GNUNET_SERVICE_CORE_NEIGHBOURS_H
#define GNUNET_SERVICE_CORE_NEIGHBOURS_H

#include "gnunet_util_lib.h"

/**
 * Transmit the given message to the given target.  Note that a
 * non-control messages should only be transmitted after a
 * #GSC_SESSION_solicit() call was made (that call is always invoked
 * when the message queue is empty).  Outbound quotas and memory
 * bounds will then be enfoced (as #GSC_SESSION_solicit() is only called
 * if sufficient bandwidth is available).
 *
 * @param target peer that should receive the message (must be connected)
 * @param msg message to transmit
 * @param timeout by when should the transmission be done?
 */
void
GSC_NEIGHBOURS_transmit (const struct GNUNET_PeerIdentity *target,
                         const struct GNUNET_MessageHeader *msg,
                         struct GNUNET_TIME_Relative timeout);


/**
 * Check if the given neighbour has excess bandwidth available.
 *
 * @param target neighbour to check
 * @return #GNUNET_YES if excess bandwidth is available, #GNUNET_NO if not
 */
int
GSC_NEIGHBOURS_check_excess_bandwidth (const struct GNUNET_PeerIdentity *target);


/**
 * Check how many messages are queued for the given neighbour.
 *
 * @param target neighbour to check
 * @return number of items in the message queue
 */
unsigned int
GSC_NEIGHBOURS_get_queue_size (const struct GNUNET_PeerIdentity *target);


/**
 * Initialize neighbours subsystem.
 */
int
GSC_NEIGHBOURS_init (void);


/**
 * Shutdown neighbours subsystem.
 */
void
GSC_NEIGHBOURS_done (void);


#endif
