/******************************************************************************
 *
 * listener.c - part of smap
 *
 * $Id: listener.c,v 1.4 2007-11-03 18:14:52 hscholz Exp $
 *
 *****************************************************************************/
#ifndef _LISTENER_C
#define _LISTENER_C
 
#include "config.h"

unsigned int running;

/******************************************************************************
 *
 * listener_sighandler()
 *
 * signal handler inside listener pthread
 *
 *****************************************************************************/

void listener_sighandler(int sig) {

	FUNCTION(__FILE__, __FUNCTION__);
	switch (sig) {
	case SIGINT:
		running = 0; /* stop listener */
		break;
	default:
		/* ignore */
		break;
	}
}

/******************************************************************************
 *
 * listener_start()
 *
 * main function of listener thread
 * does all the fingerprinting stuff for received messages
 *
 *****************************************************************************/

void *listener_start(void *nothing) {
	extern unsigned int running;
	extern int udp_sendsock;
#ifdef RAW_SOCKET
	extern int udp_rawsock;
#endif
	struct timeval timeout;
	fd_set fdset;
	int i;
	char msg[SIPLEN];
	unsigned int callid;
	task_t *task = NULL;
	state_t *state = NULL;

	struct ip	*ippkt;
	struct icmp	*icmppkt;
	char		*errstr;

	running = 1;

	while (running) {

		/* initialize a few values per received message */
		callid = 0;
		ippkt = NULL;
		icmppkt = NULL;
		errstr = NULL;

		FD_ZERO(&fdset);
		FD_SET(udp_sendsock, &fdset);
#ifdef RAW_SOCKET
		if (HAVE_ROOT) {
			FD_SET(udp_rawsock, &fdset);
		}
#endif

		set_timeout(&timeout, config.timeout * 10); /* XXX no need to hurry */

		if (select(FD_SETSIZE, &fdset, NULL, NULL, &timeout) > 0) {

			/* try all data sources */
			if (FD_ISSET(udp_sendsock, &fdset)) {
				i = read(udp_sendsock, msg, SIPLEN);
				if (i <= 0) {
					printf("broken read from sendsock\n");
					goto nextmsg;
				}
			} else if (HAVE_ROOT && FD_ISSET(udp_rawsock, &fdset)) {
				i = read(udp_rawsock, msg, SIPLEN);
				if (i <= 0) {
					printf("broken read from rawsock\n");
					goto nextmsg;
				}
			}

			/* check if response is ICMP
			 * further tests might not make sense for ICMP messages
			 */
			ippkt = (struct ip *) msg;
			if ((ippkt->ip_v == 4) && (ippkt->ip_p == IPPROTO_ICMP)) {
				/* is ICMP */
				icmppkt = (struct icmp *) (msg + (ippkt->ip_hl * 4));
				assert(icmppkt);
				if ((errstr = icmp_message_str(icmppkt)) != NULL) {
					error(ERR_DEBUG, "ICMP error %s", errstr);
					free(errstr);
				} else {
					error(ERR_DEBUG,
						"unknown ICMP message received (%d/%d)",
						icmppkt->icmp_type, icmppkt->icmp_code);
				}
				/* increment ICMP stats if appropriate */
				if (icmppkt->icmp_type == ICMP_ECHOREPLY) {
					/* lookup task for this message */
					task = task_get_by_target(&ippkt->ip_src);
					if (task) {
						if (!(task->results & RES_ICMP_REACH)) {
							task->results |= RES_ICMP_REACH;
							stats.up++;
						}
					}
				}
				goto nextmsg;
			}

			/* use Call-ID to match response to requests */
			callid = response_getcallid(msg);
			if (callid == 0) {
				error(ERR_NOTICE, "could not parse Call-ID");
				goto nextmsg;
			}
			state = state_lookup_by_callid(callid);
			if (!state) {
				error(ERR_DEBUG, "state_lookup_by_callid: no match");
				goto nextmsg;
			}
			task = state_lookup_task_by_callid(callid);
			if (!task) {
				error(ERR_NOTICE, "state_lookup_task_by_callid: no match");
				/* assume bad message and ignore */
				goto nextmsg;
			}
			error(ERR_DEBUG, "response belongs to task %d (%s)\n", task->tid, inet_ntoa(task->target));

			/* message does belong to a task so process it and
			 * respond with an ACK if needed
			 */
			ack_o_matic(task, msg);	

			/* if not marked as SIP enabled yet do so later
			 * but at least mark as SIP for stats
			 */
			if (!(task->results & RES_SIP_ENABLED))
				stats.sip++;

			/* process response content */
			if (task->scan_type & SCAN_ENABLED) {
				/* only checking if host is SIP enabled */
				task->results |= RES_SIP_ENABLED;
				task_finish(task);
			} else if (task->scan_type & SCAN_FINGERPRINT) {
				/* do full fingerprint */
				int sipmethod, reason;
				fingerprint *fp;
				fp = task->fp;
				assert(fp);

				sipmethod = sip_getcseqmethod(msg);
				if (!sipmethod) {
					error(ERR_NOTICE, "cannot determine request for response");
					goto nextmsg;
				}
				if (!response_reason(&reason, msg)) {
					error(ERR_NOTICE, "cannot determine reason code");
					goto nextmsg;
				}

				/* save that we got a response for our request */
				if (state->test == SIP_OPTIONS)
					fp->options = reason;
				else if (state->test == SIP_NEWMETHOD)
					fp->newmethod = reason;
				else if (state->test == SIP_BROKENFROMTO)
					fp->brokenfromto = reason;
				else if (state->test == SIP_PRACK)
					fp->prack = reason;
				else if (state->test == SIP_INVITE)
					fp->invite = reason;
				else if (state->test == SIP_PING)
					fp->ping = reason;
				else if (state->test == SIP_TMP)
					fp->tmp = reason;

				task->results |= RES_SIP_ENABLED;
				if (!fp->header_ua)
					fp->header_ua = (char *) sip_getheader(msg, "\r\nUser-Agent");
				if (!fp->header_server)
					fp->header_server = (char *) sip_getheader(msg, "\r\nServer");

				/* fingerprint tests ... */
				if ((fp->accept_class == NO_RELEVANCE) || (fp->accept_class == NO_RESPONSE))
					test_accept(&fp->accept_class, msg);

				if ((fp->allow_class == NO_RELEVANCE) || (fp->allow_class == NO_RESPONSE))
					test_allow(&fp->allow_class, msg);

				if (fp->supported_class == NO_RELEVANCE)
					test_supported(&fp->supported_class, msg);

				if (fp->via_class == NO_RELEVANCE)
					test_via(&fp->via_class, msg);

				if (task->scan_type == SIP_OPTIONS) {
					test_headers(&fp->hoe_class, msg);
				}

			}
			else {
				printf("unknown scan type!!!\n");
			}
		}
nextmsg:
		/* next loop step */
		;
	}

	error(ERR_ERROR, "listener finished");
	return NULL;
}

#endif /* _LISTENER_C */
