/******************************************************************************
 *
 * test_via.c - part of smap
 * 
 * $Id: test_via.c,v 1.6 2007-10-23 19:10:49 hscholz Exp $
 *****************************************************************************/
#ifndef _TEST_VIA_C
#define _TEST_VIA_C

#include "config.h"

#define VIA_LEN	256

enum { VIATRANS_UNDEF = 0, VIATRANS_LOWER=1, VIATRANS_UPPER=2 };

struct via_header_t {
	int tcap;			/* transport capitalization */
	char str[VIA_LEN];	/* tag presence and order */
#ifdef VIA_BRANCH
	int blen;			/* Branch presence and length */
	int blenmin, blenmax;
	int bfmt;			/* Branch format */
#endif
	int fpid;	/* fingerprint ID for fingerprints.db */
};

/******************************************************************************
 *
 * Via header fingerprint meta database
 *
 *
 *****************************************************************************/

#ifdef VIA_BRANCH
struct via_header_t via_headers[] = {
	/* Fritz!Box Fon */
	{ 2, "branch;rport;alias;received", 13, 0, 0, 0, 1 },
	/* do not remove */
	{ 0, "", 0, 0, 0, 0, 0 }
};
#else
struct via_header_t via_headers[] = {
	/* Fritz!Box Fon */
	{ 2, "branch;rport;alias;received", 1 },
	/* Asterisk */
	{ 2, "branch;received;rport", 2 },
	/* PA168V PA1688 series phones */
	{ 2, "branch;rport;alias", 3},
	/* Samsung SMT-G32x0 */
	{ 2, "rport;branch;alias", 4 },
	/* Samsung SMT-G33x0 new firmware */
	{ 2, "received;rport;branch;alias", 5 },
	/* do not remove */
	{ 0, "", 0 }
};

#endif /* VIA_BRANCH */


/******************************************************************************
 *
 * test_via()
 *
 * Locate and analyze the Via-Header inside the passed SIP message
 *
 * Criteria:
 *
 * - transport protocol capitalization
 * - tag presence + order
 * - branch tag presence + length
 * - branch tag format
 *
 * o Transport Protocol Capitalization
 *
 *   Some UACs write all lower or all uppercase transport names, i.e.
 *   SIP/2.0/udp vs. SIP/2.0/UDP. We only look for lower vs. upper not
 *   TCP vs. UDP.
 *
 * o Tag Presence and Order
 *
 *   Different UACs set different tags in a different order. We simply
 *   look at the presence/order of them
 *
 * o Branch tag Presence and Length
 *
 *   Some clients do not set a branch tag at all (assume length 0) while
 *   others have a fixed length or length range
 *
 * o Branch tag Format
 *
 *   Branch tags differ format as some have clearly visible ASCII dividers
 *   while others look like plain integers after the fixed prefix
 *
 * Caveats:
 *
 * - we should check if the Via header contains the target IP but on the
 *   other hand there are so many broken UACs out there that add a local
 *   IP (i.e. behind NAT) and would mess up this test.
 *
 *****************************************************************************/

int test_via(int *res, char *buf) {

	char *begin, *p, *n, *endp, *tmpa, *tmpb;
	int len, pos, fpstrlen, cmpnum, end;

	struct via_header_t fp, cmpfp;

	fp.str[0] = '\0';
	fpstrlen = 0;
	fp.tcap = VIATRANS_UNDEF;
#ifdef VIA_BRANCH
	fp.blen = fp.bfmt = -1;
#endif

	FUNCTION(__FILE__, __FUNCTION__);

	p = strcasestr(buf, "\r\nVia:");
	if (p == NULL) {
		*res = NO_RESPONSE;
		error(DEBUG, "message does not contain a Via header");
		return 0;
	}
	p += 2; /* skip leading \r\n */
	n = strstr(p, "\r\n");
	if (n == NULL) {
		*res = NO_RESPONSE;
		error(ERR_ERROR, "cannot locate end of Via header");
		return 0;
	}
	begin = p;
	len = (int) (n-p);
	if (DEBUG) error(ERR_NOTICE, "%s: \"%.*s\"", __FUNCTION__, len, begin);

	//printf("header: %.*s\n", len, begin);
	/* transport capitalization */

	p = strcasestr(begin, "SIP/2.0");
	if (p) {
		p+= 8; /* skip 'SIP/2.0/' */
		if ((*p == 'u') || (*p == 't')) {
			fp.tcap = VIATRANS_LOWER;
		} else if ((*p == 'U') || (*p == 'T')) {
			fp.tcap = VIATRANS_UPPER;
		} else {
			error(ERR_NOTICE, "Via transport neither lower nor upper");
		}
	}
	if (DEBUG || IS_LEARNING)
		error(ERR_NOTICE, "%s: transport capitalization: %d",
			__FUNCTION__, fp.tcap);

	/* Tag stuff */	
	p = begin;
	pos = 0;
	n = strnstr(begin, ";", len);
	if (!n) return 0; /* no tag found */
	pos = (int) (n-begin);
	while (pos < len) {
		n++; pos++; /* skip ; */

		/* check if this tag has a param */
		tmpa = strnstr(n, ";", len-pos);
		tmpb = strnstr(n, "=", len-pos);
		if (tmpa && tmpb) {
			/* terminate at next tag or param depending on position */
			endp = (tmpa > tmpb) ? tmpb : tmpa;
		} else if (tmpa) {
			/* terminate at next tag */
			endp = tmpa;
		} else if (tmpb) {
			/* terminate at value */
			endp = tmpb;
		} else {
			/* no further tag or tag parameter found */
			endp = (begin + len);
		}

#ifdef VIA_BRANCH
		/* branch presence + length */
		if (!strncasecmp(n, "branch", (int) (endp - n))) {
			int branchlen;
			char *branchend;
			/* this is branch so look for the param */
			/* tmpa and tmpb are still valid to use them */
			if (!tmpb || (tmpa < tmpb)) {
				error(ERR_DEBUG, "Via header contains branch w/o param");
				goto skipbranch;
			} else if (!tmpa) {
				branchend = strstr(n, "\r\n");
				if (!branchend) {
					error(ERR_NOTICE, "Cannot find end of Via header");
					goto skipbranch;
				}
			} else {
				branchend = tmpa;
			}
			
			branchlen = ((int) (branchend - n)) - strlen("branch=");
			if (branchlen >0 ) fp.blen = branchlen; 
		}
skipbranch:
#endif /* VIA_BRANCH */


		/* check length */
		if ((fpstrlen + (endp -n) + 1) >= VIA_LEN) {
			error(ERR_NOTICE, "increase VIA_LEN in %s", __FILE__);
			break;
        }

		/* append to fp string */
		if (fp.str[0] != '\0') strncat(fp.str, ";", 1);
		strncat(fp.str, n, (endp-n));
		fpstrlen += (endp-n) + 1;

		/* next tag */
		n = strnstr(n, ";", len-pos);
		if (!n) pos = len; /* bail out of loop */
		pos = (int) (n-begin);
		
	}

	/* start fingerprint comparison */
	cmpnum = end = 0;
	cmpfp = via_headers[cmpnum];
	while ((cmpfp.str[0] != '\0') && !end) {
		int matches = 0;

		if ((strlen(cmpfp.str) == (fpstrlen-1)) &&
			!strncmp(cmpfp.str, fp.str, fpstrlen)) { matches++; }
		if (fp.tcap == cmpfp.tcap) { matches++; }
#ifdef VIA_BRANCH
		if (cmpfp.blen != 0) {
			/* fixed branch length */
			if (cmpfp.blen == fp.blen) { matches++; }
		} else {
			/* variable branch length */
			if ((fp.blen >= cmpfp.blenmin) && (fp.blen <= cmpfp.blenmax)) {
				matches++;
			}
		}
		/* TODO: add bfmt */
#endif

#ifdef VIA_BRANCH
		if (matches == 3) { /* FIXME: 3 aka all test succeeded */	
#else
		if (matches == 2) {
#endif /* VIA_BRANCH */
			end++; /* break from test list */
			*res = cmpfp.fpid; /* set result */
		}

		/* next */
		cmpfp = via_headers[++cmpnum];
	}
	if (!end && IS_LEARNING) {
		error(ERR_NOTICE, "%s: \"%.*s\"", __FUNCTION__, fpstrlen, fp.str);
		error(ERR_NOTICE, "%s: Please add new cmpstr", __FUNCTION__);
	}

	return (end) ? 1 : 0;
}

#endif /* _TEST_VIA_C */
