/*
 * yamonvars reads the YAMON environment variables from
 * the file /dev/yamonenv or the file pointed to by the
 * environment variable YAMONENVFILE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#define DEFAULT_YAMON_ENV_FILE   "/dev/mtd/3"


// control byte definitions:
#define SYSENV_RECORD_IS_FREE  (0xff)
#define SYSENV_RECORD_IN_USE   (0x42)


#define FLAG_ALL_VARIABLES   (1<<0)
#define FLAG_ALL_ENTRIES     (1<<1)
#define FLAG_VERBOSE         (1<<2)


struct s_yamon_entry
{
  struct s_yamon_entry *next;

  unsigned char control;
  unsigned char checksum;
  unsigned char local_checksum;
  unsigned char id;
  unsigned char valuesize;
  unsigned char size;
  unsigned char *name;
  unsigned char *value;
  long           offset;
};
typedef struct s_yamon_entry yamon_entry;


struct s_yamon_env
{
  int  fh;
  yamon_entry *first;
  yamon_entry *last;
  yamon_entry *active_entry[256]; //points to the last entry for that id or is NULL.
};
typedef struct s_yamon_env  yamon_env;



yamon_env * yamonenv_openfile (char *name, int rw);
void  yamonenv_close (yamon_env *env);

yamon_entry *yamon_delete_entry (yamon_entry *e);
int yamon_read_environment (yamon_env *env);
int yamon_append_entry (yamon_env *env, yamon_entry *e);
yamon_entry *yamon_find_entry  (yamon_env *env, char *name);
yamon_entry *yamon_first_entry(yamon_env *env);
yamon_entry *yamon_next_entry (yamon_env *, yamon_entry *e);
yamon_entry *yamon_entry_by_id(yamon_env *, int);
int yamon_find_unused_id  (yamon_env *env);

yamon_entry *yamon_create_entry ();
yamon_entry *yamon_create_entry_from_buffer (unsigned char *buffer);
yamon_entry *yamon_create_new_entry (yamon_env *env, char *name, unsigned char *value, long valuelen);
int  yamon_write_entry (yamon_env *env, yamon_entry *e);




static int  flags = 0;

#define BE_VERBOSE  (flags&FLAG_VERBOSE)



yamon_env * yamonenv_openfile (char *name, int rw)
{
  yamon_env *env = (yamon_env *)malloc(sizeof(yamon_env));
  if (env)
    {
      memset (env, 0, sizeof(yamon_env));
      env->fh = open (name, rw ? O_RDWR : O_RDONLY);
      if (env->fh != -1)
	{
	  if (yamon_read_environment (env) == 0)
	    return env;
	  else if (BE_VERBOSE)
	    fprintf (stderr, "failed read environment\n");
	}
      else if (BE_VERBOSE)
	fprintf (stderr, "failed to open environment file\n");
      free(env);
    }
  else
    if (BE_VERBOSE)
      fprintf (stderr, "failed to alloc environment structure\n");

  return NULL;
}



void  yamonenv_close (yamon_env *env)
{
  if (env)
    {
      if (env->fh != -1)
	close(env->fh);

      while (env->first)
	env->first = yamon_delete_entry(env->first);

      free (env);
    }
}


yamon_entry *yamon_delete_entry (yamon_entry *e)
{
  if (e)
    {
      yamon_entry *e1 = e->next;
      e->next = NULL;
      free(e);
      return e1;
    }
  return NULL;
}



int yamon_read_environment (yamon_env *env)
{
  if (env)
    {
      long  offset = 0L;
      unsigned char buffer[0x80];

      lseek (env->fh, 0, SEEK_SET);
      while (read (env->fh, buffer, 0x80) >= 0x80 && buffer[0] == SYSENV_RECORD_IN_USE)
	{
	  yamon_entry *e = yamon_create_entry_from_buffer(buffer);
	  if (e)
	    {
	      e->offset = offset;
	      yamon_append_entry (env, e);
	    }

	  offset += 0x80;
	}
      return 0;
    }
  return -1;
}



int yamon_append_entry (yamon_env *env, yamon_entry *e)
{
  if (env && e)
    {
      if (env->last)
	{
	  env->last->next = e;
	  env->last = e;
	}
      else
	env->first = env->last = e;
      e->next = NULL;
      env->active_entry[e->id] = e;
      return 0;
    }
  return -1;
}



yamon_entry *yamon_create_entry ()
{
  yamon_entry *e = (yamon_entry *)malloc(sizeof(yamon_entry));
  if (e)
    {
      memset (e, 0, sizeof(yamon_entry));
    }
  return e;
}


yamon_entry *yamon_create_entry_from_buffer (unsigned char *buffer)
{
  yamon_entry *e = yamon_create_entry ();
  if (e)
    {
      unsigned char i;
      unsigned char checksum = 0;
      //control: buffer[0]; /* SYSENV_RECORD_IS_FREE or SYSENV_RECORD_IN_USE */
      // chksum: buffer[1]; /* checksum */
      // id....: buffer[2]; /* unique env-variable id */
      // size..: buffer[3]; /* total size for name and value with terminating 0 */
      // name..: buffer[4..x]
      // value.: buffer[x..y]
      /*
       * if chksum and size are zero, this means that the variable with that id has been unset
       */

      for (i=0; i < buffer[3]; i++)
	checksum += buffer[4+i];

      e->control = buffer[0];
      e->checksum = buffer[1];
      e->local_checksum = checksum;
      e->id = buffer[2];
      e->size = buffer[3];
      e->name = strdup (&buffer[4]);
      e->valuesize = e->size - (strlen(e->name)+1);
      e->value = strdup (&buffer[4+1+strlen(&buffer[4])]);
      // memcpy (e->value, &buffer[...], e->valuesize);
    }
  return e;
}



yamon_entry *yamon_create_new_entry (yamon_env *env, char *name, unsigned char *value, long valuesize)
{
  int id = -1;
  yamon_entry *ee = yamon_find_entry  (env, name);
  if (ee)
    id = ee->id;
  else
    id = yamon_find_unused_id(env);
  if (id >= 0)
    {
      yamon_entry *e = yamon_create_entry ();
      if (e)
	{
	  e->name = strdup(name);
	  if (e->name)
	    {
	      e->value = malloc (valuesize);
	      if (e->value)
		{
		  e->control = SYSENV_RECORD_IN_USE;
		  e->id = id;
		  e->valuesize = valuesize;
		  memcpy (e->value, value, valuesize);
		  e->checksum = e->local_checksum = 0;
		  return e;

		  free(e->value);
		  e->value=NULL;
		}
	      free(e->name);
	      e->name=NULL;
	    }
	  free(e);
	  e = NULL;
	}
    }
  return NULL;
}



int  yamon_write_entry (yamon_env *env, yamon_entry *e)
{
  if (env && e)
    {
      int namesize = strlen(e->name)+1;
      if (e->valuesize < (0x80 - 4 - namesize))
	{
	  long offset = env->last ? env->last->offset : 0;

	  unsigned char i;
	  unsigned char checksum = 0;
	  unsigned char buffer[0x80];
	  memset (buffer, 0xff, sizeof(buffer));

	  for (i=0; i < namesize; i++)
	    checksum += e->name[i];
	  for (i=0; i < e->valuesize; i++)
	    checksum += e->value[i];
	  e->checksum = e->local_checksum = checksum;
	  e->size = strlen(e->name)+1+e->valuesize;

	  buffer[0] = e->control;
	  buffer[1] = e->checksum;
	  buffer[2] = e->id;
	  buffer[3] = e->size;
	  memcpy (&buffer[4], e->name, namesize);
	  memcpy (&buffer[4+namesize], e->value, e->valuesize);


	  if (yamon_append_entry (env, e) == 0)
	    {
	      offset += 0x80;
	      if (lseek (env->fh, offset, SEEK_SET) == offset)
		{
		  if (write (env->fh, buffer, 0x80)==0x80)
		    {
		      if (BE_VERBOSE)
			printf ("write succeded\n");
		      return 0;
		    }
		  fprintf (stderr, "failed to write entry (write)\n");
		}
	      else
		fprintf (stderr, "failed to write entry (seek)\n");
	    }
	  else
	    fprintf (stderr, "failed to write entry (append)\n");
	}
      else
	fprintf (stderr, "failed to write entry (value to big)\n");
    }
  else
    fprintf (stderr, "failed to write entry (invalid arg)\n");
  return -1;
}






yamon_entry *yamon_find_entry  (yamon_env *env, char *name)
{
  if (env)
    {
      int i;
      for (i=0; i<256; i++)
	{
	  yamon_entry *e = yamon_entry_by_id(env, i);
	  if (e && strcmp(e->name, name)==0)
	    return env->active_entry[i];
	}
    }
  return NULL;
}



int yamon_find_unused_id  (yamon_env *env)
{
  if (env)
    {
      int  i;
      unsigned long  ids[256/32];
      yamon_entry *e = yamon_first_entry(env);
      while (e)
	{
	  ids[e->id/32] |= 1<<(e->id%32);
	  e = yamon_next_entry (env, e);
	}

      for (i=0; i<256; i++)
	if ((ids[i/32] & 1<<(i%32))==0)
	  return i;
    }
  return -1;
}



yamon_entry *yamon_entry_by_id(yamon_env *env, int id)
{
  if (env && id >=0 && id<256)
    {
      yamon_entry *e = env->active_entry[id];
      if (e && e->size>0 && e->checksum > 0)
	return env->active_entry[id];
    }
  return NULL;
}




yamon_entry *yamon_first_entry(yamon_env *env)
{
  return env ? env->first : NULL;
}



yamon_entry *yamon_next_entry (yamon_env *env, yamon_entry *e)
{
  return e ? e->next : NULL;
}






void about (char *appname)
{
  printf ("%s -a       show all variables.\n", appname);
  printf ("%s -A       show all entries as history.\n", appname);
  printf ("%s <name>   show the values of the variable name.\n", appname);
}



int main (int argc, char **argv)
{
  char *name = NULL;
  char *value = NULL;
  int  i;

  char *yamon_env_file = getenv("YAMONENVFILE");
  if (yamon_env_file == NULL)
    yamon_env_file = DEFAULT_YAMON_ENV_FILE;

  for (i=1; i<argc; i++)
    {
      if (argv[i][0]=='-')
        {
          switch (argv[i][1])
	    {
            case 'a': /* all */
              flags |= FLAG_ALL_VARIABLES;
              break;

            case 'A': /* realy all */
              flags |= FLAG_ALL_ENTRIES;
              break;
	      
	    case 'v':
	      flags |= FLAG_VERBOSE;
	      break;

            default:
              fprintf (stderr, "unknown option \"%s\"\n", argv[i]);
              break;
	    }
        }
      else
	{
	  if (name)
	    value = argv[i];
	  else
	    name = argv[i];
	}
    }


  yamon_env * env = yamonenv_openfile (yamon_env_file, (name && value)?1:0);
  if (env)
    {
      if (flags & FLAG_ALL_VARIABLES)
	{
	  for (i=0; i<256; i++)
	    {
	      yamon_entry *e = yamon_entry_by_id(env, i);
	      if (e)
		printf ("%s=\"%s\"\n", e->name, e->value);
	    }
	}

      else if (flags & FLAG_ALL_ENTRIES)
	{
	  yamon_entry *e = yamon_first_entry(env);
	  while (e)
	    {
	      if (e->control == SYSENV_RECORD_IN_USE && e->valuesize && e->checksum)
		printf ("%s=%s\n", e->name, e->value);
	      e = yamon_next_entry(env, e);
	    }
	}

      else if (name && value)
	{
	  yamon_entry *e = yamon_create_new_entry (env, name, value, strlen(value)+1);
	  if (e)
	    {
	      if (yamon_write_entry (env, e) != 0)
		fprintf (stderr, "failed to write entry\n");
	    }
	  else if (BE_VERBOSE)
	    fprintf (stderr, "failed to create new entry\n");
	}

      /* argc - firstname == 1
       * 1 = argc-firstname
       */
      else if (name)
	{
	  yamon_entry *e = yamon_find_entry (env, name);
	  if (e)
	    printf ("%s\n", e->value);
	}

      else
	about(argv[0]);

      yamonenv_close (env);
    }
  else
    fprintf (stderr, "failed to access yamon environment\n");
  return 0;
}


