/******************************************************************************

                              Copyright (c) 2010
                            Lantiq Deutschland GmbH
                     Am Campeon 3; 85579 Neubiberg, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

*******************************************************************************/

/*
   Module      : lib_fifo.c
   Date        : 2003-08-16
   Description : This library implements a general FIFO functionality.
                 The FIFO stores data pointer and length information of all
                 its elements in a preallocated ring buffer of element
                 structures.
   Remarks     : The FIFO management structure contains the following
                 information:
                 pTop         - pointer to the top element's structure
                 pBottom      - pointer to the bottom element's structure
                 pGet         - pointer to the next element to be given
                 pPut         - pointer to the next element to be stored
                 fifoSize     - number of element structures in the ring buffer
                 fifoElements - number of element structures containing valid
                                data pointers and length information
                 The concept of the preallocated ring buffer allows the
                 fifoSize to grow after its initialisation while still
                 providing a very efficient element handling.
                 Note: The dynamic growth this is not implemented yet.
                 This module used malloc and free, respectively kmalloc and
                 kfree for Linux.
*/

/* ============================= */
/* Includes                      */
/* ============================= */
#include "lib_fifo.h"
#include "ifxos_memory_alloc.h"
#ifdef LINUX
/* if linux/slab.h is not available, use the precessor linux/malloc.h */
#include <linux/slab.h>
#elif VXWORKS
#include <sys_drv_debug.h>
#endif /* LINUX */

/* ============================= */
/* Local Macros & Definitions    */
/* ============================= */


/* ============================= */
/* Global variable definition    */
/* ============================= */

/* ============================= */
/* Global function declaration   */
/* ============================= */

/* ============================= */
/* Local function declaration    */
/* ============================= */

/* ============================= */
/* Local variable definition     */
/* ============================= */

/* ============================= */
/* Local function definition     */
/* ============================= */


/*******************************************************************************
Description:
   Initialize the FIFO structure
Arguments:
   none
Return:
   fifo handle
*******************************************************************************/
FIFO_ID* fifoInit(const IFX_uint32_t initialElements)
{
   FIFO_ID        *pf;
   tFifoElement   *pe;

   /* Allocate memory for the FIFO management structure */
   pf = (FIFO_ID *) IFXOS_BlockAlloc (sizeof(FIFO_ID));
   if (pf == NULL)
      return NULL;
   /* Initialize the FIFO management structure and allocate memory for */
   /* the amount of elements the FIFO shall be able to handle */
   pf->fifoElements        = 0;
   pf->fifoSize            = initialElements;
   pf->pBottom = (tFifoElement*)
                  IFXOS_BlockAlloc (initialElements * sizeof(tFifoElement));
   if (pf->pBottom == NULL)
   {
      IFXOS_BlockFree (pf);
      return (NULL);
   }
   pf->pTop                = pf->pBottom + (initialElements-1);
   /* Initialize the FIFO elements */
   pe                      = pf->pBottom;
   while (pf->pTop != pe)
   {
      pe->pNext            = pe+1;
      pe->pData            = NULL;
      pe++;
   }
   pf->pTop->pNext         = pf->pBottom;
   pf->pTop->pData         = NULL;
   pf->pPut                = pf->pBottom;
   pf->pGet                = pf->pBottom;
   pf->pLastPut            = NULL;
   return pf;
}

/*******************************************************************************
Description:
   Reset the FIFO (!! deletes ALL references to elements !!)
   This routine is not releasing the pre allocated memory for fifo elements.
   This has to be handled by the client application.
Arguments:
   pf     - a pointer to the FIFO management structure
Return:
   SUCCESS
*******************************************************************************/
IFX_int32_t fifoReset(FIFO_ID *pf)
{
   pf->pPut                = pf->pBottom;
   pf->pGet                = pf->pBottom;
   pf->pLastPut            = NULL;
   pf->fifoElements        = 0;
   return IFX_SUCCESS;
}

/*******************************************************************************
Description:
   Add a new element to the FIFO
Arguments:
   pf     - a pointer to the FIFO management structure
   pData  - a pointer to the data to be stored in the FIFO
   len    - a length information to be stored together with the data pointer
            in the FIFO
Return:
   SUCCESS or ERROR if no more element can be added to the FIFO
*******************************************************************************/
IFX_int32_t fifoPut(FIFO_ID *pf, const IFX_void_t *pData, const IFX_uint32_t len)
{
   tFifoElement *pe, *pNext;

   if (NULL == pf)
      return IFX_ERROR;

   pe    = pf->pPut;
   pNext = pe->pNext;

   /* check if the FIFO has room for a new element */
   if (pNext == pf->pGet)
      return IFX_ERROR;

   /* add the new element */
   pe->length              = len;
   pe->pData               = (IFX_void_t *) pData;
   pf->pLastPut            = pf->pPut;
   pf->pPut                = pNext;
   pf->fifoElements++;

   return IFX_SUCCESS;
}

/*******************************************************************************
Description:
   Retrieve the first element from the FIFO
Arguments:
   pf     - a pointer to the FIFO management structure
   pLen   - a pointer to retrieve the stored length information,
            can be NULL on calling to ignore the parameter
Return:
   A pointer to the data element stored in the FIFO or NULL if no more
   elements are available or the pointer to the FIFO management structure
   was NULL.
Remarks:
   pLen is undefined if return value is NULL
*******************************************************************************/
IFX_void_t* fifoGet(FIFO_ID *pf, IFX_uint32_t *pLen)
{
   IFX_void_t *pData = NULL;

   if (pf == NULL)
      return NULL;

   /* check if the FIFO has an element available */
   if (pf->pPut != pf->pGet)
   {
      pData                   = pf->pGet->pData;
      pf->pGet->pData         = NULL;
      if (pLen != NULL)
         *pLen                = pf->pGet->length;
      pf->pGet                = pf->pGet->pNext;
      pf->fifoElements--;
   }
   return pData;
}

/*******************************************************************************
Description:
   Retrieve the first element from the FIFO without element removing
Arguments:
   pf     - a pointer to the FIFO management structure
   pLen   - a pointer to retrieve the stored length information,
            can be NULL on calling to ignore the parameter
Return:
   A pointer to the data element stored in the FIFO or NULL if no more
   elements are available or the pointer to the FIFO management structure
   was NULL.
Remarks:
   pLen is undefined if return value is NULL
*******************************************************************************/
IFX_void_t* fifoPeek (FIFO_ID *pf, IFX_uint32_t *pLen)
{
   IFX_void_t *pData = NULL;

   if (pf == NULL)
      return NULL;

   /* check if the FIFO has an element available */
   if (pf->pPut != pf->pGet)
   {
      pData                   = pf->pGet->pData;
      if (pLen != NULL)
         *pLen                = pf->pGet->length;
   }
   return pData;
}

/*******************************************************************************
Description:
   Retrieve the last element from the FIFO without removing the element
Arguments:
   pf     - a pointer to the FIFO management structure
   pLen   - a pointer to retrieve the stored length information,
            can be NULL on calling to ignore the parameter
Return:
   A pointer to the data element stored in the FIFO or NULL if no more
   elements are available or the pointer to the FIFO management structure
   was NULL.
Remarks:
   pLen is undefined if return value is NULL
*******************************************************************************/
IFX_void_t* fifoPeekTail (FIFO_ID *pf, IFX_uint32_t *pLen)
{
   IFX_void_t *pData = NULL;

   if (pf == NULL)
      return NULL;

   /* check if the FIFO has an element available */
   if (pf->pPut != pf->pGet)
   {
      pData                   = pf->pLastPut->pData;
      if (pLen != NULL)
         *pLen                = pf->pLastPut->length;
   }

   return pData;
}

/* This routine will check fifo is empty or not.
  * If fifo empty, return 1, else return 0.
  */
IFX_uint8_t fifoEmpty(FIFO_ID *pf)
{
   IFX_uint8_t ret = 0;

   if (pf->pPut ==  pf->pGet)
      ret = 1;

   return ret;
}

/*******************************************************************************
Description:
   This routines frees all allocated resource. It terminates the fifo
   instance.
Arguments:
   pf     - a pointer to the FIFO management structure
Return:
   This function returns with an ERROR, if there are still elements in
   the fifo available. Please call 'fifoFree' before release the fifo.
*******************************************************************************/
IFX_int32_t fifoFree(FIFO_ID *pf)
{
   /* check if the 'read' and the 'write' index point to the same fifo */
   /* element. Otherwise the fifo is not empty and can not be released */
   /* by this function. Please call explicite the 'fifoReset()' to     */
   /* reset the fifo before release the resources of the fifo.         */
   if (pf->pPut != pf->pGet)
   {
      return IFX_ERROR;
   }

   /* free all resources, which were allocated by the fifo */
   IFXOS_BlockFree (pf->pBottom);
   IFXOS_BlockFree (pf);

   return IFX_SUCCESS;
}

/*******************************************************************************
Description:
   get the size of the FIFO (overall number of elements)
Arguments:
   pf     - a pointer to the FIFO management structure
Return:
   the number of elements that can be max. stored in the fifo
*******************************************************************************/
IFX_uint32_t fifoSize(const FIFO_ID *pf)
{
   return(pf->fifoSize);
}

/*******************************************************************************
Description:
   get the number of elements currently in the FIFO
Arguments:
   pf     - a pointer to the FIFO management structure
Return:
   the number of elements currently in the FIFO
*******************************************************************************/
IFX_uint32_t fifoElements(const FIFO_ID *pf)
{
   return(pf->fifoElements);
}


#ifdef DEBUG
/*******************************************************************************
Description:
   Debug function
   Check the integrity of the FIFO by going through the linked list of
   FIFO elements checking that the number of linked elements equals the number
   of elements in the FIFO management structure.
Arguments:
   pf     - a pointer to the FIFO management structure
Return:
   SUCCESS or
   -n    - integrity error (negative number of cnt)
*******************************************************************************/
IFX_int32_t fifoIntegrity(const FIFO_ID *pf)
{
   tFifoElement *pe;
   IFX_uint32_t cnt = 0;

   pe = pf->pGet;
   while (pe != pf->pPut)
   {
      pe = pe->pNext;
      cnt++;
   }
   if (cnt != pf->fifoElements) {
      printf("ERROR fifoIntegrity fifoElements %d != cnt %d\n",
              pf->fifoElements, cnt);
      return -(IFX_int32_t)cnt;
   }
   cnt = 0;
   pe = pf->pBottom;
   do
   {
      pe = pe->pNext;
      cnt++;
   } while (pe != pf->pBottom);
   if (cnt != pf->fifoSize) {
      printf("ERROR fifoIntegrity fifoSize %d != cnt %d\n",
              pf->fifoSize, cnt);
      return -(IFX_int32_t)cnt;
   }

   return IFX_SUCCESS;
}
#endif /* DEBUG */
