/****************************************************************************

                            Copyright (c) 2007-2009
                            Infineon Technologies AG
                     Am Campeon 1-12; 81726 Munich, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

********************************************************************************
   Module       : lib_tapi_lt_vmmc.c
   Description  : GR909 calculation to avoid floating point operations
                  inside the kernel
*******************************************************************************/

/* ============================= */
/* Includes                      */
/* ============================= */

/* system includes */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>

#include "ifx_types.h"
#include "drv_tapi_io.h"
#include "lib_tapi_lt_gr909.h"

#if (defined (VMMC) && defined (TAPI_GR909))

/* ============================= */
/* Defines                       */
/* ============================= */
/* square root of 2 */
#define IFX_SQRT_2 1.41421356237
/* PI */
#define IFX_PI     3.14159265359
/* GIT */
#define IFX_GIT    0.9653

/* HPT and FEMF Formulas */
/* formula for calculation of HPT_AC_R2G, HPT_AC_T2G,
                              FEMF_AC_R2G, FEMF_AC_T2G,
                              HPT_AC_T2R and FEMF_AC_T2R :
   Vresult[Vrms] = Val*(R3+R1) / 32768 / 1.8844 / sqrt(2) / R3
*/
#define HPT_FEMF_AC_CALC(res,val,r_1,r_2) \
   (res)=(val)*(r_2+r_1)/32768.0/1.8844/IFX_SQRT_2/(r_2)

/* formula for calculation of HPT_DC_R2G, HPT_DC_T2G,
                              FEMF_DC_R2G and FEMF_DC_T2G :
   Vresult[Vrms] = (VAL*(R2+R1)/R2/32768/1.8844)+0.7
*/
#define HPT_FEMF_DC_R2G_T2G_CALC(res,val,r_1,r_2) \
   (res)=((val)*(r_2+r_1)/r_2/32768.0/1.8844)+0.7

/* formula for calculation of HPT_DC_T2R and FEMF_DC_T2R :
   Vresult[Vrms] = (VAL*(R2+R1)/R2/32768/1.8844)
*/
#define HPT_FEMF_DC_T2R_CALC(res,val,r_1,r_2) \
   (res)=((val)*(r_2+r_1)/r_2/32768.0/1.8844)

/* RFT and ROH Formulas */
/* formula for calculation of RFT_R2G, RFT_T2G, RFT_T2R,
 *                            ROH_T2R_L and ROH_T2R_H :
   Rresult[Ohm] = VAL*(R2+R1)/R2/26.12244898
*/
#define RFT_ROH_CALC(res,val,r_1,r_2) \
   (res)=(val)*(r_2+r_1)/(r_2)/26.12244898

/* RIT Formulas */
/* Formula for RIT calculations:
   Zresult[Ohm] = VAL*(R2+R1)/R2/105.3497998
*/
#define RIT_CALC(res,val,r_1,r_2) \
   (res)=(val)*(r_2+r_1)/r_2/105.3497998*IFX_GIT

/* ============================= */
/* Local variable definition     */
/* ============================= */

/* default system configuration (SLIC DC) */
IFX_LT_GR909_CFG_t vmmc_cfg =
{
   /* default :  R1 = 1.5 MOhm  */
   1500000.0,
   /* default :  R2 = 3.3 kOhm */
   3289.0,
   /* default :  R3 = 3.3 kOhm */
   3286.0
};

/* ============================= */
/* Global variable definition    */
/* ============================= */

/* ============================= */
/* Local function declaration    */
/* ============================= */

static IFX_void_t lt_vmmc_getres_hpt  (IFX_TAPI_GR909_RESULT_t *pTapiRes,
                                      IFX_LT_GR909_HPT_t    *p_hpt);
static IFX_void_t lt_vmmc_getres_femf (IFX_TAPI_GR909_RESULT_t *pTapiRes,
                                      IFX_LT_GR909_FEMF_t   *p_femf);
static IFX_void_t lt_vmmc_getres_rft (IFX_TAPI_GR909_RESULT_t *pTapiRes,
                                     IFX_LT_GR909_RFT_t    *p_rft);
static IFX_void_t lt_vmmc_getres_roh  (IFX_TAPI_GR909_RESULT_t *pTapiRes,
                                      IFX_LT_GR909_ROH_t    *p_roh);
static IFX_void_t lt_vmmc_getres_rit  (IFX_TAPI_GR909_RESULT_t *pTapiRes,
                                      IFX_LT_GR909_RIT_t    *p_rit);
IFX_void_t VMMC_LT_GR909_CalcResults (IFX_LT_GR909_RESULT_t *p_res,
                                       IFX_TAPI_GR909_RESULT_t *pTapiRes);

/* ============================= */
/* Local function definition     */
/* ============================= */

/**
   Calculates HPT results out of low level results
\param pTapiRes - low level results, in
\param p_hpt - HPT results, out
*/
static IFX_void_t lt_vmmc_getres_hpt (IFX_TAPI_GR909_RESULT_t *pTapiRes,
                                     IFX_LT_GR909_HPT_t    *p_hpt)
{
   /* test passed ? */
   if (pTapiRes->passed & IFX_TAPI_GR909_HPT)
      p_hpt->b_result = IFX_TRUE;

   /* HPT AC RING wire to GND result */
   HPT_FEMF_AC_CALC(p_hpt->f_hpt_ac_r2g, pTapiRes->HPT_AC_R2G, vmmc_cfg.f_R1,
                                                                    vmmc_cfg.f_R2);
   /* HPT AC TIP wire to GND result */
   HPT_FEMF_AC_CALC(p_hpt->f_hpt_ac_t2g, pTapiRes->HPT_AC_T2G, vmmc_cfg.f_R1,
                                                                    vmmc_cfg.f_R2);
   /* HPT AC TIP wire to RING wire result */
   HPT_FEMF_AC_CALC(p_hpt->f_hpt_ac_t2r, pTapiRes->HPT_AC_T2R, vmmc_cfg.f_R1,
                                                                    vmmc_cfg.f_R3);
   /* HPT DC RING wire to GND result */
   HPT_FEMF_DC_R2G_T2G_CALC(p_hpt->f_hpt_dc_r2g, pTapiRes->HPT_DC_R2G,
                                                          vmmc_cfg.f_R1, vmmc_cfg.f_R2);
   /* HPT DC TIP wire to GND result */
   HPT_FEMF_DC_R2G_T2G_CALC(p_hpt->f_hpt_dc_t2g, pTapiRes->HPT_DC_T2G,
                                                          vmmc_cfg.f_R1, vmmc_cfg.f_R2);
   /* HPT DC TIP wire to RING wire result */
   HPT_FEMF_DC_T2R_CALC(p_hpt->f_hpt_dc_t2r, pTapiRes->HPT_DC_T2R,
                                                          vmmc_cfg.f_R1, vmmc_cfg.f_R3);

   return;
}

/**
   Calculates FEMF results out of low level results
\param pTapiRes - low level results, in
\param p_hpt - FEMF results, out
*/
static IFX_void_t lt_vmmc_getres_femf (IFX_TAPI_GR909_RESULT_t *pTapiRes,
                                      IFX_LT_GR909_FEMF_t   *p_femf)
{
   /* test passed ? */
   if (pTapiRes->passed & IFX_TAPI_GR909_FEMF)
      p_femf->b_result = IFX_TRUE;

   /* FEMF AC RING wire to GND result */
   HPT_FEMF_AC_CALC(p_femf->f_femf_ac_r2g, pTapiRes->FEMF_AC_R2G, vmmc_cfg.f_R1,
                                                                     vmmc_cfg.f_R2);
   /* FEMF AC TIP wire to GND result */
   HPT_FEMF_AC_CALC(p_femf->f_femf_ac_t2g, pTapiRes->FEMF_AC_T2G, vmmc_cfg.f_R1,
                                                                     vmmc_cfg.f_R2);
   /* FEMF AC TIP wire to RING wire result */
   HPT_FEMF_AC_CALC(p_femf->f_femf_ac_t2r, pTapiRes->FEMF_AC_T2R, vmmc_cfg.f_R1,
                                                                     vmmc_cfg.f_R3);
   /* FEMF DC RING wire to GND result */
   HPT_FEMF_DC_R2G_T2G_CALC(p_femf->f_femf_dc_r2g, pTapiRes->FEMF_DC_R2G,
                                                           vmmc_cfg.f_R1, vmmc_cfg.f_R2);
   /* FEMF DC TIP wire to GND result */
   HPT_FEMF_DC_R2G_T2G_CALC(p_femf->f_femf_dc_t2g, pTapiRes->FEMF_DC_T2G,
                                                           vmmc_cfg.f_R1, vmmc_cfg.f_R2);
   /* FEMF DC TIP wire to RING wire result */
   HPT_FEMF_DC_T2R_CALC(p_femf->f_femf_dc_t2r, pTapiRes->FEMF_DC_T2R,
                                                           vmmc_cfg.f_R1, vmmc_cfg.f_R3);

   return;
}

/**
   Calculates RFT results out of low level results
\param pTapiRes - low level results, in
\param p_hpt - RFT results, out
*/
static IFX_void_t lt_vmmc_getres_rft (IFX_TAPI_GR909_RESULT_t *pTapiRes,
                                     IFX_LT_GR909_RFT_t    *p_rft)
{
   /* test passed ? */
   if (pTapiRes->passed & IFX_TAPI_GR909_RFT)
      p_rft->b_result = IFX_TRUE;

   /* RFT RING wire to GND result */
   RFT_ROH_CALC(p_rft->f_rft_r2g, pTapiRes->RFT_R2G, vmmc_cfg.f_R1, vmmc_cfg.f_R2);
   /* RFT TIP wire to GND result */
   RFT_ROH_CALC(p_rft->f_rft_t2g, pTapiRes->RFT_T2G, vmmc_cfg.f_R1, vmmc_cfg.f_R2);
   /* RFT TIP wire to RING wire result */
   RFT_ROH_CALC(p_rft->f_rft_t2r, pTapiRes->RFT_T2R, vmmc_cfg.f_R1, vmmc_cfg.f_R3);

   return;
}

/**
   Calculates ROH results out of low level results
\param pTapiRes - low level results, in
\param p_hpt - ROH results, out
*/
static IFX_void_t lt_vmmc_getres_roh  (IFX_TAPI_GR909_RESULT_t *pTapiRes,
                                     IFX_LT_GR909_ROH_t    *p_roh)
{
   /* test passed ? */
   if (pTapiRes->passed & IFX_TAPI_GR909_ROH)
      p_roh->b_result = IFX_TRUE;

   /* ROH TIP wire to RING wire result for low voltage */
   RFT_ROH_CALC(p_roh->f_roh_t2r_l, pTapiRes->ROH_T2R_L, vmmc_cfg.f_R1, vmmc_cfg.f_R3);
   /* ROH TIP wire to RING wire result for high voltage */
   RFT_ROH_CALC(p_roh->f_roh_t2r_h, pTapiRes->ROH_T2R_H, vmmc_cfg.f_R1, vmmc_cfg.f_R3);

   return;
}

/**
   Calculates RIT results out of low level results
\param pTapiRes - low level results, in
\param p_hpt   - RIT results, out
*/
static IFX_void_t lt_vmmc_getres_rit  (IFX_TAPI_GR909_RESULT_t *pTapiRes,
                                     IFX_LT_GR909_RIT_t    *p_rit)
{
   /* test passed ? */
   if (pTapiRes->passed & IFX_TAPI_GR909_RIT)
      p_rit->b_result = IFX_TRUE;

   /* RIT result */
   RIT_CALC(p_rit->f_rit_res,pTapiRes->RIT_RES, vmmc_cfg.f_R1, vmmc_cfg.f_R3);

   return;
}

/**
   Calculates floating point GR909 measurement result values
   from TAPI values.

   \param p_res  - ptr to \ref IFX_LT_GR909_RESULT_t structure
   \param pTapiRes  - ptr to \ref IFX_TAPI_GR909_RESULT_t structure
   \return
      none
   \remark
      Result is filled in by the pointer p_res.
*/
IFX_void_t VMMC_LT_GR909_CalcResults (IFX_LT_GR909_RESULT_t *p_res,
                                       IFX_TAPI_GR909_RESULT_t *meas_res)
{
   /* HPT results valid ? */
   if (meas_res->valid & IFX_TAPI_GR909_HPT)
   {
      p_res->valid_mask |= IFX_LT_GR909_HPT_MASK;
      lt_vmmc_getres_hpt (meas_res, &p_res->hpt);
   }
   /* FEMF results valid ? */
   if (meas_res->valid & IFX_TAPI_GR909_FEMF)
   {
      p_res->valid_mask |= IFX_LT_GR909_FEMF_MASK;
      lt_vmmc_getres_femf (meas_res, &p_res->femf);
   }
   /* RFT results valid ? */
   if (meas_res->valid & IFX_TAPI_GR909_RFT)
   {
      p_res->valid_mask |= IFX_LT_GR909_RFT_MASK;
      lt_vmmc_getres_rft (meas_res, &p_res->rft);
   }
   /* ROH results valid ? */
   if (meas_res->valid & IFX_TAPI_GR909_ROH)
   {
      p_res->valid_mask |= IFX_LT_GR909_ROH_MASK;
      lt_vmmc_getres_roh (meas_res, &p_res->roh);
   }
   /* RIT results valid ? */
   if (meas_res->valid & IFX_TAPI_GR909_RIT)
   {
      p_res->valid_mask |= IFX_LT_GR909_RIT_MASK;
      lt_vmmc_getres_rit (meas_res, &p_res->rit);
   }
}
#endif /* VMMC && TAPI_GR909 */
