/******************************************************************************
 *
 * stun.c - part of smap
 *
 * simple half-way STUN implementation sending a Binding Request to obtain
 * the external IP of ourself.
 *
 * $Id: stun.c,v 1.4 2007-10-07 10:15:12 hscholz Exp $
 * 
 *****************************************************************************/

#ifndef _STUN_C
#define _STUN_C
 
#include "config.h"
#include "stun.h"

/******************************************************************************
 *
 * stun_build_request()
 *
 * build a simple STUN Binding Request message
 * return 1 if successful, 0 in case of trouble
 *
 *****************************************************************************/

int stun_build_request(StunMessage **reqp, int *len) {

	StunMessage *req;
	int i;

	*len = STUN_REQ_LEN;
	*reqp = malloc(*len);
	if (*reqp == NULL) {
		perror("malloc");
		return 0;
	}

	req = *reqp;
	req->msgType = htons(MT_BINDREQ);
	req->msgLen = htons(0);		/* we don't count STUN_REQ_LEN Bytes */

	for (i = 0; i <= 15 ; i++) {
		/* srandom() or srand() should have been called already */
#ifdef HAVE_RANDOM
		req->msgId[i] = (unsigned char) (random() % 0xFF);
#else
		req->msgId[i] = (unsigned char) (rand() % 0xFF);
#endif
	}
	return 1;
}

/******************************************************************************
 *
 * stun_getip()
 *
 * build a single STUN Binding request to obtain the local IP/port combination
 * assuming we can use source port DEFAULT_SIP_PORT.
 *
 *****************************************************************************/

int stun_getip(char *targetip) {

	int sock;
	struct sockaddr_in saddr, taddr;
	struct timeval timeout;
	fd_set fdset;
	int one = 1; /* setsockopt() */

	StunMessage *stunreq;
	int			stunlen;

	char		recvbuf[BUFSIZ];
	int			recvlen;
	unsigned int	offset;
	StunMessage *stunres;
	StunAttribute *sattr;
	StunAttrAddress *addr;
	struct in_addr in;
	int localiplen;

	sock = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP);
	if (!sock) {
		perror("socket");
		return 0;
	}

	/* reuse address if needed for consecutive requests, ... */
    if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &one, sizeof(one)) != 0)
        perror("setsockopt");
	/* Linux doesn't have SO_REUSEPORT but SO_REUSEADDR should be fine */
#if defined (SO_REUSEPORT)
    if (setsockopt(sock, SOL_SOCKET, SO_REUSEPORT, &one, sizeof(one)) != 0)
        perror("setsockopt");
#endif

	saddr.sin_family = PF_INET;
	saddr.sin_port = htons(DEFAULT_SIP_PORT);
	saddr.sin_addr.s_addr = INADDR_ANY;

	if (bind(sock, (const struct sockaddr *) &saddr, sizeof(saddr)) != 0) {
		perror("bind");
		return 0;
	}

	/* build stun request */
	stun_build_request((StunMessage **) &stunreq, &stunlen);

	/* send request */
	taddr.sin_family = PF_INET;	
	taddr.sin_port = htons(3478);
	taddr.sin_addr.s_addr = inet_addr(targetip);
	if (sendto(sock, (void *) stunreq, stunlen, 0,
		(const struct sockaddr *) &taddr, sizeof(taddr)) != stunlen) {
		perror("sendto");
		free(stunreq);
		return 0;
	}
	free(stunreq);

	/* wait for response */
	set_timeout(&timeout, config.timeout);
	FD_ZERO(&fdset);
	FD_SET(sock, &fdset);

	if (select(sock+1, &fdset, NULL, NULL, &timeout) <= 0) {
		error(ERR_DEBUG, "STUN timeout");
		return 0;
	}

	/* read response */
	recvlen = read(sock, &recvbuf, BUFSIZ);

	/* close socket */
	close(sock);

	/* parse response */
	if (recvlen < STUN_REQ_LEN) {
		perror("read");
		return 0;
	}

	stunres = (StunMessage *) recvbuf;
	if (stunres->msgType != MT_BINDRES) {
		error(ERR_DEBUG, "STUN: wrong STUN message type");
		return 0;
	}
	if (stunres->msgLen == 0) {
		error(ERR_DEBUG, "STUN: no attributes in response");
		return 0;
	}	
	offset = STUN_REQ_LEN;
	while ((offset + 8) <= (ntohs(stunres->msgLen) + STUN_REQ_LEN)) {
		/* next attribute */
		sattr = (StunAttribute *) (recvbuf + offset);
		if (ntohs(sattr->attrType) == AT_MAPPED_ADDRESS) {
			addr = (StunAttrAddress *) (recvbuf + offset + 4);
			if (addr->family != 0x01) {
				break;
			}
			in.s_addr = addr->addr;
			localiplen = strlen(inet_ntoa(in));
			config.localip = malloc(localiplen + 1);
			if (config.localip == NULL)
				break;
			snprintf(config.localip, localiplen + 1, "%s", inet_ntoa(in));
			error(ERR_DEBUG, "local IP: %s\n", config.localip);
			
			break; /* we found the key, so stop looking for it */
		}
		/* advance to next attribute */
		offset+= 4;
		offset+= ntohs(sattr->attrLen);
	}
	/* return 1 if successful */
	return (config.localip == NULL) ? 0 : 1;
}
#endif /* _STUN_C */
