/******************************************************************************
 *
 * tasks.c - part of smap
 *
 * $Id: tasks.c,v 1.2 2007-10-07 10:45:20 hscholz Exp $
 *****************************************************************************/
#ifndef _TASKS_C
#define _TASKS_C

#include "config.h"

void task_list(void) {
	extern task_t *tasks;
	extern pthread_mutex_t tasklock;
	task_t *task;

	pthread_mutex_lock(&tasklock);
	task = tasks;
	printf("=== TASK LIST ===\n");
	while (task) {
		printf("[%s] %d: state %d\n",
			inet_ntoa(task->target), task->tid, task->state);
		task = task->next;
	}
	pthread_mutex_unlock(&tasklock);
	printf("=== END OF TASK LIST ===\n");
}

task_t *task_get_by_id(int tid) {
	extern task_t *tasks;
	extern pthread_mutex_t tasklock;
	task_t *task;

	pthread_mutex_lock(&tasklock);
	task = tasks;
	while (task) {
		if (task->tid == tid) {
			pthread_mutex_unlock(&tasklock);
			return task;
		}
		task = task->next;
	}
	pthread_mutex_unlock(&tasklock);
	return NULL;
}

task_t *task_get_by_target(struct in_addr *ip) {
	extern pthread_mutex_t tasklock;
	extern task_t *tasks;
	task_t *task;

	if (!ip) return NULL;

	pthread_mutex_lock(&tasklock);
	task = tasks;
	while (task) {
		if (ip->s_addr == task->target.s_addr) {
			pthread_mutex_unlock(&tasklock);
			return task;
		}
		task = task->next;
	}

	pthread_mutex_unlock(&tasklock);
	return NULL;
}

/******************************************************************************
 *
 * task_list_append()
 *
 * append an existing task to the list of to be process tasks
 *
 *****************************************************************************/

int task_list_append(task_t *newtask) {
	extern pthread_mutex_t tasklock;
	extern task_t *tasks;
	extern unsigned int active_tasks;
	task_t *task;

	pthread_mutex_lock(&tasklock);
	active_tasks++;
	if (tasks == NULL) {
		tasks = newtask;
	} else {
		task = tasks;
		while (task->next) {
			task = task->next;
		}
		task->next = newtask;	
		newtask->prev = task;
	}
	pthread_mutex_unlock(&tasklock);
	return 1;
}

/******************************************************************************
 *  
 *  task_add()
 *
 *  create a new task based on scan type and target
 *
 *****************************************************************************/

int task_add (unsigned int type, struct in_addr target) {

	task_t *task;

	task = malloc(sizeof(task_t));
	if (!task) {
		fprintf(stderr, "malloc failed\n");
		return -1;
	}

	task->prev = NULL;
	task->next = NULL;
	task->tid = ++last_task_id;
	task->state = S_START;
	task->scan_type = type;
	task->results = RES_EMPTY;
	task->target = target;
	task->transport = TRANSPORT_UDP;
	task->test = 0;
	gettimeofday(&task->lastmod, NULL);
	task->fp = NULL;
	if (type == SCAN_FINGERPRINT) {
		/* create empty fingerprint */
		task->fp = malloc(sizeof(fingerprint));
		if (!task->fp) {
			error(ERR_FATAL, "malloc() failed");
		}
		fingerprint_init(task->fp);
	}
	task_list_append(task);

	return 1;
}

/******************************************************************************
 *
 * task_finish()
 *
 * a task finished, so gracefully shut down task
 *
 *  - schedule state expiry
 *  - print scan results
 *
 *****************************************************************************/

void task_finish(task_t *task) {

	/* XXX mark all states as about to expire */

	results_print(task);
	task_destroy(task->tid);
	return;
}

void task_destroy(unsigned int tid) {
	extern pthread_mutex_t tasklock;
	extern unsigned int active_tasks;
	extern task_t *tasks;
	task_t *task;
	task_t *tprev, *tnext;

	assert(active_tasks);
	if (!tid) return;

	error(ERR_DEBUG, "destroying task %d\n", tid);
	pthread_mutex_lock(&tasklock);
	task = tasks;
	while (task) {
		if (task->tid == tid)
			goto found;
		task = task->next;
	}
	error(ERR_ERROR, "task to be destroyed NOT found");
	pthread_mutex_unlock(&tasklock);
	return; /* not found */

found:
	active_tasks--;
	tprev = task->prev;
	tnext = task->next;
	if (tprev) tprev->next = tnext;
	if (tnext) tnext->prev = tprev;

	if (task == tasks) {
			/* head of list to be removed, so change to next */
			tasks = tnext;
	}

	/* remove all state entries belonging to this task */

	state_remove_by_tid(tid);
	pthread_mutex_unlock(&tasklock);
	free(task);
	return;
}
#endif /* _TASKS_C */
