/******************************************************************************
 *
 * test_headers.c - part of smap
 * 
 * $Id: test_headers.c,v 1.18 2008-08-20 13:02:57 hscholz Exp $
 *****************************************************************************/
#ifndef _TEST_HEADERS_C
#define _TEST_HEADERS_C

#include "config.h"

/******************************************************************************
 *
 * header order fingerprint meta database
 *
 * this list contains all different types of header order/existance
 * combinations found in live-traffic.
 *
 * FIXME:
 * This test only applies to the SIP_OPTIONS request since header order
 * tends to change on different requests.
 *
 * How to add more?
 * - extend hoe_headers[], don't forget new int (third param)
 * - add param to fingerprint database where appropriate
 *
 *****************************************************************************/

#define HEADERS_LEN		256
#define HEADERS_DESCR		256

struct hoe_header_t {
	char str[HEADERS_LEN];
	unsigned int class;
};

struct hoe_header_t hoe_headers[] = {
	/* LANCOM 1[78]xx */
	{"Via:From:To:Call-ID:CSeq:Max-Forwards:User-Agent:Server:Allow:Content-Length:", 1},
	/* Asterisk PBX */
	{"Via:From:To:Call-ID:CSeq:User-Agent:Allow:Contact:Accept:Content-Length:", 2},
	/* Cisco AS5400 HPX, IOS 12.x */
	{"Via:From:To:Date:Call-ID:Server:CSeq:Allow:Accept:Allow-Events:Content-Length:Content-Type:", 3},
	/* Cisco AS5400 HPX, IOS 12.x */
	{"Via:From:To:Date:Call-ID:Server:Content-Type:CSeq:Supported:Allow:Accept:Allow-Events:Content-Length:", 4},
	/* Thomson SpeedTouch 716 */
	{"From:To:Call-ID:CSeq:Via:User-Agent:Content-Length:", 5},
	/* Thomson SpeedTouch 780 */
	{"From:To:Call-ID:CSeq:Via:Supported:Allow:User-Agent:Accept:Content-Length:", 7},
	/* Epygi Quadro SIP User Agent */
	{"Via:To:From:CSeq:Call-ID:Accept:Allow:Supported:Server:Content-Length:", 6},
	/* Intertex IX67 Series*/
	{"Via:From:To:Call-ID:CSeq:Contact:Content-Length:Max-Forwards:User-Agent:Accept:", 7}, 
	/* Acon Virtual PBX */
	{"Via:To:From:Call-ID:CSeq:User-Agent:Content-Length:", 8 },
	/* TRACER 3.95 */
	{"Via:From:To:Call-ID:CSeq:Allow:User-Agent:Max-Forwards:Accept:Supported:Content-Length:", 9},
	/* AVM ??.04.01 */
	{"Via:From:To:Call-ID:CSeq:User-Agent:Content-Length:", 10 },
	/* Siemens IAD SLI-5790 ... running Asterisk */
	{"Via:From:To:Call-ID:CSeq:User-Agent:Allow:Max-Forwards:Contact:Accept:Content-Length:", 11},
	/* Snom 200 3.56z */
	{"Via:From:To:Call-ID:CSeq:Contact:User-Agent:Accept-Language:Accept:Allow:Allow-Events:Supported:Content-Length:", 12},
	/* linphone */
	{"Via:From:To:Call-ID:CSeq:Allow:Content-Length:", 13},
	/* twinkle 0.7 */
	{"Via:To:From:Call-ID:CSeq:Server:Content-Length:", 14},
	/* SJphone 1.60.299a */
	{"Via:Content-Length:Allow:Call-ID:CSeq:From:Server:To:", 15},
	/* Siemens SX541 1.67 */
	{"Via:From:To:Call-ID:CSeq:Content-Length:", 16},
	/* Draytek */
	{"Via:From:To:Call-ID:CSeq:content-type:Allow:Content-Length:", 17},
	/* kphone */
	{"Via:From:CSeq:Content-Type:Call-ID:To:Content-Length:User-Agent:Allow:Contact:", 18},
	/* Vlines accessVoIP */
	{"Via:From:To:Call-ID:CSeq:User-Agent:Allow:Accept:Content-Length:", 19},
	/* Samsung SMT-G3xxx */
	{"From:To:Call-ID:CSeq:Via:Supported:Content-Length:", 20},
	/* Siemens C450 IP */
	{"Via:From:To:Call-ID:CSeq:Accept:Accept-Encoding:Accept-Language:Content-Length:", 21},
	/* Sipura SPA2000-3.1.3*/
	{"To:From:Call-ID:CSeq:Via:Server:Content-Length:", 22},
	/* Snom 360 */
	{"Via:From:To:Call-ID:CSeq:Contact:User-Agent:Accept-Language:Accept:Allow:Allow-Events:Supported:Content-Length:", 23},
	/* LANCOM */
	{"Via:From:To:Call-ID:CSeq:Max-Forwards:User-Agent:Server:Allow:Supported:Content-Length:", 24},
	/* HiPath n000 */
	{"From:To:Via:Call-ID:CSeq:Content-Length:Allow:Content-Type:User-Agent:", 25},
	/* Asterisk SVN trunk r56579 */
	{"Via:From:To:Call-ID:CSeq:User-Agent:Allow:Supported:Accept:Content-Length:", 26},
	/* PA168V PA1688 series phones */
	{"Via:Call-ID:CSeq:From:To:Contact:Allow:Accept:Supported:Content-Length:", 27},
	/* Arcor Twintel D910 series */
	{"Via:From:To:Call-ID:User-Agent:CSeq:Accept:Accept-Language:Allow:Content-Length:", 28},
	/* Cisco 79x0 8.0 series */
	{"Via:From:To:Call-ID:Date:CSeq:Server:Allow:Accept:Accept-Encoding:Accept-Language:Supported:Content-Length:Content-Type:Content-Disposition:", 29},
	/* */
	{"Allow:INVITE,ACK,OPTIONS,BYE,CANCEL,REGISTER,INFO,PRACK,REFER", 30},
	/* do not remove */
	{"", 0}
};

/******************************************************************************
 *
 * test_hoe()
 *
 * compile a string identifing the order/existance of headers inside
 * a SIP message.
 * Notes:
 *  - the first line (request/response) is skipped
 *  - Via, Route and Record-Route are inserted into the list but only once
 *    per header
 *  - we stop looking for headers once we reached an empty line
 *  - headers are matches case-sensitive
 *  - headers end at the first ':' of each line
 *
 *****************************************************************************/

int test_headers(int *res, char *buf) {

	char *line, *eol, *n;
	int end, headerlen, hdrnum;
	struct hoe_header_t hdr;	

	char cmpstr[HEADERS_LEN];
	int cmplen;

	FUNCTION(__FILE__, __FUNCTION__);

	/* build string to compare against hoe_headers[] */
	bzero(&cmpstr, HEADERS_LEN);
	cmplen = 0;

	line = buf;
	/* skip first line */
	line = strstr(line, "\r\n");
	line += 2;
	while (line != NULL) {
		/* locate end of line */
		eol = strstr(line, "\r\n");

		/* break on CRLF between header and body */
		if (!strncmp(line, "\r\n", 2)) break;

		/* locate header seperator ':' or bail out */
		n = strstr(line, ":");
		if (!n) break;

		headerlen = (int) (n - line + 1); /* add : */
		//printf("HEADER: '%.*s'\n", headerlen, line);

		/* check space */
		if ((cmplen + headerlen) >= HEADERS_LEN) {
			error(ERR_NOTICE, "increase HEADERS_LEN in %s", __FILE__);
			break;
		}
	
		/* check if header is already in list
		 * used to skip Via, Route and Record-Route
		 */

		if (!(
			(strncasecmp("Via", line, 3) ||
			strncasecmp("Route", line, 5) ||
			strncasecmp("Record-Route", line, 12)) &&
			(strnstr(cmpstr, line, headerlen) != NULL))) {

			/* append to cmpstr */
			strncat(cmpstr, line, headerlen);
			cmplen += headerlen;
		}

		line = eol; line += 2; /* next line */
	}
	if (cmpstr == NULL) return 0;
	if (IS_LEARNING || DEBUG)
		error(ERR_NOTICE, "%s: cmpstr: \"%.*s\"", __FUNCTION__, cmplen, cmpstr);

	/* lookup hoe (header order/existance) class */
	hdrnum = end = 0;
	hdr = hoe_headers[hdrnum];
	while ((hdr.str[0] != 0) && !end) {
		if (!strncmp(hdr.str, cmpstr, cmplen) && (strlen(hdr.str) == cmplen)) {
			end++; /* correct header string found */
			*res = hdr.class;
		}
		hdr = hoe_headers[++hdrnum];
	}
	if (!end && IS_LEARNING)
		error(ERR_NOTICE, "%s: Please add new cmpstr", __FUNCTION__);
	return (end) ? 1 : 0;
}

#endif /* _TEST_HEADERS_C */
