/******************************************************************************

                              Copyright (c) 2009
                            Lantiq Deutschland GmbH
                     Am Campeon 3; 85579 Neubiberg, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

 ******************************************************************************/

/*
   \file drv_vmmc_audio.c
   This file contains the definitions of the functions for the AUDIO module.
*/

/* ============================= */
/* Check if feature is enabled   */
/* ============================= */
#include "drv_api.h"
#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)


/* ============================= */
/* Includes                      */
/* ============================= */
/* #include "drv_tapi.h" */
#include "drv_vmmc_api.h"
#include "drv_vmmc_audio_priv.h"
#include "drv_vmmc_audio.h"
#include "drv_vmmc_con.h"

/* select voice parameter header file according to parameter usage definition */
#if defined(IPP_USE_EVALUATION_SYSTEM)
/* set by --enable-ipp-eval-system */
#include "ipp_voice_params_eval.h"
#else
#include "ipp_voice_params_ref.h"
#endif


/* ============================= */
/* Configuration                 */
/* ============================= */
#define AUDIO_TONE_TABLE_ACCESS


/* ============================= */
/* Local Macros & Definitions    */
/* ============================= */
#define SAMPLING_UNKNOWN -1


/* ============================= */
/* Local variable definition     */
/* ============================= */

/** variable to store voice parameter sets */
static VMMC_OpModeParamSet_t voice_params;
/** variable to store ring parameter sets */
static VMMC_RingParamSet_t   ring_param_set;

static VMMC_IPPHONE_VOLUME_LEVEL   current_volume     = VMMC_IPPHONE_VOL_UNKNOWN;
#if 0
static VMMC_IPPHONE_MUTE_STATE     current_mute_state = VMMC_IPPHONE_MUTE_UNKNOWN;
static VMMC_IPPHONE_OPERATION_MODE current_op_mode    = VMMC_IPPHONE_IDLE_MODE;
#endif

/* ============================= */
/* Global variable definition    */
/* ============================= */


/* ============================= */
/* External function declaration */
/* ============================= */
extern IFX_int32_t VMMC_SIG_UTG_SetCoeff( IFX_TAPI_TONE_SIMPLE_t const *pTone,
                                          RES_UTG_COEF_t *pUtgCoef);


/* ============================= */
/* Local function declaration    */
/* ============================= */
static void vmmc_audio_AudioChannel_Init(VMMC_AUDIOCH_t  *pAudioCh);

static IFX_return_t vmmc_audio_Chan_CpPersParams(const AUDIO_CHAN_t *pSrc,
                                                 AUDIO_CHAN_t *pDst);

static IFX_return_t vmmc_audio_EqualizerDisable( VMMC_CHANNEL  *pCh,
                                             AUDIO_CHAN_t *pMsg );

static IFX_return_t vmmc_audio_SpkDisable( VMMC_CHANNEL  *pCh,
                                              AUDIO_SPK_CTRL_t *pMsg );

static IFX_return_t vmmc_audio_DisableChannel( VMMC_CHANNEL  *pCh );

static
IFX_return_t vmmc_audio_Conv_Opmode( VMMC_CHANNEL *pCh, IFX_uint32_t opmode_tapi,
                                     VMMC_IPPHONE_OPERATION_MODE* pOpmodeVmmc );
static
IFX_return_t vmmc_audio_OpmodeConfigure( VMMC_CHANNEL *pCh,
                                         VMMC_IPPHONE_OPERATION_MODE opmode,
                                         IFX_int32_t  sampling_mode );
static
IFX_return_t vmmc_audio_OperationModeDispatch( VMMC_CHANNEL  *pCh, VMMC_IPPHONE_OPERATION_MODE opmode );

static
IFX_int32_t vmmc_audio_DetermineOperationMode( VMMC_IPPHONE_OPERATION_MODE current_opmode,
                                               VMMC_IPPHONE_OPERATION_MODE req_opmode,
                                               OPMODE_SMPL req_smpl,
                                               IFX_int32_t *pHF_room_type);
static IFX_return_t vmmc_audio_MuteCfg( VMMC_CHANNEL  *pCh,
                                        AUDIO_CHAN_t *pMsg,
                                        IFX_boolean_t mute );

static IFX_void_t vmmc_audio_UpdOpModeParams_AFEInOut( VMMC_CHANNEL  *pCh );

static IFX_int32_t vmmc_audio_Get_Tag_Volume(IFX_uint16_t tag,
                                             VMMC_IPPHONE_VOLUME_LEVEL *vol);
static IFX_int32_t vmmc_audio_Get_Tag_Operation_Mode(IFX_uint16_t tag,
                                                     VMMC_IPPHONE_OPERATION_MODE *opmode);

static
IFX_int32_t vmmc_audio_StoreOpModeParameters( IFX_uint8_t *pData,
                                              IFX_uint32_t size,
                                              IFX_int32_t  row_index,
                                              IFX_int32_t  col_index );
static
IFX_int32_t vmmc_audio_SendOpModeParamMessages( VMMC_CHANNEL      *pCh,
                                                VMMC_OpModeCtrl_t *opm_params,
                                                IFX_boolean_t     voice_level_mask,
                                                VMMC_IPPHONE_MUTE_STATE req_mute_state,
                                                IFX_boolean_t     invert_order);
static
IFX_int32_t vmmc_audio_SendOpModeParamSet( VMMC_CHANNEL      *pCh,
                                           VMMC_OpModeCtrl_t *opm_params,
                                           IFX_boolean_t     voice_level_change,
                                           VMMC_IPPHONE_MUTE_STATE req_mute_state,
                                           IFX_boolean_t invert_order);

static IFX_int32_t vmmc_audio_Get_Ring_Tag_Volume( IFX_uint16_t tag,
                                                   VMMC_IPPHONE_VOLUME_LEVEL *vol);
static IFX_int32_t vmmc_audio_SendRingParamMessages( VMMC_CHANNEL      *pCh,
                                                     VMMC_RingVolCfg_t *params);
static IFX_int32_t vmmc_audio_SendRingParamSet( VMMC_CHANNEL      *pCh,
                                                VMMC_RingVolCfg_t *params );

static IFX_int32_t vmmc_audio_StoreRingParameters( IFX_uint8_t *pData,
                                                   IFX_uint32_t size,
                                                   IFX_int32_t  col_index );

#ifdef VMMC_HOST_DEBUG
static IFX_void_t print_VMMC_OpModeCtrl( VMMC_OpModeCtrl_t *params );
#endif


/**
  Adds pcm  channel to the auxiliary input of AUDIO

\param
   pCh  - pointer to VMMC channel structure
\param
   nThisCh  - data channel number

\return
   IFX_SUCCESS or IFX_ERROR
\remark
Audio channel by default would be made ON by EN= 1 since its a incall anounce ment
*******************************************************************************/
IFX_return_t VMMC_TAPI_LL_Pcm_Add_Aux(VMMC_CHANNEL *pCh, IFX_uint32_t nThisCh)
{
   VMMC_DEVICE   *pDev   = (VMMC_DEVICE*) (pCh->pParent);
   AUDIO_CHAN_t *pAudio = &(pDev->pChannel[0].pAUDIO->audio_ch);
   IFX_uint32_t  *pCmd   = ((IFX_uint32_t *)pAudio), nCount = (sizeof(AUDIO_CHAN_t)- CMD_HDR_CNT);

   TRACE(VMMC, DBG_LEVEL_LOW, ("%s\n", __FUNCTION__));

   if(pAudio->IA != ECMD_IX_EMPTY)
   {
      TRACE (VMMC, DBG_LEVEL_LOW, ("INCALL ANOUNCEMENT SLOT NOT FREE \n"));
      return IFX_ERROR;
   }
   pAudio->IA = ECMD_IX_PCM_OUT0 + nThisCh ;
   /*pAudio->TR = AUDIO_CHAN_TR_ANNOUNCE; incall anouncement ON */
   pAudio->EN = 1; /*to be safer side make sure that audio chnl is enabled since incall anouncement*/
   if(CmdWrite (pDev, pCmd, nCount) != IFX_SUCCESS)
   {
      TRACE (VMMC, DBG_LEVEL_LOW, ("drv_vmmc_audio.c: VMMC_TAPI_LL_Pcm_Add_Aux : CMD write failed\n"));
      pAudio->IA = ECMD_IX_EMPTY;
     /*
     pAudio->TR = 0;
     pAudio->EN = 0;*/
      return IFX_ERROR;
   }
   return IFX_SUCCESS;
}


/**
  Configures audio channel

\param
   pCh  - pointer to VMMC device structure

\return
   IFX_SUCCESS or IFX_ERROR
\remark

*******************************************************************************/
IFX_return_t VMMC_TAPI_LL_AUDIO_Channel_Cfg( VMMC_DEVICE *pDev )
{
   VMMC_IPPHONE_VOLUME_LEVEL   current_vol;
   VMMC_IPPHONE_OPERATION_MODE current_opmode;
   VMMC_IPPHONE_MUTE_STATE     current_mute_state;
   AUDIO_CHAN_t *pAudio;
   int ret = IFX_SUCCESS, ch;
   IFX_uint32_t  *pCmd   = ((IFX_uint32_t *)pAudio);
   IFX_uint32_t  nCount = (sizeof(AUDIO_CHAN_t)- CMD_HDR_CNT);
   AUDIO_CHAN_t tmp_audio_chan_msg;

   TRACE(VMMC, DBG_LEVEL_LOW, ("%s\n", __FUNCTION__));

   for (ch = 0; ret == IFX_SUCCESS && ch < pDev->caps.nAudioCnt; ++ch)
   {
      current_opmode     = pDev->pChannel[ch].pAUDIO->current_opmode;
      current_vol        = pDev->pChannel[ch].pAUDIO->current_vol;
      current_mute_state = pDev->pChannel[ch].pAUDIO->current_mute_state;

      TRACE(VMMC, DBG_LEVEL_LOW, \
           ("%s - current_opmode=%d\n", __FUNCTION__, current_opmode));
      TRACE(VMMC, DBG_LEVEL_LOW, \
         ("%s - ica_active    =%d\n", \
         __FUNCTION__, pDev->pChannel[ch].pAUDIO->ica_active));
      TRACE(VMMC, DBG_LEVEL_LOW, \
         ("%s - call_active   =%d\n", \
         __FUNCTION__, pDev->pChannel[ch].pAUDIO->call_active));
      TRACE(VMMC, DBG_LEVEL_LOW, \
         ("%s - ringing_active=%d\n", \
         __FUNCTION__, pDev->pChannel[ch].pAUDIO->ringing_active));

      TRACE(VMMC, DBG_LEVEL_LOW, \
           ("%s - current_opmode=%d\n", __FUNCTION__, current_opmode));
      if( (current_opmode==VMMC_IPPHONE_IDLE_MODE) &&
          (pDev->pChannel[ch].pAUDIO->ica_active == IFX_FALSE) &&
          (pDev->pChannel[ch].pAUDIO->call_active == IFX_FALSE) &&
          (pDev->pChannel[ch].pAUDIO->ringing_active == IFX_FALSE) )
      {
         TRACE(VMMC, DBG_LEVEL_LOW, \
            ("%s - use idle parameter audio channel message\n", __FUNCTION__));
         memcpy((void*)&tmp_audio_chan_msg,
                &(pDev->pChannel[ch].pAUDIO->param_set_ptr->idle_params.params.ad_chan),
                sizeof(AUDIO_CHAN_t));
      }
      else
      {
         TRACE(VMMC, DBG_LEVEL_LOW, \
            ("%s - use audio channel message previously used\n", __FUNCTION__));
         memcpy((void*)&tmp_audio_chan_msg,
                &(pDev->pChannel[ch].pAUDIO->audio_ch),
                   sizeof(AUDIO_CHAN_t));
      }

      if (VMMC_CON_Get_AUDIO_SignalStatus(pDev,ch) ||
          VMMC_CON_Get_AUDIO_AUX_SignalStatus(pDev,ch) )
      {
         pAudio = &tmp_audio_chan_msg;
         /* AUDIO CH message */
         pAudio->CHAN = ch;
         pAudio->I1 = VMMC_CON_Get_AUDIO_SignalInput(pDev, ch, 0);
         pAudio->I2 = VMMC_CON_Get_AUDIO_SignalInput(pDev, ch, 1);
         pAudio->I3 = VMMC_CON_Get_AUDIO_SignalInput(pDev, ch, 2);
         pAudio->I4 = VMMC_CON_Get_AUDIO_SignalInput(pDev, ch, 3);
         pAudio->I5 = VMMC_CON_Get_AUDIO_SignalInput(pDev, ch, 4);
         pAudio->IA = VMMC_CON_Get_AUDIO_AUX_SignalInput(pDev, ch);
         pCmd = (IFX_uint32_t*) pAudio;

         if((ret = CmdWrite(pDev, pCmd, nCount)) != IFX_SUCCESS)
         {
            TRACE(VMMC, DBG_LEVEL_HIGH,
               ("%s - Sending Audio Channel Message failed\n", __FUNCTION__));
         }
         else
         {
            VMMC_CON_Set_AUDIO_SignalStatus (pDev,ch, 0);
            VMMC_CON_Set_AUDIO_AUX_SignalStatus (pDev,ch, 0);
            memcpy( &(pDev->pChannel[0].pAUDIO->audio_ch),
                    &(tmp_audio_chan_msg),
                    sizeof(AUDIO_CHAN_t) );
         }
      }
      else
      {
         TRACE(VMMC, DBG_LEVEL_LOW,
               ("%s - VMMC_CON_Get_AUDIO_SignalStatus returned 0\n", __FUNCTION__));
      }

      if (VMMC_CON_Get_AUDIO_Loop0_SignalStatus(pDev,ch)) {
                pDev->pChannel[ch].pAUDIO->audio_misc.I1=
                    VMMC_CON_Get_AUDIO_Loop0_SignalInput(pDev, ch);
                    VMMC_CON_Set_AUDIO_Loop0_SignalStatus(pDev, ch, 0);
      }
      if (VMMC_CON_Get_AUDIO_Loop1_SignalStatus(pDev,ch)) {
                pDev->pChannel[ch].pAUDIO->audio_misc.I2=
                    VMMC_CON_Get_AUDIO_Loop1_SignalInput(pDev, ch);
                    VMMC_CON_Set_AUDIO_Loop1_SignalStatus(pDev, ch, 0);
      }
   }
   return IFX_SUCCESS;
}

/**
  Inits audio channel

\param
   pAudioCh  - pointer to audio channel structure

\return
   IFX_SUCCESS or IFX_ERROR
\remark
only required values are filled rest are filled with zero during MALLOC and cmd is not written
to DSP  now it will be written when user invokes with a ioctl
*******************************************************************************/
static void vmmc_audio_AudioChannel_Init(VMMC_AUDIOCH_t  *pAudioCh)
{
   AUDIO_CHAN_t *pAudio = &(pAudioCh->audio_ch);
   AUDIO_MISC_CTRL_t *pAudioMisc = &(pAudioCh->audio_misc);

   TRACE(VMMC,DBG_LEVEL_LOW, ("%s\n", __FUNCTION__));

   pAudioCh->param_set_ptr      = (VMMC_OpModeParamSet_t*) &voice_params;
   pAudioCh->current_vol        = VMMC_AUDIO_DEFAULT_VOLUME;
   pAudioCh->current_opmode     = VMMC_IPPHONE_IDLE_MODE;
   pAudioCh->current_mute_state = VMMC_IPPHONE_UNMUTED;
   pAudioCh->current_hf_room_type = VMMC_IPPHONE_NORMAL_ROOM;
   pAudioCh->call_active        = IFX_FALSE;
   pAudioCh->ring_param_ptr     = (VMMC_RingParamSet_t*) &ring_param_set;
   pAudioCh->current_ring_vol   = VMMC_AUDIO_DEFAULT_RING_VOLUME;
   pAudioCh->ringing_active     = IFX_FALSE;
   pAudioCh->ring_table_index   = 26;  /* default ringing tone */
   pAudioCh->ica_active     = IFX_FALSE;


   /* set AFE inputs and outputs to default values */
#if defined(IPP_USE_EVALUATION_SYSTEM)
   /* select HF AFE Pins for INCAIP2 evaluation system */
   pAudioCh->nHFMic = AUDIO_CHAN_MIP3_MIN3;
   pAudioCh->nHFOut = AUDIO_CHAN_LSP1_LSN1;
#else
   /* select HF AFE Pins for INCAIP2 reference system */
   pAudioCh->nHFMic = AUDIO_CHAN_MIP4_MIN4;
   pAudioCh->nHFOut = AUDIO_CHAN_LSP2_LSN2;
#endif
   pAudioCh->nHNMic = AUDIO_CHAN_MIP1_MIN1;
   pAudioCh->nHNOut = AUDIO_CHAN_HOP1_HON1;
   pAudioCh->nHDMic = AUDIO_CHAN_MIP2_MIN2;
   pAudioCh->nHDOut = AUDIO_CHAN_HOP2_HON2;

   /*during allocation set to 0, so only appr parms are filled*/

   /*Init audio ctrl cmd start*/

   /** Command */
   pAudio->CMD = CMD_EOP; /*6*/
   /** Mode of Subcommand */
   pAudio->MOD = 4;
   /** EDSP Command */
   pAudio->ECMD = 1;
   /** Length */
   pAudio->LENGTH = 24;

   /** Analog Gain 1 */
   pAudio->AG1 = 1;/*6;*/
   /** Analog Gain 2 */
   pAudio->AG2 =3; /*0;*/
   /** Analog Gain 3 */
   pAudio->AG3 = 3; /* 0;*/

   /** Headset Detection */
   pAudio->HSD = 1;

   /*Instruction 2*/

   /** Digital Gain 1 */
   pAudio->DG1 = 0x7fff;/*0x2000;*/
   /** Digital Gain 2 */
   pAudio->DG2 =0x7fff;/* 0x2000;*/

   /*Instruction 3*/
   /** Digital Gain 3 */
   pAudio->DG3 = 0x7fff;/*0x2000;*/

   /** Side tone Gain */
   pAudio->SG = 0x0100;/*0x2000;*/

   /*Instruction 4*/
   /** AGC1 enable */
   pAudio->AGC1 = 0;
   /** AGC resource number */
   pAudio->RAGC1 = 0;
   /** AGC2 enable */
   pAudio->AGC2 = 0;
   /** AGC2 resource number */
   pAudio->RAGC2 = 0;
   /** Equalizer 1 enable */
   pAudio->EQ1 = 0;
   /** Equalizer 1 resource number */
   pAudio->REQU1 = 0;
   /** Equalizer 2 enable */
   pAudio->EQ2 = 0;
   /** Equalizer 2 resource number */
   pAudio->REQU2 = 0;
   /** Equalizer 3 enable */
   pAudio->EQ3 = 0;
   /** Equalizer 3 resource number */
   pAudio->REQU3 = 0;
   pAudio->TGI2 = 1;
   /** AFE In- and Output Pins */
   pAudio->HFMIC = pAudioCh->nHFMic;
   pAudio->HEMIC = pAudioCh->nHDMic;
   pAudio->HSMIC = pAudioCh->nHNMic;
   pAudio->HFRCV = pAudioCh->nHFOut;
   pAudio->HERCV = pAudioCh->nHDOut;
   pAudio->HSRCV = pAudioCh->nHNOut;

   /*Init audio ctrl cmd end*/

   /* Init audio misc ctrl cmd start */
   memset( pAudioMisc, 0, sizeof(AUDIO_MISC_CTRL_t));
   /** Command */
   pAudioMisc->CMD = CMD_EOP; /* 6 */
   /** Mode of Subcommand */
   pAudioMisc->MOD = 3;
   /** EDSP Command */
   pAudioMisc->ECMD = 11;
   /** Length */
   pAudioMisc->LENGTH = 8;
   /** Feature */
   pAudioMisc->FEATURE = AUDIO_TEST_LOOP_MISC_FEAT;
   /** Disable Loop */
   pAudioMisc->EN = 0;
   /** Output Signals */
   pAudioMisc->O1 = ECMD_IX_TEST_OUT1;
   pAudioMisc->O2 = ECMD_IX_TEST_OUT2;
   pAudioMisc->O3 = ECMD_IX_TEST_OUT3;
   pAudioMisc->O4 = ECMD_IX_TEST_OUT4;

   /* Init audio misc ctrl cmd end */
   return;
}


/**
   Stop AUDIO on this channel

   \param  pCh             Pointer to the VMMC channel structure.

   \return
   - VMMC_statusOk         If successful
   - VMMC_statusCmdWr      Writing the command has failed
*/
IFX_int32_t VMMC_AUDIO_ChStop (VMMC_CHANNEL *pCh)
{
   IFX_int32_t       ret   = VMMC_statusOk;

   /* calling function should ensure valid parameters */
   VMMC_ASSERT(pCh != IFX_NULL);

   if (pCh->pAUDIO != IFX_NULL)
   {
      /* todo: check if ringing needs to be stopped separately */

      /* force these flags to false to make sure that the disable function will
         actually switch off this channel */
      pCh->pAUDIO->ica_active = IFX_FALSE;
      pCh->pAUDIO->ringing_active = IFX_FALSE;

      ret = vmmc_audio_DisableChannel( pCh );
   }

   RETURN_STATUS(ret);
}


/**
  Inits audio channel

\param
   pAudioCh  - pointer to audio channel structure

\return
   IFX_SUCCESS or IFX_ERROR
\remark
only required values are filled rest are filled with zero during MALLOC and cmd is not written
to DSP  now it will be written when user invokes with a ioctl
*******************************************************************************/
void TAPI_LL_SetDefEqu_IIR_FIRCoef(VMMC_CHANNEL *pCh)
{
   RES_EQU_IIR_COEF_t iir[3];
   RES_EQU_FIR_COEF_t fir[3];
   int i;
   VMMC_DEVICE    *pDev = pCh->pParent;
   IFX_int32_t    ret   = IFX_SUCCESS;
   IFX_uint32_t   *pCmd, nCount;
   memset(iir,0,sizeof(iir));
   memset(fir,0,sizeof(fir));
   for(i=0;i<3;i++)
   {
      /*IIR cmds*/
     /** Command */
     iir[i].CMD = 6;
     /** Mode of Subcommand */
     iir[i].MOD = 6;
     /** EDSP Command */
     iir[i].ECMD = 18;
     /** Length */
     iir[i].LENGTH = 36;
     /** Coefficient BI1 of the FIR part of the IIR filter */
      iir[i].BI1 = 0x7fff;
     /** Output gain of the IIR filter */
     iir[i].CI = 0x80;
     iir[i].CHAN = i;

     /*FIR cmds*/
        /** Command */
      fir[i].CMD = 6;
      /** Mode of Subcommand */
      fir[i].MOD = 6;
      /** EDSP Command */
      fir[i].ECMD = 19;
      /** Length */
      fir[i].LENGTH = 36;
     /** Coefficient BF1 of the FIR part of the FIR filter */
      fir[i].BF1 = 0x7fff;
     fir[i].CHAN = i;
      /** Output gain of the IIR filter */
      fir[i].CF = 0x80;

     pCmd = ((IFX_uint32_t *) (iir+i));
     nCount = sizeof (RES_EQU_IIR_COEF_t) - CMD_HDR_CNT;
     ret = CmdWrite (pDev, pCmd, nCount);

     pCmd = ((IFX_uint32_t *) (fir+i));
     nCount = sizeof (RES_EQU_FIR_COEF_t) - CMD_HDR_CNT;
     ret = CmdWrite (pDev, pCmd, nCount);
   }
}

/**
   allocate data structures of the AUDIO module per channel
\param
   pDev         pointer to the device structure
\return
   IFX_SUCCESS or IFX_ERROR in case the stucture could not be created
\remark
*/
IFX_return_t VMMC_AUDIO_Allocate_Ch_Structures(VMMC_DEVICE *pDev)
{
   IFX_uint8_t i;

   TRACE (VMMC,DBG_LEVEL_LOW, ("%s\n", __FUNCTION__));

   for (i = 0; i < pDev->caps.nAudioCnt; i++)
   {
       if (pDev->pChannel[i].pAUDIO != IFX_NULL)
      {
         VMMC_OS_Free(pDev->pChannel[i].pAUDIO);
      }
      pDev->pChannel[i].pAUDIO = VMMC_OS_Malloc (sizeof(VMMC_AUDIOCH_t));
      if (pDev->pChannel[i].pAUDIO == NULL)
      {
         return IFX_ERROR;
      }
      memset(pDev->pChannel[i].pAUDIO, 0, sizeof(VMMC_AUDIOCH_t));
      /* prepare cmd but not written to dsp, since f/w down load is done by app*/
      vmmc_audio_AudioChannel_Init(pDev->pChannel[i].pAUDIO);
      VMMC_CON_Init_AudioCh (&pDev->pChannel[i]);
   }
   return IFX_SUCCESS;
}


/**
  free data structures of the AUDIO module on the given channel
  \param  pCh           Pointer to the VMMC channel structure
  \return VMMC_statusOk
*/
IFX_int32_t VMMC_AUDIO_Free_Ch_Structures (VMMC_CHANNEL *pCh)
{
    if (pCh->pAUDIO != IFX_NULL)
   {
      VMMC_OS_Free(pCh->pAUDIO);
   }

   return VMMC_statusOk;
}


/**
   Updates all internal data stores with the requested AFE In- and Output
   Pin Configuration.
\param
   pLLChannel         pointer to channel structure.
\param
   pAFECfg            pointer to AFE pin configuration structure.

\return
   IFX_SUCCESS or IFX_ERROR in case of invalid pin configuration request.
\remark
*/
IFX_return_t VMMC_TAPI_LL_AUDIO_AFE_Cfg_Set( IFX_TAPI_LL_CH_t *pLLChannel,
                                            IFX_TAPI_AUDIO_AFE_CFG_SET_t *pAFECfg)
{
   VMMC_CHANNEL  *pCh    = (VMMC_CHANNEL *) pLLChannel;

   if( pAFECfg == IFX_NULL )
   {
      TRACE( VMMC, DBG_LEVEL_LOW, ("%s - Null Pointer passed\n", __FUNCTION__));
      return IFX_ERROR;
   }

   if( pCh->pAUDIO->nHFMic == IFX_TAPI_AUDIO_AFE_PIN_MIC1 ||
       pCh->pAUDIO->nHFMic == IFX_TAPI_AUDIO_AFE_PIN_MIC2 )
   {
      TRACE( VMMC, DBG_LEVEL_LOW, \
             ("%s - %d not allowed as Handsfree Microphone Input\n", \
              __FUNCTION__, pCh->pAUDIO->nHFMic));
      return IFX_ERROR;
   }

   if( pCh->pAUDIO->nHNMic == IFX_TAPI_AUDIO_AFE_PIN_MIC4 ||
       pCh->pAUDIO->nHDMic == IFX_TAPI_AUDIO_AFE_PIN_MIC4 )
   {
      TRACE( VMMC, DBG_LEVEL_LOW, \
             ("%s - %d not allowed as Handset/Headset Microphone Input\n", \
              __FUNCTION__, IFX_TAPI_AUDIO_AFE_PIN_MIC4));
      return IFX_ERROR;
   }

   /** Handsfree Microphone. */
   pCh->pAUDIO->nHFMic = (IFX_uint8_t) pAFECfg->nHFMic;
   /** Handsfree Output. */
   pCh->pAUDIO->nHFOut = (IFX_uint8_t) pAFECfg->nHFOut;
   /** Handset Microphone. */
   pCh->pAUDIO->nHNMic = (IFX_uint8_t) pAFECfg->nHNMic;
   /** Handset Output. */
   pCh->pAUDIO->nHNOut = (IFX_uint8_t) pAFECfg->nHNOut;
   /** Headset Microphone. */
   pCh->pAUDIO->nHDMic = (IFX_uint8_t) pAFECfg->nHDMic;
   /** Headset Output. */
   pCh->pAUDIO->nHDOut = (IFX_uint8_t) pAFECfg->nHDOut;

   /* update audio channel messages of voice and ringing parameter arrays */
   vmmc_audio_UpdOpModeParams_AFEInOut( pCh );

   return IFX_SUCCESS;
}


/**
  Sets volume level

\param
   pLLChannel  - pointer to low level channel structure
\param
   level       - requested volume level (1...8)

\return
   IFX_SUCCESS or IFX_ERROR
\remark
*******************************************************************************/
IFX_return_t VMMC_TAPI_LL_AUDIO_Volume_Set(IFX_TAPI_LL_CH_t *pLLChannel,IFX_uint32_t level)
{
   VMMC_CHANNEL  *pCh    = (VMMC_CHANNEL *) pLLChannel;
   VMMC_IPPHONE_VOLUME_LEVEL   current_vol;
   VMMC_IPPHONE_OPERATION_MODE current_opmode;
   VMMC_IPPHONE_MUTE_STATE     current_mute_state;
   VMMC_IPPHONE_VOLUME_LEVEL   req_vol;
   IFX_return_t                err=IFX_ERROR;
   VMMC_OpModeCtrl_t           tmp_opmode_params;

   TRACE(VMMC, DBG_LEVEL_LOW, \
         ("%s( vol_level = %u )\n", __FUNCTION__, level));

   /* check requested volume level and convert it to enumeration type */
   if( ((level-1)<((IFX_uint32_t)VMMC_IPPHONE_VOL1)) ||
       ((level-1)>((IFX_uint32_t)VMMC_IPPHONE_VOL8)) )
   {
      TRACE(VMMC, DBG_LEVEL_HIGH,
               ("%s - Invalid volume level %d passed \n", __FUNCTION__, level));
      return err;
   }
   req_vol = (VMMC_IPPHONE_VOLUME_LEVEL)(level-1);

   VMMC_OS_MutexGet (&pCh->chAcc);

   /* get current operation mode, volume and mute state
      for plausibilty checks and parameter set selection */
   current_opmode     = pCh->pAUDIO->current_opmode;
   current_vol        = pCh->pAUDIO->current_vol;
   current_mute_state = pCh->pAUDIO->current_mute_state;

   if( pCh->pAUDIO->call_active == IFX_FALSE )
   {
      /* in case of no active call just store the requested volume */
      pCh->pAUDIO->current_vol = req_vol;
      VMMC_OS_MutexRelease (&pCh->chAcc);
      return IFX_SUCCESS;
   }
   else
   {
         /* in case of active call send configuration messages
            for current mode and requested volume */
         TRACE(VMMC, DBG_LEVEL_LOW, \
            ("%s - send opmode_params[%d][%d] )\n", __FUNCTION__, current_opmode, req_vol));
         /* copy requested voice parameters to local variable to avoid
            that parameter array gets overwritten. */
         memcpy( (void*)&tmp_opmode_params,
                 (void*)&(pCh->pAUDIO->param_set_ptr->opmode_params[current_opmode][req_vol]),
                 sizeof(VMMC_OpModeCtrl_t) );
      /* ensure that audio channel inputs remain */
      vmmc_audio_Chan_CpPersParams( &(pCh->pAUDIO->audio_ch),
                                    &(tmp_opmode_params.params.ad_chan) );

         err=vmmc_audio_SendOpModeParamSet( pCh,
                                            &tmp_opmode_params,
                                            IFX_TRUE,
                                            current_mute_state,
                                            IFX_FALSE);
      }

   if(err != IFX_SUCCESS)
   {
      TRACE(VMMC, DBG_LEVEL_HIGH,
            ("%s - Sending Parameter Set for Volume Change failed\n", __FUNCTION__));
   }
   else
   {
      /* update current volume and store used audio channel message */
      pCh->pAUDIO->current_vol = req_vol;
      /* store audio channel message last used */
      memcpy( &(pCh->pAUDIO->audio_ch),
              &(tmp_opmode_params.params.ad_chan),
              sizeof(AUDIO_CHAN_t) );
   }
   VMMC_OS_MutexRelease (&pCh->chAcc);
   return err;

}



/**
  Returns whether the current passed operation mode is a handsfree mode.

\param
   mode  - mode to be checked

\return
   IFX_TRUE in case of handsfree mode passed or IFX_FALSE otherwise
\remark
*******************************************************************************/

IFX_boolean_t VMMC_AUDIO_HF_Mode( VMMC_IPPHONE_OPERATION_MODE mode )
{

   if( mode == VMMC_IPPHONE_HANDSFREE_MUFFLED_MODE ||
       mode == VMMC_IPPHONE_HANDSFREE_NORMAL_MODE ||
       mode == VMMC_IPPHONE_HANDSFREE_ECHOIC_MODE ||
       mode == VMMC_IPPHONE_HANDSFREE_WIDEBAND_MUFFLED_MODE ||
       mode == VMMC_IPPHONE_HANDSFREE_WIDEBAND_NORMAL_MODE ||
       mode == VMMC_IPPHONE_HANDSFREE_WIDEBAND_ECHOIC_MODE )
   {
      return IFX_TRUE;
   }
   else
      return IFX_FALSE;
}



/**
  Copies persistent parameters from source audio channel message to destination

\param
   pSrc  - pointer to source audio channel message structure
\param
   pDst  - pointer to destination audio channel message structure to be updated

\return
   IFX_SUCCESS or IFX_ERROR

\remarks
   I1 to I5, IA and TR fields are copied to destination.
*******************************************************************************/
static IFX_return_t vmmc_audio_Chan_CpPersParams(const AUDIO_CHAN_t *pSrc, AUDIO_CHAN_t *pDst)
{
   TRACE(VMMC, DBG_LEVEL_LOW, ("vmmc_audio_Chan_CpPersParams\n"));

   if(pSrc == IFX_NULL || pDst == IFX_NULL)
      return IFX_ERROR;

   pDst->I1 = pSrc->I1;
   pDst->I2 = pSrc->I2;
   pDst->I3 = pSrc->I3;
   pDst->I4 = pSrc->I4;
   pDst->I5 = pSrc->I5;
   pDst->IA = pSrc->IA;
   pDst->TR = pSrc->TR;

   TRACE(VMMC, DBG_LEVEL_LOW, ("pSrc->I1=%X\n", pSrc->I1));
   TRACE(VMMC, DBG_LEVEL_LOW, ("pSrc->I2=%X\n", pSrc->I2));
   TRACE(VMMC, DBG_LEVEL_LOW, ("pSrc->I3=%X\n", pSrc->I3));
   TRACE(VMMC, DBG_LEVEL_LOW, ("pSrc->I4=%X\n", pSrc->I4));
   TRACE(VMMC, DBG_LEVEL_LOW, ("pSrc->I5=%X\n", pSrc->I5));
   TRACE(VMMC, DBG_LEVEL_LOW, ("pSrc->IA=%X\n", pSrc->IA));
   TRACE(VMMC, DBG_LEVEL_LOW, ("pSrc->TR=%X\n", pSrc->TR));

   return IFX_SUCCESS;
}


/**
  Converts TAPI-HL format room type to VMMC room type used internally.

\param
   RoomTypeTapi  - TAPI HL room type (used by IOCTL)
\param
   pRoomTypeVmmc - pointer to vmmc room type value

\return
   IFX_SUCCESS or IFX_ERROR

\remarks

*******************************************************************************/
static
IFX_return_t VMMC_AUDIO_Conv_RoomType( IFX_uint32_t RoomTypeTapi,
                                       VMMC_IPPHONE_HF_ROOM_TYPE* pRoomTypeVmmc )
{
   IFX_return_t retval=IFX_SUCCESS;

   switch(RoomTypeTapi)
   {
      case IFX_TAPI_AUDIO_ROOM_TYPE_MUFFLED:
         *pRoomTypeVmmc = VMMC_IPPHONE_MUFFLED_ROOM;
         break;
      case IFX_TAPI_AUDIO_ROOM_TYPE_MEDIUM:
         *pRoomTypeVmmc = VMMC_IPPHONE_NORMAL_ROOM;
         break;
      case IFX_TAPI_AUDIO_ROOM_TYPE_ECHOIC:
         *pRoomTypeVmmc = VMMC_IPPHONE_ECHOIC_ROOM;
         break;
      default:
         TRACE(VMMC, DBG_LEVEL_HIGH,
            ("%s - Invalid room type %u passed !\n", __FUNCTION__, RoomTypeTapi));
         retval=IFX_ERROR;
         break;
   }

   return retval;
}


/**
  Converts TAPI-HL format operation mode to VMMC operation mode used internally.

\param
   pCh          - pointer to VMMC channel structure
\param
   opmode_tapi  - pointer to source audio channel message structure
\param
   pOpmodeVmmc  - pointer to vmmc operation mode value

\return
   IFX_SUCCESS or IFX_ERROR

\remarks
   For Handsfree mode conversion the room type is taken into account.
*******************************************************************************/
static
IFX_return_t vmmc_audio_Conv_Opmode( VMMC_CHANNEL                *pCh,
                                     IFX_uint32_t                 opmode_tapi,
                                     VMMC_IPPHONE_OPERATION_MODE* pOpmodeVmmc )
{
   IFX_return_t retval=IFX_SUCCESS;

   switch(opmode_tapi)
   {
      case IFX_TAPI_AUDIO_MODE_DISABLED:
         *pOpmodeVmmc = VMMC_IPPHONE_IDLE_MODE;
         break;
      case IFX_TAPI_AUDIO_MODE_HANDSET:
         *pOpmodeVmmc = VMMC_IPPHONE_HANDSET_MODE;
         break;
      case IFX_TAPI_AUDIO_MODE_HANDSET_OPENL:
         *pOpmodeVmmc = VMMC_IPPHONE_HANDSET_OPENLISTENING_MODE;
         break;
      case IFX_TAPI_AUDIO_MODE_HEADSET:
         *pOpmodeVmmc = VMMC_IPPHONE_HEADSET_MODE;
         break;
      case IFX_TAPI_AUDIO_MODE_HEADSET_OPENL:
         *pOpmodeVmmc = VMMC_IPPHONE_HEADSET_OPENLISTENING_MODE;
         break;
      case IFX_TAPI_AUDIO_MODE_HANDSFREE:
         switch(pCh->pAUDIO->current_hf_room_type)
         {
            case VMMC_IPPHONE_NORMAL_ROOM:
               *pOpmodeVmmc = VMMC_IPPHONE_HANDSFREE_NORMAL_MODE;
               break;
            case VMMC_IPPHONE_MUFFLED_ROOM:
               *pOpmodeVmmc = VMMC_IPPHONE_HANDSFREE_MUFFLED_MODE;
               break;
            case VMMC_IPPHONE_ECHOIC_ROOM:
               *pOpmodeVmmc = VMMC_IPPHONE_HANDSFREE_ECHOIC_MODE;
               break;
            default:
               retval=IFX_ERROR;
               break;
         }
         break;
      default:
         retval=IFX_ERROR;
         break;

   }

   return retval;

}



/**
  Switches Audio Channel to requested operation mode

\param
   pLLChannel  - pointer to low level  channel structure
\param
   mode        - phone operation mode (\ref IFX_TAPI_AUDIO_MODE_t)

\return
   IFX_SUCCESS or IFX_ERROR

\remarks
   When switching to the requested new operation mode
   the previously used volume is used.
*******************************************************************************/
IFX_return_t VMMC_TAPI_LL_AUDIO_Mode_Set(IFX_TAPI_LL_CH_t *pLLChannel,IFX_uint32_t mode)
{
   VMMC_CHANNEL  *pCh    = (VMMC_CHANNEL *) pLLChannel;
   VMMC_IPPHONE_OPERATION_MODE opmode;
   IFX_return_t                err;

   TRACE(VMMC, DBG_LEVEL_LOW, \
         ("%s( pCh->nChannel=%d,  mode=%u )\n", \
         __FUNCTION__,pCh->nChannel, mode));

   /* convert passed opmode value to internally used enumeration value */
   if( (err=vmmc_audio_Conv_Opmode( pCh, mode, &opmode ))
          == IFX_SUCCESS)
   {
      VMMC_OS_MutexGet (&pCh->chAcc);


      /* dispatch and configure new operation mode */
      err=vmmc_audio_OperationModeDispatch(pCh, opmode);
      VMMC_OS_MutexRelease (&pCh->chAcc);
   }

   return err;


}


/**
  Sets room type for activated handsfree audio mode

\param
   pLLChannel  - pointer to low level channel channel structure
\param
   type  - room type (\ref IFX_TAPI_AUDIO_ROOM_TYPE_t)

\return
   IFX_SUCCESS or IFX_ERROR
\remarks
   Handsfree Operating mode has to be active when a room type change
   is requested by means of this function.
*******************************************************************************/
IFX_return_t VMMC_AUDIO_Room_Set( VMMC_CHANNEL                *pCh,
                                  VMMC_IPPHONE_HF_ROOM_TYPE   vmmc_room_type)
{
   VMMC_IPPHONE_VOLUME_LEVEL   current_vol;
   VMMC_IPPHONE_OPERATION_MODE current_opmode;
   VMMC_IPPHONE_MUTE_STATE     current_mute_state;
   VMMC_IPPHONE_OPERATION_MODE req_opmode;
   IFX_return_t                err=IFX_ERROR;

   TRACE(VMMC, DBG_LEVEL_LOW,
         ("%s( room_type = %d )\n", __FUNCTION__, vmmc_room_type));

   /* get current operation mode, volume and mute state
      for plausibilty checks and voice parameter set selection */
   current_opmode     = pCh->pAUDIO->current_opmode;
   current_vol        = pCh->pAUDIO->current_vol;
   current_mute_state = pCh->pAUDIO->current_mute_state;

   if( VMMC_AUDIO_HF_Mode(current_opmode) )
   {

      /* compare current and requested operation mode in order to determine
        if firmware messages have to be sent to establish the requested handsfree
        room type */
      switch(vmmc_room_type)
      {
         case VMMC_IPPHONE_MUFFLED_ROOM:
            if( (current_opmode == VMMC_IPPHONE_HANDSFREE_MUFFLED_MODE) ||
                (current_opmode == VMMC_IPPHONE_HANDSFREE_WIDEBAND_MUFFLED_MODE) )
            {
               TRACE(VMMC, DBG_LEVEL_HIGH,
                     ("%s - Already operating in muffled mode\n", __FUNCTION__));
               return IFX_SUCCESS;
            }
            if( (current_opmode == VMMC_IPPHONE_HANDSFREE_NORMAL_MODE) ||
                (current_opmode == VMMC_IPPHONE_HANDSFREE_ECHOIC_MODE) )
               req_opmode = VMMC_IPPHONE_HANDSFREE_MUFFLED_MODE;
            else
               if( (current_opmode == VMMC_IPPHONE_HANDSFREE_WIDEBAND_NORMAL_MODE) ||
                   (current_opmode == VMMC_IPPHONE_HANDSFREE_WIDEBAND_ECHOIC_MODE))
                  req_opmode = VMMC_IPPHONE_HANDSFREE_WIDEBAND_MUFFLED_MODE;
               else
                  return IFX_ERROR;
            break;

         case VMMC_IPPHONE_NORMAL_ROOM:
            if( (current_opmode == VMMC_IPPHONE_HANDSFREE_NORMAL_MODE) ||
                (current_opmode == VMMC_IPPHONE_HANDSFREE_WIDEBAND_NORMAL_MODE) )
            {
               TRACE(VMMC, DBG_LEVEL_HIGH,
                     ("%s - Already operating in normal mode\n", __FUNCTION__));
               return IFX_SUCCESS;
            }
            if( (current_opmode == VMMC_IPPHONE_HANDSFREE_MUFFLED_MODE) ||
                (current_opmode == VMMC_IPPHONE_HANDSFREE_ECHOIC_MODE) )
               req_opmode = VMMC_IPPHONE_HANDSFREE_NORMAL_MODE;
            else
               if( (current_opmode == VMMC_IPPHONE_HANDSFREE_WIDEBAND_MUFFLED_MODE) ||
                   (current_opmode == VMMC_IPPHONE_HANDSFREE_WIDEBAND_ECHOIC_MODE))
                  req_opmode = VMMC_IPPHONE_HANDSFREE_WIDEBAND_NORMAL_MODE;
               else
                  return IFX_ERROR;
            break;

         case VMMC_IPPHONE_ECHOIC_ROOM:
            if( (current_opmode == VMMC_IPPHONE_HANDSFREE_ECHOIC_MODE) ||
                (current_opmode == VMMC_IPPHONE_HANDSFREE_WIDEBAND_ECHOIC_MODE) )
            {
               TRACE(VMMC, DBG_LEVEL_HIGH,
                     ("%s - Already operating in echoic mode\n", __FUNCTION__));
               return IFX_SUCCESS;
            }
            if( (current_opmode == VMMC_IPPHONE_HANDSFREE_MUFFLED_MODE) ||
                (current_opmode == VMMC_IPPHONE_HANDSFREE_NORMAL_MODE) )
               req_opmode = VMMC_IPPHONE_HANDSFREE_ECHOIC_MODE;
            else
               if( (current_opmode == VMMC_IPPHONE_HANDSFREE_WIDEBAND_MUFFLED_MODE) ||
                   (current_opmode == VMMC_IPPHONE_HANDSFREE_WIDEBAND_NORMAL_MODE))
                  req_opmode = VMMC_IPPHONE_HANDSFREE_WIDEBAND_ECHOIC_MODE;
               else
                  return IFX_ERROR;
            break;

         default:
            TRACE(VMMC, DBG_LEVEL_HIGH,
                  ("%s - Invalid room type %u passed\n", __FUNCTION__, vmmc_room_type));
            return IFX_ERROR;
      }

      /* configure new HF room type configuration */
      err = vmmc_audio_OpmodeConfigure( pCh, req_opmode, SAMPLING_UNKNOWN );
   }
   else
   {
      TRACE(VMMC, DBG_LEVEL_LOW, ("%s - not in HF opmode\n", __FUNCTION__));
      err=IFX_SUCCESS;
   }

   /* update hf room type */
   pCh->pAUDIO->current_hf_room_type = vmmc_room_type;

   return err;
}


/**
  Sets room type for activated handsfree audio mode

\param
   pLLChannel  - pointer to low level channel channel structure
\param
   type  - room type (\ref IFX_TAPI_AUDIO_ROOM_TYPE_t)

\return
   IFX_SUCCESS or IFX_ERROR
\remarks
   Handsfree Operating mode has to be active when a room type change
   is requested by means of this function.
*******************************************************************************/
IFX_return_t VMMC_TAPI_LL_AUDIO_Room_Set( IFX_TAPI_LL_CH_t *pLLChannel,
                                         IFX_uint32_t     type)
{
   VMMC_CHANNEL               *pCh = (VMMC_CHANNEL *) pLLChannel;
   IFX_return_t               err = IFX_ERROR;
   VMMC_IPPHONE_HF_ROOM_TYPE  vmmc_room_type;

   TRACE(VMMC, DBG_LEVEL_LOW,
         ("%s( room_type = %u )\n", __FUNCTION__, type));

   if( type!=IFX_TAPI_AUDIO_ROOM_TYPE_MUFFLED &&
       type!=IFX_TAPI_AUDIO_ROOM_TYPE_MEDIUM &&
       type!=IFX_TAPI_AUDIO_ROOM_TYPE_ECHOIC )
   {
      TRACE(VMMC, DBG_LEVEL_HIGH,
            ("%s - Invalid room type %u passed\n", __FUNCTION__, type));
      return err;
   }


   if( (err=VMMC_AUDIO_Conv_RoomType( type, &vmmc_room_type )) == IFX_SUCCESS )
   {
      /* call HF room type configuration protected by semaphore */
      VMMC_OS_MutexGet (&pCh->chAcc);
      err = VMMC_AUDIO_Room_Set( pCh, vmmc_room_type );
      VMMC_OS_MutexRelease (&pCh->chAcc);
   }
   else
   {
       TRACE(VMMC, DBG_LEVEL_HIGH,
               ("%s - room type conversion failed\n", __FUNCTION__));
   }
   return err;
}


/**
  Enables or disables MUTE in audio channel.

\param
   pLLChannel  - pointer to low level channel structure
\param
   oprn        - operation to enable (IFX_ENABLE) or disable (IFX_DISABLE) muting.

\return
   IFX_SUCCESS or IFX_ERROR
\remark
*******************************************************************************/
IFX_return_t VMMC_TAPI_LL_AUDIO_Mute_Set(IFX_TAPI_LL_CH_t *pLLChannel,IFX_uint32_t oprn)
{
   VMMC_CHANNEL  *pCh    = (VMMC_CHANNEL *) pLLChannel;
   VMMC_IPPHONE_VOLUME_LEVEL   current_vol;
   VMMC_IPPHONE_OPERATION_MODE current_opmode;
   VMMC_IPPHONE_MUTE_STATE     current_mute_state;
   VMMC_IPPHONE_MUTE_STATE     req_mute_state;
   AUDIO_CHAN_t                tmp_audio_chan_msg;
   IFX_return_t                err=IFX_ERROR;

   TRACE(VMMC, DBG_LEVEL_LOW,
         ("%s( mute = %u )\n", __FUNCTION__, oprn));

   /* get current operation mode, volume and mute state
      for plausibilty checks and voice parameter set selection */
   current_opmode     = pCh->pAUDIO->current_opmode;
   current_vol        = pCh->pAUDIO->current_vol;
   current_mute_state = pCh->pAUDIO->current_mute_state;

   if( current_opmode == VMMC_IPPHONE_IDLE_MODE)
   {
      TRACE(VMMC, DBG_LEVEL_HIGH,
            ("%s - Phone is Idle, (Un)Muting not possible\n", __FUNCTION__));
      return err;
   }

   switch(oprn)
   {
      case IFX_DISABLE:
      {
         if( current_mute_state == VMMC_IPPHONE_UNMUTED )
         {
            TRACE(VMMC, DBG_LEVEL_HIGH,
                  ("%s - Phone is already unmuted\n", __FUNCTION__));
            return IFX_SUCCESS;
         }
         req_mute_state = VMMC_IPPHONE_UNMUTED;
         break;
      }
      case IFX_ENABLE:
      {
         if( current_mute_state == VMMC_IPPHONE_MUTED )
         {
            TRACE(VMMC, DBG_LEVEL_HIGH,
                  ("%s - Phone is already muted\n", __FUNCTION__));
            return IFX_SUCCESS;
         }
         req_mute_state = VMMC_IPPHONE_MUTED;
         break;
      }
      default:
      {
         return IFX_ERROR;
      }
   }


   /* get last used audio channel message and set mute bit according to request
      and update mute state in channel structure. */
   memcpy( &tmp_audio_chan_msg, &(pCh->pAUDIO->audio_ch), sizeof(AUDIO_CHAN_t) );
   err = vmmc_audio_MuteCfg( pCh, &tmp_audio_chan_msg,
                                (IFX_boolean_t) req_mute_state );

   if(err==IFX_SUCCESS)
   {
      pCh->pAUDIO->current_mute_state = req_mute_state;
   }
   else
      pCh->pAUDIO->current_mute_state = VMMC_IPPHONE_MUTE_UNKNOWN;

   return err;

}


/**
   configures UTG coefficients based on the index, enables AUDIO channel if not active and
   then enables UTG in AUDIO channel

\param
   pLLChannel  - pointer to low level  channel structure
\param
   index  - index to tone table

\return
   IFX_SUCCESS or IFX_ERROR
\remark
   This is enabled only when audio channel is not active. If its enabled means some
   call is going on and it's not nice to play a ringing tone in speaker for a
   second call, user has to use tone (call waiting)play which will be  heard
   in audio channel.
*******************************************************************************/
IFX_return_t VMMC_TAPI_LL_AUDIO_Ring_Start(IFX_TAPI_LL_CH_t *pLLChannel,IFX_uint32_t index)
{
   VMMC_CHANNEL      *pCh    = (VMMC_CHANNEL *) pLLChannel;
   IFX_int32_t       ret     = IFX_SUCCESS;
   IFX_uint32_t      req_vol = pCh->pAUDIO->current_ring_vol;
   VMMC_RingVolCfg_t tmp_ring_par;
   AUDIO_CHAN_t                tmp_audio_chan_msg;
   AUDIO_CHAN_t                *pAudio = &(pCh->pAUDIO->audio_ch);
   VMMC_IPPHONE_OPERATION_MODE current_opmode;
#ifdef AUDIO_TONE_TABLE_ACCESS
   COMPLEX_TONE               *pToneCoeff;
   IFX_uint32_t               ring_index = index;
#endif
   TRACE(VMMC, DBG_LEVEL_LOW, \
         ("%s - index=%u\n", __FUNCTION__, index));
#if 0
   /* check if a tone is already being played */
   if( pCh->pAUDIO->ringing_active == IFX_TRUE )
   {
      TRACE(VMMC, DBG_LEVEL_HIGH, \
         ("%s - Ringing already activated\n", __FUNCTION__));
      return ret;
   }
#endif


#ifdef AUDIO_TONE_TABLE_ACCESS
   /*
    * check passed table index and use default if parameter is out of range
    */
   if (ring_index >= TAPI_MAX_TONE_CODE || ring_index < 0)
   {
      TRACE(VMMC, DBG_LEVEL_HIGH,
            ("DRV_ERROR: audio tone index out of range -> use default\n"));
      ring_index = pCh->pAUDIO->ring_table_index;
/*      return IFX_ERROR;  */
   }
   pToneCoeff = &(pCh->pParent->pTapiDev->pToneTbl[ring_index]);
#endif
   current_opmode     = pCh->pAUDIO->current_opmode;
   /* copy requested ring parameters to local variable to avoid
      that parameter array gets overwritten. */
   memcpy( &tmp_ring_par,
           &(pCh->pAUDIO->ring_param_ptr->ring_params[req_vol]),
           sizeof(VMMC_RingVolCfg_t));

   if( pCh->pAUDIO->call_active == IFX_TRUE )
   {
      /* disable audio channel equalizers before sending Ringing parameters */
      memcpy( &tmp_audio_chan_msg, pAudio, sizeof(AUDIO_CHAN_t));
      vmmc_audio_EqualizerDisable( pCh, &tmp_audio_chan_msg );


      if( VMMC_AUDIO_HF_Mode(current_opmode) == IFX_TRUE)
      {
         /* for handsfree modes adapt operation mode and TG Inputs of
            ringing parameter sets */
         tmp_ring_par.ad_chan.OP = AUDIO_CHAN_HANDS_FREE;
         tmp_ring_par.ad_chan.TGI3 = 0;
         tmp_ring_par.ad_chan.TGI4 = 1;
      }
   }

   /* ensure that current inputs of audio channel persist */
   vmmc_audio_Chan_CpPersParams( pAudio, &(tmp_ring_par.ad_chan));

#ifdef AUDIO_TONE_TABLE_ACCESS
   if( pToneCoeff->type == TAPI_TONE_TYPE_SIMPLE )
   {
      if(VMMC_SIG_UTG_SetCoeff( &(pToneCoeff->tone.simple), &(tmp_ring_par.utg_coeff))
         != IFX_SUCCESS)
      {
         TRACE(VMMC, DBG_LEVEL_HIGH, ("Failed to set UTG Coefficients\n"));
         return IFX_ERROR;
      }
   }
   else
   {
      TRACE(VMMC, DBG_LEVEL_HIGH,
         ("Currently only simple tones supported for audio channel ringing\n"));
      return IFX_ERROR;
   }
   pCh->pAUDIO->ring_table_index = index;
#endif

   TRACE(VMMC, DBG_LEVEL_LOW, \
      ("%s - Send Ringing Parameters for volume %u\n", __FUNCTION__, req_vol));
   /* send ring parameter messages for requested volume */
   ret=vmmc_audio_SendRingParamSet( pCh, &(tmp_ring_par));
   if(ret == IFX_SUCCESS)
   {
      pCh->pAUDIO->current_ring_vol = req_vol;
      TRACE(VMMC, DBG_LEVEL_LOW, \
         ("%s - ringing_active = IFX_TRUE\n", __FUNCTION__));
      pCh->pAUDIO->ringing_active   = IFX_TRUE;
      /* store audio channel message used last */
      memcpy( &(pCh->pAUDIO->audio_ch),
              &(tmp_ring_par.ad_chan),
              sizeof(AUDIO_CHAN_t) );
   }
   else
   {
      TRACE(VMMC, DBG_LEVEL_HIGH, \
         ("%s - vmmc_audio_SendRingParamSet() failed\n", __FUNCTION__));
   }

   return ret;
}



/**
   Disables the UTG which is directly connected to AUDIO and also disables the audio channel

\param
   pLLChannel  - pointer to low level channle structure
\param
   index  - index to the tone table which has to be played.

\return
   IFX_SUCCESS or IFX_ERROR
\remark
   This param is not needed since its not being used.
*******************************************************************************/
IFX_return_t VMMC_TAPI_LL_AUDIO_Ring_Stop(IFX_TAPI_LL_CH_t *pLLChannel,IFX_uint32_t index)
{
   VMMC_CHANNEL      *pCh    = (VMMC_CHANNEL *) pLLChannel;
   VMMC_DEVICE       *pDev   = (VMMC_DEVICE*) (pCh->pParent);
   IFX_int32_t       ret     = IFX_SUCCESS;
   IFX_uint32_t      *pCmd, nCount;
   AUDIO_UTG_CTRL_t  *putg;
   AUDIO_UTG_CTRL_t  tmp_adutg_cmd;
   AUDIO_CHAN_t      *pAudio;
   AUDIO_CHAN_t      tmp_adchan_cmd;

   IFX_uint32_t req_vol = pCh->pAUDIO->current_ring_vol;

   TRACE(VMMC, DBG_LEVEL_LOW, ("%s\n", __FUNCTION__));
#if 0
   if( pCh->pAUDIO->ringing_active == IFX_FALSE )
   {
      TRACE(VMMC, DBG_LEVEL_LOW, \
         ("%s - Ringing already deactivated\n", __FUNCTION__));
      return ret;
   }
#endif
   /* disable UTG in audio channel*/
   memcpy( &tmp_adutg_cmd,
           &(pCh->pAUDIO->ring_param_ptr->ring_params[req_vol].ad_utg),
           sizeof(tmp_adutg_cmd));
   putg = &tmp_adutg_cmd;
   putg->EN = 0;

   nCount = sizeof(AUDIO_UTG_CTRL_t)- CMD_HDR_CNT;
   pCmd =((IFX_uint32_t *)putg);
   if((ret = CmdWrite(pDev, pCmd, nCount)) != IFX_SUCCESS)
   {
      TRACE(VMMC, DBG_LEVEL_HIGH, \
         ("%s - Failed to disable utg in audio channel\n", __FUNCTION__));
      return IFX_ERROR;
   }

   if( (pCh->pAUDIO->call_active == IFX_FALSE) &&
       (pCh->pAUDIO->ica_active == IFX_FALSE) )
   {
      /* disable audio channel */
      memcpy( &(tmp_adchan_cmd),
              &(pCh->pAUDIO->audio_ch),
              sizeof(AUDIO_CHAN_t) );
      pAudio = &tmp_adchan_cmd;
      nCount = sizeof(AUDIO_CHAN_t)- CMD_HDR_CNT;
      pAudio->EN = 0;
      pCmd =((IFX_uint32_t *)pAudio);
      if((ret = CmdWrite(pDev, pCmd, nCount)) != IFX_SUCCESS)
      {
         TRACE(VMMC, DBG_LEVEL_HIGH, \
            ("%s - Failed to disable audio channel\n", __FUNCTION__));
          return IFX_ERROR;
      }

      /* store audio channel message used last and update ringing state */
      memcpy( &(pCh->pAUDIO->audio_ch),
              &(tmp_adchan_cmd),
              sizeof(AUDIO_CHAN_t) );

      TRACE(VMMC, DBG_LEVEL_LOW, \
         ("%s - ringing_active = IFX_FALSE\n", __FUNCTION__));
      pCh->pAUDIO->ringing_active = IFX_FALSE;
   }

   return IFX_SUCCESS;

}

/**
   Sets the ring volume starts ringing by sending assigned fw messages

\param
   pLLChannel  - pointer to low level  channel structure
\param
   volume      - ring volume level (1...8)

\return
   IFX_SUCCESS or IFX_ERROR
\remark

*******************************************************************************/
IFX_return_t VMMC_TAPI_LL_AUDIO_Ring_Volume_Set( IFX_TAPI_LL_CH_t *pLLChannel,
                                                IFX_uint32_t     level)
{
   IFX_int32_t               ret = IFX_ERROR;
   VMMC_IPPHONE_VOLUME_LEVEL current_ring_vol;
   VMMC_IPPHONE_VOLUME_LEVEL req_vol;
   VMMC_CHANNEL              *pCh = (VMMC_CHANNEL *) pLLChannel;
   VMMC_DEVICE               *pDev   = (VMMC_DEVICE*) (pCh->pParent);
   IFX_uint32_t              *pCmd, nCount;
   VMMC_RingVolCfg_t         tmp_ring_par;
   COMPLEX_TONE              *pToneCoeff;

   TRACE(VMMC, DBG_LEVEL_LOW, ("%s - level = %u\n", __FUNCTION__, level));

   /* check requested volume level and convert it to enumeration type */
   if( ((level-1)<((IFX_uint32_t)VMMC_IPPHONE_VOL1)) ||
       ((level-1)>((IFX_uint32_t)VMMC_IPPHONE_VOL8)) )
   {
      TRACE(VMMC, DBG_LEVEL_HIGH, \
            ("%s - Invalid volume level %u passed \n", __FUNCTION__, level));
      return ret;
   }

   req_vol          = (VMMC_IPPHONE_VOLUME_LEVEL)(level-1);
   current_ring_vol = pCh->pAUDIO->current_ring_vol;

   if( pCh->pAUDIO->ringing_active == IFX_FALSE)
   {
      /* in case of inactive ringing store new ring volume to channel structure */
      pCh->pAUDIO->current_ring_vol = req_vol;
      ret = IFX_SUCCESS;
   }
   else
   {
      /*
       * in case of active ringing disable audio channel utg
       * before sending volume dependent ring parameter set
       */
      memcpy( &tmp_ring_par,
              &(pCh->pAUDIO->ring_param_ptr->ring_params[req_vol]),
              sizeof(VMMC_RingVolCfg_t));

         nCount = sizeof(AUDIO_UTG_CTRL_t)- CMD_HDR_CNT;
         tmp_ring_par.ad_utg.EN = 0;
         /* set msg pointer to audio channel utg message and send it to VCPU */
         pCmd =((IFX_uint32_t *)&tmp_ring_par.ad_utg);
         if((ret = CmdWrite (pDev, pCmd, nCount)) == IFX_SUCCESS)
         {
            tmp_ring_par.ad_utg.EN = 1;

         /*
          * take coefficients from tone table and copy them to
          *  utg coefficients message
          */
         pToneCoeff = &(pCh->pParent->pTapiDev->pToneTbl[pCh->pAUDIO->ring_table_index]);
         if( pToneCoeff->type == TAPI_TONE_TYPE_SIMPLE )
         {
            if(VMMC_SIG_UTG_SetCoeff( &(pToneCoeff->tone.simple), &(tmp_ring_par.utg_coeff))
               != IFX_SUCCESS)
            {
               TRACE(VMMC, DBG_LEVEL_HIGH,
                  ("Failed to set UTG Coefficients\n"));
               return IFX_ERROR;
            }
         }
         else
         {
            TRACE(VMMC, DBG_LEVEL_HIGH,
               ("Only simple tones supported for audio channel ringing\n"));
            return IFX_ERROR;
         }

            /* write ringing configuration messages assigned to new volume
               and update ringing status */
         ret=vmmc_audio_SendRingParamSet( pCh, &(tmp_ring_par));
            if(ret == IFX_SUCCESS)
            {
               pCh->pAUDIO->current_ring_vol = req_vol;
            }
         }
         else
         {
            TRACE(VMMC, DBG_LEVEL_HIGH,
               ("%s - Disabling utg failed\n", __FUNCTION__));
         }
      }
   return ret;
}


/**
   Configures the mute Bit of the passed audio channel message

\param
   pCh    - pointer to low level channel structure
\param
   pMsg   - pointer to audio channel message to be modified
\param
   mute   - control bit to enable (IFX_TRUE) or disable (IFX_FALSE) muting

\return
   IFX_SUCCESS or IFX_ERROR
\remarks

*******************************************************************************/
IFX_return_t vmmc_audio_MuteCfg( VMMC_CHANNEL   *pCh,
                                 AUDIO_CHAN_t   *pMsg,
                                 IFX_boolean_t  mute )
{
   VMMC_DEVICE  *pDev   = (VMMC_DEVICE*) (pCh->pParent);
   IFX_uint32_t *pCmd;
   IFX_uint32_t nCount = (sizeof(AUDIO_CHAN_t)- CMD_HDR_CNT);
   IFX_return_t ret = IFX_ERROR;

   if( pMsg != IFX_NULL )
   {
      pMsg->MUTE = mute;
      pCmd = (IFX_uint32_t *) pMsg;
      ret = CmdWrite(pDev, pCmd, nCount);
      if( ret == IFX_SUCCESS )
         memcpy( &(pCh->pAUDIO->audio_ch), pCmd, sizeof(AUDIO_CHAN_t) );
   }

   return ret;
}

IFX_return_t VMMC_AUDIO_MuteCfg( VMMC_CHANNEL  *pCh, IFX_boolean_t mute )
{
   VMMC_DEVICE  *pDev = pCh->pParent;
   AUDIO_CHAN_t  nMsg = pCh->pAUDIO->audio_ch;
   IFX_uint32_t nCount = (sizeof(AUDIO_CHAN_t)- CMD_HDR_CNT);
   IFX_return_t ret;

   nMsg.MUTE = mute;
   ret = CmdWrite(pDev, (IFX_uint32_t *)&nMsg, nCount);

   if( ret == IFX_SUCCESS )
     memcpy( &pCh->pAUDIO->audio_ch, &nMsg, sizeof(AUDIO_CHAN_t) );

   return ret;
}



/**
   Disables the equalizers in the passed audio channel message

\param
   pCh    - pointer to low level channel structure
\param
   pMsg   - pointer to audio channel message to be modified

\return
   IFX_SUCCESS or IFX_ERROR
\remarks
   Audio Channel Message is stored in channel structure after usage.
*******************************************************************************/
static IFX_return_t vmmc_audio_EqualizerDisable( VMMC_CHANNEL  *pCh,
                                                    AUDIO_CHAN_t *pMsg )
{
   VMMC_DEVICE  *pDev   = (VMMC_DEVICE*) (pCh->pParent);
   IFX_uint32_t *pCmd;
   IFX_uint32_t nCount = (sizeof(AUDIO_CHAN_t)- CMD_HDR_CNT);
   IFX_return_t ret = IFX_ERROR;

   if( pMsg != IFX_NULL )
   {
      pMsg->EQ1 = 0;
      pMsg->EQ2 = 0;
      pMsg->EQ3 = 0;
      pMsg->EQ4 = 0;
      pCmd = (IFX_uint32_t *) pMsg;
      ret = CmdWrite(pDev, pCmd, nCount);
      if( ret == IFX_SUCCESS )
         memcpy( &(pCh->pAUDIO->audio_ch), pCmd, sizeof(AUDIO_CHAN_t) );
   }

   return ret;
}


/**
   Disables the speakerphone in the passed audio channel spk message

\param
   pCh    - pointer to low level channel structure
\param
   pMsg   - pointer to audio channel speakerphone message to be modified

\return
   IFX_SUCCESS or IFX_ERROR
\remark
   Assumes that the passed message has been prepared for sending in advance.
*******************************************************************************/
static IFX_return_t vmmc_audio_SpkDisable( VMMC_CHANNEL  *pCh,
                                              AUDIO_SPK_CTRL_t *pMsg )
{
   VMMC_DEVICE  *pDev   = (VMMC_DEVICE*) (pCh->pParent);
   IFX_uint32_t *pCmd;
   IFX_uint32_t nCount = (sizeof(AUDIO_SPK_CTRL_t)- CMD_HDR_CNT);
   IFX_return_t ret = IFX_ERROR;

   if( pMsg != IFX_NULL )
   {
      pMsg->EN=0;
      pCmd = (IFX_uint32_t *) pMsg;
      ret = CmdWrite(pDev, pCmd, nCount);
   }
   return ret;
}


/**
   Enables or disables audio channel test mode

\param
   pLLChannel  - pointer to low leve channel structure
\param
   param  - pointer to mode structure

\return
   IFX_SUCCESS or IFX_ERROR
*******************************************************************************/
IFX_return_t VMMC_TAPI_LL_AUDIO_Test_Mode_Set (IFX_TAPI_LL_CH_t *pLLChannel,IFX_uint32_t param)
{
   VMMC_CHANNEL      *pCh  = (VMMC_CHANNEL *) pLLChannel;
   VMMC_DEVICE       *pDev = (VMMC_DEVICE *) (pCh->pParent);
   IFX_TAPI_AUDIO_TEST_MODE_t *mode = (IFX_TAPI_AUDIO_TEST_MODE_t *) param;
   IFX_int32_t       ret   = IFX_SUCCESS;
   AUDIO_MISC_CTRL_t tmp_audio_misc_msg;
   AUDIO_MISC_CTRL_t *pAudioMisc = &tmp_audio_misc_msg;
   IFX_uint32_t      *pCmd;
   IFX_uint32_t      nCount = (sizeof(AUDIO_MISC_CTRL_t)- CMD_HDR_CNT);

   /* Get current message from storage */
   memcpy( &tmp_audio_misc_msg, &(pCh->pAUDIO->audio_misc), sizeof(AUDIO_MISC_CTRL_t));

   /* Disable test mode if no port is used */
   if ((mode->nTestPort0 == IFX_TAPI_AUDIO_TEST_DISABLED) &&
       (mode->nTestPort1 == IFX_TAPI_AUDIO_TEST_DISABLED)) {
        pAudioMisc->EN=0;
   }
   else {
        pAudioMisc->EN=1;
   }

   /* Check for disabled ports */
   if (mode->nTestPort0 == IFX_TAPI_AUDIO_TEST_DISABLED) {
        pAudioMisc->PD0=0;
        pAudioMisc->ADC0I=0;
        pAudioMisc->I1=0;
        pAudioMisc->O3=0;
        pAudioMisc->O2=0;
        pAudioMisc->O4=0;
   }
   if (mode->nTestPort1 == IFX_TAPI_AUDIO_TEST_DISABLED) {
        pAudioMisc->PD1=0;
        pAudioMisc->ADC1I=0;
        pAudioMisc->I2=0;
        pAudioMisc->O1=0;
        pAudioMisc->O2=0;
        pAudioMisc->O4=0;
   }

   /* Setup ports for loop mode if needed */
   if (mode->nTestPort0 == IFX_TAPI_AUDIO_TEST_LOOP) {
        pAudioMisc->PD0=1;
        pAudioMisc->ADC0I=1;
        pAudioMisc->O2=0;
        pAudioMisc->O3=ECMD_IX_TEST_OUT3;
        pAudioMisc->O4=0;
   }
   if (mode->nTestPort1 == IFX_TAPI_AUDIO_TEST_LOOP) {
        pAudioMisc->PD1=1;
        pAudioMisc->ADC1I=1;
        pAudioMisc->O1=ECMD_IX_TEST_OUT1;
        pAudioMisc->O2=0;
        pAudioMisc->O4=0;
   }

   /* Setup ports for disgnostic mode if needed */
   if (mode->nTestPort0 == IFX_TAPI_AUDIO_TEST_DIAGNOSTIC) {
        pAudioMisc->PD0=0;
        pAudioMisc->ADC0I=0;
   }
   if (mode->nTestPort1 == IFX_TAPI_AUDIO_TEST_DIAGNOSTIC) {
        pAudioMisc->PD1=0;
        pAudioMisc->ADC1I=0;
   }

   pCmd = ((IFX_uint32_t *)&tmp_audio_misc_msg);
#if 0
   printk("%s %s %d\n",__FILE__,__FUNCTION__,__LINE__);
   printk("%s %s %d 0x%08x\n",__FILE__,__FUNCTION__,__LINE__,*(u32*)(pCmd+0));
   printk("%s %s %d 0x%08x\n",__FILE__,__FUNCTION__,__LINE__,*(u32*)(pCmd+1));
   printk("%s %s %d 0x%08x\n",__FILE__,__FUNCTION__,__LINE__,*(u32*)(pCmd+2));
#endif
   ret = CmdWrite(pDev, pCmd, nCount);

   return ret;
}

/**
   Disables Audio Channel

\param
   pCh    - pointer to low level channel structure

\return
   IFX_SUCCESS or IFX_ERROR
\remark
   Disabling is done by first disabling active audio algorithms. In the second
   step the audio channel is switched off.
*******************************************************************************/
static
IFX_return_t vmmc_audio_DisableChannel( VMMC_CHANNEL  *pCh )
{
   VMMC_DEVICE        *pDev   = (VMMC_DEVICE*) (pCh->pParent);
   VMMC_OpModeCtrl_t  tmp_opmode_params;
   VMMC_OpModeCtrl_t  *opmode_param_ptr;
   AUDIO_CHAN_t       *pAudio = &(pCh->pAUDIO->audio_ch);
   IFX_uint32_t       *pCmd;
   IFX_uint32_t       nCount = (sizeof(AUDIO_CHAN_t)- CMD_HDR_CNT);
   VMMC_IPPHONE_MUTE_STATE     current_mute_state;

   if(pCh==IFX_NULL)
      return IFX_ERROR;

   current_mute_state = pCh->pAUDIO->current_mute_state;

   if( !pCh->pAUDIO->ica_active && !pCh->pAUDIO->ringing_active)
   {
      /* in case of IDLE mode request and no auxiliary channel in use
         set parameter pointer to idle parameter set */
      TRACE(VMMC, DBG_LEVEL_LOW, \
         ("%s - select idle parameter set\n", __FUNCTION__));

      /* disable active audio algorithms by sending idle parameter set
         leave channel enabled and take ISR bit from last audio channel
         message used */
      memcpy( (void*)&tmp_opmode_params,
              (void*)&(pCh->pAUDIO->param_set_ptr->idle_params),
              sizeof(VMMC_OpModeCtrl_t) );

      tmp_opmode_params.params.ad_chan.EN  = 1;
      tmp_opmode_params.params.ad_chan.ISR = pCh->pAUDIO->audio_ch.ISR;
      vmmc_audio_Chan_CpPersParams( pAudio, &(tmp_opmode_params.params.ad_chan));

      opmode_param_ptr = &tmp_opmode_params;
      TRACE(VMMC, DBG_LEVEL_LOW, \
         ("%s - disable active audio algorithms\n", __FUNCTION__));
      if(vmmc_audio_SendOpModeParamSet( pCh, opmode_param_ptr,
                                        IFX_FALSE,
                                        current_mute_state,
                                        IFX_FALSE) != IFX_SUCCESS)
      {
         TRACE(VMMC, DBG_LEVEL_HIGH,
            ("%s - Disabling audio algorithms failed\n", __FUNCTION__));
         return IFX_ERROR;
      }

      /* as 2nd step disable audio channel */
      tmp_opmode_params.params.ad_chan.EN  = 0;
      pCmd  = (IFX_uint32_t *)&(tmp_opmode_params.params.ad_chan);
      TRACE(VMMC, DBG_LEVEL_LOW, \
         ("%s - disable audio channel\n", __FUNCTION__));
      if(CmdWrite(pDev, pCmd, nCount) != IFX_SUCCESS)
      {
         TRACE(VMMC, DBG_LEVEL_HIGH, \
            ("%s - Failed to disable audio channel\n", __FUNCTION__));
         return IFX_ERROR;
      }
      /* store audio channel message for later usage and update states */
      memcpy( &(pCh->pAUDIO->audio_ch),
              &(opmode_param_ptr->params.ad_chan),
              sizeof(AUDIO_CHAN_t) );

      pCh->pAUDIO->current_opmode = VMMC_IPPHONE_IDLE_MODE;
      pCh->pAUDIO->call_active    = IFX_FALSE;
   }
   else
      TRACE(VMMC, DBG_LEVEL_LOW, \
         ("%s - ICA/Ringing active, do not disable Audio Channel\n", __FUNCTION__));

   return IFX_SUCCESS;

}


/**
   Returns if the audio is active on this channel.

\param
   pCh - pointer to VMMC channel structure

\return
   IFX_TRUE or IFX_FALSE
*******************************************************************************/
IFX_boolean_t VMMC_AUDIO_IsChannelActive( VMMC_CHANNEL  *pCh )
{
   if (pCh->pAUDIO != IFX_NULL)
   {
      return pCh->pAUDIO->audio_ch.EN ? IFX_TRUE : IFX_FALSE;
   }

   return IFX_FALSE;
}


/**
   Enables or disables Incall voice announcement feature on audio channel.

\param
   pLLChannel - pointer to low level channel structure
\param
   oprn       - Enable Off Hook Voice Announcement, enable In Call Announcement
                or disable these features (\ref IFX_TAPI_AUDIO_ICA_t).

\return
   IFX_SUCCESS or IFX_ERROR
\remarks
*******************************************************************************/
IFX_return_t VMMC_TAPI_LL_AUDIO_ICA (IFX_TAPI_LL_CH_t *pLLChannel,IFX_uint32_t oprn)
{
   VMMC_CHANNEL  *pCh    = (VMMC_CHANNEL *) pLLChannel;
   VMMC_DEVICE   *pDev   = (VMMC_DEVICE*) (pCh->pParent);
   IFX_int32_t   ret     = IFX_SUCCESS;
   AUDIO_CHAN_t                *pAudio = &(pCh->pAUDIO->audio_ch);
   AUDIO_CHAN_t                tmp_audio_chan_msg;
   AUDIO_SPK_CTRL_t            tmp_audio_spk_msg;
   IFX_uint32_t                *pCmd;
   IFX_uint32_t                nCount = (sizeof(AUDIO_CHAN_t)- CMD_HDR_CNT);
   VMMC_IPPHONE_OPERATION_MODE current_opmode = pCh->pAUDIO->current_opmode;
   VMMC_IPPHONE_VOLUME_LEVEL   current_vol = pCh->pAUDIO->current_vol;
   VMMC_IPPHONE_OPERATION_MODE new_mode = current_opmode;
   VMMC_OpModeCtrl_t           tmp_opmode_params;
   VMMC_IPPHONE_MUTE_STATE     current_mute_state = pCh->pAUDIO->current_mute_state;
   /* Flag to control sending of complete parameter set (OHVA) or
      sending of audio channel message only (ICA) */
   IFX_boolean_t               send_audio_msg_only = IFX_FALSE;


   TRACE(VMMC, DBG_LEVEL_LOW,
      ("%s - oprn                       = %u\n", __FUNCTION__, oprn));
   TRACE(VMMC, DBG_LEVEL_LOW,
      ("%s - current_opmode             = %d\n", __FUNCTION__, current_opmode));
   TRACE(VMMC, DBG_LEVEL_LOW,
      ("%s - pCh->pAUDIO->call_active   = %d\n",
      __FUNCTION__, pCh->pAUDIO->call_active));
   TRACE(VMMC, DBG_LEVEL_LOW,
      ("%s - pCh->pAUDIO->ring_active   = %d\n", __FUNCTION__, pCh->pAUDIO->ringing_active));
   TRACE(VMMC, DBG_LEVEL_LOW,
      ("%s - pCh->pAUDIO->ica_active    = %d\n", __FUNCTION__, pCh->pAUDIO->ica_active));

   if( (oprn != IFX_TAPI_AUDIO_ICA_DISABLED) &&
       (oprn != IFX_TAPI_AUDIO_ICA_OUT) &&
       (oprn != IFX_TAPI_AUDIO_ICA_INOUT) )
   {
      TRACE(VMMC, DBG_LEVEL_HIGH, \
         ("%s - Invalid mode (%u) passed for ICA \n", __FUNCTION__, oprn));
      return IFX_ERROR;
   }

   /* plausibility checks: enable request although state is enabled or
                           disable request although state is disabled */
   if( (pDev->pChannel[0].pAUDIO->ica_active == IFX_TRUE) &&
       ((oprn == IFX_TAPI_AUDIO_ICA_OUT) || (oprn == IFX_TAPI_AUDIO_ICA_INOUT)) )
   {
      TRACE(VMMC, DBG_LEVEL_LOW, ("INCALL ANOUNCEMENT already enabled \n"));
      return IFX_SUCCESS;
   }
   if( (pDev->pChannel[0].pAUDIO->ica_active == IFX_FALSE) &&
       (oprn == IFX_TAPI_AUDIO_ICA_DISABLED) )
   {
      TRACE(VMMC, DBG_LEVEL_LOW, ("INCALL ANOUNCEMENT already disabled \n"));
      return IFX_SUCCESS;
   }

   if(oprn == IFX_TAPI_AUDIO_ICA_DISABLED)
   {
      TRACE(VMMC, DBG_LEVEL_LOW,
            ("%s - IFX_TAPI_AUDIO_ICA_DISABLED\n", __FUNCTION__));

      /* disable ICVA and return to operation mode that was valid
         before ICVA was started.  Switch off audio channel
         completely if ICVA is running standalone -> IDLE Mode  */
      switch(current_opmode)
      {
         case VMMC_IPPHONE_IDLE_MODE:
            if( !pCh->pAUDIO->ringing_active )
               return IFX_ERROR;
            break;

         case VMMC_IPPHONE_HANDSET_MODE:
         case VMMC_IPPHONE_HEADSET_MODE:
         case VMMC_IPPHONE_HANDSET_WIDEBAND_MODE:
         case VMMC_IPPHONE_HEADSET_WIDEBAND_MODE:
            new_mode = current_opmode;
            break;

         case VMMC_IPPHONE_ICVA_HANDSET_MODE:
            if( pCh->pAUDIO->call_active )
               /* switch back to handset mode */
               new_mode = VMMC_IPPHONE_HANDSET_MODE;
            break;
         case VMMC_IPPHONE_ICVA_HEADSET_MODE:
            if( pCh->pAUDIO->call_active )
               new_mode = VMMC_IPPHONE_HEADSET_MODE;
            break;
         case VMMC_IPPHONE_ICVA_HANDSET_WIDEBAND_MODE:
            if( pCh->pAUDIO->call_active )
               new_mode = VMMC_IPPHONE_HANDSET_WIDEBAND_MODE;
            break;
         case VMMC_IPPHONE_ICVA_HEADSET_WIDEBAND_MODE:
            if( pCh->pAUDIO->call_active )
               new_mode = VMMC_IPPHONE_HEADSET_WIDEBAND_MODE;
            break;

         case VMMC_IPPHONE_HANDSFREE_MUFFLED_MODE:
         case VMMC_IPPHONE_HANDSFREE_NORMAL_MODE:
         case VMMC_IPPHONE_HANDSFREE_ECHOIC_MODE:
         case VMMC_IPPHONE_HANDSFREE_WIDEBAND_MUFFLED_MODE:
         case VMMC_IPPHONE_HANDSFREE_WIDEBAND_NORMAL_MODE:
         case VMMC_IPPHONE_HANDSFREE_WIDEBAND_ECHOIC_MODE:
            VMMC_CON_Interchange_AudioDataChannels( pDev );
            new_mode = current_opmode;
            pCh->pAUDIO->ica_active     = IFX_FALSE;
            return IFX_SUCCESS;
         default:
             TRACE(VMMC, DBG_LEVEL_HIGH,
                   ("%s - Unable to disable ICA/OHVA in mode %d\n",
                   __FUNCTION__, pDev->pChannel[0].pAUDIO->current_opmode));
             return IFX_ERROR;
       }

       if( !pCh->pAUDIO->call_active )
       {
          /* retrieve idle parameter set and take audio channel speakerphone
             message from this parameter set for disabling */
          new_mode = VMMC_IPPHONE_IDLE_MODE;
          memcpy( &tmp_opmode_params,
                  &(pDev->pChannel[0].pAUDIO->param_set_ptr->idle_params),
                  sizeof(VMMC_OpModeCtrl_t));
          memcpy( &tmp_audio_spk_msg,
                  &(pDev->pChannel[0].pAUDIO->param_set_ptr->idle_params.params.ad_spk),
                  sizeof(AUDIO_SPK_CTRL_t));
       }
       else
       {
          /* retrieve parameter set for calculated mode and current volume
             and take audio channel speakerphone message for disabling */
          memcpy( &tmp_opmode_params,
                  &(pDev->pChannel[0].pAUDIO->param_set_ptr-> \
                  opmode_params[new_mode][current_vol]),
                  sizeof(VMMC_OpModeCtrl_t));
          memcpy( &tmp_audio_spk_msg,
                  &(pDev->pChannel[0].pAUDIO->param_set_ptr-> \
                  opmode_params[new_mode][current_vol].params.ad_spk),
                  sizeof(AUDIO_SPK_CTRL_t));

       }

       /* ensure that Inputs from last audio channel message persist */
       if(vmmc_audio_Chan_CpPersParams( pAudio,
                                        &(tmp_opmode_params.params.ad_chan)) != IFX_SUCCESS)
       {
          TRACE(VMMC, DBG_LEVEL_HIGH,
                ("%s - failed to copy audio channel parameters(2)\n", __FUNCTION__));
          return IFX_ERROR;
       }

       /* get last used audio channel message and disable equalizers and
          speakerphone before sending parameter set of requested mode */
       memcpy( &tmp_audio_chan_msg, pAudio, sizeof(AUDIO_CHAN_t));
       vmmc_audio_EqualizerDisable(pCh, &tmp_audio_chan_msg);
       vmmc_audio_SpkDisable( pCh, &tmp_audio_spk_msg );

       memcpy( &tmp_audio_chan_msg,
               &(tmp_opmode_params.params.ad_chan),
               sizeof(AUDIO_CHAN_t));
       send_audio_msg_only = IFX_FALSE;

   }
   else  /* oprn != IFX_TAPI_AUDIO_ICA_DISABLED */
   {
      /*
       * Enable ICA or OHVA dep. requested mode
       */
      if( oprn == IFX_TAPI_AUDIO_ICA_INOUT )
      {
         /*
          * Enable OHVA
          */
         TRACE(VMMC, DBG_LEVEL_LOW,
               ("%s - IFX_TAPI_AUDIO_ICA_INOUT\n", __FUNCTION__));
         if( pCh->pAUDIO->ringing_active )
         {
            /* If ringing is active stop it during ICA/OHVA call.
               Leave status active for later re-enable when ICA/OHVA
               is disabled again. */
            VMMC_TAPI_LL_AUDIO_Ring_Stop( pCh, 0);
            pCh->pAUDIO->ringing_active = IFX_TRUE;
         }

         /* OHVA enable request */
         switch(current_opmode)
         {
            case VMMC_IPPHONE_IDLE_MODE:
               /* request for ICVA standalone mode
                  -> we configure handset+ICVA parameters nevertheless because
                  the normal audio channel should not be connected to a data channel.*/
               new_mode = VMMC_IPPHONE_ICVA_HANDSET_MODE;
               break;
            case VMMC_IPPHONE_HANDSET_MODE:
               new_mode = VMMC_IPPHONE_ICVA_HANDSET_MODE;
               break;
            case VMMC_IPPHONE_HEADSET_MODE:
               new_mode = VMMC_IPPHONE_ICVA_HEADSET_MODE;
               break;
            case VMMC_IPPHONE_HANDSET_WIDEBAND_MODE:
               new_mode = VMMC_IPPHONE_ICVA_HANDSET_WIDEBAND_MODE;
               break;
            case VMMC_IPPHONE_HEADSET_WIDEBAND_MODE:
               new_mode = VMMC_IPPHONE_ICVA_HEADSET_WIDEBAND_MODE;
               break;

            case VMMC_IPPHONE_HANDSFREE_MUFFLED_MODE:
            case VMMC_IPPHONE_HANDSFREE_NORMAL_MODE:
            case VMMC_IPPHONE_HANDSFREE_ECHOIC_MODE:
            case VMMC_IPPHONE_HANDSFREE_WIDEBAND_MUFFLED_MODE:
            case VMMC_IPPHONE_HANDSFREE_WIDEBAND_NORMAL_MODE:
            case VMMC_IPPHONE_HANDSFREE_WIDEBAND_ECHOIC_MODE:
               /* in case of OHVA during handsfree call the handsfree call
                  is switched off by interchanging the data channels connected
                  to the audio module. */
               VMMC_CON_Interchange_AudioDataChannels( pDev );
               new_mode = current_opmode;
               pCh->pAUDIO->ica_active     = IFX_TRUE;
               return IFX_SUCCESS;
               break;

            default:
                TRACE(VMMC, DBG_LEVEL_HIGH,
                      ("%s - Unable to start ICA/OHVA in mode %d\n", __FUNCTION__, pDev->pChannel[0].pAUDIO->current_opmode));
                return IFX_ERROR;
         }


         /* disable audio channel equalizers before sending OHVA parameters */
         memcpy( &tmp_audio_chan_msg, pAudio, sizeof(AUDIO_CHAN_t));
         vmmc_audio_EqualizerDisable( pCh, &tmp_audio_chan_msg );

         /* disable speakerphone based on speakerphone message from
            parameter set to be used */
         memcpy( &tmp_audio_spk_msg,
                 &(pDev->pChannel[0].pAUDIO->param_set_ptr-> \
                  opmode_params[new_mode][current_vol].params.ad_spk),
                 sizeof(AUDIO_SPK_CTRL_t));

         vmmc_audio_SpkDisable( pCh, &tmp_audio_spk_msg );

         /* retrieve parameter set for calculated OHVA mode and current volume */
         memcpy( &tmp_opmode_params,
                 &(pDev->pChannel[0].pAUDIO->param_set_ptr-> \
                 opmode_params[new_mode][current_vol]),
                 sizeof(VMMC_OpModeCtrl_t));

         if(vmmc_audio_Chan_CpPersParams( pAudio,
                                          &(tmp_opmode_params.params.ad_chan)) != IFX_SUCCESS)
         {
             TRACE(VMMC, DBG_LEVEL_HIGH,
                   ("%s - failed to copy audio channel parameters(2)\n", __FUNCTION__));
             return IFX_ERROR;
         }
         memcpy( &tmp_audio_chan_msg,
                 &(tmp_opmode_params.params.ad_chan),
                 sizeof(AUDIO_CHAN_t));
          send_audio_msg_only = IFX_FALSE;
      }
      else /* oprn == IFX_TAPI_AUDIO_ICA_OUT -> ICA */
      {
         /*
          * Enable ICA
          */
         TRACE(VMMC, DBG_LEVEL_LOW,
               ("%s - IFX_TAPI_AUDIO_ICA_OUT\n", __FUNCTION__));
         if( pCh->pAUDIO->ringing_active )
         {
            /* enable ICA when ringing is active */
            memcpy( &tmp_audio_chan_msg,
                    &(pDev->pChannel[0].pAUDIO->param_set_ptr-> \
                    opmode_params[VMMC_IPPHONE_HANDSET_MODE][VMMC_AUDIO_DEFAULT_ICA_VOLUME].params.ad_chan),
                    sizeof(AUDIO_CHAN_t));
            /* ensure that current TR bit persists and ringing continues */
            if(vmmc_audio_Chan_CpPersParams( pAudio,
                                             &tmp_audio_chan_msg) != IFX_SUCCESS)
            {
               TRACE(VMMC, DBG_LEVEL_HIGH,
                     ("%s - failed to copy audio channel parameters(2)\n", __FUNCTION__));
               return IFX_ERROR;
            }

         }
         else
         if( pCh->pAUDIO->call_active )
         {
            /* enable ICA when call is active */
            memcpy( &tmp_audio_chan_msg, pAudio, sizeof(AUDIO_CHAN_t));
         }
         else
         {
            /* enable ICA when neither call nor ringing is active */
            memcpy( &tmp_audio_chan_msg,
                    &(pDev->pChannel[0].pAUDIO->param_set_ptr-> \
                    opmode_params[VMMC_IPPHONE_HANDSET_MODE][VMMC_AUDIO_DEFAULT_ICA_VOLUME].params.ad_chan),
                    sizeof(AUDIO_CHAN_t));
         }

         tmp_audio_chan_msg.TR = AUDIO_CHAN_TR_ANNOUNCE;
         tmp_audio_chan_msg.EN = 1;
         send_audio_msg_only = IFX_TRUE;
      }
   }

   if( send_audio_msg_only == IFX_FALSE )
   {
      if((ret=vmmc_audio_SendOpModeParamSet( pCh, &tmp_opmode_params,
                                             IFX_FALSE,
                                             current_mute_state,
                                             IFX_FALSE)) != IFX_SUCCESS)
      {
         TRACE(VMMC, DBG_LEVEL_HIGH,
               ("%s - Sending Opmode parameters failed\n", __FUNCTION__));
      }
      /* set message pointer for correct storage of last used audio channel message */
      pCmd = (IFX_uint32_t *)&(tmp_opmode_params.params.ad_chan);

   }
   else
   {
      pCmd = ((IFX_uint32_t *)&tmp_audio_chan_msg);
      ret = CmdWrite(pDev, pCmd, nCount);
   }

   if(ret == IFX_SUCCESS)
   {
      /* in case of successful configuration update ICA state and
         store audio channel message used last */
      if( oprn==IFX_TAPI_AUDIO_ICA_DISABLED )
      {
         TRACE(VMMC, DBG_LEVEL_LOW, \
            ("%s - set ica_active to IFX_FALSE\n", __FUNCTION__));
         pCh->pAUDIO->current_opmode = new_mode;
         pCh->pAUDIO->ica_active     = IFX_FALSE;

         if( pCh->pAUDIO->ringing_active )
         {
            /* ICA disable request during active ringing */
            TRACE(VMMC, DBG_LEVEL_LOW, \
                  ("%s - ringing active\n", __FUNCTION__));
            VMMC_TAPI_LL_AUDIO_Ring_Start( pCh, 0 );
         }
      }
      else
      {
         TRACE(VMMC, DBG_LEVEL_LOW, \
            ("%s - set ica_active to IFX_TRUE\n", __FUNCTION__));
         pCh->pAUDIO->current_opmode = new_mode;
         pCh->pAUDIO->ica_active     = IFX_TRUE;
      }
      /* store used audio channel message */
      memcpy( &(pCh->pAUDIO->audio_ch), pCmd, sizeof(AUDIO_CHAN_t) );
   }
   else
      TRACE(VMMC, DBG_LEVEL_HIGH, \
         ("%s - sending audio channel command for ICA failed\n", __FUNCTION__));
   return ret;
}


/**
   Configures requested operation mode.

\param
   pCh    - pointer to VMMC_CHANNEL
\param
   opmode - requested operation mode
\return
   IFX_SUCCESS or IFX_ERROR
\remarks
  This function configures all 'active' operation modes.
  For simplification reasons the idle mode is not configured by this function
  but by vmmc_audio_DisableChannel.
*******************************************************************************/
static
IFX_return_t vmmc_audio_OpmodeConfigure( VMMC_CHANNEL *pCh,
                                         VMMC_IPPHONE_OPERATION_MODE opmode,
                                         IFX_int32_t  sampling_mode)
{
   VMMC_IPPHONE_VOLUME_LEVEL   req_vol;
   VMMC_IPPHONE_MUTE_STATE     req_mute_state;
   VMMC_IPPHONE_OPERATION_MODE req_opmode, current_opmode;
   IFX_int32_t                 row_index=IFX_ERROR;
   IFX_int32_t                 col_index=IFX_ERROR;
   IFX_int32_t                 room_type;
   static IFX_int32_t          current_sampling_mode = NB_8_KHZ;
   VMMC_OpModeCtrl_t           tmp_idle_params;
   VMMC_OpModeCtrl_t           tmp_opmode_params;
   IFX_boolean_t               req_sampling_16KHZ = IFX_FALSE;
   IFX_return_t                retval=IFX_ERROR;
   AUDIO_CHAN_t                *pAudio = &(pCh->pAUDIO->audio_ch);
   VMMC_OpModeCtrl_t           *opmode_param_ptr;
#if 0
   AUDIO_CHAN_t                tmp_audio_chan_msg;
#endif

   current_opmode = pCh->pAUDIO->current_opmode;
   req_vol        = pCh->pAUDIO->current_vol;
   req_mute_state = pCh->pAUDIO->current_mute_state;
   req_opmode     = opmode;


   if( (sampling_mode != WB_16_KHZ) && (sampling_mode != NB_8_KHZ) )
   {
      sampling_mode = current_sampling_mode;
   }
   else
   {
      current_sampling_mode = sampling_mode;
   }

   if(sampling_mode == WB_16_KHZ)
      req_sampling_16KHZ = IFX_TRUE;
   else
      req_sampling_16KHZ = IFX_FALSE;

   TRACE(VMMC, DBG_LEVEL_LOW, \
      ("%s - req_sampling_16KHZ            =%d\n", __FUNCTION__,req_sampling_16KHZ));
   TRACE(VMMC, DBG_LEVEL_LOW, \
      ("%s - current_opmode     =%d\n", __FUNCTION__, current_opmode));
   TRACE(VMMC, DBG_LEVEL_LOW, \
      ("%s - req_opmode     =%d\n", __FUNCTION__, req_opmode));
   TRACE(VMMC, DBG_LEVEL_LOW, \
      ("%s - req_vol        =%d\n", __FUNCTION__,req_vol));
   TRACE(VMMC, DBG_LEVEL_LOW, \
      ("%s - req_mute_state =%d\n", __FUNCTION__,req_mute_state));


   if( pCh->pAUDIO->ica_active )
   {
      /* during active incall announcement block
         activation of loudspeaker dependent operation modes */
      if( req_opmode == VMMC_IPPHONE_HANDSFREE_MUFFLED_MODE ||
          req_opmode == VMMC_IPPHONE_HANDSFREE_NORMAL_MODE ||
          req_opmode == VMMC_IPPHONE_HANDSFREE_ECHOIC_MODE ||
          req_opmode == VMMC_IPPHONE_HANDSFREE_WIDEBAND_MUFFLED_MODE ||
          req_opmode == VMMC_IPPHONE_HANDSFREE_WIDEBAND_NORMAL_MODE ||
          req_opmode == VMMC_IPPHONE_HANDSFREE_WIDEBAND_ECHOIC_MODE ||
          req_opmode == VMMC_IPPHONE_HANDSET_OPENLISTENING_MODE ||
          req_opmode == VMMC_IPPHONE_HANDSET_WIDEBAND_OPENLISTENING_MODE ||
          req_opmode == VMMC_IPPHONE_HEADSET_OPENLISTENING_MODE ||
          req_opmode == VMMC_IPPHONE_HEADSET_WIDEBAND_OPENLISTENING_MODE )
      {
         TRACE(VMMC, DBG_LEVEL_HIGH, \
            ("%s - Can not enable mode=%u when ICA is active\n", \
               __FUNCTION__, req_opmode));
         return IFX_ERROR;
      }
   }

   /*
    * determine new operation mode depending on requested mode, current mode
    * and requested sampling rate. The returned operation mode is the row index
    * to the 2-dim. voice parameter array.
    */
   row_index = vmmc_audio_DetermineOperationMode( current_opmode,
                                                  req_opmode,
                                                  req_sampling_16KHZ,
                                                  (IFX_int32_t*) &room_type);
   if( row_index == IFX_ERROR )
   {
      TRACE(VMMC, DBG_LEVEL_HIGH, \
            ("%s - Can not determine operation mode\n", __FUNCTION__));
      return IFX_ERROR;
   }

   /* set array access column */
   col_index=(IFX_int32_t)req_vol;

   if(current_opmode == (VMMC_IPPHONE_OPERATION_MODE) row_index )
   {
      TRACE(VMMC, DBG_LEVEL_LOW,
               ("%s - Requested Mode already activated\n", __FUNCTION__));
      return IFX_SUCCESS;
   }

   if(pAudio->EN)
   {
      /* in case of active audio channel send idle parameter set
         before setting new operation mode */
      TRACE(VMMC, DBG_LEVEL_LOW,
               ("%s - Sending Idle parameters\n", __FUNCTION__));
      memcpy( &tmp_idle_params,
              &pCh->pAUDIO->param_set_ptr->idle_params,
              sizeof(VMMC_OpModeCtrl_t));

      /* ensure that current inputs of audio channel persist */
      vmmc_audio_Chan_CpPersParams( pAudio, &(tmp_idle_params.params.ad_chan));

      if( pAudio->ISR == 1 )
      {
         tmp_idle_params.params.ad_chan.ISR = 1;
         tmp_idle_params.params.ad_spk.ISR  = 1;
      }
      else
      {
         tmp_idle_params.params.ad_chan.ISR = 0;
         tmp_idle_params.params.ad_spk.ISR  = 0;
      }

      /* if audio channel is enabled disable it
         before enabling the requested operation mode */
      if((retval=vmmc_audio_SendOpModeParamSet( pCh,
                                                &tmp_idle_params,
                                                IFX_FALSE,
                                                req_mute_state,
                                                IFX_TRUE)) != IFX_SUCCESS)
      {
         TRACE(VMMC, DBG_LEVEL_HIGH,
               ("%s - Sending Idle parameters failed\n", __FUNCTION__));
         return retval;
      }
      /* store last audio channel message used
         as base for later  configurations */
      memcpy( &(pCh->pAUDIO->audio_ch),
              &(tmp_idle_params.params.ad_chan),
              sizeof(AUDIO_CHAN_t) );

   }


   memcpy( (void*)&tmp_opmode_params,
           (void*)&(pCh->pAUDIO->param_set_ptr->opmode_params[row_index][col_index]),
           sizeof(VMMC_OpModeCtrl_t) );
   /* ensure that current inputs of audio channel persist */
   vmmc_audio_Chan_CpPersParams( pAudio, &(tmp_opmode_params.params.ad_chan));
   opmode_param_ptr = &tmp_opmode_params;

   TRACE(VMMC, DBG_LEVEL_LOW, \
               ("%s - Sending Opmode parameters for \n", __FUNCTION__));
   TRACE(VMMC, DBG_LEVEL_LOW, \
      ("row_index=%d, col_index=%d\n",  row_index, col_index));
   /* send firmware messages assigned to requested operation mode and volume */
   if((retval=vmmc_audio_SendOpModeParamSet( pCh, opmode_param_ptr,
                                             IFX_FALSE,
                                             req_mute_state,
                                             IFX_FALSE)) != IFX_SUCCESS)
   {
         TRACE(VMMC, DBG_LEVEL_HIGH,
               ("%s - Sending Opmode parameters failed\n", __FUNCTION__));
   }
   else
   {
      /* store last audio channel message used
         as base for later  configurations */
      memcpy( &(pCh->pAUDIO->audio_ch),
              &(opmode_param_ptr->params.ad_chan),
              sizeof(AUDIO_CHAN_t) );
      /* update call, opmode and volume states */
      pCh->pAUDIO->current_opmode = (VMMC_IPPHONE_OPERATION_MODE) row_index;
      pCh->pAUDIO->current_vol    = (VMMC_IPPHONE_VOLUME_LEVEL) col_index;
      pCh->pAUDIO->call_active    = IFX_TRUE;
   }
   return retval;
}

/**
   Dispatches requested operation mode.

\param
   pCh    - pointer to VMMC_CHANNEL
\param
   opmode - requested operation mode
\return
   IFX_SUCCESS or IFX_ERROR
\remarks
  The function dispatches the request for idle operation mode and
  all further (active) operation modes.
*******************************************************************************/
static
IFX_return_t vmmc_audio_OperationModeDispatch( VMMC_CHANNEL  *pCh,
                                               VMMC_IPPHONE_OPERATION_MODE opmode )
{
   VMMC_IPPHONE_OPERATION_MODE current_opmode;
   IFX_return_t                retval = IFX_ERROR;

   /*
    * get current state information that is necessary for
    * operation mode configuration
    */
   current_opmode     = pCh->pAUDIO->current_opmode;

   TRACE(VMMC, DBG_LEVEL_LOW, \
      ("%s - current_opmode     =%d\n", __FUNCTION__,current_opmode));

   if( opmode == current_opmode )
   {
      /* requested opmode is already active */
      retval = IFX_SUCCESS;
   }
   else
   {
      if( opmode == VMMC_IPPHONE_IDLE_MODE )
      {
         /* configure audio channel for idle mode */
         retval = vmmc_audio_DisableChannel( pCh );
      }
      else
      {
         /* configure audio channel for active operation mode */
         retval = vmmc_audio_OpmodeConfigure( pCh, opmode, SAMPLING_UNKNOWN );
      }
   }
   return retval;

}


/**
   Determines new operation mode.

\param
   current_opmode - current operation mode
\param
   req_opmode     - requested operation mode
\param
   req_smpl       - requested sampling rate
\param
   pHF_room_type  - pointer to room type setting in case of active handsfree mode
\return
   Derived new operation mode or IFX_ERROR in case of invalid parameter
   combinations passed.
\remarks
  The new operation mode is determined based on the requested operation mode,
  the current operation mode and the requested sampling rate (8/16kHz).
*******************************************************************************/
IFX_int32_t vmmc_audio_DetermineOperationMode( VMMC_IPPHONE_OPERATION_MODE current_opmode,
                                               VMMC_IPPHONE_OPERATION_MODE req_opmode,
                                               OPMODE_SMPL req_smpl,
                                               IFX_int32_t *pHF_room_type)
{
   IFX_int32_t   opmode = IFX_ERROR;

   TRACE(VMMC, DBG_LEVEL_LOW, \
       ("%s - current_opmode=%d\n", __FUNCTION__, current_opmode));
   TRACE(VMMC, DBG_LEVEL_LOW, \
       ("%s - req_opmode    =%d\n", __FUNCTION__, req_opmode));
   TRACE(VMMC, DBG_LEVEL_LOW, \
       ("%s - req_smpl      =%d\n", __FUNCTION__, req_smpl));

   if( req_smpl == WB_16_KHZ )
   {
      /*
       * determine wideband operation mode
       * in case of active ICVA mode switch to ICVA wideband mode
       */
      switch(req_opmode)
      {
         case VMMC_IPPHONE_HANDSET_MODE:
         case VMMC_IPPHONE_HANDSET_WIDEBAND_MODE:
            if( current_opmode == VMMC_IPPHONE_ICVA_HEADSET_MODE  ||
                current_opmode == VMMC_IPPHONE_ICVA_HEADSET_WIDEBAND_MODE )
            {
               /* switch from headset ICVA (NB and WB) to handset WB ICVA */
               opmode=(IFX_int32_t) VMMC_IPPHONE_ICVA_HANDSET_WIDEBAND_MODE;
            }
            else
               opmode=(IFX_int32_t) VMMC_IPPHONE_HANDSET_WIDEBAND_MODE;
            break;
         case VMMC_IPPHONE_HEADSET_MODE:
         case VMMC_IPPHONE_HEADSET_WIDEBAND_MODE:
            if( current_opmode == VMMC_IPPHONE_ICVA_HANDSET_MODE  ||
                current_opmode == VMMC_IPPHONE_ICVA_HANDSET_WIDEBAND_MODE)
            {
               /* switch from handset ICVA (NB and WB) to headset WB ICVA */
               opmode=(IFX_int32_t) VMMC_IPPHONE_ICVA_HEADSET_WIDEBAND_MODE;
            }
            else
               opmode=(IFX_int32_t) VMMC_IPPHONE_HEADSET_WIDEBAND_MODE;
            break;
         case VMMC_IPPHONE_HANDSFREE_MUFFLED_MODE:
         case VMMC_IPPHONE_HANDSFREE_WIDEBAND_MUFFLED_MODE:
            opmode=(IFX_int32_t) VMMC_IPPHONE_HANDSFREE_WIDEBAND_MUFFLED_MODE;
            *pHF_room_type = (IFX_int32_t)VMMC_IPPHONE_MUFFLED_ROOM;
            break;
         case VMMC_IPPHONE_HANDSFREE_NORMAL_MODE:
         case VMMC_IPPHONE_HANDSFREE_WIDEBAND_NORMAL_MODE:
            opmode=(IFX_int32_t) VMMC_IPPHONE_HANDSFREE_WIDEBAND_NORMAL_MODE;
            *pHF_room_type = (IFX_int32_t)VMMC_IPPHONE_NORMAL_ROOM;
            break;
         case VMMC_IPPHONE_HANDSFREE_ECHOIC_MODE:
         case VMMC_IPPHONE_HANDSFREE_WIDEBAND_ECHOIC_MODE:
            opmode=(IFX_int32_t) VMMC_IPPHONE_HANDSFREE_WIDEBAND_ECHOIC_MODE;
            *pHF_room_type = (IFX_int32_t)VMMC_IPPHONE_ECHOIC_ROOM;
            break;
         case VMMC_IPPHONE_HANDSET_OPENLISTENING_MODE:
         case VMMC_IPPHONE_HANDSET_WIDEBAND_OPENLISTENING_MODE:
            opmode=(IFX_int32_t) VMMC_IPPHONE_HANDSET_WIDEBAND_OPENLISTENING_MODE;
            break;
         case VMMC_IPPHONE_HEADSET_OPENLISTENING_MODE:
         case VMMC_IPPHONE_HEADSET_WIDEBAND_OPENLISTENING_MODE:
            opmode=(IFX_int32_t) VMMC_IPPHONE_HEADSET_WIDEBAND_OPENLISTENING_MODE;
            break;

         case VMMC_IPPHONE_ICVA_HANDSET_MODE:
         case VMMC_IPPHONE_ICVA_HANDSET_WIDEBAND_MODE:
            opmode=(IFX_int32_t) VMMC_IPPHONE_ICVA_HANDSET_WIDEBAND_MODE;
            break;
         case VMMC_IPPHONE_ICVA_HEADSET_WIDEBAND_MODE:
         case VMMC_IPPHONE_ICVA_HEADSET_MODE:
            opmode=(IFX_int32_t) VMMC_IPPHONE_ICVA_HEADSET_WIDEBAND_MODE;
            break;

         default:
            TRACE(VMMC, DBG_LEVEL_HIGH, \
                  ("%s - Invalid operation mode request %d for Wideband transition\n", \
                  __FUNCTION__,req_opmode));
            break;
      }
   }
   else  /* req_smpl == NB_8_KHZ */
   {
      /*
       * determine narrowband operation mode
       * in case of active ICVA mode switch to ICVA narrowband mode
       */
      switch(req_opmode)
      {
         case VMMC_IPPHONE_HANDSET_MODE:
         case VMMC_IPPHONE_HANDSET_WIDEBAND_MODE:
            if( current_opmode == VMMC_IPPHONE_ICVA_HEADSET_MODE ||
                current_opmode == VMMC_IPPHONE_ICVA_HEADSET_WIDEBAND_MODE )
            {
               opmode=(IFX_int32_t) VMMC_IPPHONE_ICVA_HANDSET_MODE;
            }
            else
               opmode=(IFX_int32_t) VMMC_IPPHONE_HANDSET_MODE;
            break;
         case VMMC_IPPHONE_HEADSET_MODE:
         case VMMC_IPPHONE_HEADSET_WIDEBAND_MODE:
            if( current_opmode == VMMC_IPPHONE_ICVA_HANDSET_MODE  ||
                current_opmode == VMMC_IPPHONE_ICVA_HANDSET_WIDEBAND_MODE )
            {
               opmode=(IFX_int32_t) VMMC_IPPHONE_ICVA_HEADSET_MODE;
            }
            else
               opmode=(IFX_int32_t) VMMC_IPPHONE_HEADSET_MODE;
            break;
         case VMMC_IPPHONE_HANDSFREE_MUFFLED_MODE:
         case VMMC_IPPHONE_HANDSFREE_WIDEBAND_MUFFLED_MODE:
            opmode=(IFX_int32_t) VMMC_IPPHONE_HANDSFREE_MUFFLED_MODE;
            *pHF_room_type = (IFX_int32_t)VMMC_IPPHONE_MUFFLED_ROOM;
            break;
         case VMMC_IPPHONE_HANDSFREE_NORMAL_MODE:
         case VMMC_IPPHONE_HANDSFREE_WIDEBAND_NORMAL_MODE:
            opmode=(IFX_int32_t) VMMC_IPPHONE_HANDSFREE_NORMAL_MODE;
            *pHF_room_type = (IFX_int32_t)VMMC_IPPHONE_NORMAL_ROOM;
            break;
         case VMMC_IPPHONE_HANDSFREE_ECHOIC_MODE:
         case VMMC_IPPHONE_HANDSFREE_WIDEBAND_ECHOIC_MODE:
            opmode=(IFX_int32_t) VMMC_IPPHONE_HANDSFREE_ECHOIC_MODE;
            *pHF_room_type = (IFX_int32_t)VMMC_IPPHONE_ECHOIC_ROOM;
            break;
         case VMMC_IPPHONE_HANDSET_OPENLISTENING_MODE:
         case VMMC_IPPHONE_HANDSET_WIDEBAND_OPENLISTENING_MODE:
            opmode=(IFX_int32_t) VMMC_IPPHONE_HANDSET_OPENLISTENING_MODE;
            break;
         case VMMC_IPPHONE_HEADSET_OPENLISTENING_MODE:
         case VMMC_IPPHONE_HEADSET_WIDEBAND_OPENLISTENING_MODE:
            opmode=(IFX_int32_t) VMMC_IPPHONE_HEADSET_OPENLISTENING_MODE;
            break;

         case VMMC_IPPHONE_ICVA_HANDSET_MODE:
         case VMMC_IPPHONE_ICVA_HANDSET_WIDEBAND_MODE:
            opmode=(IFX_int32_t) VMMC_IPPHONE_ICVA_HANDSET_MODE;
            break;
         case VMMC_IPPHONE_ICVA_HEADSET_WIDEBAND_MODE:
         case VMMC_IPPHONE_ICVA_HEADSET_MODE:
            opmode=(IFX_int32_t) VMMC_IPPHONE_ICVA_HEADSET_MODE;
            break;
         default:
            TRACE(VMMC, DBG_LEVEL_HIGH, \
                  ("%s - Invalid operation mode request %d for Narrowband transition\n", \
                  __FUNCTION__, req_opmode));
            break;
      }
   }
   return opmode;
}


/******* +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ ************/

#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
/**
   Gets requested operation mode from the passed BBD tag.
\param
   tag  - 16 bit bbd subblock tag that comprises operation mode, volume
                 and mute state.
\param
   vol  - pointer to requested operation mode information retrieved from tag.
\return
   IFX_SUCCESS or IFX_ERROR
\remarks
*/
static IFX_int32_t vmmc_audio_Get_Tag_Operation_Mode(IFX_uint16_t tag,
                                                     VMMC_IPPHONE_OPERATION_MODE *opmode)
{
   IFX_int32_t retval = IFX_SUCCESS;
   IFX_uint16_t  tmp_tag;

   TRACE(VMMC,DBG_LEVEL_LOW, ("%s\n", __FUNCTION__));

   /* clear mute bit if set */
   if(tag & BBD_VMMC_IPPHONE_MUTE)
   {
      tmp_tag = tag & ~BBD_VMMC_IPPHONE_MUTE;
   }
   else
      tmp_tag = tag;

   if( tmp_tag == BBD_VMMC_IPPHONE_IDLE )
   {
      *opmode = VMMC_IPPHONE_IDLE_MODE;
   }
   else
   if( BBD_VMMC_IPPHONE_HANDS_VOL1 <= tmp_tag && tmp_tag <= BBD_VMMC_IPPHONE_HANDS_VOL8 )
   {
      *opmode = VMMC_IPPHONE_HANDSET_MODE;
   }
   else
   if( BBD_VMMC_IPPHONE_HEADS_VOL1 <= tmp_tag && tmp_tag <= BBD_VMMC_IPPHONE_HEADS_VOL8 )
   {
      *opmode = VMMC_IPPHONE_HEADSET_MODE;
   }
   else
   if( BBD_VMMC_IPPHONE_HANDSF_MUFFLED_VOL1 <= tmp_tag &&
       tmp_tag <= BBD_VMMC_IPPHONE_HANDSF_MUFFLED_VOL8 )
   {
      *opmode = VMMC_IPPHONE_HANDSFREE_MUFFLED_MODE;
   }
   else
   if( BBD_VMMC_IPPHONE_HANDSF_NORMAL_VOL1 <= tmp_tag &&
       tmp_tag <= BBD_VMMC_IPPHONE_HANDSF_NORMAL_VOL8 )
   {
      *opmode = VMMC_IPPHONE_HANDSFREE_NORMAL_MODE;
   }
   else
   if( BBD_VMMC_IPPHONE_HANDSF_ECHOIC_VOL1 <= tmp_tag &&
       tmp_tag <= BBD_VMMC_IPPHONE_HANDSF_ECHOIC_VOL8 )
   {
      *opmode = VMMC_IPPHONE_HANDSFREE_ECHOIC_MODE;
   }
   else
   if( BBD_VMMC_IPPHONE_HANDS_OL_VOL1 <= tmp_tag &&
       tmp_tag <= BBD_VMMC_IPPHONE_HANDS_OL_VOL8 )
   {
      *opmode = VMMC_IPPHONE_HANDSET_OPENLISTENING_MODE;
   }
   else
   if( BBD_VMMC_IPPHONE_HEADS_OL_VOL1 <= tmp_tag &&
       tmp_tag <= BBD_VMMC_IPPHONE_HEADS_OL_VOL8 )
   {
      *opmode = VMMC_IPPHONE_HEADSET_OPENLISTENING_MODE;
   }
   else
   if( BBD_VMMC_IPPHONE_HANDS_WB_VOL1 <= tmp_tag &&
       tmp_tag <= BBD_VMMC_IPPHONE_HANDS_WB_VOL8 )
   {
      *opmode = VMMC_IPPHONE_HANDSET_WIDEBAND_MODE;
   }
   else
   if( BBD_VMMC_IPPHONE_HEADS_WB_VOL1 <= tmp_tag &&
       tmp_tag <= BBD_VMMC_IPPHONE_HEADS_WB_VOL8 )
   {
      *opmode = VMMC_IPPHONE_HEADSET_WIDEBAND_MODE;
   }
   else
   if( BBD_VMMC_IPPHONE_HANDSF_WB_MUFFLED_VOL1 <= tmp_tag &&
       tmp_tag <= BBD_VMMC_IPPHONE_HANDSF_WB_MUFFLED_VOL8 )
   {
      *opmode = VMMC_IPPHONE_HANDSFREE_WIDEBAND_MUFFLED_MODE;
   }
   else
   if( BBD_VMMC_IPPHONE_HANDSF_WB_NORMAL_VOL1 <= tmp_tag &&
       tmp_tag <= BBD_VMMC_IPPHONE_HANDSF_WB_NORMAL_VOL8 )
   {
      *opmode = VMMC_IPPHONE_HANDSFREE_WIDEBAND_NORMAL_MODE;
   }
   else
   if( BBD_VMMC_IPPHONE_HANDSF_WB_ECHOIC_VOL1 <= tmp_tag &&
       tmp_tag <= BBD_VMMC_IPPHONE_HANDSF_WB_ECHOIC_VOL8 )
   {
      *opmode = VMMC_IPPHONE_HANDSFREE_WIDEBAND_ECHOIC_MODE;
   }
   else
   if( BBD_VMMC_IPPHONE_HANDS_OL_WB_VOL1 <= tmp_tag &&
       tmp_tag <= BBD_VMMC_IPPHONE_HANDS_OL_WB_VOL8 )
   {
      *opmode = VMMC_IPPHONE_HANDSET_WIDEBAND_OPENLISTENING_MODE;
   }
   else
   if( BBD_VMMC_IPPHONE_HEADS_OL_WB_VOL1 <= tmp_tag &&
       tmp_tag <= BBD_VMMC_IPPHONE_HEADS_OL_WB_VOL8 )
   {
      *opmode = VMMC_IPPHONE_HEADSET_WIDEBAND_OPENLISTENING_MODE;
   }
   else
   if( BBD_VMMC_IPPHONE_HANDS_ICVA_VOL1 <= tmp_tag &&
       tmp_tag <= BBD_VMMC_IPPHONE_HANDS_ICVA_VOL8 )
   {
      *opmode = VMMC_IPPHONE_ICVA_HANDSET_MODE;
   }
   else
   if( BBD_VMMC_IPPHONE_HEADS_ICVA_VOL1 <= tmp_tag &&
       tmp_tag <= BBD_VMMC_IPPHONE_HEADS_ICVA_VOL8 )
   {
      *opmode = VMMC_IPPHONE_ICVA_HEADSET_MODE;
   }
   else
   if( BBD_VMMC_IPPHONE_HANDS_WB_ICVA_VOL1 <= tmp_tag &&
       tmp_tag <= BBD_VMMC_IPPHONE_HANDS_WB_ICVA_VOL8 )
   {
      *opmode = VMMC_IPPHONE_ICVA_HANDSET_WIDEBAND_MODE;
   }
   else
   if( BBD_VMMC_IPPHONE_HEADS_WB_ICVA_VOL1 <= tmp_tag &&
       tmp_tag <= BBD_VMMC_IPPHONE_HEADS_WB_ICVA_VOL8 )
   {
      *opmode = VMMC_IPPHONE_ICVA_HEADSET_WIDEBAND_MODE;
   }
   else
   {
      TRACE(VMMC,DBG_LEVEL_HIGH, ("%s - Invalid tag 0x%X\n", __FUNCTION__, tag));
      retval = IFX_ERROR;
   }

   return retval;
}

/**
   Gets requested volume from the passed BBD tag.
\param
   tag  - 16 bit bbd subblock tag that comprises operation mode, volume
                 and mute state.
\param
   vol  - pointer to requested volume information retrieved from tag.
\return
   IFX_SUCCESS or IFX_ERROR
\remarks
*/
static IFX_int32_t vmmc_audio_Get_Tag_Volume(IFX_uint16_t tag, VMMC_IPPHONE_VOLUME_LEVEL *vol)
{
   IFX_uint16_t  tmp_tag;
   IFX_int32_t retval = IFX_SUCCESS;

   TRACE(VMMC,DBG_LEVEL_LOW, ("%s\n", __FUNCTION__));

   /* clear mute bit if set */
   if(tag & BBD_VMMC_IPPHONE_MUTE)
   {
      tmp_tag = tag & ~BBD_VMMC_IPPHONE_MUTE;
   }
   else
      tmp_tag = tag;

   switch(tmp_tag & 0xF)
   {
      case 0:  /* Idle Mode requested -> return current volume */
         *vol = current_volume;
         break;

      case 1:
         *vol = VMMC_IPPHONE_VOL1;
         break;
      case 2:
         *vol = VMMC_IPPHONE_VOL2;
         break;
      case 3:
         *vol = VMMC_IPPHONE_VOL3;
         break;
      case 4:
         *vol = VMMC_IPPHONE_VOL4;
         break;
      case 5:
         *vol = VMMC_IPPHONE_VOL5;
         break;
      case 6:
         *vol = VMMC_IPPHONE_VOL6;
         break;
      case 7:
         *vol = VMMC_IPPHONE_VOL7;
         break;
      case 8:
         *vol = VMMC_IPPHONE_VOL8;
         break;
      default:
         retval = IFX_ERROR;
         break;

   }
   return retval;

}


/**
   Updates Audio Channel Messages of all parameter sets with AFE- In and Outputs
   stored in Audio Channel structure.
\param
   pCh         - pointer to channel structure.
\return
   Nothing.
\remarks
   Function is called everytime AFE-CFG IOCTL is called.
*/
static IFX_void_t vmmc_audio_UpdOpModeParams_AFEInOut( VMMC_CHANNEL  *pCh )
{
   IFX_int32_t mode, vol;

   TRACE(VMMC, DBG_LEVEL_LOW,
         ("%s - Update HFMIC with %u\n", __FUNCTION__, pCh->pAUDIO->nHFMic));
   TRACE(VMMC, DBG_LEVEL_LOW,
         ("%s - Update HFRCV with %u\n", __FUNCTION__, pCh->pAUDIO->nHFOut));
   TRACE(VMMC, DBG_LEVEL_LOW,
         ("%s - Update HSMIC with %u\n", __FUNCTION__, pCh->pAUDIO->nHNMic));
   TRACE(VMMC, DBG_LEVEL_LOW,
         ("%s - Update HSRCV with %u\n", __FUNCTION__, pCh->pAUDIO->nHNOut));
   TRACE(VMMC, DBG_LEVEL_LOW,
         ("%s - Update HEMIC with %u\n", __FUNCTION__, pCh->pAUDIO->nHDMic));
   TRACE(VMMC, DBG_LEVEL_LOW,
         ("%s - Update HERCV with %u\n", __FUNCTION__, pCh->pAUDIO->nHDOut));

   for(vol=0; vol<VMMC_IPPHONE_NR_VOL_LEVELS; vol++)
   {
      /* update opmode parameters' audio channel messages */
      for(mode=0; mode<VMMC_IPPHONE_NR_OPMODES; mode++)
      {
         voice_params.opmode_params[mode][vol].params.ad_chan.HFMIC =
            pCh->pAUDIO->nHFMic;
         voice_params.opmode_params[mode][vol].params.ad_chan.HFRCV =
            pCh->pAUDIO->nHFOut;
         voice_params.opmode_params[mode][vol].params.ad_chan.HSMIC =
            pCh->pAUDIO->nHNMic;
         voice_params.opmode_params[mode][vol].params.ad_chan.HSRCV =
            pCh->pAUDIO->nHNOut;
         voice_params.opmode_params[mode][vol].params.ad_chan.HEMIC =
            pCh->pAUDIO->nHDMic;
         voice_params.opmode_params[mode][vol].params.ad_chan.HERCV =
            pCh->pAUDIO->nHDOut;
      }
      /* update ringing parameters' audio channel messages */
      ring_param_set.ring_params[vol].ad_chan.HFMIC =
            pCh->pAUDIO->nHFMic;
      ring_param_set.ring_params[vol].ad_chan.HFRCV =
            pCh->pAUDIO->nHFOut;
      ring_param_set.ring_params[vol].ad_chan.HSMIC =
            pCh->pAUDIO->nHNMic;
      ring_param_set.ring_params[vol].ad_chan.HSRCV =
            pCh->pAUDIO->nHNOut;
      ring_param_set.ring_params[vol].ad_chan.HEMIC =
            pCh->pAUDIO->nHDMic;
      ring_param_set.ring_params[vol].ad_chan.HERCV =
            pCh->pAUDIO->nHDOut;
   }
   /* update idle operation mode parameters' audio channel message */
   voice_params.idle_params.params.ad_chan.HFMIC = pCh->pAUDIO->nHFMic;
   voice_params.idle_params.params.ad_chan.HFRCV = pCh->pAUDIO->nHFOut;
   voice_params.idle_params.params.ad_chan.HSMIC = pCh->pAUDIO->nHNMic;
   voice_params.idle_params.params.ad_chan.HSRCV = pCh->pAUDIO->nHNOut;;
   voice_params.idle_params.params.ad_chan.HEMIC = pCh->pAUDIO->nHDMic;
   voice_params.idle_params.params.ad_chan.HERCV = pCh->pAUDIO->nHDOut;

   /* update last used audio channel message */
   pCh->pAUDIO->audio_ch.HFMIC = pCh->pAUDIO->nHFMic;
   pCh->pAUDIO->audio_ch.HFRCV = pCh->pAUDIO->nHFOut;
   pCh->pAUDIO->audio_ch.HSMIC = pCh->pAUDIO->nHNMic;
   pCh->pAUDIO->audio_ch.HSRCV = pCh->pAUDIO->nHNOut;;
   pCh->pAUDIO->audio_ch.HEMIC = pCh->pAUDIO->nHDMic;
   pCh->pAUDIO->audio_ch.HERCV = pCh->pAUDIO->nHDOut;
}




/**
   Initialize array of operation mode and volume dependent voice parameters.
\param
   None.
\return
   IFX_SUCCESS or IFX_ERROR
\remarks

*/
IFX_int32_t VMMC_AUDIO_InitOpModeParameters( IFX_void_t )
{
   TRACE(VMMC,DBG_LEVEL_LOW, ("%s\n", __FUNCTION__));
   IFX_int32_t voice_par_size = sizeof(voice_params.opmode_params[VMMC_IPPHONE_HANDSET_MODE][0]);

   memset((IFX_void_t*) &voice_params, 0, sizeof(VMMC_OpModeParamSet_t));

   /* check idle parameter set for correct size */
   if(sizeof(ip2_idle) != voice_par_size)
   {
      TRACE(VMMC,DBG_LEVEL_LOW, \
         ("%s - Invalid size of idle parameter set in ipp_voice_params.h\n", __FUNCTION__));
      return IFX_ERROR;
   }
   /* set idle mode parameters */
   memcpy(&(voice_params.idle_params), &ip2_idle, sizeof(VMMC_OpModeCtrl_t));

   /* check handset parameter set for correct size */
   if( (sizeof(ip2_hs_nb_0) != voice_par_size) ||
       (sizeof(ip2_hs_nb_1) != voice_par_size) ||
       (sizeof(ip2_hs_nb_2) != voice_par_size) ||
       (sizeof(ip2_hs_nb_3) != voice_par_size) ||
       (sizeof(ip2_hs_nb_4) != voice_par_size) ||
       (sizeof(ip2_hs_nb_5) != voice_par_size) ||
       (sizeof(ip2_hs_nb_6) != voice_par_size) ||
       (sizeof(ip2_hs_nb_7) != voice_par_size) )
   {
      TRACE(VMMC,DBG_LEVEL_HIGH, \
         ("%s - Invalid size of handset parameter set in ipp_voice_params.h\n", __FUNCTION__));
      return IFX_ERROR;
   }
   /* set parameters for handset mode, narrowband, volumes 1 ... 8 */
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_MODE][0],
          ip2_hs_nb_0,
          sizeof(ip2_hs_nb_0));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_MODE][1],
          ip2_hs_nb_1,
          sizeof(ip2_hs_nb_1));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_MODE][2],
          ip2_hs_nb_2,
          sizeof(ip2_hs_nb_2));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_MODE][3],
          ip2_hs_nb_3,
          sizeof(ip2_hs_nb_3));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_MODE][4],
          ip2_hs_nb_4,
          sizeof(ip2_hs_nb_4));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_MODE][5],
          ip2_hs_nb_5,
          sizeof(ip2_hs_nb_5));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_MODE][6],
          ip2_hs_nb_6,
          sizeof(ip2_hs_nb_6));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_MODE][7],
          ip2_hs_nb_7,
          sizeof(ip2_hs_nb_7));

   /* check headset parameter set for correct size */
   if( (sizeof(ip2_he_nb_0) != voice_par_size) ||
       (sizeof(ip2_he_nb_1) != voice_par_size) ||
       (sizeof(ip2_he_nb_2) != voice_par_size) ||
       (sizeof(ip2_he_nb_3) != voice_par_size) ||
       (sizeof(ip2_he_nb_4) != voice_par_size) ||
       (sizeof(ip2_he_nb_5) != voice_par_size) ||
       (sizeof(ip2_he_nb_6) != voice_par_size) ||
       (sizeof(ip2_he_nb_7) != voice_par_size) )
   {
      TRACE(VMMC,DBG_LEVEL_HIGH, \
         ("%s - Invalid size of headset parameter set in ipp_voice_params.h\n", __FUNCTION__));
      return IFX_ERROR;
   }
   /* set parameters for headset mode, narrowband, volumes 1 ... 8 */
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_MODE][0],
          ip2_he_nb_0,
          sizeof(ip2_he_nb_0));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_MODE][1],
          ip2_he_nb_1,
          sizeof(ip2_he_nb_1));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_MODE][2],
          ip2_he_nb_2,
          sizeof(ip2_he_nb_2));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_MODE][3],
          ip2_he_nb_3,
          sizeof(ip2_he_nb_3));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_MODE][4],
          ip2_he_nb_4,
          sizeof(ip2_he_nb_4));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_MODE][5],
          ip2_he_nb_5,
          sizeof(ip2_he_nb_5));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_MODE][6],
          ip2_he_nb_6,
          sizeof(ip2_he_nb_6));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_MODE][7],
          ip2_he_nb_7,
          sizeof(ip2_he_nb_7));

   /* check handsfree-nb-muffled parameter set for correct size */
   if( (sizeof(ip2_hf_nb_muf_0) != voice_par_size) ||
       (sizeof(ip2_hf_nb_muf_1) != voice_par_size) ||
       (sizeof(ip2_hf_nb_muf_2) != voice_par_size) ||
       (sizeof(ip2_hf_nb_muf_3) != voice_par_size) ||
       (sizeof(ip2_hf_nb_muf_4) != voice_par_size) ||
       (sizeof(ip2_hf_nb_muf_5) != voice_par_size) ||
       (sizeof(ip2_hf_nb_muf_6) != voice_par_size) ||
       (sizeof(ip2_hf_nb_muf_7) != voice_par_size) )
   {
      TRACE(VMMC,DBG_LEVEL_HIGH, \
         ("%s - Invalid size of hf-nb-muffled parameter set in ipp_voice_params.h\n", __FUNCTION__));
      return IFX_ERROR;
   }
   /* set parameters for handsfree mode, narrowband, muffled room,
      volumes 1 ... 8 */
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_MUFFLED_MODE][0],
          ip2_hf_nb_muf_0,
          sizeof(ip2_hf_nb_muf_0));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_MUFFLED_MODE][1],
          ip2_hf_nb_muf_1,
          sizeof(ip2_hf_nb_muf_1));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_MUFFLED_MODE][2],
          ip2_hf_nb_muf_2,
          sizeof(ip2_hf_nb_muf_2));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_MUFFLED_MODE][3],
          ip2_hf_nb_muf_3,
          sizeof(ip2_hf_nb_muf_3));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_MUFFLED_MODE][4],
          ip2_hf_nb_muf_4,
          sizeof(ip2_hf_nb_muf_4));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_MUFFLED_MODE][5],
          ip2_hf_nb_muf_5,
          sizeof(ip2_hf_nb_muf_5));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_MUFFLED_MODE][6],
          ip2_hf_nb_muf_6,
          sizeof(ip2_hf_nb_muf_6));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_MUFFLED_MODE][7],
          ip2_hf_nb_muf_7,
          sizeof(ip2_hf_nb_muf_7));

   /* check handsfree-nb-muffled parameter set for correct size */
   if( (sizeof(ip2_hf_nb_nrm_0) != voice_par_size) ||
       (sizeof(ip2_hf_nb_nrm_1) != voice_par_size) ||
       (sizeof(ip2_hf_nb_nrm_2) != voice_par_size) ||
       (sizeof(ip2_hf_nb_nrm_3) != voice_par_size) ||
       (sizeof(ip2_hf_nb_nrm_4) != voice_par_size) ||
       (sizeof(ip2_hf_nb_nrm_5) != voice_par_size) ||
       (sizeof(ip2_hf_nb_nrm_6) != voice_par_size) ||
       (sizeof(ip2_hf_nb_nrm_7) != voice_par_size) )
   {
      TRACE(VMMC,DBG_LEVEL_HIGH, \
         ("%s - Invalid size of hf-nb-normal parameter set in ipp_voice_params.h\n", __FUNCTION__));
      return IFX_ERROR;
   }
   /* set parameters for handsfree mode, narrowband, normal room,
      volumes 1 ... 8 */
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_NORMAL_MODE][0],
          ip2_hf_nb_nrm_0,
          sizeof(ip2_hf_nb_nrm_0));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_NORMAL_MODE][1],
          ip2_hf_nb_nrm_1,
          sizeof(ip2_hf_nb_nrm_1));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_NORMAL_MODE][2],
          ip2_hf_nb_nrm_2,
          sizeof(ip2_hf_nb_nrm_2));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_NORMAL_MODE][3],
          ip2_hf_nb_nrm_3,
          sizeof(ip2_hf_nb_nrm_3));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_NORMAL_MODE][4],
          ip2_hf_nb_nrm_4,
          sizeof(ip2_hf_nb_nrm_4));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_NORMAL_MODE][5],
          ip2_hf_nb_nrm_5,
          sizeof(ip2_hf_nb_nrm_5));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_NORMAL_MODE][6],
          ip2_hf_nb_nrm_6,
          sizeof(ip2_hf_nb_nrm_6));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_NORMAL_MODE][7],
          ip2_hf_nb_nrm_7,
          sizeof(ip2_hf_nb_nrm_7));

   /* check handsfree-nb-echoic parameter set for correct size */
   if( (sizeof(ip2_hf_nb_ech_0) != voice_par_size) ||
       (sizeof(ip2_hf_nb_ech_1) != voice_par_size) ||
       (sizeof(ip2_hf_nb_ech_2) != voice_par_size) ||
       (sizeof(ip2_hf_nb_ech_3) != voice_par_size) ||
       (sizeof(ip2_hf_nb_ech_4) != voice_par_size) ||
       (sizeof(ip2_hf_nb_ech_5) != voice_par_size) ||
       (sizeof(ip2_hf_nb_ech_6) != voice_par_size) ||
       (sizeof(ip2_hf_nb_ech_7) != voice_par_size) )
   {
      TRACE(VMMC,DBG_LEVEL_HIGH, \
         ("%s - Invalid size of hf-nb-echoic parameter set in ipp_voice_params.h\n",
          __FUNCTION__));
      return IFX_ERROR;
   }
   /* set parameters for handsfree mode, narrowband, echoic room,
      volumes 1 ... 8 */
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_ECHOIC_MODE][0],
          ip2_hf_nb_ech_0,
          sizeof(ip2_hf_nb_ech_0));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_ECHOIC_MODE][1],
          ip2_hf_nb_ech_1,
          sizeof(ip2_hf_nb_ech_1));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_ECHOIC_MODE][2],
          ip2_hf_nb_ech_2,
          sizeof(ip2_hf_nb_ech_2));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_ECHOIC_MODE][3],
          ip2_hf_nb_ech_3,
          sizeof(ip2_hf_nb_ech_3));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_ECHOIC_MODE][4],
          ip2_hf_nb_ech_4,
          sizeof(ip2_hf_nb_ech_4));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_ECHOIC_MODE][5],
          ip2_hf_nb_ech_5,
          sizeof(ip2_hf_nb_ech_5));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_ECHOIC_MODE][6],
          ip2_hf_nb_ech_6,
          sizeof(ip2_hf_nb_ech_6));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_ECHOIC_MODE][7],
          ip2_hf_nb_ech_7,
          sizeof(ip2_hf_nb_ech_7));

   /* check handset-nb-openlistening parameter set for correct size */
   if( (sizeof(ip2_hs_nb_ol_0) != voice_par_size) ||
       (sizeof(ip2_hs_nb_ol_1) != voice_par_size) ||
       (sizeof(ip2_hs_nb_ol_2) != voice_par_size) ||
       (sizeof(ip2_hs_nb_ol_3) != voice_par_size) ||
       (sizeof(ip2_hs_nb_ol_4) != voice_par_size) ||
       (sizeof(ip2_hs_nb_ol_5) != voice_par_size) ||
       (sizeof(ip2_hs_nb_ol_6) != voice_par_size) ||
       (sizeof(ip2_hs_nb_ol_7) != voice_par_size) )
   {
      TRACE(VMMC,DBG_LEVEL_HIGH, \
         ("%s - Invalid size of hs-nb-openlistening parameter set in ipp_voice_params.h\n",
          __FUNCTION__));
      return IFX_ERROR;
   }
    /* set parameters for handset mode, narrowband,
       open listening, volumes 1 ... 8 */
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_OPENLISTENING_MODE][0],
          ip2_hs_nb_ol_0,
          sizeof(ip2_hs_nb_ol_0));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_OPENLISTENING_MODE][1],
          ip2_hs_nb_ol_1,
          sizeof(ip2_hs_nb_ol_1));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_OPENLISTENING_MODE][2],
          ip2_hs_nb_ol_2,
          sizeof(ip2_hs_nb_ol_2));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_OPENLISTENING_MODE][3],
          ip2_hs_nb_ol_3,
          sizeof(ip2_hs_nb_ol_3));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_OPENLISTENING_MODE][4],
          ip2_hs_nb_ol_4,
          sizeof(ip2_hs_nb_ol_4));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_OPENLISTENING_MODE][5],
          ip2_hs_nb_ol_5,
          sizeof(ip2_hs_nb_ol_5));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_OPENLISTENING_MODE][6],
          ip2_hs_nb_ol_6,
          sizeof(ip2_hs_nb_ol_6));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_OPENLISTENING_MODE][7],
          ip2_hs_nb_ol_7,
          sizeof(ip2_hs_nb_ol_7));


   /* check headset-nb-openlistening parameter set for correct size */
   if( (sizeof(ip2_he_nb_ol_0) != voice_par_size) ||
       (sizeof(ip2_he_nb_ol_1) != voice_par_size) ||
       (sizeof(ip2_he_nb_ol_2) != voice_par_size) ||
       (sizeof(ip2_he_nb_ol_3) != voice_par_size) ||
       (sizeof(ip2_he_nb_ol_4) != voice_par_size) ||
       (sizeof(ip2_he_nb_ol_5) != voice_par_size) ||
       (sizeof(ip2_he_nb_ol_6) != voice_par_size) ||
       (sizeof(ip2_he_nb_ol_7) != voice_par_size) )
   {
      TRACE(VMMC,DBG_LEVEL_HIGH, \
         ("%s - Invalid size of he-nb-openlistening parameter set in ipp_voice_params.h\n",
          __FUNCTION__));
      return IFX_ERROR;
   }
    /* set parameters for headset mode, narrowband,
       open listening, volumes 1 ... 8 */
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_OPENLISTENING_MODE][0],
          ip2_he_nb_ol_0,
          sizeof(ip2_he_nb_ol_0));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_OPENLISTENING_MODE][1],
          ip2_he_nb_ol_1,
          sizeof(ip2_he_nb_ol_1));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_OPENLISTENING_MODE][2],
          ip2_he_nb_ol_2,
          sizeof(ip2_he_nb_ol_2));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_OPENLISTENING_MODE][3],
          ip2_he_nb_ol_3,
          sizeof(ip2_he_nb_ol_3));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_OPENLISTENING_MODE][4],
          ip2_he_nb_ol_4,
          sizeof(ip2_he_nb_ol_4));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_OPENLISTENING_MODE][5],
          ip2_he_nb_ol_5,
          sizeof(ip2_he_nb_ol_5));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_OPENLISTENING_MODE][6],
          ip2_he_nb_ol_6,
          sizeof(ip2_he_nb_ol_6));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_OPENLISTENING_MODE][7],
          ip2_he_nb_ol_7,
          sizeof(ip2_he_nb_ol_7));

   /* check handset-nb-openlistening parameter set for correct size */
   if( (sizeof(ip2_hs_wb_0) != voice_par_size) ||
       (sizeof(ip2_hs_wb_1) != voice_par_size) ||
       (sizeof(ip2_hs_wb_2) != voice_par_size) ||
       (sizeof(ip2_hs_wb_3) != voice_par_size) ||
       (sizeof(ip2_hs_wb_4) != voice_par_size) ||
       (sizeof(ip2_hs_wb_5) != voice_par_size) ||
       (sizeof(ip2_hs_wb_6) != voice_par_size) ||
       (sizeof(ip2_hs_wb_7) != voice_par_size) )
   {
      TRACE(VMMC,DBG_LEVEL_HIGH, \
         ("%s - Invalid size of handset-wideband parameter set in ipp_voice_params.h\n",
          __FUNCTION__));
      return IFX_ERROR;
   }
    /* set parameters for handset mode, wideband,
       volumes 1 ... 8 */
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_WIDEBAND_MODE][0],
          ip2_hs_wb_0,
          sizeof(ip2_hs_wb_0));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_WIDEBAND_MODE][1],
          ip2_hs_wb_1,
          sizeof(ip2_hs_wb_1));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_WIDEBAND_MODE][2],
          ip2_hs_wb_2,
          sizeof(ip2_hs_wb_2));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_WIDEBAND_MODE][3],
          ip2_hs_wb_3,
          sizeof(ip2_hs_wb_3));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_WIDEBAND_MODE][4],
          ip2_hs_wb_4,
          sizeof(ip2_hs_wb_4));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_WIDEBAND_MODE][5],
          ip2_hs_wb_5,
          sizeof(ip2_hs_wb_5));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_WIDEBAND_MODE][6],
          ip2_hs_wb_6,
          sizeof(ip2_hs_wb_6));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_WIDEBAND_MODE][7],
          ip2_hs_wb_7,
          sizeof(ip2_hs_wb_7));

   /* check headset-wb-openlistening parameter set for correct size */
   if( (sizeof(ip2_he_wb_0) != voice_par_size) ||
       (sizeof(ip2_he_wb_1) != voice_par_size) ||
       (sizeof(ip2_he_wb_2) != voice_par_size) ||
       (sizeof(ip2_he_wb_3) != voice_par_size) ||
       (sizeof(ip2_he_wb_4) != voice_par_size) ||
       (sizeof(ip2_he_wb_5) != voice_par_size) ||
       (sizeof(ip2_he_wb_6) != voice_par_size) ||
       (sizeof(ip2_he_wb_7) != voice_par_size) )
   {
      TRACE(VMMC,DBG_LEVEL_HIGH, \
         ("%s - Invalid size of headset-wb parameter set in ipp_voice_params.h\n",
          __FUNCTION__));
      return IFX_ERROR;
   }
    /* set parameters for headset mode, wideband,
       volumes 1 ... 8 */
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_WIDEBAND_MODE][0],
          ip2_he_wb_0,
          sizeof(ip2_he_wb_0));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_WIDEBAND_MODE][1],
          ip2_he_wb_1,
          sizeof(ip2_he_wb_1));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_WIDEBAND_MODE][2],
          ip2_he_wb_2,
          sizeof(ip2_he_wb_2));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_WIDEBAND_MODE][3],
          ip2_he_wb_3,
          sizeof(ip2_he_wb_3));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_WIDEBAND_MODE][4],
          ip2_he_wb_4,
          sizeof(ip2_he_wb_4));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_WIDEBAND_MODE][5],
          ip2_he_wb_5,
          sizeof(ip2_he_wb_5));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_WIDEBAND_MODE][6],
          ip2_he_wb_6,
          sizeof(ip2_he_wb_6));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_WIDEBAND_MODE][7],
          ip2_he_wb_7,
          sizeof(ip2_he_wb_7));

   /* check handsfree-wb-muffled parameter set for correct size */
   if( (sizeof(ip2_hf_wb_muf_0) != voice_par_size) ||
       (sizeof(ip2_hf_wb_muf_1) != voice_par_size) ||
       (sizeof(ip2_hf_wb_muf_2) != voice_par_size) ||
       (sizeof(ip2_hf_wb_muf_3) != voice_par_size) ||
       (sizeof(ip2_hf_wb_muf_4) != voice_par_size) ||
       (sizeof(ip2_hf_wb_muf_5) != voice_par_size) ||
       (sizeof(ip2_hf_wb_muf_6) != voice_par_size) ||
       (sizeof(ip2_hf_wb_muf_7) != voice_par_size) )
   {
      TRACE(VMMC,DBG_LEVEL_HIGH, \
         ("%s - Invalid size of handsfree-wb-muffled parameter set in ipp_voice_params.h\n",
          __FUNCTION__));
      return IFX_ERROR;
   }
   /* set parameters for handsfree mode, wideband, muffled room,
      volumes 1 ... 8 */
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_WIDEBAND_MUFFLED_MODE][0],
          ip2_hf_wb_muf_0,
          sizeof(ip2_hf_wb_muf_0));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_WIDEBAND_MUFFLED_MODE][1],
          ip2_hf_wb_muf_1,
          sizeof(ip2_hf_wb_muf_1));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_WIDEBAND_MUFFLED_MODE][2],
          ip2_hf_wb_muf_2,
          sizeof(ip2_hf_wb_muf_2));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_WIDEBAND_MUFFLED_MODE][3],
          ip2_hf_wb_muf_3,
          sizeof(ip2_hf_wb_muf_3));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_WIDEBAND_MUFFLED_MODE][4],
          ip2_hf_wb_muf_4,
          sizeof(ip2_hf_wb_muf_4));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_WIDEBAND_MUFFLED_MODE][5],
          ip2_hf_wb_muf_5,
          sizeof(ip2_hf_wb_muf_5));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_WIDEBAND_MUFFLED_MODE][6],
          ip2_hf_wb_muf_6,
          sizeof(ip2_hf_wb_muf_6));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_WIDEBAND_MUFFLED_MODE][7],
          ip2_hf_wb_muf_7,
          sizeof(ip2_hf_wb_muf_7));

   /* check handsfree-wb-normal parameter set for correct size */
   if( (sizeof(ip2_hf_wb_nrm_0) != voice_par_size) ||
       (sizeof(ip2_hf_wb_nrm_1) != voice_par_size) ||
       (sizeof(ip2_hf_wb_nrm_2) != voice_par_size) ||
       (sizeof(ip2_hf_wb_nrm_3) != voice_par_size) ||
       (sizeof(ip2_hf_wb_nrm_4) != voice_par_size) ||
       (sizeof(ip2_hf_wb_nrm_5) != voice_par_size) ||
       (sizeof(ip2_hf_wb_nrm_6) != voice_par_size) ||
       (sizeof(ip2_hf_wb_nrm_7) != voice_par_size) )
   {
      TRACE(VMMC,DBG_LEVEL_HIGH, \
         ("%s - Invalid size of handsfree-wb-normal parameter set in ipp_voice_params.h\n",
          __FUNCTION__));
      return IFX_ERROR;
   }
   /* set parameters for handsfree mode, wideband, normal room,
      volumes 1 ... 8 */
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_WIDEBAND_NORMAL_MODE][0],
          ip2_hf_wb_nrm_0,
          sizeof(ip2_hf_wb_nrm_0));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_WIDEBAND_NORMAL_MODE][1],
          ip2_hf_wb_nrm_1,
          sizeof(ip2_hf_wb_nrm_1));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_WIDEBAND_NORMAL_MODE][2],
          ip2_hf_wb_nrm_2,
          sizeof(ip2_hf_wb_nrm_2));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_WIDEBAND_NORMAL_MODE][3],
          ip2_hf_wb_nrm_3,
          sizeof(ip2_hf_wb_nrm_3));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_WIDEBAND_NORMAL_MODE][4],
          ip2_hf_wb_nrm_4,
          sizeof(ip2_hf_wb_nrm_4));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_WIDEBAND_NORMAL_MODE][5],
          ip2_hf_wb_nrm_5,
          sizeof(ip2_hf_wb_nrm_5));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_WIDEBAND_NORMAL_MODE][6],
          ip2_hf_wb_nrm_6,
          sizeof(ip2_hf_wb_nrm_6));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_WIDEBAND_NORMAL_MODE][7],
          ip2_hf_wb_nrm_7,
          sizeof(ip2_hf_wb_nrm_7));

   /* check handsfree-wb-echoic parameter set for correct size */
   if( (sizeof(ip2_hf_wb_ech_0) != voice_par_size) ||
       (sizeof(ip2_hf_wb_ech_1) != voice_par_size) ||
       (sizeof(ip2_hf_wb_ech_2) != voice_par_size) ||
       (sizeof(ip2_hf_wb_ech_3) != voice_par_size) ||
       (sizeof(ip2_hf_wb_ech_4) != voice_par_size) ||
       (sizeof(ip2_hf_wb_ech_5) != voice_par_size) ||
       (sizeof(ip2_hf_wb_ech_6) != voice_par_size) ||
       (sizeof(ip2_hf_wb_ech_7) != voice_par_size) )
   {
      TRACE(VMMC,DBG_LEVEL_HIGH, \
         ("%s - Invalid size of handsfree-wb-echoic parameter set in ipp_voice_params.h\n",
          __FUNCTION__));
      return IFX_ERROR;
   }
   /* set parameters for handsfree mode, wideband, echoic room,
      volumes 1 ... 8 */
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_WIDEBAND_ECHOIC_MODE][0],
          ip2_hf_wb_ech_0,
          sizeof(ip2_hf_wb_ech_0));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_WIDEBAND_ECHOIC_MODE][1],
          ip2_hf_wb_ech_1,
          sizeof(ip2_hf_wb_ech_1));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_WIDEBAND_ECHOIC_MODE][2],
          ip2_hf_wb_ech_2,
          sizeof(ip2_hf_wb_ech_2));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_WIDEBAND_ECHOIC_MODE][3],
          ip2_hf_wb_ech_3,
          sizeof(ip2_hf_wb_ech_3));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_WIDEBAND_ECHOIC_MODE][4],
          ip2_hf_wb_ech_4,
          sizeof(ip2_hf_wb_ech_4));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_WIDEBAND_ECHOIC_MODE][5],
          ip2_hf_wb_ech_5,
          sizeof(ip2_hf_wb_ech_5));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_WIDEBAND_ECHOIC_MODE][6],
          ip2_hf_wb_ech_6,
          sizeof(ip2_hf_wb_ech_6));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSFREE_WIDEBAND_ECHOIC_MODE][7],
          ip2_hf_wb_ech_7,
          sizeof(ip2_hf_wb_ech_7));

   /* check handset-wb-openlistening parameter set for correct size */
   if( (sizeof(ip2_hs_wb_ol_0) != voice_par_size) ||
       (sizeof(ip2_hs_wb_ol_1) != voice_par_size) ||
       (sizeof(ip2_hs_wb_ol_2) != voice_par_size) ||
       (sizeof(ip2_hs_wb_ol_3) != voice_par_size) ||
       (sizeof(ip2_hs_wb_ol_4) != voice_par_size) ||
       (sizeof(ip2_hs_wb_ol_5) != voice_par_size) ||
       (sizeof(ip2_hs_wb_ol_6) != voice_par_size) ||
       (sizeof(ip2_hs_wb_ol_7) != voice_par_size) )
   {
      TRACE(VMMC,DBG_LEVEL_HIGH, \
         ("%s - Invalid size of handset-wb-openlistening parameter set in ipp_voice_params.h\n",
          __FUNCTION__));
      return IFX_ERROR;
   }
    /* set parameters for handset mode, wideband,
       open listening, volumes 1 ... 8 */
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_WIDEBAND_OPENLISTENING_MODE][0],
          ip2_hs_wb_ol_0,
          sizeof(ip2_hs_wb_ol_0));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_WIDEBAND_OPENLISTENING_MODE][1],
          ip2_hs_wb_ol_1,
          sizeof(ip2_hs_wb_ol_1));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_WIDEBAND_OPENLISTENING_MODE][2],
          ip2_hs_wb_ol_2,
          sizeof(ip2_hs_wb_ol_2));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_WIDEBAND_OPENLISTENING_MODE][3],
          ip2_hs_wb_ol_3,
          sizeof(ip2_hs_wb_ol_3));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_WIDEBAND_OPENLISTENING_MODE][4],
          ip2_hs_wb_ol_4,
          sizeof(ip2_hs_wb_ol_4));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_WIDEBAND_OPENLISTENING_MODE][5],
          ip2_hs_wb_ol_5,
          sizeof(ip2_hs_wb_ol_5));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_WIDEBAND_OPENLISTENING_MODE][6],
          ip2_hs_wb_ol_6,
          sizeof(ip2_hs_wb_ol_6));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HANDSET_WIDEBAND_OPENLISTENING_MODE][7],
          ip2_hs_wb_ol_7,
          sizeof(ip2_hs_wb_ol_7));


   /* check headset-wb-openlistening parameter set for correct size */
   if( (sizeof(ip2_he_wb_ol_0) != voice_par_size) ||
       (sizeof(ip2_he_wb_ol_1) != voice_par_size) ||
       (sizeof(ip2_he_wb_ol_2) != voice_par_size) ||
       (sizeof(ip2_he_wb_ol_3) != voice_par_size) ||
       (sizeof(ip2_he_wb_ol_4) != voice_par_size) ||
       (sizeof(ip2_he_wb_ol_5) != voice_par_size) ||
       (sizeof(ip2_he_wb_ol_6) != voice_par_size) ||
       (sizeof(ip2_he_wb_ol_7) != voice_par_size) )
   {
      TRACE(VMMC,DBG_LEVEL_HIGH, \
         ("%s - Invalid size of headset-wb-openlistening parameter set in ipp_voice_params.h\n",
          __FUNCTION__));
      return IFX_ERROR;
   }
    /* set parameters for headset mode, wideband,
       open listening, volumes 1 ... 8 */
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_WIDEBAND_OPENLISTENING_MODE][0],
          ip2_he_wb_ol_0,
          sizeof(ip2_he_wb_ol_0));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_WIDEBAND_OPENLISTENING_MODE][1],
          ip2_he_wb_ol_1,
          sizeof(ip2_he_wb_ol_1));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_WIDEBAND_OPENLISTENING_MODE][2],
          ip2_he_wb_ol_2,
          sizeof(ip2_he_wb_ol_2));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_WIDEBAND_OPENLISTENING_MODE][3],
          ip2_he_wb_ol_3,
          sizeof(ip2_he_wb_ol_3));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_WIDEBAND_OPENLISTENING_MODE][4],
          ip2_he_wb_ol_4,
          sizeof(ip2_he_wb_ol_4));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_WIDEBAND_OPENLISTENING_MODE][5],
          ip2_he_wb_ol_5,
          sizeof(ip2_he_wb_ol_5));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_WIDEBAND_OPENLISTENING_MODE][6],
          ip2_he_wb_ol_6,
          sizeof(ip2_he_wb_ol_6));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_HEADSET_WIDEBAND_OPENLISTENING_MODE][7],
          ip2_he_wb_ol_7,
          sizeof(ip2_he_wb_ol_7));

   /* check handset icva parameter set for correct size */
   if( (sizeof(ip2_hs_nb_icva_0) != voice_par_size) ||
       (sizeof(ip2_hs_nb_icva_1) != voice_par_size) ||
       (sizeof(ip2_hs_nb_icva_2) != voice_par_size) ||
       (sizeof(ip2_hs_nb_icva_3) != voice_par_size) ||
       (sizeof(ip2_hs_nb_icva_4) != voice_par_size) ||
       (sizeof(ip2_hs_nb_icva_5) != voice_par_size) ||
       (sizeof(ip2_hs_nb_icva_6) != voice_par_size) ||
       (sizeof(ip2_hs_nb_icva_7) != voice_par_size) )
   {
      TRACE(VMMC,DBG_LEVEL_HIGH, \
         ("%s - Invalid size of handset OHVA parameter set in ipp_voice_params.h\n", __FUNCTION__));
      return IFX_ERROR;
   }
   /* set parameters for handset mode, narrowband, volumes 1 ... 8 */
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HANDSET_MODE][0],
          ip2_hs_nb_icva_0,
          sizeof(ip2_hs_nb_icva_0));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HANDSET_MODE][1],
          ip2_hs_nb_icva_1,
          sizeof(ip2_hs_nb_icva_1));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HANDSET_MODE][2],
          ip2_hs_nb_icva_2,
          sizeof(ip2_hs_nb_icva_2));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HANDSET_MODE][3],
          ip2_hs_nb_icva_3,
          sizeof(ip2_hs_nb_icva_3));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HANDSET_MODE][4],
          ip2_hs_nb_icva_4,
          sizeof(ip2_hs_nb_icva_4));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HANDSET_MODE][5],
          ip2_hs_nb_icva_5,
          sizeof(ip2_hs_nb_icva_5));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HANDSET_MODE][6],
          ip2_hs_nb_icva_6,
          sizeof(ip2_hs_nb_icva_6));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HANDSET_MODE][7],
          ip2_hs_nb_icva_7,
          sizeof(ip2_hs_nb_icva_7));

   /* check handset wideband ohva parameter set for correct size */
   if( (sizeof(ip2_hs_wb_icva_0) != voice_par_size) ||
       (sizeof(ip2_hs_wb_icva_1) != voice_par_size) ||
       (sizeof(ip2_hs_wb_icva_2) != voice_par_size) ||
       (sizeof(ip2_hs_wb_icva_3) != voice_par_size) ||
       (sizeof(ip2_hs_wb_icva_4) != voice_par_size) ||
       (sizeof(ip2_hs_wb_icva_5) != voice_par_size) ||
       (sizeof(ip2_hs_wb_icva_6) != voice_par_size) ||
       (sizeof(ip2_hs_wb_icva_7) != voice_par_size) )
   {
      TRACE(VMMC,DBG_LEVEL_HIGH, \
         ("%s - Invalid size of handset WB OHVA parameter set in ipp_voice_params.h\n", __FUNCTION__));
      return IFX_ERROR;
   }
   /* set parameters for handset mode, narrowband, volumes 1 ... 8 */
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HANDSET_WIDEBAND_MODE][0],
          ip2_hs_wb_icva_0,
          sizeof(ip2_hs_wb_icva_0));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HANDSET_WIDEBAND_MODE][1],
          ip2_hs_wb_icva_1,
          sizeof(ip2_hs_wb_icva_1));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HANDSET_WIDEBAND_MODE][2],
          ip2_hs_wb_icva_2,
          sizeof(ip2_hs_wb_icva_2));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HANDSET_WIDEBAND_MODE][3],
          ip2_hs_wb_icva_3,
          sizeof(ip2_hs_wb_icva_3));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HANDSET_WIDEBAND_MODE][4],
          ip2_hs_wb_icva_4,
          sizeof(ip2_hs_wb_icva_4));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HANDSET_WIDEBAND_MODE][5],
          ip2_hs_wb_icva_5,
          sizeof(ip2_hs_wb_icva_5));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HANDSET_WIDEBAND_MODE][6],
          ip2_hs_wb_icva_6,
          sizeof(ip2_hs_wb_icva_6));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HANDSET_WIDEBAND_MODE][7],
          ip2_hs_wb_icva_7,
          sizeof(ip2_hs_wb_icva_7));


  /* check headset icva parameter set for correct size */
   if( (sizeof(ip2_he_nb_icva_0) != voice_par_size) ||
       (sizeof(ip2_he_nb_icva_1) != voice_par_size) ||
       (sizeof(ip2_he_nb_icva_2) != voice_par_size) ||
       (sizeof(ip2_he_nb_icva_3) != voice_par_size) ||
       (sizeof(ip2_he_nb_icva_4) != voice_par_size) ||
       (sizeof(ip2_he_nb_icva_5) != voice_par_size) ||
       (sizeof(ip2_he_nb_icva_6) != voice_par_size) ||
       (sizeof(ip2_he_nb_icva_7) != voice_par_size) )
   {
      TRACE(VMMC,DBG_LEVEL_HIGH, \
         ("%s - Invalid size of headset OHVA parameter set in ipp_voice_params.h\n", __FUNCTION__));
      return IFX_ERROR;
   }
   /* set parameters for headset mode icva, narrowband, volumes 1 ... 8 */
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HEADSET_MODE][0],
          ip2_he_nb_icva_0,
          sizeof(ip2_he_nb_icva_0));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HEADSET_MODE][1],
          ip2_he_nb_icva_1,
          sizeof(ip2_he_nb_icva_1));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HEADSET_MODE][2],
          ip2_he_nb_icva_2,
          sizeof(ip2_he_nb_icva_2));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HEADSET_MODE][3],
          ip2_he_nb_icva_3,
          sizeof(ip2_he_nb_icva_3));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HEADSET_MODE][4],
          ip2_he_nb_icva_4,
          sizeof(ip2_he_nb_icva_4));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HEADSET_MODE][5],
          ip2_he_nb_icva_5,
          sizeof(ip2_he_nb_icva_5));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HEADSET_MODE][6],
          ip2_he_nb_icva_6,
          sizeof(ip2_he_nb_icva_6));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HEADSET_MODE][7],
          ip2_he_nb_icva_7,
          sizeof(ip2_he_nb_icva_7));


   /* check headset wideband ohva parameter set for correct size */
   if( (sizeof(ip2_he_wb_icva_0) != voice_par_size) ||
       (sizeof(ip2_he_wb_icva_1) != voice_par_size) ||
       (sizeof(ip2_he_wb_icva_2) != voice_par_size) ||
       (sizeof(ip2_he_wb_icva_3) != voice_par_size) ||
       (sizeof(ip2_he_wb_icva_4) != voice_par_size) ||
       (sizeof(ip2_he_wb_icva_5) != voice_par_size) ||
       (sizeof(ip2_he_wb_icva_6) != voice_par_size) ||
       (sizeof(ip2_he_wb_icva_7) != voice_par_size) )
   {
      TRACE(VMMC,DBG_LEVEL_HIGH, \
         ("%s - Invalid size of headset WB OHVA parameter set in ipp_voice_params.h\n", __FUNCTION__));
      return IFX_ERROR;
   }
   /* set parameters for headset mode, wideband, volumes 1 ... 8 */
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HEADSET_WIDEBAND_MODE][0],
          ip2_he_wb_icva_0,
          sizeof(ip2_he_wb_icva_0));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HEADSET_WIDEBAND_MODE][1],
          ip2_he_wb_icva_1,
          sizeof(ip2_he_wb_icva_1));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HEADSET_WIDEBAND_MODE][2],
          ip2_he_wb_icva_2,
          sizeof(ip2_he_wb_icva_2));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HEADSET_WIDEBAND_MODE][3],
          ip2_he_wb_icva_3,
          sizeof(ip2_he_wb_icva_3));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HEADSET_WIDEBAND_MODE][4],
          ip2_he_wb_icva_4,
          sizeof(ip2_he_wb_icva_4));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HEADSET_WIDEBAND_MODE][5],
          ip2_he_wb_icva_5,
          sizeof(ip2_he_wb_icva_5));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HEADSET_WIDEBAND_MODE][6],
          ip2_he_wb_icva_6,
          sizeof(ip2_he_wb_icva_6));
   memcpy(&voice_params.opmode_params[VMMC_IPPHONE_ICVA_HEADSET_WIDEBAND_MODE][7],
          ip2_he_wb_icva_7,
          sizeof(ip2_he_wb_icva_7));


   return IFX_SUCCESS;

}

/**
   Store Voice Parameter Set to Operation Mode Parameter Array.
\param
   pData       - pointer to bbd payload (set of fw messages).
\param
   size        - size of the BBD_AUDIO_OPMODE block.
\param
   row_index   - row index for 2 dimensional array access (opmode).
\param
   col_index   - column index for 2 dimensional array access (volume).
\return
   IFX_SUCCESS or IFX_ERROR
\remarks
   It is assumed that pData points to the beginning of a complete
   voice parameter set as described in VMMC_OpModeCtrl_t structure.
*/
static
IFX_int32_t vmmc_audio_StoreOpModeParameters( IFX_uint8_t *pData,
                                              IFX_uint32_t size,
                                              IFX_int32_t  row_index,
                                              IFX_int32_t  col_index )
{
   IFX_int32_t        err = IFX_ERROR;
   VMMC_OpModeCtrl_t  *opm_param_ptr;
   IFX_uint8_t        *pData_tmp=pData;

   TRACE(VMMC,DBG_LEVEL_LOW, ("%s(size = %u)\n", __FUNCTION__, size));

   if( row_index < VMMC_IPPHONE_IDLE_MODE ||
       row_index >= VMMC_IPPHONE_NR_OPMODES )
      return err;

   if( col_index < VMMC_IPPHONE_VOL_UNKNOWN ||
       col_index >= VMMC_IPPHONE_NR_VOL_LEVELS )
      return err;

   if( pData == NULL )
      return err;

   if(row_index == VMMC_IPPHONE_IDLE_MODE)
      opm_param_ptr = &(voice_params.idle_params);
   else
      opm_param_ptr = &(voice_params.opmode_params[row_index][col_index]);


   if(size==BBD_VMMC_AUDIO_OPMODE_DEFAULT_SIZE)
   {

      memcpy((IFX_uint8_t*)&(opm_param_ptr->valid), pData_tmp, sizeof(IFX_uint32_t));
      pData_tmp+=sizeof(IFX_uint32_t);
      memcpy((IFX_uint8_t*)&(opm_param_ptr->volume_level_change), pData_tmp, sizeof(IFX_uint32_t));
      pData_tmp+=sizeof(IFX_uint32_t);
      memcpy((IFX_uint8_t*)&(opm_param_ptr->params), pData_tmp, sizeof(VMMC_OperatingModeCfg_t));
      err = IFX_SUCCESS;

      if(row_index == VMMC_IPPHONE_IDLE_MODE)
      {
         TRACE(VMMC,DBG_LEVEL_LOW,
               ("Stored Parameter set for Idle Mode\n"));
      }
      else
      {
         TRACE(VMMC,DBG_LEVEL_LOW,
               ("Stored Parameter set for Opmode %d and volume %d\n",
               row_index, col_index));
      }
   }
   else
   {
      TRACE(VMMC,DBG_LEVEL_HIGH, ("Variable BBD Block sizes not supported !!!\n"));
   }

   return err;
}


#if 0 /* not used ? */
/**
   Get Voice Parameter Set from Operation Mode Parameter Array
   according to the passed row and column.

\param
   row_index   - row index for 2 dimensional array access (opmode).
\param
   col_index   - column index for 2 dimensional array access (volume).
\param
   pData       - pointer to opmode parameter set retrieved from
                 opmode_params array.
\return
   IFX_SUCCESS or IFX_ERROR in case of invalid parameters
\remarks
*/
IFX_int32_t VMMC_GetOpModeParameters( IFX_int32_t       row_index,
                                      IFX_int32_t       col_index,
                                      VMMC_OpModeCtrl_t *opm_param_ptr )
{
   TRACE(VMMC,DBG_LEVEL_LOW, ("%s\n", __FUNCTION__));

   if( row_index < VMMC_IPPHONE_IDLE_MODE ||
       row_index >= VMMC_IPPHONE_NR_OPMODES )
   {
      TRACE(VMMC,DBG_LEVEL_HIGH, ("%s - invalid parameter row_index = %d\n",
            __FUNCTION__, row_index));
      return IFX_ERROR;
   }

   if( col_index < VMMC_IPPHONE_VOL_UNKNOWN ||
       col_index >= VMMC_IPPHONE_NR_VOL_LEVELS )
   {
      TRACE(VMMC,DBG_LEVEL_HIGH, ("%s - invalid parameter col_index = %d\n",
            __FUNCTION__, col_index));
      return IFX_ERROR;
   }

   if( opm_param_ptr == NULL )
   {
      TRACE(VMMC,DBG_LEVEL_HIGH, ("%s - invalid parameter opm_param_ptr = NULL\n",
            __FUNCTION__));
      return IFX_ERROR;
   }

   /* idle mode parameters are retrieved seperately because they are not
      comprised in the opmode_params array */
   if( row_index == VMMC_IPPHONE_IDLE_MODE )
   {
      *opm_param_ptr = voice_params.idle_params;
      TRACE(VMMC,DBG_LEVEL_LOW,
            ("%s - retrieved Idle_parameter\n", __FUNCTION__));
   }
   else
      if(col_index != VMMC_IPPHONE_VOL_UNKNOWN)
      {
         /* unknown volume is only allowed with IDLE request */
         *opm_param_ptr = voice_params.opmode_params[row_index][col_index];
         TRACE(VMMC,DBG_LEVEL_LOW, ("%s - retrieved voice_parameter[%d][%d]\n",
              __FUNCTION__, row_index, col_index));
      }
      else
         return IFX_ERROR;

   return IFX_SUCCESS;
}
#endif


/**
   Sends Voice Parameter configuration messages to firmware according to
   selected mask.

\param
   pCh          - handle to channel
\param
   opm_params   - pointer to operation mode parameters for requested mode
                  and volume.
\param
   voice_level_mask   - voice change mask selector (use voice change mask(IFX_TRUE)
                        or opmode change mask(IFX_FALSE) to control
                        sending of FW messages).
\param
   req_mute_state     - mute selector.
\param
   invert_order       - controls the order the messages audio channel control
                        and audio channel speakerphone are sent. In case of
                        IFX_TRUE the speakerphone message is sent before the
                        audio channel message.
\return
   IFX_SUCCESS or IFX_ERROR in case of invalid parameters
\remarks
*/
static
IFX_int32_t vmmc_audio_SendOpModeParamMessages( VMMC_CHANNEL      *pCh,
                                                VMMC_OpModeCtrl_t *opm_params,
                                                IFX_boolean_t     voice_level_mask,
                                                VMMC_IPPHONE_MUTE_STATE req_mute_state,
                                                IFX_boolean_t     invert_order )
{
   IFX_int32_t   err = IFX_ERROR;
   IFX_uint32_t  mask;
   IFX_uint32_t  bitpos;
   VMMC_DEVICE*  pDev = pCh->pParent;

   TRACE(VMMC,DBG_LEVEL_LOW, ("%s\n", __FUNCTION__));
#ifdef VMMC_HOST_DEBUG
   print_VMMC_OpModeCtrl(opm_params);
#endif

   if(opm_params == NULL || pCh == NULL )
      return err;

   if(invert_order!=IFX_TRUE && invert_order!=IFX_FALSE)
   {
      return err;
   }

   /* get mask that controls which messages have to be sent.
      In case of requested voice level change the voice message control
      word is chosen, otherwise operation mode change mask is used */
   if(voice_level_mask)
   {
      mask=(IFX_uint32_t)opm_params->volume_level_change;
   }
   else
   {
      mask=(IFX_uint32_t)opm_params->valid;
   }

   if(mask == 0x00000000)
   {
      TRACE(VMMC,DBG_LEVEL_HIGH, ("%s: Invalid mask passed 0x%04X\n", __FUNCTION__, mask));
#ifdef VMMC_HOST_DEBUG
      print_VMMC_OpModeCtrl(opm_params);
#endif
      return err;
   }

   bitpos=0x00000001;
   /* within this while loop all mask bits are checked for being set.
      In case of a set mask bit the related firmware message is sent. The
      procedure starts with the LSB and proceeds with the more significant bits  */
   while(bitpos!=0)  /* after 32 bitshifts bitpos is 0 */
   {
      TRACE(VMMC,DBG_LEVEL_LOW, ("%s: mask=0x%08X, bitpos=0x%08X\n",
                                  __FUNCTION__, mask, bitpos));
      /* if mask bit is set at current bitpos send assigned FW message */
      switch(mask&bitpos)
      {
         case EQU1_IIR_POS:
            TRACE(VMMC,DBG_LEVEL_LOW,
                  ("%s - send equ1_iir fw command\n", __FUNCTION__));
            err=CmdWrite( pDev, (IFX_uint32_t*) &(opm_params->params.equ1_iir),
                               sizeof(RES_EQU_IIR_COEF_t)-4);
            break;
         case EQU1_FIR_POS:
            TRACE(VMMC,DBG_LEVEL_LOW,
                  ("%s - send equ1_fir fw command\n", __FUNCTION__));
            err=CmdWrite( pDev, (IFX_uint32_t*) &(opm_params->params.equ1_fir),
                               sizeof(RES_EQU_FIR_COEF_t)-4);
            break;
         case EQU2_IIR_POS:
            TRACE(VMMC,DBG_LEVEL_LOW,
                  ("%s - send equ2_iir fw command\n", __FUNCTION__));
            err=CmdWrite( pDev, (IFX_uint32_t*) &(opm_params->params.equ2_iir),
                               sizeof(RES_EQU_IIR_COEF_t)-4);
            break;
         case EQU2_FIR_POS:
            TRACE(VMMC,DBG_LEVEL_LOW,
                  ("%s - send equ2_fir fw command\n", __FUNCTION__));
            err=CmdWrite( pDev, (IFX_uint32_t*) &(opm_params->params.equ2_fir),
                               sizeof(RES_EQU_FIR_COEF_t)-4);
            break;
         case EQU3_IIR_POS:
            TRACE(VMMC,DBG_LEVEL_LOW,
                  ("%s - send equ3_iir fw command\n", __FUNCTION__));
            err=CmdWrite( pDev, (IFX_uint32_t*) &(opm_params->params.equ3_iir),
                               sizeof(RES_EQU_IIR_COEF_t)-4);
            break;
         case EQU3_FIR_POS:
            TRACE(VMMC,DBG_LEVEL_LOW,
                  ("%s - send equ3_fir fw command\n", __FUNCTION__));
            err=CmdWrite( pDev, (IFX_uint32_t*) &(opm_params->params.equ3_fir),
                               sizeof(RES_EQU_FIR_COEF_t)-4);
            break;

         case EQU4_IIR_POS:
            TRACE(VMMC,DBG_LEVEL_LOW,
                  ("%s - send equ4_iir fw command\n", __FUNCTION__));
            err=CmdWrite( pDev, (IFX_uint32_t*) &(opm_params->params.equ4_iir),
                               sizeof(RES_EQU_IIR_COEF_t)-4);
            break;
         case EQU4_FIR_POS:
            TRACE(VMMC,DBG_LEVEL_LOW,
                  ("%s - send equ4_fir fw command\n", __FUNCTION__));
            err=CmdWrite( pDev, (IFX_uint32_t*) &(opm_params->params.equ4_fir),
                               sizeof(RES_EQU_FIR_COEF_t)-4);
            break;
         case AGC1_POS:
               TRACE(VMMC,DBG_LEVEL_LOW,
                     ("%s - send agc1 fw command\n", __FUNCTION__));
               err=CmdWrite( pDev, (IFX_uint32_t*) &(opm_params->params.agc1),
                             sizeof(RES_AGC_COEF_t)-4);
            break;
         case AGC2_POS:
               /* download AGC2 coefficients */
               TRACE(VMMC,DBG_LEVEL_LOW,
                     ("%s - send agc2 fw command\n", __FUNCTION__));
               err=CmdWrite( pDev, (IFX_uint32_t*) &(opm_params->params.agc2),
                             sizeof(RES_AGC_COEF_t)-4);
            break;
         case SPK_HDK_POS:
            TRACE(VMMC,DBG_LEVEL_LOW,
                  ("%s - send spk_hdk fw command\n", __FUNCTION__));
            err=CmdWrite( pDev, (IFX_uint32_t*) &(opm_params->params.spk_hdk),
                               sizeof(RES_SPK_HDX_COEF_t)-4);
            break;
         case AGCX_POS:
            TRACE(VMMC,DBG_LEVEL_LOW,
                  ("%s - send spk_agcx fw command\n", __FUNCTION__));
            err=CmdWrite( pDev, (IFX_uint32_t*) &(opm_params->params.spk_agcx),
                               sizeof(RES_SPK_AGCX_COEF_t)-4);
            break;
         case AGCR_POS:
            TRACE(VMMC,DBG_LEVEL_LOW,
                  ("%s - send spk_agcr fw command\n", __FUNCTION__));
            err=CmdWrite( pDev, (IFX_uint32_t*) &(opm_params->params.spk_agcr),
                               sizeof(AUDIO_SPK_AGCR_COEF_t)-4);
            break;
         case FDX_POS:
            TRACE(VMMC,DBG_LEVEL_LOW,
                  ("%s - send spk_fdx fw command\n", __FUNCTION__));
            err=CmdWrite( pDev, (IFX_uint32_t*) &(opm_params->params.spk_fdx),
                               sizeof(RES_SPK_FDX_COEF_t)-4);
            break;
         case WF_POS:
            TRACE(VMMC,DBG_LEVEL_LOW,
                  ("%s - send spk_wf fw command\n", __FUNCTION__));
            err=CmdWrite( pDev, (IFX_uint32_t*) &(opm_params->params.spk_wf),
                               sizeof(RES_SPK_WF_COEF_t)-4);
            break;
         case NR_POS:
            TRACE(VMMC,DBG_LEVEL_LOW,
                  ("%s - send spk_nr fw command\n", __FUNCTION__));
            err=CmdWrite( pDev, (IFX_uint32_t*) &(opm_params->params.spk_nr),
                               sizeof(RES_SPK_NR_COEF_t)-4);
            break;
         case NLP_POS:
            TRACE(VMMC,DBG_LEVEL_LOW,
                  ("%s - send spk_nlp fw command\n", __FUNCTION__));
            err=CmdWrite( pDev, (IFX_uint32_t*) &(opm_params->params.spk_nlp),
                               sizeof(RES_SPK_NLP_COEF_t)-4);
            break;

         case HEC_POS:
            TRACE(VMMC,DBG_LEVEL_LOW,
                  ("%s - send hec_coeff fw command\n", __FUNCTION__));
            err=CmdWrite( pDev, (IFX_uint32_t*) &(opm_params->params.spk_hec),
                               sizeof(RES_HEC_COEF_t)-4);
            break;

         case ACHAN_POS:
            if( invert_order == IFX_TRUE )
            {
               TRACE(VMMC,DBG_LEVEL_LOW, \
                     ("%s - invert_order\n", __FUNCTION__));
               /* if invert order is requested send the audio channel
                  speakerphone message before sending the audio channel msg */
               TRACE(VMMC,DBG_LEVEL_LOW, \
                     ("%s - send ad_spk fw command\n", __FUNCTION__));
               err=CmdWrite( pDev, (IFX_uint32_t*) &(opm_params->params.ad_spk),
                             sizeof(AUDIO_SPK_CTRL_t)-4);
            }


            if( req_mute_state == VMMC_IPPHONE_MUTED )
            {
               TRACE(VMMC,DBG_LEVEL_LOW, \
                     ("%s - set mute bit in audio_chan fw command\n", __FUNCTION__));
               /* set mute bit in audio channel command if requested */
               opm_params->params.ad_chan.MUTE = 1;
            }
            else
            {
               TRACE(VMMC,DBG_LEVEL_LOW, \
                     ("%s - clear mute bit in audio_chan fw command\n", __FUNCTION__));
               /* clear mute bit in audio channel command if requested */
               opm_params->params.ad_chan.MUTE = 0;
            }

            if(err == IFX_SUCCESS)
            {
               TRACE(VMMC,DBG_LEVEL_LOW,
                     ("%s - send audio_chan fw command\n", __FUNCTION__));
               err=CmdWrite( pDev,
                            (IFX_uint32_t*) &(opm_params->params.ad_chan),
                            sizeof(AUDIO_CHAN_t)-4);
            }
            break;

         case ASPK_POS:
            if( invert_order == IFX_FALSE )
            {
               TRACE(VMMC,DBG_LEVEL_LOW, \
                     ("%s -invert_order == IFX_FALSE\n", __FUNCTION__));
               TRACE(VMMC,DBG_LEVEL_LOW, \
                     ("%s - send ad_spk fw command\n", __FUNCTION__));
               err=CmdWrite( pDev, (IFX_uint32_t*) &(opm_params->params.ad_spk),
                             sizeof(AUDIO_SPK_CTRL_t)-4);
            }
            else
               /* skip sending of audio channel speakerphone message */
               TRACE(VMMC,DBG_LEVEL_LOW,
                     ("%s - skip sending of audio channel speakerphone message\n", __FUNCTION__));
            break;

         case 0x00000000:
            err = IFX_SUCCESS;
            break;

         default:
            if(mask&bitpos)
               TRACE(VMMC,DBG_LEVEL_HIGH,
                     ("%s - Invalid mask set: 0x%04uX\n", __FUNCTION__, mask&bitpos));
            break;

      }
      if( err != IFX_SUCCESS )
      {
         TRACE(VMMC,DBG_LEVEL_HIGH, \
               ("%s - Sending FW Message of cmd %u failed\n", \
               __FUNCTION__, mask&bitpos));
#ifdef VMMC_HOST_DEBUG
         print_VMMC_OpModeCtrl(opm_params);
#endif
         return err;
      }
      bitpos<<=1;
   }

   err = IFX_SUCCESS;

   return err;
}

#ifdef VMMC_HOST_DEBUG
static IFX_void_t print_VMMC_OpModeCtrl( VMMC_OpModeCtrl_t *params )
{
   IFX_uint32_t *data_ptr;
   IFX_uint32_t i;

   TRACE(VMMC,DBG_LEVEL_HIGH,("params->valid_mode = 0x%08X\n", params->valid));
   TRACE(VMMC,DBG_LEVEL_HIGH,("params->valid_vol  = 0x%08X\n", \
         params->volume_level_change));
   TRACE(VMMC,DBG_LEVEL_HIGH,("data\n"));
   data_ptr = (IFX_uint32_t*)&(params->params);
   for(i=0;i<(sizeof(VMMC_OperatingModeCfg_t)>>2); i++)
   {
      TRACE(VMMC,DBG_LEVEL_HIGH,("0x%08X\n", *(data_ptr+i)));
   }
   return;
}
#endif

/**
   Sends Set of Voice Parameter configuration messages to firmware.

\param
   pCh        - handle to channel
\param
   opm_params   - pointer to operation mode parameters for requested mode
                  and volume.
\param
   voice_level_change   - voice change only selector.
\param
   req_mute_state       - mute selector.
\return
   IFX_SUCCESS or IFX_ERROR in case of invalid parameters
\remarks
*/
static
IFX_int32_t vmmc_audio_SendOpModeParamSet( VMMC_CHANNEL      *pCh,
                                           VMMC_OpModeCtrl_t *opm_params,
                                           IFX_boolean_t     voice_level_change,
                                           VMMC_IPPHONE_MUTE_STATE req_mute_state,
                                           IFX_boolean_t           invert_order)
{
   IFX_int32_t   err = IFX_ERROR;

   TRACE(VMMC,DBG_LEVEL_LOW, ("%s\n", __FUNCTION__));

   /* send the passed voice parameter set to FW */
   err = vmmc_audio_SendOpModeParamMessages( pCh, opm_params, voice_level_change,
                                             req_mute_state, invert_order );
   return err;

}


/**
   Stores passed Audio Coefficients in Operation Mode Parameter array after
   determination of row and column from tag.
\param
   pCh          - handle to channel
\param
   bbd_opmode   - handle to bbd audio operation mode block
\return
   IFX_SUCCESS or IFX_ERROR
\remarks
   - It is assumed that the given bbd block is valid and that the data
     representation in the bbd block is according to BBD specification.
*/
IFX_int32_t VMMC_AUDIO_BBD_AudioOpmode(VMMC_CHANNEL *pCh,
                                       bbd_block_t  *bbd_opmode)
{
   IFX_int32_t  err = IFX_ERROR;
   VMMC_IPPHONE_OPERATION_MODE  new_op_mode;
   VMMC_IPPHONE_VOLUME_LEVEL    new_volume;

   TRACE(VMMC,DBG_LEVEL_LOW, ("%s\n", __FUNCTION__));

   /* get Requested operation mode and volume from passed bbd tag to access
      operation mode parameter array. */
   if( ((err = vmmc_audio_Get_Tag_Operation_Mode(bbd_opmode->tag,
                                                 &new_op_mode)) != IFX_SUCCESS) ||
       ((err = vmmc_audio_Get_Tag_Volume(bbd_opmode->tag,
                                         &new_volume)) != IFX_SUCCESS) )
   {
      TRACE(VMMC,DBG_LEVEL_HIGH,
            ("%s - retrieval of requested mode/volume failed\n", __FUNCTION__));
      return err;
   }

   /* BBD_AUDIO_OPMODE Block incl. Parameters
      -> Store passed Parameters to Operation Mode Parameter Array */
   err = vmmc_audio_StoreOpModeParameters( bbd_opmode->pData,
                                           bbd_opmode->size,
                                           (IFX_int32_t) new_op_mode,
                                           (IFX_int32_t) new_volume);

   return err;
}


/**
   Initialize array of ring parameters.
\param
   None.
\return
   IFX_SUCCESS or IFX_ERROR
\remarks
   It is assumed that pData points to the beginning of a complete
   voice parameter set as described in VMMC_OpModeCtrl_t structure.
*/
IFX_int32_t VMMC_AUDIO_InitRingParameters( IFX_void_t )
{
   IFX_int32_t ring_par_size = sizeof(ring_param_set.ring_params[0]);

   TRACE(VMMC,DBG_LEVEL_LOW, ("%s\n", __FUNCTION__));


   memset((IFX_void_t*) &ring_param_set, 0, sizeof(VMMC_RingParamSet_t));

   if( (sizeof(ip2_rng_vol_0) != ring_par_size) ||
       (sizeof(ip2_rng_vol_1) != ring_par_size) ||
       (sizeof(ip2_rng_vol_2) != ring_par_size) ||
       (sizeof(ip2_rng_vol_3) != ring_par_size) ||
       (sizeof(ip2_rng_vol_4) != ring_par_size) ||
       (sizeof(ip2_rng_vol_5) != ring_par_size) ||
       (sizeof(ip2_rng_vol_6) != ring_par_size) ||
       (sizeof(ip2_rng_vol_7) != ring_par_size) )
   {
      TRACE(VMMC,DBG_LEVEL_HIGH, \
         ("%s - Invalid size of ring parameter set in ipp_voice_params.h\n", __FUNCTION__));
      return IFX_ERROR;
   }

   /* parameters fring volumes 1 ... 8 */
   memcpy(&ring_param_set.ring_params[0], ip2_rng_vol_0,
          sizeof(ip2_rng_vol_0));
   memcpy(&ring_param_set.ring_params[1], ip2_rng_vol_1,
          sizeof(ip2_rng_vol_1));
   memcpy(&ring_param_set.ring_params[2], ip2_rng_vol_2,
          sizeof(ip2_rng_vol_2));
   memcpy(&ring_param_set.ring_params[3], ip2_rng_vol_3,
          sizeof(ip2_rng_vol_3));
   memcpy(&ring_param_set.ring_params[4], ip2_rng_vol_4,
          sizeof(ip2_rng_vol_4));
   memcpy(&ring_param_set.ring_params[5], ip2_rng_vol_5,
          sizeof(ip2_rng_vol_5));
   memcpy(&ring_param_set.ring_params[6], ip2_rng_vol_6,
          sizeof(ip2_rng_vol_6));
   memcpy(&ring_param_set.ring_params[7], ip2_rng_vol_7,
          sizeof(ip2_rng_vol_7));

   return IFX_SUCCESS;
}



/**
   Gets requested volume from the passed Ring BBD tag.
\param
   tag  - 16 bit bbd subblock tag that comprises the ring volume.
\param
   vol  - pointer volume information retrieved from tag.
\return
   IFX_SUCCESS or IFX_ERROR
\remarks
*/
static IFX_int32_t vmmc_audio_Get_Ring_Tag_Volume( IFX_uint16_t tag,
                                                   VMMC_IPPHONE_VOLUME_LEVEL *vol)
{
   TRACE(VMMC,DBG_LEVEL_LOW, ("%s\n", __FUNCTION__));

   if( tag < BBD_VMMC_IPPHONE_RINGTONE_VOL1 ||
       tag > BBD_VMMC_IPPHONE_RINGTONE_VOL8 )
   {
      TRACE(VMMC,DBG_LEVEL_HIGH,
            ("%s - Invalid Ringtone Tag 0x%04X\n", __FUNCTION__, tag));
      TRACE(VMMC,DBG_LEVEL_HIGH, \
            ("%s - has to be between 0x%04X and 0x%04X\n", \
            __FUNCTION__, BBD_VMMC_IPPHONE_RINGTONE_VOL1, BBD_VMMC_IPPHONE_RINGTONE_VOL8));
      return IFX_ERROR;
   }

   switch(tag & 0xF)
   {
      case 1:
         *vol = VMMC_IPPHONE_VOL1;
         break;
      case 2:
         *vol = VMMC_IPPHONE_VOL2;
         break;
      case 3:
         *vol = VMMC_IPPHONE_VOL3;
         break;
      case 4:
         *vol = VMMC_IPPHONE_VOL4;
         break;
      case 5:
         *vol = VMMC_IPPHONE_VOL5;
         break;
      case 6:
         *vol = VMMC_IPPHONE_VOL6;
         break;
      case 7:
         *vol = VMMC_IPPHONE_VOL7;
         break;
      case 8:
         *vol = VMMC_IPPHONE_VOL8;
         break;
      default:
         return IFX_ERROR;
         break;

   }
   return IFX_SUCCESS;;

}





/**
   Sends Ring Parameter configuration messages to firmware according to
   selected mask.

\param
   pCh          - handle to channel
\param
   opm_params   - pointer to ring parameters for requested volume.
\return
   IFX_SUCCESS or IFX_ERROR in case of invalid parameters
\remarks
*/
static IFX_int32_t vmmc_audio_SendRingParamMessages( VMMC_CHANNEL      *pCh,
                                                     VMMC_RingVolCfg_t *params)
{
   IFX_int32_t   err = IFX_ERROR;
   VMMC_DEVICE*  pDev = pCh->pParent;

   TRACE(VMMC,DBG_LEVEL_LOW, ("%s\n", __FUNCTION__));


   if(params == NULL || pCh == NULL )
      return err;

   TRACE(VMMC,DBG_LEVEL_LOW,
               ("%s - send utg coefficients fw command\n", __FUNCTION__));
   err=CmdWrite( pDev,
                 (IFX_uint32_t*) &(params->utg_coeff),
                 sizeof(RES_UTG_COEF_t)-4);

   if( err == IFX_SUCCESS )
   {
      TRACE(VMMC,DBG_LEVEL_LOW,
            ("%s - send audio channel fw command\n", __FUNCTION__));
      err=CmdWrite( pDev,
                    (IFX_uint32_t*) &(params->ad_chan),
                    sizeof(AUDIO_CHAN_t)-4);

      if( err == IFX_SUCCESS )
      {
         TRACE(VMMC,DBG_LEVEL_LOW,
               ("%s - send utg ctrl fw command\n", __FUNCTION__));
         err=CmdWrite( pDev,
                       (IFX_uint32_t*) &(params->ad_utg),
                       sizeof(AUDIO_UTG_CTRL_t)-4);
      }
   }

   return err;
}





/**
   Sends Set of Ring Parameter configuration messages to firmware.

\param
   pCh        - handle to channel
\param
   opm_params   - pointer to operation mode parameters for requested mode
                  and volume.
\param
   voice_level_change   - voice change only selector.
\param
   req_mute_state       - mute selector.
\return
   IFX_SUCCESS or IFX_ERROR in case of invalid parameters
\remarks
*/
static IFX_int32_t vmmc_audio_SendRingParamSet( VMMC_CHANNEL      *pCh,
                                                VMMC_RingVolCfg_t *params )
{
   IFX_int32_t  err = IFX_ERROR;


   TRACE(VMMC,DBG_LEVEL_LOW, ("%s\n", __FUNCTION__));

   /* send the passed ring parameter set to FW */
   err = vmmc_audio_SendRingParamMessages( pCh, params );
   return err;

}



/**
   Store Ring Parameter Set Ring Parameter Array.
\param
   pData       - pointer to bbd payload (set of fw messages).
\param
   size        - size of the BBD_AUDIO_RING block.
\param
   col_index   - index for Ring Parameter array access (volume).
\return
   IFX_SUCCESS or IFX_ERROR
\remarks
   It is assumed that pData points to the beginning of a complete
   ring parameter set as described in VMMC_RingVolCfg_t structure.
*/
IFX_int32_t vmmc_audio_StoreRingParameters( IFX_uint8_t *pData,
                                            IFX_uint32_t size,
                                            IFX_int32_t  col_index )
{
   VMMC_RingVolCfg_t  *ring_param_ptr;
   IFX_uint8_t        *pData_tmp=pData;

   TRACE(VMMC,DBG_LEVEL_LOW,
         ("%s(size = %u, index=%d)\n", __FUNCTION__, size, col_index));

   if( col_index < VMMC_IPPHONE_VOL1 ||
       col_index >= VMMC_IPPHONE_NR_VOL_LEVELS )
      return IFX_ERROR;

   if( pData == NULL )
      return IFX_ERROR;

   ring_param_ptr = &(ring_param_set.ring_params[col_index]);
   memcpy((IFX_uint8_t*)ring_param_ptr, pData_tmp, sizeof(VMMC_RingVolCfg_t));

   return IFX_SUCCESS;
}



/**
   Stores passed Ring Coefficients in Ring Parameter array after
   determination of column from tag.
\param
   pCh          - handle to channel
\param
   bbd_ring     - handle to bbd ring block
\return
   IFX_SUCCESS or IFX_ERROR
\remarks
   - It is assumed that the given bbd block is valid and that the data
     representation in the bbd block is according to BBD specification.
*/
IFX_int32_t VMMC_AUDIO_BBD_AudioRing(VMMC_CHANNEL *pCh,
                                     bbd_block_t  *bbd_ring)
{
   IFX_int32_t  err = IFX_ERROR;
   VMMC_IPPHONE_VOLUME_LEVEL req_vol;

   TRACE(VMMC,DBG_LEVEL_LOW, ("%s \n", __FUNCTION__));

   if( (err = vmmc_audio_Get_Ring_Tag_Volume( bbd_ring->tag, &req_vol)) != IFX_SUCCESS )
   {
      TRACE(VMMC,DBG_LEVEL_HIGH,
            ("%s - Getting volume from Ring Tag failed \n", __FUNCTION__));
      return err;
   }

   /* BBD_AUDIO_RING Block incl. Parameters
      -> Store passed Parameters to Ring Mode Parameter Array */
   err = vmmc_audio_StoreRingParameters( bbd_ring->pData,
                                         bbd_ring->size,
                                         (IFX_int32_t) req_vol);
   if(err)
       TRACE(VMMC,DBG_LEVEL_HIGH, \
             ("%s - vmmc_audio_StoreRingParameters failed\n", __FUNCTION__));

   return err;
}
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO) */



/******* +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ ************/


/**
   Configures the Audio Channel to operate in wideband or narrowband mode

\param
   pCh    - Pointer to the VMMC channel structure.
\param
   action - action to be executed (set or check)
\param
   mode   - requested operation mode (16kHz or 8 kHz)

\return
   If action is SM_SET: IFX_SUCCESS or IFX_ERROR.
   If action is SM_CHECK: IFX_TRUE when module would do a switch or IFX_FALSE
                          if nothing needs to be done.
\remarks
   If necessary, Audio Channel is disabled before configuration.
*******************************************************************************/
IFX_int32_t VMMC_AUDIO_SamplingMode( VMMC_CHANNEL *pCh,
                                     SM_ACTION action,
                                     OPMODE_SMPL smpl_mode )
{
   VMMC_DEVICE                 *pDev;
   AUDIO_CHAN_t                *pAudio;
   IFX_boolean_t               change_isr = IFX_FALSE;
   IFX_return_t                ret        = IFX_SUCCESS;
   VMMC_IPPHONE_OPERATION_MODE current_opmode;
   IFX_uint32_t                opmode = 0;
   IFX_int32_t                 room_type = -1;

   if( pCh == IFX_NULL )
   {
      TRACE(VMMC, DBG_LEVEL_HIGH, ("%s - NULL Pointer passed\n", __FUNCTION__));
      return IFX_ERROR;
   }
   pDev = pCh->pParent;

   pAudio = &(pDev->pChannel[0].pAUDIO->audio_ch);
   current_opmode = pDev->pChannel[0].pAUDIO->current_opmode;

   /* check if passed FW module already operates in requested mode */
   if( ((smpl_mode == WB_16_KHZ) && (pAudio->ISR == 0)) ||
       ((smpl_mode == NB_8_KHZ)  && (pAudio->ISR == 1)) )
      change_isr = IFX_TRUE;

   /* If action is check return if module needs a switch */
   if (action != SM_SET)
   {
      return change_isr;
   }

   if( change_isr )
   {
      /* get new audio operation mode dep. on current operation mode and
         requested sampling rate (additionally room_type is returned
         in case of handsfree mode) */
      opmode = vmmc_audio_DetermineOperationMode( current_opmode,
                                                  current_opmode,
                                                  smpl_mode,
                                                  (IFX_int32_t*) &room_type );

      if(opmode!=IFX_ERROR)
      {
         /* request audio mode change */
         TRACE(VMMC, DBG_LEVEL_LOW, \
              ("current_opmode=%d -> new mode=%u\n", current_opmode, opmode));

         if( opmode == VMMC_IPPHONE_IDLE_MODE )
            ret = vmmc_audio_DisableChannel( &(pDev->pChannel[0]) );
         else
            ret = vmmc_audio_OpmodeConfigure( &(pDev->pChannel[0]), opmode, smpl_mode );

         if( ret != IFX_SUCCESS )
         {
            TRACE(VMMC, DBG_LEVEL_HIGH, \
                 ("%s - failed to set audio mode\n", __FUNCTION__));
         }
         else
         {
            TRACE(VMMC, DBG_LEVEL_LOW, \
                  ("%s - audio parameters successfully set\n", \
                  __FUNCTION__));

            /* check if room type settings have to be adapted in handsfree mode */
            if( room_type != -1 )
            {
               if( (ret=VMMC_AUDIO_Room_Set( &(pDev->pChannel[0]),
                                             (VMMC_IPPHONE_HF_ROOM_TYPE)room_type)) != IFX_SUCCESS )
               {
                  TRACE(VMMC, DBG_LEVEL_HIGH, \
                       ("%s - VMMC_AUDIO_Room_Set() failed\n", __FUNCTION__));
               }
            }
         }
      }
      else   /* opmode==IFX_ERROR */
      {
         ret = IFX_ERROR;
      }
   }
   else  /* change_isr == IFX_FALSE */
      TRACE(VMMC, DBG_LEVEL_LOW, ("Nothing to be done for Audio Channel\n"));

   return ret;
}


/**
   Registers all audio related functions

\param
   pAudio  - pointer to driver context audio channel structure

\return
   void
\remark
*******************************************************************************/
void VMMC_AUDIO_Func_Register(IFX_TAPI_DRV_CTX_AUDIO_t *pAudio)
{
   TRACE(VMMC,DBG_LEVEL_LOW, ("%s \n", __FUNCTION__));


   /* Fill the function pointers of audio module */
   pAudio->Volume_Set             = VMMC_TAPI_LL_AUDIO_Volume_Set;
   pAudio->Mode_Set               = VMMC_TAPI_LL_AUDIO_Mode_Set;
   pAudio->Room_Set               = VMMC_TAPI_LL_AUDIO_Room_Set;
   pAudio->Mute_Set               = VMMC_TAPI_LL_AUDIO_Mute_Set;
   pAudio->Ring_Start             = VMMC_TAPI_LL_AUDIO_Ring_Start;
   /*pAudio->Ring_Start             = VMMC_TAPI_LL_SIG_UTG_AUDIO_Play;*/
   pAudio->Ring_Stop              = VMMC_TAPI_LL_AUDIO_Ring_Stop;
   pAudio->Ring_Volume_Set        = VMMC_TAPI_LL_AUDIO_Ring_Volume_Set;
   pAudio->Incall_Anouncement     = VMMC_TAPI_LL_AUDIO_ICA;
   pAudio->AFE_Cfg_Set            = VMMC_TAPI_LL_AUDIO_AFE_Cfg_Set;
   pAudio->Test_Mode_Set          = VMMC_TAPI_LL_AUDIO_Test_Mode_Set;
}

#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO) */
