/*
     This file is part of GNUnet.
     Copyright (C) 2010-2015 GNUnet e.V.

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 3, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
     Boston, MA 02110-1301, USA.
*/

/**
 * @file transport/gnunet-service-transport_clients.c
 * @brief communication with clients (core service and monitors)
 * @author Christian Grothoff
 */
#include "platform.h"
#include "gnunet-service-transport_blacklist.h"
#include "gnunet-service-transport_clients.h"
#include "gnunet-service-transport_hello.h"
#include "gnunet-service-transport_neighbours.h"
#include "gnunet-service-transport_plugins.h"
#include "gnunet-service-transport_validation.h"
#include "gnunet-service-transport_manipulation.h"
#include "gnunet-service-transport.h"
#include "transport.h"


/**
 * How many messages can we have pending for a given client process
 * before we start to drop incoming messages?  We typically should
 * have only one client and so this would be the primary buffer for
  * messages, so the number should be chosen rather generously.
 *
 * The expectation here is that most of the time the queue is large
 * enough so that a drop is virtually never required.  Note that
 * this value must be about as large as 'TOTAL_MSGS' in the
 * 'test_transport_api_reliability.c', otherwise that testcase may
 * fail.
 */
#define MAX_PENDING (128 * 1024)


/**
 * Linked list of messages to be transmitted to the client.  Each
 * entry is followed by the actual message.
 */
struct ClientMessageQueueEntry
{
  /**
   * This is a doubly-linked list.
   */
  struct ClientMessageQueueEntry *next;

  /**
   * This is a doubly-linked list.
   */
  struct ClientMessageQueueEntry *prev;
};


/**
 * Client connected to the transport service.
 */
struct TransportClient
{

  /**
   * This is a doubly-linked list.
   */
  struct TransportClient *next;

  /**
   * This is a doubly-linked list.
   */
  struct TransportClient *prev;

  /**
   * Handle to the client.
   */
  struct GNUNET_SERVER_Client *client;

  /**
   * Linked list of messages yet to be transmitted to
   * the client.
   */
  struct ClientMessageQueueEntry *message_queue_head;

  /**
   * Tail of linked list of messages yet to be transmitted to the
   * client.
   */
  struct ClientMessageQueueEntry *message_queue_tail;

  /**
   * Current transmit request handle.
   */
  struct GNUNET_SERVER_TransmitHandle *th;

  /**
   * Length of the list of messages pending for this client.
   */
  unsigned int message_count;

  /**
   * Is this client interested in payload messages?
   */
  int send_payload;
};


/**
 * Context for address to string operations
 */
struct AddressToStringContext
{
  /**
   * This is a doubly-linked list.
   */
  struct AddressToStringContext *next;

  /**
   * This is a doubly-linked list.
   */
  struct AddressToStringContext *prev;

  /**
   * Transmission context
   */
  struct GNUNET_SERVER_TransmitContext* tc;
};


/**
 * Client monitoring changes of active addresses or validations
 * of our neighbours. Which type is being monitored depends on the
 * DLL this struct is in.
 */
struct MonitoringClient
{
  /**
   * This is a doubly-linked list.
   */
  struct MonitoringClient *next;

  /**
   * This is a doubly-linked list.
   */
  struct MonitoringClient *prev;

  /**
   * Handle to the client.
   */
  struct GNUNET_SERVER_Client *client;

  /**
   * Peer identity to monitor the addresses of.
   * Zero to monitor all neighrours.
   */
  struct GNUNET_PeerIdentity peer;

};


/**
 * Closure for #handle_send_transmit_continuation()
 */
struct SendTransmitContinuationContext
{
  /**
   * Client that made the request.
   */
  struct GNUNET_SERVER_Client *client;

  /**
   * Peer that was the target.
   */
  struct GNUNET_PeerIdentity target;

  /**
   * At what time did we receive the message?
   */
  struct GNUNET_TIME_Absolute send_time;

  /**
   * Unique ID, for logging.
   */
  unsigned long long uuid;

  /**
   * Set to #GNUNET_YES if the connection for @e target goes
   * down and we thus must no longer send the
   * #GNUNET_MESSAGE_TYPE_TRANSPORT_SEND_OK message.
   */
  int down;
};


/**
 * Head of linked list of all clients to this service.
 */
static struct TransportClient *clients_head;

/**
 * Tail of linked list of all clients to this service.
 */
static struct TransportClient *clients_tail;

/**
 * Map of peer identities to active send transmit continuation
 * contexts. Used to flag contexts as 'dead' when a connection goes
 * down. Values are of type `struct SendTransmitContinuationContext
 * *`.
 */
static struct GNUNET_CONTAINER_MultiPeerMap *active_stccs;

/**
 * Head of linked list of all pending address iterations
 */
static struct AddressToStringContext *a2s_head;

/**
 * Tail of linked list of all pending address iterations
 */
static struct AddressToStringContext *a2s_tail;

/**
 * Head of linked list of monitoring clients.
 */
static struct MonitoringClient *peer_monitoring_clients_head;

/**
 * Tail of linked list of monitoring clients.
 */
static struct MonitoringClient *peer_monitoring_clients_tail;

/**
 * Notification context, to send updates on changes to active addresses
 * of our neighbours.
 */
static struct GNUNET_SERVER_NotificationContext *peer_nc;

/**
 * Notification context, to send updates on changes to active addresses
 * of our neighbours.
 */
static struct GNUNET_SERVER_NotificationContext *val_nc;

/**
 * Notification context, to send updates on changes to active plugin
 * connections.
 */
static struct GNUNET_SERVER_NotificationContext *plugin_nc;

/**
 * Plugin monitoring client we are currently syncing, NULL if all
 * monitoring clients are in sync.
 */
static struct GNUNET_SERVER_Client *sync_client;

/**
 * Peer identity that is all zeros, used as a way to indicate
 * "all peers".  Used for comparissons.
 */
static struct GNUNET_PeerIdentity all_zeros;


/**
 * Find the internal handle associated with the given client handle.
 *
 * @param client server's client handle to look up
 * @return internal client handle
 */
static struct TransportClient *
lookup_client (struct GNUNET_SERVER_Client *client)
{
  return GNUNET_SERVER_client_get_user_context (client,
                                                struct TransportClient);
}


/**
 * Create the internal handle for the given server client handle.
 *
 * @param client server's client handle to create our internal handle for
 * @return fresh internal client handle
 */
static struct TransportClient *
setup_client (struct GNUNET_SERVER_Client *client)
{
  struct TransportClient *tc;

  GNUNET_assert (NULL == lookup_client (client));
  tc = GNUNET_new (struct TransportClient);
  tc->client = client;
  GNUNET_SERVER_client_set_user_context (client, tc);
  GNUNET_CONTAINER_DLL_insert (clients_head,
                               clients_tail,
                               tc);
  GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
              "Client %p connected\n",
              tc);
  return tc;
}


/**
 * Find the handle to the monitoring client associated with the given
 * client handle.
 *
 * @param head the head of the client queue to look in
 * @param client server's client handle to look up
 * @return handle to the monitoring client
 */
static struct MonitoringClient *
lookup_monitoring_client (struct MonitoringClient *head,
                          struct GNUNET_SERVER_Client *client)
{
  struct MonitoringClient *mc;

  for (mc = head; NULL != mc; mc = mc->next)
    if (mc->client == client)
      return mc;
  return NULL;
}


/**
 * Setup a new monitoring client using the given server client handle and
 * the peer identity.
 *
 * @param client server's client handle to create our internal handle for
 * @param peer identity of the peer to monitor the addresses of,
 *             zero to monitor all neighrours.
 * @return handle to the new monitoring client
 */
static struct MonitoringClient *
setup_peer_monitoring_client (struct GNUNET_SERVER_Client *client,
                              const struct GNUNET_PeerIdentity *peer)
{
  struct MonitoringClient *mc;

  GNUNET_assert (NULL ==
                 lookup_monitoring_client (peer_monitoring_clients_head,
                                           client));
  mc = GNUNET_new (struct MonitoringClient);
  mc->client = client;
  mc->peer = *peer;
  GNUNET_CONTAINER_DLL_insert (peer_monitoring_clients_head,
                               peer_monitoring_clients_tail,
                               mc);
  GNUNET_SERVER_client_mark_monitor (client);
  GNUNET_SERVER_notification_context_add (peer_nc,
                                          client);
  if (0 != memcmp (peer,
                   &all_zeros,
                   sizeof (struct GNUNET_PeerIdentity)))
    GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
                "Client %p started monitoring of the peer `%s'\n",
                mc,
                GNUNET_i2s (peer));
  else
    GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
                "Client %p started monitoring all peers\n",
                mc);
  return mc;
}


/**
 * Function called to notify a client about the socket being ready to
 * queue more data.  @a buf will be NULL and @a size zero if the socket
 * was closed for writing in the meantime.
 *
 * @param cls closure
 * @param size number of bytes available in @a buf
 * @param buf where the callee should write the message
 * @return number of bytes written to @a buf
 */
static size_t
transmit_to_client_callback (void *cls,
                             size_t size,
                             void *buf)
{
  struct TransportClient *tc = cls;
  struct ClientMessageQueueEntry *q;
  const struct GNUNET_MessageHeader *msg;
  char *cbuf;
  uint16_t msize;
  size_t tsize;

  tc->th = NULL;
  if (NULL == buf)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
                "Transmission to client failed, closing connection.\n");
    return 0;
  }
  cbuf = buf;
  tsize = 0;
  while (NULL != (q = tc->message_queue_head))
  {
    msg = (const struct GNUNET_MessageHeader *) &q[1];
    msize = ntohs (msg->size);
    if (msize + tsize > size)
      break;
    GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
                "Transmitting message of type %u to client %p.\n",
                ntohs (msg->type),
                tc);
    GNUNET_CONTAINER_DLL_remove (tc->message_queue_head,
                                 tc->message_queue_tail,
                                 q);
    tc->message_count--;
    GNUNET_memcpy (&cbuf[tsize], msg, msize);
    GNUNET_free (q);
    tsize += msize;
  }
  if (NULL != q)
  {
    GNUNET_assert (msize >= sizeof (struct GNUNET_MessageHeader));
    tc->th =
        GNUNET_SERVER_notify_transmit_ready (tc->client, msize,
                                             GNUNET_TIME_UNIT_FOREVER_REL,
                                             &transmit_to_client_callback, tc);
    GNUNET_assert (NULL != tc->th);
  }
  return tsize;
}


/**
 * Queue the given message for transmission to the given client
 *
 * @param tc target of the message
 * @param msg message to transmit
 * @param may_drop #GNUNET_YES if the message can be dropped
 */
static void
unicast (struct TransportClient *tc,
         const struct GNUNET_MessageHeader *msg,
         int may_drop)
{
  struct ClientMessageQueueEntry *q;
  uint16_t msize;

  if (NULL == msg)
  {
    GNUNET_break (0);
    return;
  }
  if ( (tc->message_count >= MAX_PENDING) &&
       (GNUNET_YES == may_drop) )
  {
    GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
                "Dropping message of type %u and size %u, have %u/%u messages pending\n",
                ntohs (msg->type),
                ntohs (msg->size),
                tc->message_count,
                MAX_PENDING);
    GNUNET_STATISTICS_update (GST_stats,
                              gettext_noop
                              ("# messages dropped due to slow client"), 1,
                              GNUNET_NO);
    return;
  }
  msize = ntohs (msg->size);
  GNUNET_assert (msize >= sizeof (struct GNUNET_MessageHeader));
  q = GNUNET_malloc (sizeof (struct ClientMessageQueueEntry) + msize);
  GNUNET_memcpy (&q[1], msg, msize);
  GNUNET_CONTAINER_DLL_insert_tail (tc->message_queue_head,
                                    tc->message_queue_tail,
                                    q);
  tc->message_count++;
  if (NULL != tc->th)
    return;
  tc->th =
      GNUNET_SERVER_notify_transmit_ready (tc->client, msize,
                                           GNUNET_TIME_UNIT_FOREVER_REL,
                                           &transmit_to_client_callback, tc);
  GNUNET_assert (NULL != tc->th);
}


/**
 * Called whenever a client is disconnected.  Frees our
 * resources associated with that client.
 *
 * @param cls closure, NULL
 * @param client identification of the client
 */
static void
client_disconnect_notification (void *cls,
                                struct GNUNET_SERVER_Client *client)
{
  struct TransportClient *tc;
  struct MonitoringClient *mc;
  struct ClientMessageQueueEntry *mqe;

  if (NULL == client)
    return;
  mc = lookup_monitoring_client (peer_monitoring_clients_head,
                                 client);
  if (NULL != mc)
  {
    GNUNET_CONTAINER_DLL_remove (peer_monitoring_clients_head,
                                 peer_monitoring_clients_tail,
                                 mc);
    GNUNET_free (mc);
  }
  tc = lookup_client (client);
  if (NULL == tc)
    return;
  GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
              "Client %p disconnected, cleaning up.\n",
              tc);
  while (NULL != (mqe = tc->message_queue_head))
  {
    GNUNET_CONTAINER_DLL_remove (tc->message_queue_head,
                                 tc->message_queue_tail,
                                 mqe);
    tc->message_count--;
    GNUNET_free (mqe);
  }
  GNUNET_CONTAINER_DLL_remove (clients_head,
                               clients_tail,
                               tc);
  GNUNET_SERVER_client_set_user_context (client, NULL);
  if (NULL != tc->th)
  {
    GNUNET_SERVER_notify_transmit_ready_cancel (tc->th);
    tc->th = NULL;
  }
  GNUNET_break (0 == tc->message_count);
  GNUNET_free (tc);
}


/**
 * Function called for each of our connected neighbours.  Notify the
 * client about the existing neighbour.
 *
 * @param cls the `struct TransportClient *` to notify
 * @param peer identity of the neighbour
 * @param address the address
 * @param state the current state of the peer
 * @param state_timeout the time out for the state
 * @param bandwidth_in inbound bandwidth in NBO
 * @param bandwidth_out outbound bandwidth in NBO
 */
static void
notify_client_about_neighbour (void *cls,
                               const struct GNUNET_PeerIdentity *peer,
                               const struct GNUNET_HELLO_Address *address,
                               enum GNUNET_TRANSPORT_PeerState state,
                               struct GNUNET_TIME_Absolute state_timeout,
                               struct GNUNET_BANDWIDTH_Value32NBO bandwidth_in,
                               struct GNUNET_BANDWIDTH_Value32NBO bandwidth_out)
{
  struct TransportClient *tc = cls;
  struct ConnectInfoMessage cim;

  if (GNUNET_NO == GST_neighbours_test_connected (peer))
    return;
  cim.header.size = htons (sizeof (struct ConnectInfoMessage));
  cim.header.type = htons (GNUNET_MESSAGE_TYPE_TRANSPORT_CONNECT);
  cim.id = *peer;
  cim.quota_in = bandwidth_in;
  cim.quota_out = bandwidth_out;
  unicast (tc, &cim.header, GNUNET_NO);
}


/**
 * Initialize a normal client.  We got a start message from this
 * client, add him to the list of clients for broadcasting of inbound
 * messages.
 *
 * @param cls unused
 * @param client the client
 * @param message the start message that was sent
 */
static void
clients_handle_start (void *cls,
                      struct GNUNET_SERVER_Client *client,
                      const struct GNUNET_MessageHeader *message)
{
  const struct StartMessage *start;
  const struct GNUNET_MessageHeader *hello;
  struct TransportClient *tc;
  uint32_t options;

  tc = lookup_client (client);
  if (NULL != tc)
  {
    /* got 'start' twice from the same client, not allowed */
    GNUNET_break (0);
    GNUNET_SERVER_receive_done (client,
                                GNUNET_SYSERR);
    return;
  }
  GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
              "Client %p sent START\n",
              client);
  start = (const struct StartMessage *) message;
  options = ntohl (start->options);
  if ((0 != (1 & options)) &&
      (0 !=
       memcmp (&start->self,
               &GST_my_identity,
               sizeof (struct GNUNET_PeerIdentity))))
  {
    /* client thinks this is a different peer, reject */
    GNUNET_break (0);
    GNUNET_SERVER_receive_done (client,
                                GNUNET_SYSERR);
    return;
  }
  tc = setup_client (client);
  tc->send_payload = (0 != (2 & options));
  hello = GST_hello_get ();
  if (NULL != hello)
    unicast (tc,
             hello,
             GNUNET_NO);
  GST_neighbours_iterate (&notify_client_about_neighbour,
                          tc);
  GNUNET_SERVER_receive_done (client,
                              GNUNET_OK);
}


/**
 * Client sent us a HELLO.  Process the request.
 *
 * @param cls unused
 * @param client the client
 * @param message the HELLO message
 */
static void
clients_handle_hello (void *cls,
                      struct GNUNET_SERVER_Client *client,
                      const struct GNUNET_MessageHeader *message)
{
  GST_validation_handle_hello (message);
  GNUNET_SERVER_receive_done (client, GNUNET_OK);
}


/**
 * Function called after the transmission is done.  Notify the client that it is
 * OK to send the next message.
 *
 * @param cls closure
 * @param success #GNUNET_OK on success, #GNUNET_NO on failure, #GNUNET_SYSERR if we're not connected
 * @param bytes_payload bytes payload sent
 * @param bytes_on_wire bytes sent on wire
 */
static void
handle_send_transmit_continuation (void *cls,
                                   int success,
                                   size_t bytes_payload,
                                   size_t bytes_on_wire)
{
  struct SendTransmitContinuationContext *stcc = cls;
  struct SendOkMessage send_ok_msg;
  struct GNUNET_TIME_Relative delay;
  const struct GNUNET_HELLO_Address *addr;

  delay = GNUNET_TIME_absolute_get_duration (stcc->send_time);
  addr = GST_neighbour_get_current_address (&stcc->target);
  if (delay.rel_value_us > GNUNET_CONSTANTS_LATENCY_WARN.rel_value_us)
    GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
                "It took us %s to send %u/%u bytes to %s (%d, %s)\n",
                GNUNET_STRINGS_relative_time_to_string (delay,
                                                        GNUNET_YES),
                (unsigned int) bytes_payload,
                (unsigned int) bytes_on_wire,
                GNUNET_i2s (&stcc->target),
                success,
                (NULL != addr) ? addr->transport_name : "%");
  else
    GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
                "It took us %s to send %u/%u bytes to %s (%d, %s)\n",
                GNUNET_STRINGS_relative_time_to_string (delay,
                                                        GNUNET_YES),
                (unsigned int) bytes_payload,
                (unsigned int) bytes_on_wire,
                GNUNET_i2s (&stcc->target),
                success,
                (NULL != addr) ? addr->transport_name : "%");

  if (GNUNET_NO == stcc->down)
  {
    /* Only send confirmation if we are still connected */
    GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
                "Sending SEND_OK for transmission request %llu\n",
                stcc->uuid);
    send_ok_msg.header.size = htons (sizeof (send_ok_msg));
    send_ok_msg.header.type = htons (GNUNET_MESSAGE_TYPE_TRANSPORT_SEND_OK);
    send_ok_msg.bytes_msg = htonl (bytes_payload);
    send_ok_msg.bytes_physical = htonl (bytes_on_wire);
    send_ok_msg.success = htonl (success);
    send_ok_msg.peer = stcc->target;
    GST_clients_unicast (stcc->client,
                         &send_ok_msg.header,
                         GNUNET_NO);
  }
  GNUNET_SERVER_client_drop (stcc->client);
  GNUNET_assert (GNUNET_OK ==
                 GNUNET_CONTAINER_multipeermap_remove (active_stccs,
                                                       &stcc->target,
                                                       stcc));
  GNUNET_free (stcc);
}


/**
 * Client asked for transmission to a peer.  Process the request.
 *
 * @param cls unused
 * @param client the client
 * @param message the send message that was sent
 */
static void
clients_handle_send (void *cls,
                     struct GNUNET_SERVER_Client *client,
                     const struct GNUNET_MessageHeader *message)
{
  static unsigned long long uuid_gen;
  const struct OutboundMessage *obm;
  const struct GNUNET_MessageHeader *obmm;
  struct SendTransmitContinuationContext *stcc;
  uint16_t size;
  uint16_t msize;
  struct TransportClient *tc;

  tc = lookup_client (client);
  if (NULL == tc)
  {
    /* client asked for transmission before 'START' */
    GNUNET_break (0);
    GNUNET_SERVER_receive_done (client,
                                GNUNET_SYSERR);
    return;
  }

  size = ntohs (message->size);
  if (size <
      sizeof (struct OutboundMessage) + sizeof (struct GNUNET_MessageHeader))
  {
    GNUNET_break (0);
    GNUNET_SERVER_receive_done (client,
                                GNUNET_SYSERR);
    return;
  }
  obm = (const struct OutboundMessage *) message;
  obmm = (const struct GNUNET_MessageHeader *) &obm[1];
  msize = size - sizeof (struct OutboundMessage);
  if (msize < sizeof (struct GNUNET_MessageHeader))
  {
    GNUNET_break (0);
    GNUNET_SERVER_receive_done (client,
                                GNUNET_SYSERR);
    return;
  }

  if (GNUNET_NO == GST_neighbours_test_connected (&obm->peer))
  {
    /* not connected, not allowed to send; can happen due to asynchronous operations */
    GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
                "Could not send message to peer `%s': not connected\n",
                GNUNET_i2s (&obm->peer));
    GNUNET_STATISTICS_update (GST_stats,
                              gettext_noop
                              ("# bytes payload dropped (other peer was not connected)"),
                              msize, GNUNET_NO);
    GNUNET_SERVER_receive_done (client,
                                GNUNET_OK);
    return;
  }
  GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
              "Received SEND request %llu for `%s' and first message of type %u and total size %u\n",
              uuid_gen,
              GNUNET_i2s (&obm->peer),
              ntohs (obmm->type),
              msize);
  GNUNET_SERVER_receive_done (client,
                              GNUNET_OK);
  stcc = GNUNET_new (struct SendTransmitContinuationContext);
  stcc->target = obm->peer;
  stcc->client = client;
  stcc->send_time = GNUNET_TIME_absolute_get ();
  stcc->uuid = uuid_gen++;
  (void) GNUNET_CONTAINER_multipeermap_put (active_stccs,
                                            &stcc->target,
                                            stcc,
                                            GNUNET_CONTAINER_MULTIHASHMAPOPTION_MULTIPLE);
  GNUNET_SERVER_client_keep (client);
  GST_manipulation_send (&obm->peer,
                         obmm,
                         msize,
                         GNUNET_TIME_relative_ntoh (obm->timeout),
                         &handle_send_transmit_continuation,
                         stcc);
}


/**
 * Take the given address and append it to the set of results sent back to
 * the client.  This function may be called serveral times for a single
 * conversion.   The last invocation will be with a @a address of
 * NULL and a @a res of #GNUNET_OK.  Thus, to indicate conversion
 * errors, the callback might be called first with @a address NULL and
 * @a res being #GNUNET_SYSERR.  In that case, there will still be a
 * subsequent call later with @a address NULL and @a res #GNUNET_OK.
 *
 * @param cls the transmission context used (`struct GNUNET_SERVER_TransmitContext *`)
 * @param buf text to transmit (contains the human-readable address, or NULL)
 * @param res #GNUNET_OK if conversion was successful, #GNUNET_SYSERR on error,
 *            never #GNUNET_NO
 */
static void
transmit_address_to_client (void *cls,
                            const char *buf,
                            int res)
{
  struct AddressToStringContext *actx = cls;
  struct AddressToStringResultMessage *atsm;
  size_t len;
  size_t slen;

  GNUNET_assert ( (GNUNET_OK == res) ||
                  (GNUNET_SYSERR == res) );
  if (NULL == buf)
  {
    len = sizeof (struct AddressToStringResultMessage);
    atsm = GNUNET_malloc (len);
    atsm->header.size = ntohs (len);
    atsm->header.type = ntohs (GNUNET_MESSAGE_TYPE_TRANSPORT_ADDRESS_TO_STRING_REPLY);
    if (GNUNET_OK == res)
    {
      /* this was the last call, transmit */
      atsm->res = htonl (GNUNET_OK);
      atsm->addr_len = htonl (0);
      GNUNET_SERVER_transmit_context_append_message (actx->tc,
                                                     (const struct GNUNET_MessageHeader *) atsm);
      GNUNET_SERVER_transmit_context_run (actx->tc,
                                          GNUNET_TIME_UNIT_FOREVER_REL);
      GNUNET_CONTAINER_DLL_remove (a2s_head,
                                   a2s_tail,
                                   actx);
      GNUNET_free (atsm);
      GNUNET_free (actx);
      return;
    }
    if (GNUNET_SYSERR == res)
    {
      /* address conversion failed, but there will be more callbacks */
      atsm->res = htonl (GNUNET_SYSERR);
      atsm->addr_len = htonl (0);
      GNUNET_SERVER_transmit_context_append_message (actx->tc,
                                                     (const struct GNUNET_MessageHeader *) atsm);
      GNUNET_free (atsm);
      return;
    }
  }
  GNUNET_assert (GNUNET_OK == res);
  /* succesful conversion, append*/
  slen = strlen (buf) + 1;
  len = sizeof (struct AddressToStringResultMessage) + slen;
  atsm = GNUNET_malloc (len);
  atsm->header.size = ntohs (len);
  atsm->header.type = ntohs (GNUNET_MESSAGE_TYPE_TRANSPORT_ADDRESS_TO_STRING_REPLY);
  atsm->res = htonl (GNUNET_YES);
  atsm->addr_len = htonl (slen);
  GNUNET_memcpy (&atsm[1],
          buf,
          slen);
  GNUNET_SERVER_transmit_context_append_message (actx->tc,
                                                 (const struct GNUNET_MessageHeader *) atsm);
  GNUNET_free (atsm);
}


/**
 * Client asked to resolve an address.  Process the request.
 *
 * @param cls unused
 * @param client the client
 * @param message the resolution request
 */
static void
clients_handle_address_to_string (void *cls,
                                  struct GNUNET_SERVER_Client *client,
                                  const struct GNUNET_MessageHeader *message)
{
  const struct AddressLookupMessage *alum;
  struct GNUNET_TRANSPORT_PluginFunctions *papi;
  const char *plugin_name;
  const char *address;
  uint32_t address_len;
  uint16_t size;
  struct GNUNET_SERVER_TransmitContext *tc;
  struct AddressToStringContext *actx;
  struct AddressToStringResultMessage atsm;
  struct GNUNET_TIME_Relative rtimeout;
  int32_t numeric;

  size = ntohs (message->size);
  if (size < sizeof (struct AddressLookupMessage))
  {
    GNUNET_break (0);
    GNUNET_SERVER_receive_done (client, GNUNET_SYSERR);
    return;
  }
  alum = (const struct AddressLookupMessage *) message;
  address_len = ntohs (alum->addrlen);
  if (size <= sizeof (struct AddressLookupMessage) + address_len)
  {
    GNUNET_break (0);
    GNUNET_SERVER_receive_done (client, GNUNET_SYSERR);
    return;
  }
  address = (const char *) &alum[1];
  plugin_name = (const char *) &address[address_len];
  if ('\0' != plugin_name[size - sizeof (struct AddressLookupMessage) - address_len - 1])
  {
    GNUNET_break (0);
    GNUNET_SERVER_receive_done (client, GNUNET_SYSERR);
    return;
  }
  rtimeout = GNUNET_TIME_relative_ntoh (alum->timeout);
  numeric = ntohs (alum->numeric_only);
  tc = GNUNET_SERVER_transmit_context_create (client);
  papi = GST_plugins_printer_find (plugin_name);
  if (NULL == papi)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
                "Failed to find plugin `%s'\n",
                plugin_name);
    atsm.header.size = ntohs (sizeof (struct AddressToStringResultMessage));
    atsm.header.type = ntohs (GNUNET_MESSAGE_TYPE_TRANSPORT_ADDRESS_TO_STRING_REPLY);
    atsm.res = htonl (GNUNET_SYSERR);
    atsm.addr_len = htonl (0);
    GNUNET_SERVER_transmit_context_append_message (tc,
                                                   &atsm.header);
    atsm.header.size = ntohs (sizeof (struct AddressToStringResultMessage));
    atsm.header.type = ntohs (GNUNET_MESSAGE_TYPE_TRANSPORT_ADDRESS_TO_STRING_REPLY);
    atsm.res = htonl (GNUNET_OK);
    atsm.addr_len = htonl (0);
    GNUNET_SERVER_transmit_context_append_message (tc,
                                                   &atsm.header);
    GNUNET_SERVER_transmit_context_run (tc, GNUNET_TIME_UNIT_FOREVER_REL);
    return;
  }
  actx = GNUNET_new (struct AddressToStringContext);
  actx->tc = tc;
  GNUNET_CONTAINER_DLL_insert (a2s_head, a2s_tail, actx);
  GNUNET_SERVER_disable_receive_done_warning (client);
  GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
              "Pretty-printing address of %u bytes using plugin `%s'\n",
              address_len,
              plugin_name);
  papi->address_pretty_printer (papi->cls,
                                plugin_name,
                                address, address_len,
                                numeric,
                                rtimeout,
                                &transmit_address_to_client,
                                actx);
}


/**
 * Compose #PeerIterateResponseMessage using the given peer and address.
 *
 * @param peer identity of the peer
 * @param address the address, NULL on disconnect
 * @return composed message
 */
static struct PeerIterateResponseMessage *
compose_address_iterate_response_message (const struct GNUNET_PeerIdentity *peer,
                                          const struct GNUNET_HELLO_Address *address)
{
  struct PeerIterateResponseMessage *msg;
  size_t size;
  size_t tlen;
  size_t alen;
  char *addr;

  GNUNET_assert (NULL != peer);
  if (NULL != address)
  {
    tlen = strlen (address->transport_name) + 1;
    alen = address->address_length;
  }
  else
  {
    tlen = 0;
    alen = 0;
  }
  size = (sizeof (struct PeerIterateResponseMessage) + alen + tlen);
  msg = GNUNET_malloc (size);
  msg->header.size = htons (size);
  msg->header.type =
      htons (GNUNET_MESSAGE_TYPE_TRANSPORT_MONITOR_PEER_RESPONSE);
  msg->reserved = htonl (0);
  msg->peer = *peer;
  msg->addrlen = htonl (alen);
  msg->pluginlen = htonl (tlen);

  if (NULL != address)
  {
    msg->local_address_info = htonl((uint32_t) address->local_info);
    addr = (char *) &msg[1];
    GNUNET_memcpy (addr, address->address, alen);
    GNUNET_memcpy (&addr[alen], address->transport_name, tlen);
  }
  return msg;
}


/**
 * Context for #send_validation_information() and
 * #send_peer_information().
 */
struct IterationContext
{
  /**
   * Context to use for the transmission.
   */
  struct GNUNET_SERVER_TransmitContext *tc;

  /**
   * Which peers do we care about?
   */
  struct GNUNET_PeerIdentity id;

  /**
   * #GNUNET_YES if @e id should be ignored because we want all peers.
   */
  int all;
};


/**
 * Output information of neighbours to the given client.
 *
 * @param cls the `struct PeerIterationContext *`
 * @param peer identity of the neighbour
 * @param address the address
 * @param state current state this peer is in
 * @param state_timeout timeout for the current state of the peer
 * @param bandwidth_in inbound quota in NBO
 * @param bandwidth_out outbound quota in NBO
 */
static void
send_peer_information (void *cls,
                       const struct GNUNET_PeerIdentity *peer,
                       const struct GNUNET_HELLO_Address *address,
                       enum GNUNET_TRANSPORT_PeerState state,
                       struct GNUNET_TIME_Absolute state_timeout,
                       struct GNUNET_BANDWIDTH_Value32NBO bandwidth_in,
                       struct GNUNET_BANDWIDTH_Value32NBO bandwidth_out)
{
  struct IterationContext *pc = cls;
  struct PeerIterateResponseMessage *msg;

  if ( (GNUNET_YES != pc->all) &&
       (0 != memcmp (peer, &pc->id, sizeof (pc->id))) )
    return;
  GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
              "Sending information about `%s' using address `%s' in state `%s'\n",
              GNUNET_i2s(peer),
              (NULL != address) ? GST_plugins_a2s (address) : "<none>",
              GNUNET_TRANSPORT_ps2s (state));
  msg = compose_address_iterate_response_message (peer, address);
  msg->state = htonl (state);
  msg->state_timeout = GNUNET_TIME_absolute_hton(state_timeout);
  GNUNET_SERVER_transmit_context_append_message (pc->tc, &msg->header);
  GNUNET_free (msg);
}


/**
 * Client asked to obtain information about a specific or all peers
 * Process the request.
 *
 * @param cls unused
 * @param client the client
 * @param message the peer address information request
 */
static void
clients_handle_monitor_peers (void *cls,
                              struct GNUNET_SERVER_Client *client,
                              const struct GNUNET_MessageHeader *message)
{
  struct GNUNET_SERVER_TransmitContext *tc;
  const struct PeerMonitorMessage *msg;
  struct IterationContext pc;

  msg = (const struct PeerMonitorMessage *) message;
  if ( (GNUNET_YES != ntohl (msg->one_shot)) &&
       (NULL != lookup_monitoring_client (peer_monitoring_clients_head,
                                          client)) )
  {
    GNUNET_break (0);
    GNUNET_SERVER_receive_done (client,
                                GNUNET_SYSERR);
    return;
  }
  GNUNET_SERVER_disable_receive_done_warning (client);
  GNUNET_SERVER_client_mark_monitor (client);
  pc.tc = tc = GNUNET_SERVER_transmit_context_create (client);

  /* Send initial list */
  if (0 == memcmp (&msg->peer,
                   &all_zeros,
                   sizeof (struct GNUNET_PeerIdentity)))
  {
    /* iterate over all neighbours */
    pc.all = GNUNET_YES;
    pc.id = msg->peer;
  }
  else
  {
    /* just return one neighbour */
    pc.all = GNUNET_NO;
    pc.id = msg->peer;
  }
  GST_neighbours_iterate (&send_peer_information,
                          &pc);

  if (GNUNET_YES != ntohl (msg->one_shot))
  {
    setup_peer_monitoring_client (client,
                                  &msg->peer);
  }
  else
  {
    GNUNET_SERVER_transmit_context_append_data (tc,
                                                NULL,
                                                0,
                                                GNUNET_MESSAGE_TYPE_TRANSPORT_MONITOR_PEER_RESPONSE_END);
  }
  GNUNET_SERVER_transmit_context_run (tc,
                                      GNUNET_TIME_UNIT_FOREVER_REL);
}


/**
 * Function called by the plugin with information about the
 * current sessions managed by the plugin (for monitoring).
 *
 * @param cls closure
 * @param session session handle this information is about,
 *        NULL to indicate that we are "in sync" (initial
 *        iteration complete)
 * @param info information about the state of the session,
 *        NULL if @a session is also NULL and we are
 *        merely signalling that the initial iteration is over
 */
static void
plugin_session_info_cb (void *cls,
			struct GNUNET_ATS_Session *session,
			const struct GNUNET_TRANSPORT_SessionInfo *info)
{
  struct TransportPluginMonitorMessage *msg;
  struct GNUNET_MessageHeader sync;
  size_t size;
  size_t slen;
  uint16_t alen;
  char *name;
  char *addr;

  if (0 == GNUNET_SERVER_notification_context_get_size (plugin_nc))
  {
    GST_plugins_monitor_subscribe (NULL,
                                   NULL);
    return;
  }
  if ( (NULL == info) &&
       (NULL == session) )
  {
    /* end of initial iteration */
    if (NULL != sync_client)
    {
      sync.size = htons (sizeof (struct GNUNET_MessageHeader));
      sync.type = htons (GNUNET_MESSAGE_TYPE_TRANSPORT_MONITOR_PLUGIN_SYNC);
      GNUNET_SERVER_notification_context_unicast (plugin_nc,
                                                  sync_client,
                                                  &sync,
                                                  GNUNET_NO);
      sync_client = NULL;
    }
    return;
  }
  GNUNET_assert (NULL != info);
  GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
              "Plugin event for peer %s on transport %s\n",
              GNUNET_i2s (&info->address->peer),
              info->address->transport_name);
  slen = strlen (info->address->transport_name) + 1;
  alen = info->address->address_length;
  size = sizeof (struct TransportPluginMonitorMessage) + slen + alen;
  if (size > UINT16_MAX)
  {
    GNUNET_break (0);
    return;
  }
  msg = GNUNET_malloc (size);
  msg->header.size = htons (size);
  msg->header.type = htons (GNUNET_MESSAGE_TYPE_TRANSPORT_MONITOR_PLUGIN_EVENT);
  msg->session_state = htons ((uint16_t) info->state);
  msg->is_inbound = htons ((int16_t) info->is_inbound);
  msg->msgs_pending = htonl (info->num_msg_pending);
  msg->bytes_pending = htonl (info->num_bytes_pending);
  msg->timeout = GNUNET_TIME_absolute_hton (info->session_timeout);
  msg->delay = GNUNET_TIME_absolute_hton (info->receive_delay);
  msg->peer = info->address->peer;
  msg->session_id = (uint64_t) (intptr_t) session;
  msg->plugin_name_len = htons (slen);
  msg->plugin_address_len = htons (alen);
  name = (char *) &msg[1];
  GNUNET_memcpy (name,
          info->address->transport_name,
          slen);
  addr = &name[slen];
  GNUNET_memcpy (addr,
          info->address->address,
          alen);
  if (NULL != sync_client)
    GNUNET_SERVER_notification_context_unicast (plugin_nc,
                                                sync_client,
                                                &msg->header,
                                                GNUNET_NO);
  else
    GNUNET_SERVER_notification_context_broadcast (plugin_nc,
                                                  &msg->header,
                                                  GNUNET_NO);
  GNUNET_free (msg);
}


/**
 * Client asked to obtain information about all plugin connections.
 *
 * @param cls unused
 * @param client the client
 * @param message the peer address information request
 */
static void
clients_handle_monitor_plugins (void *cls,
				struct GNUNET_SERVER_Client *client,
				const struct GNUNET_MessageHeader *message)
{
  GNUNET_SERVER_client_mark_monitor (client);
  GNUNET_SERVER_disable_receive_done_warning (client);
  GNUNET_SERVER_notification_context_add (plugin_nc,
                                          client);
  GNUNET_assert (NULL == sync_client);
  sync_client = client;
  GST_plugins_monitor_subscribe (&plugin_session_info_cb,
                                 NULL);
}


/**
 * Start handling requests from clients.
 *
 * @param server server used to accept clients from.
 */
void
GST_clients_start (struct GNUNET_SERVER_Handle *server)
{
  static const struct GNUNET_SERVER_MessageHandler handlers[] = {
    {&clients_handle_start, NULL,
     GNUNET_MESSAGE_TYPE_TRANSPORT_START, sizeof (struct StartMessage)},
    {&clients_handle_hello, NULL,
     GNUNET_MESSAGE_TYPE_HELLO, 0},
    {&clients_handle_send, NULL,
     GNUNET_MESSAGE_TYPE_TRANSPORT_SEND, 0},
    {&clients_handle_address_to_string, NULL,
     GNUNET_MESSAGE_TYPE_TRANSPORT_ADDRESS_TO_STRING, 0},
    {&clients_handle_monitor_peers, NULL,
     GNUNET_MESSAGE_TYPE_TRANSPORT_MONITOR_PEER_REQUEST,
     sizeof (struct PeerMonitorMessage)},
    {&GST_blacklist_handle_init, NULL,
     GNUNET_MESSAGE_TYPE_TRANSPORT_BLACKLIST_INIT,
     sizeof (struct GNUNET_MessageHeader)},
    {&GST_blacklist_handle_reply, NULL,
     GNUNET_MESSAGE_TYPE_TRANSPORT_BLACKLIST_REPLY,
     sizeof (struct BlacklistMessage)},
    {&GST_manipulation_set_metric, NULL,
     GNUNET_MESSAGE_TYPE_TRANSPORT_TRAFFIC_METRIC,
     sizeof (struct TrafficMetricMessage) },
    {&clients_handle_monitor_plugins, NULL,
     GNUNET_MESSAGE_TYPE_TRANSPORT_MONITOR_PLUGIN_START,
     sizeof (struct GNUNET_MessageHeader) },
    {NULL, NULL, 0, 0}
  };
  active_stccs = GNUNET_CONTAINER_multipeermap_create (128,
                                                       GNUNET_YES);
  peer_nc = GNUNET_SERVER_notification_context_create (server, 0);
  val_nc = GNUNET_SERVER_notification_context_create (server, 0);
  plugin_nc = GNUNET_SERVER_notification_context_create (server, 0);
  GNUNET_SERVER_add_handlers (server, handlers);
  GNUNET_SERVER_disconnect_notify (server,
                                   &client_disconnect_notification,
                                   NULL);
}


/**
 * Stop processing clients.
 */
void
GST_clients_stop ()
{
  struct AddressToStringContext *cur;

  while (NULL != (cur = a2s_head))
  {
    GNUNET_SERVER_transmit_context_destroy (cur->tc, GNUNET_NO);
    GNUNET_CONTAINER_DLL_remove (a2s_head, a2s_tail, cur);
    GNUNET_free (cur);
  }
  if (NULL != peer_nc)
  {
    GNUNET_SERVER_notification_context_destroy (peer_nc);
    peer_nc = NULL;
  }
  if (NULL != val_nc)
  {
    GNUNET_SERVER_notification_context_destroy (val_nc);
    val_nc = NULL;
  }
  if (NULL != plugin_nc)
  {
    GNUNET_SERVER_notification_context_destroy (plugin_nc);
    plugin_nc = NULL;
  }
  GNUNET_CONTAINER_multipeermap_destroy (active_stccs);
  active_stccs = NULL;
}


/**
 * Broadcast the given message to all of our clients.
 *
 * @param msg message to broadcast
 * @param may_drop #GNUNET_YES if the message can be dropped / is payload
 */
void
GST_clients_broadcast (const struct GNUNET_MessageHeader *msg,
                       int may_drop)
{
  struct TransportClient *tc;
  int done;

  GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
              "Asked to broadcast message of type %u with %u bytes\n",
              (unsigned int) ntohs (msg->type),
              (unsigned int) ntohs (msg->size));
  done = GNUNET_NO;
  for (tc = clients_head; NULL != tc; tc = tc->next)
  {
    if ( (GNUNET_YES == may_drop) &&
         (GNUNET_YES != tc->send_payload) )
      continue; /* skip, this client does not care about payload */
    unicast (tc, msg, may_drop);
    done = GNUNET_YES;
  }
  if (GNUNET_NO == done)
    GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
		"Message of type %u not delivered, is CORE service up?\n",
		ntohs (msg->type));
}


/**
 * Send the given message to a particular client
 *
 * @param client target of the message
 * @param msg message to transmit
 * @param may_drop #GNUNET_YES if the message can be dropped
 */
void
GST_clients_unicast (struct GNUNET_SERVER_Client *client,
                     const struct GNUNET_MessageHeader *msg,
                     int may_drop)
{
  struct TransportClient *tc;

  tc = lookup_client (client);
  if (NULL == tc)
    return;                     /* client got disconnected in the meantime, drop message */
  unicast (tc, msg, may_drop);
}


/**
 * Broadcast the new active address to all clients monitoring the peer.
 *
 * @param peer peer this update is about (never NULL)
 * @param address address, NULL on disconnect
 * @param state the current state of the peer
 * @param state_timeout the time out for the state
 */
void
GST_clients_broadcast_peer_notification (const struct GNUNET_PeerIdentity *peer,
                                         const struct GNUNET_HELLO_Address *address,
                                         enum GNUNET_TRANSPORT_PeerState state,
                                         struct GNUNET_TIME_Absolute state_timeout)
{
  struct PeerIterateResponseMessage *msg;
  struct MonitoringClient *mc;

  msg = compose_address_iterate_response_message (peer, address);
  msg->state = htonl (state);
  msg->state_timeout = GNUNET_TIME_absolute_hton (state_timeout);
  for (mc = peer_monitoring_clients_head; NULL != mc; mc = mc->next)
    if ((0 == memcmp (&mc->peer, &all_zeros,
                      sizeof (struct GNUNET_PeerIdentity))) ||
        (0 == memcmp (&mc->peer, peer,
                      sizeof (struct GNUNET_PeerIdentity))))
      GNUNET_SERVER_notification_context_unicast (peer_nc,
                                                  mc->client,
                                                  &msg->header,
                                                  GNUNET_NO);
  GNUNET_free (msg);
}


/**
 * Mark the peer as down so we don't call the continuation
 * context in the future.
 *
 * @param cls NULL
 * @param peer peer that got disconnected
 * @param value a `struct SendTransmitContinuationContext` to mark
 * @return #GNUNET_OK (continue to iterate)
 */
static int
mark_peer_down (void *cls,
                const struct GNUNET_PeerIdentity *peer,
                void *value)
{
  struct SendTransmitContinuationContext *stcc = value;

  stcc->down = GNUNET_YES;
  return GNUNET_OK;
}


/**
 * Notify all clients about a disconnect, and cancel
 * pending SEND_OK messages for this peer.
 *
 * @param peer peer that disconnected
 */
void
GST_clients_broadcast_disconnect (const struct GNUNET_PeerIdentity *peer)
{
  struct DisconnectInfoMessage disconnect_msg;

  GNUNET_CONTAINER_multipeermap_get_multiple (active_stccs,
                                              peer,
                                              &mark_peer_down,
                                              NULL);
  disconnect_msg.header.size = htons (sizeof(struct DisconnectInfoMessage));
  disconnect_msg.header.type = htons (GNUNET_MESSAGE_TYPE_TRANSPORT_DISCONNECT);
  disconnect_msg.reserved = htonl (0);
  disconnect_msg.peer = *peer;
  GST_clients_broadcast (&disconnect_msg.header,
                         GNUNET_NO);

}


/* end of file gnunet-service-transport_clients.c */
