/*   FILE: state-play.vala -- A state machine implementing lcdgrilo's menu
 * AUTHOR: W. Michael Petullo <mike@flyn.org>
 *   DATE: 01 December 2013 
 *
 * Copyright (c) 2013 W. Michael Petullo <new@flyn.org>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using Gst;

const int64 SCAN_JUMP_SECONDS = 5;
const bool  FORWARD           = true;
const bool  BACKWARD          = false;

class StatePlay : State {
	private MainLoop loop;
	private Gee.ArrayList<Grl.Media> playlist;
	private int current;
	private uint watch_id = 0;
	private static GLib.Once<StatePlay> instance;

	/* This is a singleton class. */
	private StatePlay () {}
	public static unowned StatePlay get_instance (MainLoop loop, LCDPlayer player, Gee.ArrayList<Grl.Media> playlist) {
		unowned StatePlay state_play = instance.once (() => {
			return new StatePlay ();
		});

		state_play.current = 0;
		state_play.loop = loop;
		state_play.player   = player;
		state_play.playlist = playlist;

		GLib.Idle.add (state_play.play);

		return state_play;
	}

	// Print selected item.
	public override void print_selected () {
		output (playlist[current].get_title ());
	}

	private bool bus_callback (Gst.Bus bus, Gst.Message message) {
		bool fnval = true;

                switch (message.type) {
                case MessageType.ERROR:
			// Terminate; lcdgrilo will hopefully restart.
                        GLib.Error err;
                        string debug;
                        message.parse_error (out err, out debug);
                        GLib.error ("Error: %s\n", err.message);
                case MessageType.EOS:
			// Play the next queued song.
			forward_track ();
			GLib.Idle.add (play);
			fnval = false;
                        break;
                default:
                        break;
                }

                return fnval;
        }

	// Seek to previous track.
	public override void seekback () {
		back_track ();
		GLib.Idle.add (play);
	}

	// Seek to next track.
	public override void seekforward () {
		forward_track ();
		GLib.Idle.add (play);
	}

	// Select next menu item.
	public override void next () {
		var builder = new StringBuilder ();
		int64 position = seek (FORWARD);
		builder.printf ("%" + int64.FORMAT + ":%.2" + int64.FORMAT, position / (Gst.SECOND * 60), (position / Gst.SECOND) % 60);
		
		output (builder.str);
	}

	// Select previous menu item.
	public override void previous () {
		var builder = new StringBuilder ();
		int64 position = seek (BACKWARD);
		builder.printf ("%" + int64.FORMAT + ":%.2" + int64.FORMAT, position / (Gst.SECOND * 60), (position / Gst.SECOND) % 60);
		
		output (builder.str);
	}

	private bool play () {
		var media = playlist[current];

		player.element.set_state (Gst.State.NULL);
		GLib.Source.remove(watch_id);

		player.element.uri = media.get_url ();
		player.element.current_audio = media.get_int (Grl.MetadataKey.AUDIO_TRACK);

		Gst.Bus bus = player.element.get_bus ();
		watch_id = bus.add_watch (GLib.Priority.DEFAULT, bus_callback);

		output ("Playing " + media.get_title ());

		player.element.set_state (Gst.State.PLAYING);

		return false;
	}

	// Play currently selected song; do not transition to new state.
	public override State transition () {
		GLib.Idle.add (play);
		return this;
	}

	private void forward_track () {
		current = (current + 1) % playlist.size;
                print_selected ();
	}

	private void back_track () {
		current = (current + (playlist.size - 1)) % playlist.size;
                print_selected ();
	}

	// Seek and return new absolute position in song.
	private int64 seek (bool forward) {
		int64 current_position, next_position;

		player.element.query_position (Gst.Format.TIME, out current_position);

		next_position = current_position + (SCAN_JUMP_SECONDS * Gst.SECOND * (forward ? 1 : -1));
		next_position = next_position < 0 ? 0 : next_position;

		player.element.seek_simple (Gst.Format.TIME, Gst.SeekFlags.FLUSH | Gst.SeekFlags.KEY_UNIT, next_position);

		return next_position;

		
	}
}
