/*   FILE: state.vala -- A state machine implementing lcdgrilo's menu
 * AUTHOR: W. Michael Petullo <mike@flyn.org>
 *   DATE: 01 December 2013 
 *
 * Copyright (c) 2013 W. Michael Petullo <new@flyn.org>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* The PiFace CAD provides the following inputs:
 *   From left to right:
 *     Button 0 = 0
 *     Button 1 = 1
 *     Button 2 = 2
 *     Button 3 = 3
 *     Button 4 = 4
 *   Rocker switch:
 *     Select     = 5
 *     Rock left  = 6
 *     Rock right = 7
 */

enum SwitchNumber {
	SWITCH_STOP        = 0,
	SWITCH_SEEKBACK    = 1,
	SWITCH_PAUSE_PLAY  = 2,
	SWITCH_SEEKFORWARD = 3,
	SWITCH_ESCAPE      = 4,
	SWITCH_SELECT      = 5,
	SWITCH_LEFT        = 6,
	SWITCH_RIGHT       = 7
}

abstract class State : GLib.Object {
	private static bool initialized = false;
	uint lights_out_timeout_id = 0;
	static uint output_shift = 0;
	static string output_str = "";

	// FIXME: Move button handlers to another class?
	private uint8[] lastSwitchState = { 1, 1, 1, 1, 1, 1, 1, 1 };

	protected unowned LCDPlayer player;

	// Print selected item.
	public abstract void print_selected ();

	// Stop playback.
	public void stop () {
		player.element.set_state (Gst.State.NULL);
		player.stateStack.peek ().print_selected ();
	}

	// Seek backwards.
	public abstract void seekback ();

	// Seek forward.
	public abstract void seekforward ();

	// Select next menu item.
	public abstract void next ();

	// Select previous menu item.
	public abstract void previous ();

	// Transition to next state, based on current selection.
	public abstract State transition ();

	public State () {
		if (false == initialized) {
			register_handlers ();
			start_output ();
			initialized = true;
		}
	}

	// Toggle between playing and paused.
	public void pause_play () {
		Gst.State state;
		player.element.get_state (out state, null, 0);
		if (Gst.State.PLAYING == state) {
			player.element.set_state (Gst.State.PAUSED);
			output ("Paused");
		} else if (Gst.State.PAUSED == state) {
			player.element.set_state (Gst.State.PLAYING);
			player.stateStack.peek ().print_selected ();
		}
	}

	public bool check_keyboard (IOChannel channel, IOCondition condition) {
		char []c = new char[1];

		if (condition == IOCondition.HUP) {
			GLib.stderr.printf ("Standard in closed?\n");
			return false;
		}

		try {
			size_t length = -1;
			IOStatus status = channel.read_chars (c, out length);
			if (status == IOStatus.EOF || length != 1) {
				GLib.stderr.printf ("Standard in closed?\n");
				return false;
			}
		} catch (IOChannelError e) {
			GLib.stderr.printf ("IOChannelError: %s\n", e.message);
			return false;
		} catch (ConvertError e) {
			GLib.stderr.printf ("ConvertError: %s\n", e.message);
			return false;
		}

                switch (c[0]) {
		case 'z':
			do_input (SwitchNumber.SWITCH_STOP);
			break;
		case 'q':
			do_input (SwitchNumber.SWITCH_SEEKBACK);
			break;
		case 'w':
			do_input (SwitchNumber.SWITCH_SEEKFORWARD);
			break;
                case ' ':
			do_input (SwitchNumber.SWITCH_PAUSE_PLAY);
                        break;
                case 'a':
			do_input (SwitchNumber.SWITCH_LEFT);
                        break;
                case 's':
			do_input (SwitchNumber.SWITCH_RIGHT);
                        break;
                case '\n':
			do_input (SwitchNumber.SWITCH_SELECT);
                        break;
                case 0x1b:
			do_input (SwitchNumber.SWITCH_ESCAPE);
                        break;
                default:
			if (! no_pifacecad) {
				GLib.warning ("Invalid key pressed");
			} else {
				GLib.stderr.printf ("Valid keys are:\n");
				GLib.stderr.printf ("  'z'     (stop)\n");
				GLib.stderr.printf ("  'q'     (seek backward)\n");
				GLib.stderr.printf ("  'w'     (seek forward)\n");
				GLib.stderr.printf ("  ' '     (pause/play)\n");
				GLib.stderr.printf ("  'a'     (left)\n");
				GLib.stderr.printf ("  's'     (right)\n");
				GLib.stderr.printf ("  'enter' (select)\n");
				GLib.stderr.printf ("  'esc'   (escape)\n");
			}
			break;
                }

                return true;
	}

	private bool render_output () {
		Pifacecad.lcd_clear ();

		if (output_str.length <= 15) {
			Pifacecad.lcd_write (output_str);
		} else {
			var builder = new StringBuilder ();
			uint rest = output_str.length > 30 ? 15 : output_str.length - 15;
			uint cutoff = output_str.length > 30 ? output_str.length - 30 : 0;

			// FIXME: Should be able to replace 15 with 16,
			// but libpifacecad seems to require a "cell" for '\n'.
			// See https://github.com/piface/libpifacecad/issues/2.
			builder.append (output_str.substring (output_shift, 15));
			builder.append ("\n ");
			builder.append (output_str.substring (15 + output_shift, rest));

			Pifacecad.lcd_write (builder.str);

			output_shift = output_shift < cutoff ? output_shift + 1 : 0;
		}

		return true;
	}

	private void start_output () {
		if (! no_pifacecad) {
			GLib.Timeout.add (750, render_output);
		}
	}

	public void output (string str) {
		GLib.stdout.printf ("%s\n", str);
		output_shift = 0;
		output_str = str;
		if (! no_pifacecad) {
			render_output(); /* Also registered as GLib.Timeout function for long-string animation. */
		}
	}

	protected bool lights_out () {
		GLib.debug ("Lights out.\n");
		if (! no_pifacecad) {
			Pifacecad.lcd_backlight_off ();
		}
		this.lights_out_timeout_id = 0;
		return false;
	}

	protected void register_handlers () {
		try {
			IOChannel channel = new IOChannel.unix_new (Posix.STDIN_FILENO);
			channel.set_encoding (null);
			channel.add_watch (IOCondition.IN, check_keyboard);
			Timeout.add (SWITCH_POLL_MILLISECONDS, check_switches);
		} catch (IOChannelError e) {
			GLib.error ("Channel error initializing state");
		}
	}

	protected bool check_switches () {
		if (! no_pifacecad) {
			for (uint8 switch = 0; switch < 8; switch++) {
				uint8 current = Pifacecad.read_switch (switch);

				if (0 == current && 1 == lastSwitchState[switch]) {
					do_input (switch);
				}

				lastSwitchState[switch] = current;
			}
		}

		return true;
	}

	private void do_input (uint8 op) {
		if (! no_pifacecad) {
			Pifacecad.lcd_backlight_on ();
		}

		switch (op) {
		case SwitchNumber.SWITCH_STOP:
			do_stop ();
			break;
		case SwitchNumber.SWITCH_SEEKBACK:
			do_seekback ();
			break;
		case SwitchNumber.SWITCH_PAUSE_PLAY:
			do_pause_play ();
			break;
		case SwitchNumber.SWITCH_SEEKFORWARD:
			do_seekforward ();
			break;
		case SwitchNumber.SWITCH_ESCAPE:
			do_escape ();
			break;
		case SwitchNumber.SWITCH_LEFT:
			do_left ();
			break;
		case SwitchNumber.SWITCH_RIGHT:
			do_right ();
			break;
		case SwitchNumber.SWITCH_SELECT:
			do_select ();
			break;
		default:
			GLib.warning ("Invalid button pressed");
			break;
		}

		if (0 != lights_out_timeout_id) {
			Source.remove (lights_out_timeout_id);
		}
		lights_out_timeout_id = Timeout.add_seconds (LIGHTS_OUT_SECONDS, lights_out);
	}

	private void do_stop () {
		player.stateStack.peek ().stop ();
	}

	private void do_seekback () {
		player.stateStack.peek ().seekback ();
	}

	private void do_pause_play () {
		player.stateStack.peek ().pause_play ();
	}

	private void do_seekforward () {
		player.stateStack.peek ().seekforward ();
	}

	private void do_escape () {
		if (player.stateStack.size () > 1) {
			player.stateStack.pop ();
		}
		player.stateStack.peek ().print_selected ();
	}

	private void do_left () {
		player.stateStack.peek ().previous ();
	}

	private void do_right () {
		player.stateStack.peek ().next ();
	}

	private void do_select () {
		// Peek at top, call transition, push value returned (only push if new state).
		State next = player.stateStack.peek ().transition ();
		if (player.stateStack.peek () != next) {
			player.stateStack.push (next);
		}
	}

        protected void connected_cb (Grl.Source source) {
                // FIXME: Handle connections that require authentication.

                var state = player.stateStack.pop (); // Pop dummy "Connecting to service" state.
                GLib.assert (state is StateConnecting);

                StateChooseCategory next_state = (StateChooseCategory) state.transition ();
                next_state.build_menu ();

		// Do not push if already on top of stack after popping connection (e.g., user selected "more").
		if (next_state != player.stateStack.peek()) {
			player.stateStack.push (next_state);
		}
        }
}
