
/*
 * The olsr.org Optimized Link-State Routing daemon version 2 (olsrd2)
 * Copyright (c) 2004-2015, the olsr.org team - see HISTORY file
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * * Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in
 *   the documentation and/or other materials provided with the
 *   distribution.
 * * Neither the name of olsr.org, olsrd nor the names of its
 *   contributors may be used to endorse or promote products derived
 *   from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * Visit http://www.olsr.org for more information.
 *
 * If you find this software useful feel free to make a donation
 * to the project. For more information see the website or contact
 * the copyright holders.
 *
 */

#ifndef OONF_TELNET_H_
#define OONF_TELNET_H_

#include "common/common_types.h"
#include "common/avl.h"
#include "common/list.h"
#include "common/netaddr.h"
#include "common/netaddr_acl.h"
#include "subsystems/oonf_stream_socket.h"

#define OONF_TELNET_SUBSYSTEM "telnet"

enum oonf_telnet_result {
  TELNET_RESULT_ACTIVE,
  TELNET_RESULT_CONTINOUS,
  TELNET_RESULT_INTERNAL_ERROR,
  TELNET_RESULT_QUIT,

  /*
   * this one is used internally for the telnet API,
   * it should not be returned by a command handler
   */
  _TELNET_RESULT_UNKNOWN_COMMAND,
};

/*
 * represents a cleanup handler that must be called when the
 * telnet core is shut down.
 */
struct oonf_telnet_cleanup {
  /* pointer to telnet data */
  struct oonf_telnet_data *data;

  /* callback for cleanup */
  void (*cleanup_handler)(struct oonf_telnet_cleanup *);

  /* custom data pointer for cleanup handler */
  void *custom;

  /* node for list of cleanup handlers */
  struct list_entity node;
};

/*
 * represents the data part of a telnet connection to a client
 */
struct oonf_telnet_data {
  /* address of remote communication partner */
  struct netaddr *remote;

  /* output buffer for telnet commands */
  struct autobuf *out;

  /* current telnet command and parameters */
  const char *command;
  const char *parameter;

  /* remember if echo mode is active */
  bool show_echo;

  /* millisecond timeout between commands */
  uint32_t timeout_value;

  /* callback and data to stop a continous output txt command */
  void (*stop_handler)(struct oonf_telnet_data *);
  void *stop_data[4];

  struct list_entity cleanup_list;
};

/*
 * represents a full telnet session including socket
 */
struct oonf_telnet_session {
  struct oonf_stream_session session;
  struct oonf_telnet_data data;
};

typedef enum oonf_telnet_result (*oonf_telnethandler)
    (struct oonf_telnet_data *con);

#if !defined(REMOVE_HELPTEXT)
#define TELNET_CMD(cmd, cb, helptext, args...) { .command = (cmd), .handler = (cb), .help = helptext, ##args }
#else
#define TELNET_CMD(cmd, cb, helptext, args...) { .command = (cmd), .handler = (cb), .help = "", ##args }
#endif

/* represents a telnet command */
struct oonf_telnet_command {
  /* name of telnet command */
  const char *command;

  /* help text for telnet command, NULL if it uses a custom help handler */
  const char *help;

  /* access control list for telnet command, NULL if not used */
  struct netaddr_acl *acl;

  /* handler for telnet command */
  oonf_telnethandler handler;

  /* handler for help text */
  oonf_telnethandler help_handler;

  /* node for tree of telnet commands */
  struct avl_node _node;
};

EXPORT int oonf_telnet_add(struct oonf_telnet_command *command);
EXPORT void oonf_telnet_remove(struct oonf_telnet_command *command);

EXPORT void oonf_telnet_stop(struct oonf_telnet_data *data, bool print_prompt);

EXPORT enum oonf_telnet_result oonf_telnet_execute(
    const char *cmd, const char *para,
    struct autobuf *out, struct netaddr *remote);

/**
 * Add a cleanup handler to a telnet session
 * @param data pointer to telnet data
 * @param cleanup pointer to initialized cleanup handler
 */
static INLINE void
oonf_telnet_add_cleanup(struct oonf_telnet_data *data,
    struct oonf_telnet_cleanup *cleanup) {
  cleanup->data = data;
  list_add_tail(&data->cleanup_list, &cleanup->node);
}

/**
 * Removes a cleanup handler to a telnet session
 * @param cleanup pointer to cleanup handler
 */
static INLINE void
oonf_telnet_remove_cleanup(struct oonf_telnet_cleanup *cleanup) {
  list_remove(&cleanup->node);
}

/**
 * Flushs the output stream of a telnet session. This will be only
 * necessary for continous output.
 * @param data pointer to telnet data
 */
static INLINE void
oonf_telnet_flush_session(struct oonf_telnet_data *data) {
  struct oonf_telnet_session *session;

  session = container_of(data, struct oonf_telnet_session, data);
  oonf_stream_flush(&session->session);
}

#endif /* OONF_TELNET_H_ */
