/******************************************************************************
 *
 * fingerprint.c - part of smap
 *
 *
 * $Id: fingerprint.c,v 1.18 2007-10-14 10:02:18 hscholz Exp $ 
 *****************************************************************************/

#ifndef _FINGERPRINT_C
#define _FINGERPRINT_C

#include "config.h"

fingerprint *fpdb;   /* fingerprint database */
static char *FPDBFILE_LOCAL = "/usr/local/share/smap/fingerprint.db";
static char *FPDBFILE_CWD = "fingerprint.db";
static char *FPDBFILE = "/usr/share/smap/fingerprint.db";

/* init a fingerprint structure */
int fingerprint_init(fingerprint *fp) {

	FUNCTION(__FILE__, __FUNCTION__);
	fp->name = NULL;
	fp->options = NO_RELEVANCE;
	fp->newmethod = NO_RELEVANCE;
	fp->brokenfromto = NO_RELEVANCE;
	fp->prack = NO_RELEVANCE;
	fp->invite = NO_RELEVANCE;
	fp->ping = NO_RELEVANCE;
	fp->header_ua = NULL;
	fp->header_server = NULL;
	fp->accept_class = NO_RELEVANCE;
	fp->allow_class = NO_RELEVANCE;
	fp->supported_class = NO_RELEVANCE;
	fp->via_class = NO_RELEVANCE;
	fp->hoe_class = NO_RELEVANCE;

	fp->next = NULL;

	return 1;
}

/******************************************************************************
 *
 * fingerprint_info()
 *
 * dump intel gathered inside the fp structure
 *
 *****************************************************************************/
void fingerprint_info(fingerprint fp) {

	printf("\nFINGERPRINT information:\n");

	if (fp.name)
		printf("NAME=%s\n", fp.name);

	if (fp.newmethod == NO_RESPONSE) printf("newmethod=NR\n");
	else printf("newmethod=%d\n", fp.newmethod);
	if (fp.accept_class == NO_RESPONSE) printf("accept_class=ignore\n");
	else printf("accept_class=%d\n", fp.accept_class);
	if (fp.allow_class == NO_RESPONSE) printf("allow_class=ignore\n");
	else printf("allow_class=%d\n", fp.allow_class);
	if (fp.supported_class == NO_RELEVANCE) printf("supported_class=ignore\n");
	else printf("supported_class=%d\n", fp.supported_class);
	if (fp.via_class == NO_RELEVANCE) printf("via_class=ignore\n");
	else printf("via_class=%d\n", fp.via_class);
	if (fp.hoe_class == NO_RELEVANCE) printf("hoe_class=ignore\n");
	else printf("hoe_class=%d\n", fp.hoe_class);
	if (fp.options == NO_RESPONSE) printf("options=NR\n");
	else printf("options=%d\n", fp.options);
	if (fp.brokenfromto == NO_RESPONSE) printf("brokenfromto=NR\n");
	else printf("brokenfromto=%d\n", fp.brokenfromto);
	if (fp.prack == NO_RESPONSE) printf("prack=NR\n");
	else printf("prack=%d\n", fp.prack);
	if (fp.ping == NO_RESPONSE) printf("ping=NR\n");
	else printf("ping=%d\n", fp.ping);
	if (fp.invite == NO_RESPONSE) printf("invite=NR\n");
	else printf("invite=%d\n", fp.invite);
	fingerprint_headers(&fp);

	printf("\n");
}

/******************************************************************************
 *
 * fingerprint_headers()
 *
 * print headers found in fingerprint 
 *
 *****************************************************************************/

void fingerprint_headers(fingerprint *fp) {

	FUNCTION(__FILE__, __FUNCTION__);

	if (fp->header_ua)		printf("  User-Agent: %s\n", fp->header_ua);
	if (fp->header_server)	printf("      Server: %s\n", fp->header_server);
}

/******************************************************************************
 *
 * fingerprint_lookup()
 *
 * lookup a fingerprint from fingerprinting database
 * We traverse the whole list and remember the best hit on the way
 *
 *****************************************************************************/

int fingerprint_lookup(fingerprint needle) {

	extern fingerprint *fpdb;
	fingerprint *besthit, *fp;
	unsigned int bestmatches, matches, possiblematches;

	FUNCTION(__FILE__, __FUNCTION__);

	if (fpdb == NULL)
		return 0;

	besthit = NULL; /* nothing found yet */
	bestmatches = matches = 0;

	possiblematches = 0;
	if (needle.options != NO_RELEVANCE) possiblematches++;
	if (needle.newmethod != NO_RELEVANCE) possiblematches++;
	if (needle.accept_class != NO_RELEVANCE) possiblematches++;
	if (needle.allow_class != NO_RELEVANCE) possiblematches++;
	if (needle.supported_class != NO_RELEVANCE) possiblematches++;
	if (needle.via_class != NO_RELEVANCE) possiblematches++;
	if (needle.hoe_class != NO_RELEVANCE) possiblematches++;
	if (needle.brokenfromto != NO_RELEVANCE) possiblematches++;
	if (needle.prack != NO_RELEVANCE) possiblematches++;
	if (needle.ping != NO_RELEVANCE) possiblematches++;
	if (needle.invite != NO_RELEVANCE) possiblematches++;
	if (possiblematches == 0)
		goto nohitpossible; /* shortcut to error */

	fp = fpdb;
	while (fp != NULL) {
		matches = 0;

#define HIT(x) if (config.debug > 0) { printf("FP match: %s\n", x); } 

		/* add more comparisons here once added to smap + DB */
		if (fp->options != NO_RELEVANCE) {
			if (fp->options == needle.options) {
				HIT("options")
				matches++;
			}
		}
		if (fp->newmethod != NO_RELEVANCE) {
			if (fp->newmethod == needle.newmethod) {
				HIT("newmethod")
				matches++;
			}
		}
		if (fp->accept_class != NO_RELEVANCE) {
			if (fp->accept_class == needle.accept_class) {
				HIT("accept_class")
				matches++;
			}
		}
		if (fp->allow_class != NO_RELEVANCE) {
			if (fp->allow_class == needle.allow_class) {
				HIT("allow_class")
				matches++;
			}
		}
		if (fp->supported_class != NO_RELEVANCE) {
			if (fp->supported_class == needle.supported_class) {
				HIT("supported_class")
				matches++;
			}
		}
		if (fp->via_class != NO_RELEVANCE) {
			if (fp->via_class == needle.via_class) {
				HIT("via_class")
				matches++;
			}
		}
		if (fp->hoe_class != NO_RELEVANCE) {
			if (fp->hoe_class == needle.hoe_class) {
				HIT("hoe_class")
				matches++;
			}
		}
		if (fp->brokenfromto != NO_RELEVANCE) {
			if (fp->brokenfromto == needle.brokenfromto) {
				HIT("brokenfromto")
				matches++;
			}
		}
		if (fp->prack != NO_RELEVANCE) {
			if (fp->prack == needle.prack) {
				HIT("prack")
				matches++;
			}
		}
		if (fp->ping != NO_RELEVANCE) {
			if (fp->ping == needle.ping) {
				HIT("ping")
				matches++;
			}
		}
		if (fp->invite != NO_RELEVANCE) {
			if (fp->invite == needle.invite) {
				HIT("invite")
				matches++;
			}
		}

		if ((matches > 0) && (matches == possiblematches)) {
			besthit = fp;
			break;
		}
		if (matches > bestmatches) {
			bestmatches = matches;
			besthit = fp;
		}
		error(ERR_DEBUG, "%d fp matches for '%s'", matches, fp->name);

		fp = fp->next;
	}
	/* print results
	 * mode -o: just print device name
	 * mode -O: more verbose information
	 */
	if (config.flags & FLAG_FINGERPRINT_SMALL) {
		printf("       Guess: %s\n", besthit->name);
        if (needle.header_ua)
			printf("  User-Agent: %s\n", needle.header_ua);
        if (needle.header_server)
			printf("      Server:%s\n", needle.header_server);
		return 0;
	}
	if (matches == possiblematches) {
		printf("device identified as:\n  %s\n", besthit->name);
		if (config.flags & FLAG_FPLEARNING)
			fingerprint_info(needle);
		else
			fingerprint_headers(&needle);
	} else if (bestmatches > 0) {
		printf("best guess (%.f%% sure) fingerprint:\n  %s\n",
			(float) ((bestmatches*100)/possiblematches),
			besthit->name);
		if (config.flags & FLAG_FPLEARNING)
			fingerprint_info(needle);
		else
			fingerprint_headers(&needle);
	} else {
nohitpossible:
		printf("\nAttention: could not find a match in the database\n"
			   "           Please send the following information along with\n"
			   "           client IP and type to hscholz@raisdorf.net.\n");
		fingerprint_info(needle);
	}

	return 0;
}

/******************************************************************************
 *
 * fingerprint_loaddb()
 *
 * load fingerprinting database into memory
 *
 *****************************************************************************/

#define EAT_SPACES(p) {while ((*p == ' ') || (*p == '\t')) { p++; }}

int fingerprint_loaddb(void) {

	extern fingerprint *fpdb;
	fingerprint	*fp, *lastfp;
#define LINELEN 256
	char line[LINELEN], *p, *n, *fpfile;
	FILE *file;
	int *val, iskeyvalue;

	FUNCTION(__FILE__, __FUNCTION__);

	fpdb = fp = lastfp = NULL; /* nothing there yet */

	/* try to find a fingerprint database */
	fpfile = NULL;
	if (access(FPDBFILE_LOCAL, R_OK) == 0) {
		fpfile = FPDBFILE_LOCAL;
	} else if (access(FPDBFILE, R_OK) == 0) {
		fpfile = FPDBFILE;
	} else if (access(FPDBFILE_CWD, R_OK) == 0) {
		fpfile = FPDBFILE_CWD;
	} else {
		error(ERR_NOTICE, "cannot locate fingerprint database");
		return 0;
	}

	file = fopen(fpfile, "r");
	if (!file) {
		fprintf(stderr, "cannot open fingerprint database '%s'\n", FPDBFILE);
		return 0;
	}
	while (fgets(line, LINELEN-1, file)) {

		iskeyvalue = 0;
		p = line;
		EAT_SPACES(p);

		/* new fingerprint */
		if (!strncasecmp("fingerprint", p, 11)) {
			/* format: keyword fingerprint followed by a string including
			 * white spaces
			 */
			p += 11;
			EAT_SPACES(p);
			n = strstr(p, "\r\n"); /* end of line */
			if (!n)
				n = strstr(p, "\n");
			if (!n)
				return 0; /* no end of line? -> end of file? */

			/* allocate and prepare new fingerprint */
			fp = malloc(sizeof(fingerprint));
			if (!fp) {
				perror("malloc");
				return 0;
			}
			fingerprint_init(fp);
			fp->name = malloc((int) (n-p+1));
			if (!fp->name) {
				perror("malloc");
				return 0;
			}
			memcpy(fp->name, p, (int) (n-p));
			fp->name[n-p] = '\0';

			/* append new fingerprint to FP list */
			if (fpdb == NULL) {
				/* first fp in database */
				fpdb = fp;
				lastfp = fp;
			} else {
				/* append to list */
				lastfp->next = fp;
				lastfp = fp;
			}
		} else if (!strncasecmp("options", p, 7)) {
			p+= 7;
			iskeyvalue++;
			val = &fp->options;
		} else if (!strncasecmp("accept_class", p, 12)) {
			p+= 12;
			iskeyvalue++;
			val = &fp->accept_class;
		} else if (!strncasecmp("allow_class", p, 11)) {
			p+= 11;
			iskeyvalue++;
			val = &fp->allow_class;
		} else if (!strncasecmp("supported_class", p, 15)) {
			p+= 15;
			iskeyvalue++;
			val = &fp->supported_class;
		} else if (!strncasecmp("via_class", p, 9)) {
			p+= 9;
			iskeyvalue++;
			val = &fp->via_class;
		} else if (!strncasecmp("hoe_class", p, 9)) {
			p+= 9;
			iskeyvalue++;
			val = &fp->hoe_class;
		} else if (!strncasecmp("newmethod", p, 9)) {
			p+= 9;
			iskeyvalue++;
			val = &fp->newmethod;
		} else if (!strncasecmp("brokenfromto", p, 12)) {
			p+= 12;
			iskeyvalue++;
			val = &fp->brokenfromto;
		} else if (!strncasecmp("prack", p, 5)) {
			p+= 5;
			iskeyvalue++;
			val = &fp->prack;
		} else if (!strncasecmp("ping", p, 4)) {
			p+= 4;
			iskeyvalue++;
			val = &fp->ping;
		} else if (!strncasecmp("invite", p, 6)) {
			p+= 6;
			iskeyvalue++;
			val = &fp->invite;
		}
	
		/* assign value of key/value pairs */
		if (iskeyvalue) {
			EAT_SPACES(p);
			if (*p != '=') {
				//fprintf(stderr, "ignoring bogus line\n");
				continue;
			}
			p++;
			EAT_SPACES(p);
			*val = fingerprint_getvalue(p);			
		} /* end iskeyvalue */
	}
	fclose(file);

	return 1;
}

/******************************************************************************
 *
 * fingerprint_dumpdb()
 *
 * dump the internal fingerprinting database content to stdout
 * only useful for debugging
 *
 *****************************************************************************/

void fingerprint_dumpdb(void) {

	extern fingerprint *fpdb;
	fingerprint *fp;

	FUNCTION(__FILE__, __FUNCTION__);

	fp = fpdb;
	while (fp) {
		fingerprint_info(*fp);
		fp = fp->next;
	}
}

/******************************************************************************
 *
 * fingerprint_free()
 *
 * free data dynamically allocated while filling a fingerprint structure
 *
 *****************************************************************************/

void fingerprint_free(fingerprint fp) {

	FUNCTION(__FILE__, __FUNCTION__);

	if (fp.name) free(fp.name);
	if (fp.header_ua) free(fp.header_ua);
	if (fp.header_server) free(fp.header_server);

}

/******************************************************************************
 *
 * fingerprint_destroydb()
 *
 * gracefully destroy the fingerprint db by free()ing all memory
 *
 *****************************************************************************/

void fingerprint_destroydb(void) {

	extern fingerprint *fpdb;
	fingerprint *fp, *oldfp;

	FUNCTION(__FILE__, __FUNCTION__);

	fp = fpdb;
	while (fp) {
		fingerprint_free(*fp);
		oldfp = fp;
		fp = fp->next;
		free(oldfp);
	}
}

/******************************************************************************
 *
 * fingerprint_getvalue()
 *
 * return decimal value of strig pointed to
 *        or -1 (NO_RESPONSE) in case of string 'NR'
 *		  or -2 in case of error
 *
 *****************************************************************************/

int fingerprint_getvalue(char *p) {
	unsigned int val;

	if (p == NULL)
		return -2;

	if (!strncasecmp("nr", p, 2))
		return (NO_RESPONSE); /* NR found */
	if (!strncasecmp("ignore", p, 6))
		return (NO_RELEVANCE); /* ignore found */

	val = 0;
	while ((p != NULL) && (*p >= '0') && (*p <= '9')) {
		val *= 10;
		val += *p - '0';
		p++;
	}
	return val;
}


#endif /* _FINGERPRINT_C */
