/******************************************************************************

                              Copyright (c) 2009
                            Lantiq Deutschland GmbH
                     Am Campeon 3; 85579 Neubiberg, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

******************************************************************************/

/**
   \file drv_tapi_meter.c
   TAPI Metering Service.
*/

/* ============================= */
/* Check if feature is enabled   */
/* ============================= */
#ifdef HAVE_CONFIG_H
#include <drv_config.h>
#endif

#ifdef TAPI_METERING

/* ============================= */
/* Includes                      */
/* ============================= */

#include "drv_tapi.h"
#include "drv_tapi_ll_interface.h"
#include "drv_tapi_errno.h"

/* ============================= */
/* Type declarations             */
/* ============================= */

struct TAPI_METER_DATA
{
   /* is metering currently active? */
   IFX_boolean_t      bMeterActive;
   /* last linemode befor metering start */
   IFX_uint8_t        nLastLineMode;
};

/* ============================= */
/* Local function declaration    */
/* ============================= */

/* ============================= */
/* Local function definitions    */
/* ============================= */

/* ============================= */
/* Global function definitions   */
/* ============================= */

/**
   Initialise metering on the given channel.

   Initialise the data structures and resources needed for metering.

   \param   pChannel    Pointer to TAPI_CHANNEL structure.

   \return
     - \ref IFX_SUCCESS: if successful
     - \ref IFX_ERROR: in case of an error
*/
IFX_int32_t IFX_TAPI_Meter_Initialise(TAPI_CHANNEL *pChannel)
{
   TAPI_METER_DATA_t *pTapiMeterData = pChannel->pTapiMeterData;

   /* check if channel has the required analog module */
   if (pChannel->nChannel >=  pChannel->pTapiDevice->nResource.AlmCount)
   {
      /* no analog module -> nothing to do  --  this is not an error */
      return TAPI_statusOk;
   }

   if (pTapiMeterData != IFX_NULL)
   {
      /* already initialised - do nothing */
      return TAPI_statusOk;
   }

   TAPI_OS_MutexGet (&pChannel->semTapiChDataLock);

   /* allocate data storage on the channel */
   pTapiMeterData = TAPI_OS_Malloc (sizeof (*pTapiMeterData));
   if (pTapiMeterData == IFX_NULL)
   {
      TAPI_OS_MutexRelease (&pChannel->semTapiChDataLock);
      RETURN_STATUS (TAPI_statusNoMem, 0);
   }

   /* Store pointer to data in the channel or we lose it on exit. */
   pChannel->pTapiMeterData = pTapiMeterData;
   memset (pTapiMeterData, 0x00, sizeof (*pTapiMeterData));

   TAPI_OS_MutexRelease (&pChannel->semTapiChDataLock);

   return TAPI_statusOk;
}


/**
   Cleanup metering on the given channel.

   Free the resources needed for the metering.

   \param   pChannel    Pointer to TAPI_CHANNEL structure.
*/
IFX_void_t IFX_TAPI_Meter_Cleanup (TAPI_CHANNEL *pChannel)
{
   TAPI_OS_MutexGet (&pChannel->semTapiChDataLock);

   if (pChannel->pTapiMeterData != IFX_NULL)
   {
      /* free the data storage on the channel */
      TAPI_OS_Free (pChannel->pTapiMeterData);
      pChannel->pTapiMeterData = IFX_NULL;
   }

   TAPI_OS_MutexRelease (&pChannel->semTapiChDataLock);
}


/**
   Return the status of the metering module.

   \param   pChannel    Pointer to TAPI_CHANNEL structure.

   \return
     - \ref IFX_TRUE:  if metering is active
     - \ref IFX_FALSE: if metering is not active or not installed on channel
*/
IFX_boolean_t TAPI_Phone_Meter_IsActive (TAPI_CHANNEL *pChannel)
{
   if (pChannel->pTapiMeterData != IFX_NULL)
   {
      return pChannel->pTapiMeterData->bMeterActive;
   }
   else
   {
      return IFX_FALSE;
   }
}

/**
   Stops the metering.

   \param pChannel      Pointer to TAPI_CHANNEL structure.

   \return
   TAPI_statusOk or error code
*/
IFX_int32_t TAPI_Phone_Meter_Stop (TAPI_CHANNEL *pChannel)
{
   IFX_int32_t ret = TAPI_statusOk;

   if (pChannel->pTapiMeterData == IFX_NULL)
   {
      /* metering not initialised on this channel */
      RETURN_STATUS (TAPI_statusInitMeterFail, 0);
   }

   if (pChannel->pTapiMeterData->bMeterActive == IFX_TRUE)
   {
      /* Metering is deactivated */
      pChannel->pTapiMeterData->bMeterActive = IFX_FALSE;
      /* recover the last line mode */
      ret = TAPI_Phone_Set_Linefeed (pChannel,
                                     pChannel->pTapiMeterData->nLastLineMode);
   }

   return ret;
}

/**
   Generates a metering burst.

   \param pChannel      Pointer to TAPI_CHANNEL structure.
   \param pMeterBurst   Contains the burst settings (IFX_TAPI_METER_BURST_t)

   \return
   TAPI_statusOk or error code
*/
IFX_int32_t TAPI_Phone_Meter_Burst (TAPI_CHANNEL *pChannel,
                                    IFX_TAPI_METER_BURST_t const *pMeterBurst)
{
   TAPI_METER_DATA_t *pTapiMeterData = pChannel->pTapiMeterData;
   IFX_TAPI_DRV_CTX_t *pDrvCtx = pChannel->pTapiDevice->pDevDrvCtx;
   IFX_int32_t retLL;

   if (pTapiMeterData == IFX_NULL)
   {
      /* metering not initialised on this channel */
      RETURN_STATUS (TAPI_statusInitMeterFail, 0);
   }

   /* check if meter characteristic ist set correctly */
   if (pMeterBurst->nPulses == 0)
   {
      /* bad params */
      RETURN_STATUS (TAPI_statusParam, 0);
   }

   /* check if telephone isn't ringing */
   if (IFX_TAPI_Ring_IsActive (pChannel) == IFX_TRUE)
   {
      /* channel is just ringing */
      RETURN_STATUS (TAPI_statusRingAct, 0);
   }

   /* check if line is in active state */
   if (! (pChannel->TapiOpControlData.nLineMode == IFX_TAPI_LINE_FEED_ACTIVE ||
       pChannel->TapiOpControlData.nLineMode == IFX_TAPI_LINE_FEED_ACTIVE_REV ||
       pChannel->TapiOpControlData.nLineMode == IFX_TAPI_LINE_FEED_NORMAL_AUTO ||
       pChannel->TapiOpControlData.nLineMode == IFX_TAPI_LINE_FEED_REVERSED_AUTO))
   {
      RETURN_STATUS (TAPI_statusLineNotAct, 0);
   }

   /* save line mode */
   pTapiMeterData->nLastLineMode = pChannel->TapiOpControlData.nLineMode;

   /* send first pulse */
   if (ptr_chk (pDrvCtx->ALM.Metering_Start, "pDrvCtx->ALM.Metering_Start"))
   {
      retLL = pDrvCtx->ALM.Metering_Start (pChannel->pLLChannel,
                                         pMeterBurst->nPulses);

      if (TAPI_SUCCESS (retLL))
      {
         pTapiMeterData->bMeterActive = IFX_TRUE;
      }
      else
      {
         RETURN_STATUS (TAPI_statusMeterBurstFail, retLL);
      }
   }
   else
   {
      /* metering is already active */
      RETURN_STATUS (TAPI_statusNotSupported, 0);
   }

   RETURN_STATUS (TAPI_statusOk, 0);
}

#endif /* TAPI_METERING */
