/*
 * RoutingExtension.h
 *
 * Copyright (C) 2013 IBR, TU Braunschweig
 *
 * Written-by: Johannes Morgenroth <morgenroth@ibr.cs.tu-bs.de>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#ifndef ROUTINGEXTENSION_H_
#define ROUTINGEXTENSION_H_

#include "storage/BundleResult.h"
#include "routing/NeighborDatabase.h"
#include "routing/NodeHandshake.h"
#include "core/Event.h"
#include "core/Node.h"
#include <ibrdtn/data/BundleID.h>
#include <ibrdtn/data/EID.h>

namespace dtn
{
	namespace routing
	{
		class BaseRouter;

		class RoutingResult
		 : public dtn::storage::BundleResult, public std::list<std::pair<dtn::data::MetaBundle, dtn::core::Node::Protocol> >
		{
		public:
			RoutingResult();
			virtual ~RoutingResult();

			virtual void put(const dtn::data::MetaBundle &bundle) throw ();
			virtual void put(const dtn::data::MetaBundle &bundle, const dtn::core::Node::Protocol p) throw ();
		};

		class RoutingExtension
		{
			static const std::string TAG;

		public:
			RoutingExtension();
			virtual ~RoutingExtension() = 0;

			virtual void componentUp() throw () = 0;
			virtual void componentDown() throw () = 0;

			/**
			 * Returns a tag used to identify the routing extension in filtering rules
			 */
			virtual const std::string getTag() const throw () { return "default"; }

			/**
			 * This method is called every time something has changed. The module
			 * should search again for bundles to transfer to the given peer.
			 */
			virtual void eventDataChanged(const dtn::data::EID &peer) throw () { };

			/**
			 * This method is called every time a bundle has been completed successfully
			 */
			virtual void eventTransferCompleted(const dtn::data::EID &peer, const dtn::data::MetaBundle &meta) throw () { };

			/**
			 * This method is called every time a bundle was queued
			 */
			virtual void eventBundleQueued(const dtn::data::EID &peer, const dtn::data::MetaBundle &meta) throw () { };

			/**
			 * If some data of another node is required. These method is called to collect all
			 * necessary identifier of data items.
			 * @param The EID of the other node.
			 * @param The object to put the requested identifier into.
			 */
			virtual void requestHandshake(const dtn::data::EID&, NodeHandshake&) const { };

			/**
			 * If a handshake message is received, this method is called to collect the different
			 * data items generated by the router extensions.
			 * @param The EID of the other node.
			 * @param The received handshake object.
			 * @param The response for the received handshake.
			 */
			virtual void responseHandshake(const dtn::data::EID&, const NodeHandshake&, NodeHandshake&) { };

			/**
			 * After a handshake has been completed every module can process the handshake response.
			 * @param The EID of the other node.
			 * @param The received handshake object.
			 */
			virtual void processHandshake(const dtn::data::EID&, NodeHandshake&) { };

		protected:
			/**
			 * Transfer one bundle to another node.
			 * @throw BundleNotFoundException if the bundle do not exist.
			 * @param destination The EID of the other node.
			 * @param id The ID of the bundle to transfer. This bundle must be stored in the storage.
			 */
			void transferTo(const dtn::data::EID &destination, const dtn::data::MetaBundle &meta, const dtn::core::Node::Protocol);

			BaseRouter& operator*();
		};
	} /* namespace routing */
} /* namespace dtn */
#endif /* ROUTINGEXTENSION_H_ */
