/*   FILE: lcdgrilo.vala -- A simple DAAP music player
 * AUTHOR: W. Michael Petullo <mike@flyn.org>
 *   DATE: 27 November 2013
 *
 * Copyright (c) 2013 W. Michael Petullo <new@flyn.org>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using Grl;
using Pifacecad;
using Posix;

const uint LIGHTS_OUT_SECONDS = 10;
const uint SWITCH_POLL_MILLISECONDS = 1;

bool no_pifacecad;
string tag = null;

const OptionEntry[] options = {
	{ "no-pifacecad", 'n', 0, OptionArg.NONE, ref no_pifacecad, "Do not try to use PiFace CAD for I/O", null },
	{ "tag", 't', OptionArg.NONE, OptionArg.STRING, ref tag, "Only show sources with the following tag", null },
	{ null }
};


private class LCDPlayer {
	public Stack<State> stateStack { get; set; default = null; }
	public dynamic Gst.Element element;

	private MainLoop loop;
	private Gee.ArrayList<Grl.Source> sources = new Gee.ArrayList<unowned Grl.Source> ();

	public void source_added_cb (Grl.Source source) {
		if ((source.get_supported_operations() & Grl.SupportedOps.BROWSE) == 0)
			return;

		if (tag != null) {
			if (! (tag in source.get_tags()))
				return;
		}

		sources.add (source);

		if (stateStack == null)
			return;

		// Possibly replace "No media source found" message.
		stateStack.peek ().print_selected ();
	}

	public void source_removed_cb (Grl.Source source) {
		sources.remove (source);
	}

	public LCDPlayer () {
		var registry = Grl.Registry.get_default ();

		registry.source_added.connect (source_added_cb);
		registry.source_removed.connect (source_removed_cb);

		if (tag == null) {
			try {
				registry.load_plugin_by_id ("grl-daap");
				// registry.load_plugin_by_id ("grl-jamendo");
				// registry.load_plugin_by_id ("grl-magnatune");
			} catch (GLib.Error e) {
				GLib.error ("%s", e.message);
			}
		} else {
			try {
				registry.load_all_plugins ();
			} catch (GLib.Error e) {
				GLib.error ("%s", e.message);
			}
		}

		loop    = new MainLoop ();
		element = Gst.ElementFactory.make ("playbin", "play");
		//audio and soft-volume, don't decode video
		element.flags = 0x00000002 | 0x00000010;

		stateStack = new Stack<State> ();

		// FIXME: Fix this throughout: write wrapper functions.
		if (! no_pifacecad) {
			Pifacecad.open ();
			Pifacecad.lcd_backlight_on ();

			Timeout.add_seconds (LIGHTS_OUT_SECONDS, () => {
				Pifacecad.lcd_backlight_off (); return false;
			});
		}

		stateStack.push (new StateChooseSource (loop, this, sources));

		loop.run ();

		if (! no_pifacecad) {
			Pifacecad.close ();
		}
	}
}

int main (string[] args) {     
	// FIXME: Fix these: they monopolize the results of --help.
	Gst.init (ref args);

	try {
		var opt_context = new OptionContext ("- Media player");
		opt_context.set_help_enabled (true);
		//opt_context.add_group (Grl.init_get_option_group ());
		Grl.init (ref args);
		opt_context.add_main_entries (options, null);
		opt_context.parse (ref args);
	} catch (OptionError e) {
		GLib.error ("%s", e.message);
	}

	Posix.termios old_tio, new_tio;
	Posix.tcgetattr(Posix.STDIN_FILENO, out old_tio);
	new_tio = old_tio;
	new_tio.c_lflag &=(~ICANON & ~ECHO);
	Posix.tcsetattr(STDIN_FILENO, TCSANOW, new_tio);

	new LCDPlayer ();

	Posix.tcsetattr(STDIN_FILENO, TCSANOW, old_tio);

	return 0;
}
