/******************************************************************************
 *
 * ack.c - part of smap
 * 
 * send ACK to prevent further retransmissions
 *
 * $Id: ack.c,v 1.6 2007-10-20 19:47:41 hscholz Exp $
 *****************************************************************************/
#ifndef _ACK_C
#define _ACK_C

#include "config.h"

/******************************************************************************
 *
 * ack_o_matic()
 *
 * takes a SIP response and a socket and decides whether to send a ACK
 * to acknowledge the response
 *
 * Returns:
 *
 *   0: if no msg was sent
 *   1: if ACK was sent
 *
 *****************************************************************************/

int ack_o_matic(task_t *task, char *response) {

	int len, rc;
	char buf[SIPLEN];

	char *callid, *from, *to, *via;
	int callidlen, fromlen, tolen, vialen;
	unsigned int cseq;

	if (!task || !response) return 0;

	/* before doing anything decide if we have to do anything at all */
	if (response_reason(&rc, response) != 1) {
		/* could not find a reason code so bail out */
		return 0;
	}
	if (rc < 200)	return 0;
	if (rc == 481)	return 0; /* Call Leg/Transaction Does Not Exist */

	/* obtain cseq */
	sip_getcseq(response, &cseq);

	/* obtain Call-ID */
	callid = NULL; callidlen = 0;
	sip_getheaderp(response, "\r\nCall-ID:", &callid, &callidlen);

	/* obtain From header */
	from = NULL; fromlen = 0;
	sip_getheaderp(response, "\r\nFrom:", &from, &fromlen);

	/* obtain To header */
	to = NULL; tolen = 0;
	sip_getheaderp(response, "\r\nTo:", &to, &tolen);

	/* obtain Via header
	 * RFC 3261 clearly states that we must use the same Via header as
	 * used in the initial request. Since we are mostly stateless we
	 * simply copy the Via from the response knowing that there might
	 * be additional tags (i.e. received) in there
	 */
	via = NULL; vialen = 0;
	sip_getheaderp(response, "\r\nVia:", &via, &vialen);

	len = snprintf(buf, SIPLEN, "ACK sip:%s SIP/2.0\r\n"
		"%.*s\r\n"	/* Via */
		"%.*s\r\n"	/* From */
		"%.*s\r\n"	/* To */
		"%.*s\r\n"	/* Call-ID */
		"CSeq: %d ACK\r\n"
		"Content-Length: 0\r\n"
		"User-Agent: smap %s\r\n",

		config.sip_domain,		/* SIP domain */	
		vialen, via,			/* Via header including tag */
		fromlen, from,			/* From header including tag */
		tolen, to,				/* To header including tag */
		callidlen, callid,		/* Call-ID */
		cseq,					/* CSeq */
		SMAP_VERSION			/* smap version string */
	);

	error(ERR_DEBUG, "ACK: %s\n--- end of ACK--", buf);

	if (!udp_send(task, buf)) {
		error(ERR_ERROR, "error sending ACK");
		return 0;
	}

	return 1;
}
#endif /* _ACK_C */
