/******************************************************************************
 *
 * localip.c - part of smap
 *
 * obtain the local IP address ... needed for correct SIP signalling
 *
 * $Id: localip.c,v 1.9 2007-10-07 10:15:11 hscholz Exp $ 
 *****************************************************************************/

#ifndef _LOCALIP_C
#define _LOCALIP_C

#ifdef __linux__
#if SMAP_OS == linux-ifaddrs
#define HAVE_IFADDRS
#endif
#endif

#ifdef __APPLE__
#define HAVE_IFADDRS
#endif

#ifdef __FreeBSD__
#define HAVE_IFADDRS
#endif

#ifdef __OpenBSD__
#define HAVE_IFADDRS
#endif

#include "config.h"

/******************************************************************************
 *
 * filter_localip()
 *
 * check if the given IP is loopback or multicast and thus not suitable
 * for using inside signalling.
 * We would settle for a RFC1918 address (i.e. 192.168.0.1) even though
 * it ain't perfect.
 *
 * Return 1 if the IP isn't any good.
 *
 * IP addresses could be:
 *
 * - loopback (127.0.0.0/8)
 * - multicast (224.0.0.0/4)
 *
 *****************************************************************************/

#define ADDR ntohl(sin->sin_addr.s_addr)

int filter_localip(struct sockaddr_in *sin) {

	/* loopback */
	if ((ADDR >= 2130706432u) && (ADDR <= 2147483647u))
		return 0;
	/* multicast */
	if ((ADDR >= 3758096384u) && (ADDR <= 4294967295u))
		return 0;

	return 1;
}

/******************************************************************************
 *
 * get_localip()
 *
 * store the local IP in config.localip for later usage
 * return 1 if successful, 0 in case of errors
 * 
 *****************************************************************************/

#ifdef HAVE_IFADDRS
int get_localip (void) {
	int s;
	char ip[200];
	struct ifaddrs *ifa = NULL;

 	s = socket (PF_INET, SOCK_STREAM, 0);
	if (!s) {
		if (DEBUG) perror("socket");
		return 0;
	}

	if (getifaddrs (&ifa) < 0) {
		if (DEBUG) perror ("getifaddrs");
		close(s);
		return 0;
	}

	for (; ifa; ifa = ifa->ifa_next) {
		/* we only do IPv4 now ... sorry */
		if (ifa->ifa_addr->sa_family != AF_INET)
			continue;

		if (!filter_localip((struct sockaddr_in *)ifa->ifa_addr))
			continue;

		if (getnameinfo (ifa->ifa_addr, sizeof(struct sockaddr_in),
			ip, sizeof (ip), NULL, 0, NI_NUMERICHOST) < 0) {
			if (DEBUG) perror ("getnameinfo");
				continue;
		}
		break; /* we found one */
	}
	for (; ifa; ifa = ifa->ifa_next)
		; /* freeifaddrs() crashes if we don't do this ... weird */

	config.localip = strdup(ip);
	freeifaddrs (ifa);
	close (s);
	return 1;
}
#else /* no ifaddrs.h present */

int get_localip(void) {
#warning "get_localip() uses ioctl()"

	int s, num, iface, ifacelen;
	char buf[BUFSIZ];
	struct ifconf ifc;
	struct ifreq *ifr;
	struct in_addr addr;

	s = socket(PF_INET, SOCK_DGRAM, IPPROTO_IP);
	if (!s) {
		if (DEBUG) perror("socket");
		return 0;
	}

	/* obtain number of interfaces */
	ifc.ifc_len = sizeof(buf);
	ifc.ifc_buf = buf;

	if (ioctl(s, SIOCGIFCONF, &ifc) == -1) {
		if (DEBUG) perror("ioctl");
		close(s);
		return 0;
	}
	ifr = ifc.ifc_req;
	num = ifc.ifc_len / sizeof(struct ifreq);

	addr.s_addr = 0;

	/* loop */
	for ( iface = 0; iface<num ; iface++) {
		if (ioctl(s, SIOCGIFADDR, &ifr[iface]) != 0) {
			if (DEBUG) perror("ioctl");
			continue;
		}
		if (filter_localip((struct sockaddr_in *) &(*(struct sockaddr_in *)&ifr[iface].ifr_addr)))
			continue;
	}

	ifacelen = strlen(inet_ntoa(addr));
	config.localip = malloc(ifacelen + 1);
	if (config.localip == NULL) {
		if (DEBUG) perror("malloc");
		close(s);
		return 0;
	}
	snprintf(config.localip, ifacelen + 1, "%s", inet_ntoa(addr));
	error(ERR_DEBUG, "local IP: %s", config.localip);

	close(s);
	return 1;
}

#endif /* WITH_IFADDRS */

#endif /* _LOCALIP_C */
