/******************************************************************************
 *
 * randomizer.c - part of smap
 *
 * randomizers used inside SIP message generators
 * 
 * $Id: randomizer.c,v 1.5 2007-10-07 10:15:11 hscholz Exp $ 
 *****************************************************************************/

#ifndef _RANDOMZIER_C
#define _RANDOMZIER_C

#include "config.h"

/******************************************************************************
 *
 * randomizer_init()
 *
 * initialize the pseudo random number generator
 * This method seems to be quite effective
 *
 *****************************************************************************/
void randomizer_init(void) {
	FUNCTION(__FILE__, __FUNCTION__);

#ifdef HAVE_RANDOM
	srandom((unsigned) time(NULL) ^ getpid());
#else
	srand((unsigned) time(NULL) ^ getpid());
#endif

}

/******************************************************************************
 *
 * randomizer() 
 *
 * fill the refered buffer with some random value depending on the
 * request
 * If possible (Call-ID, CSeq) the integer value will be returned.
 * On error the buffer will be zeroed out and 0 will be returned.
 * Bear in mind that 0 is a legal return code for this function.
 *
 *****************************************************************************/
unsigned int randomizer(int type, char *buf, int len) {
	unsigned int i;
	unsigned long l,m;

	FUNCTION(__FILE__, __FUNCTION__);

	switch (type) {
	case RAND_CALLID:
		/* Call-ID
		 * some integer should do it for us
		 */
		if (!buf) return 0;
#ifdef HAVE_RANDOM
		i = (unsigned int) random();
#else
		i = (unsigned int) rand();
#endif
		snprintf(buf, len, "%u", i);
		break;
	case RAND_TAG:
		/* To and From tag */
		if (!buf) return 0;
#ifdef HAVE_RANDOM
		l = (unsigned long) random();
		m = (unsigned long) random();
#else
		l = (unsigned long) rand();
		m = (unsigned long) rand();
#endif
		snprintf(buf, len, "%lx%lx", l, m);
		break;
	case RAND_CSEQ:
		/* RFC 3261 says: "MUST be expressible as a 32-bit unsigned
		 *                 integer and MUST be less than 2**31"
		 * Since some UAs might freak out if CSeq is too high we
		 * reduce the max CSeq to something smaller
		 */
#ifdef HAVE_RANDOM
		i = (unsigned long) ((random() % 0xFFFF) + 1);
#else
		i = (unsigned long) ((rand() % 0xFFFF) + 1);
#endif
		buf = NULL;
		break;
	case RAND_BRANCH:
		/* branch tag appendix ... we use a simple integer */
		buf = NULL;
#ifdef HAVE_RANDOM
		i = (unsigned long) ((random() % 0xFFFF) + 1);
#else
		i = (unsigned long) ((rand() % 0xFFFF) + 1);
#endif
		break;
	default:
		fprintf(stderr, "%s:%s: internal error\n", __FILE__, __FUNCTION__);
		if (buf)
			bzero(&buf, len);
		return 0;
		break;
	}

	/* return callid value if asked for Call-ID */
	return ((type == RAND_CALLID) || (type == RAND_CSEQ)
			|| (type == RAND_BRANCH)) ? i : 0;
}

#endif /* _RANDOMZIER_C */
