/******************************************************************************
 *
 * request.c - part of smap
 *
 * SIP message generator
 * 
 * $Id: request.c,v 1.7 2007-10-07 10:15:11 hscholz Exp $
 *****************************************************************************/
#ifndef _REQUEST_C
#define _REQUEST_C

#include "config.h"

/******************************************************************************
 *
 * request_generate()
 *
 * write a specific SIP request into buf which is expected to hold at least
 * SIPLEN Bytes.
 *
 * parameters:
 *
 * - test: symbol from SIP test enumeration
 * - buf: pointer to buffer capable of holding SIPLEN bytes
 * - localport: local port to use in signalling
 * - proto: protocol string (i.e. "UDP" or "TCP")
 * - ident: pointer to Call-ID generated inside request_generate
 *
 * Return 0 in case of error, 1 if request is usable for tests
 *
 *****************************************************************************/
int request_generate(unsigned int test, char *buf,
						unsigned int localport,
						unsigned int proto,
						unsigned int *ident) {

	int len;
	unsigned int branch;

	FUNCTION(__FILE__, __FUNCTION__);

	if ((buf == NULL) || (proto == 0) || (ident == NULL)) {
		error(ERR_ERROR, "NULL");
		return 0;
	}

	branch = randomizer(RAND_BRANCH, NULL, 0);

	switch (test) {
	case SIP_OPTIONS:
		/* SIP OPTIONS
	 	 *
		 * this generates a valid OPTIONS request
		 * The test allows us to fetch some headers from the UAS
		 * ... if it responds at all
	 	 */
		{

		char callid[CALLID_LEN];
		char fromtag[TAG_LEN];
		*ident = randomizer(RAND_CALLID, (char *) &callid, CALLID_LEN);
		randomizer(RAND_TAG, (char *) &fromtag, TAG_LEN);
		len = snprintf(buf, SIPLEN, "OPTIONS sip:smap@%s SIP/2.0\r\n"
			"Via: SIP/2.0/%s %s:%d;branch=z9hG4bK.%d;rport;alias\r\n"
			"From: <sip:smap@%s:%d>;tag=%s\r\n"
			"To: <sip:smap@%s>\r\n"
			"Call-ID: %s@%s\r\n"
			"CSeq: %d OPTIONS\r\n"
			"Contact: <sip:smap@%s:%d>\r\n"
			"Content-Length: 0\r\n"
			"Max-Forwards: 70\r\n"
			"User-Agent: smap %s\r\n"
			"Accept: text/plain\r\n\r\n",

			config.sip_domain,		/* SIP domain */
			TRANSPORT_STRING(proto),/* Via protocol */
			config.localip,			/* Via host */
			localport,				/* Via port */
			branch,					/* Via branch */
			config.localip,			/* From IP */
			localport,				/* From port */
			fromtag,				/* From tag */
			config.sip_domain,		/* To hostpart */
			callid,					/* Call-ID local part */
			config.localip,			/* Call-ID hostpart */
			randomizer(RAND_CSEQ, NULL, 0),	/* CSeq */
			config.localip,			/* Contact hostpart */
			localport,				/* Contact port */
			SMAP_VERSION			/* smap version */
			);
		}
		break;
	case SIP_NEWMETHOD:
		/* New Method test
	 	 *
		 * This test generates a valid message ... beside the invalid
		 * request type
		 *
		 * We do this to make the UAS respond with an Allow header including
		 * the features it supports. The Allow header is a major
		 * source of information.
		 *
	 	 */
		{

		char callid[CALLID_LEN];
		char fromtag[TAG_LEN];
		*ident = randomizer(RAND_CALLID, (char *) &callid, CALLID_LEN);
		randomizer(RAND_TAG, (char *) &fromtag, TAG_LEN);
		len = snprintf(buf, SIPLEN, "NEWMETHOD sip:%s SIP/2.0\r\n"
			"Via: SIP/2.0/%s %s:%d;branch=z9hG4bK.%d;rport\r\n"
			"From: <sip:smap@%s:%d>;tag=%s\r\n"
			"To: <sip:%s>\r\n"
			"Call-ID: %s@%s\r\n"
			"CSeq: %d NEWMETHOD\r\n"
			"Contact: <sip:smap@%s:%d>\r\n"
			"Content-Length: 0\r\n"
			"Max-Forwards: 70\r\n"
			"User-Agent: smap %s\r\n"
			"Accept: text/plain\r\n\r\n",

			config.sip_domain,		/* SIP domain */
			TRANSPORT_STRING(proto),/* Via protocol */
			config.localip,			/* Via host */
			localport,				/* Via port */
			branch,					/* Via branch */
			config.sip_domain,		/* From domain */
			localport,				/* From port */
			fromtag,				/* From tag */
			config.sip_domain,		/* To domain */
			callid,					/* Call-ID local part */
			config.localip,			/* Call-ID hostpart */
			randomizer(RAND_CSEQ, NULL, 0),	/* CSeq */
			config.localip,			/* Contact hostpart */
			localport,				/* Contact port */
			SMAP_VERSION			/* smap version */
			);
		}
		break;
	case SIP_BROKENFROMTO:
		/* SIP OPTIONS with broken From/To field
		 *
		 * we fire off a request with a broken From + To field by leaving out
		 * the leading 'sip:' inside the URIs
		 * the UAS should responed with a 400 class error
		 */
		{

		char callid[CALLID_LEN];
		char fromtag[TAG_LEN];
		*ident = randomizer(RAND_CALLID, (char *) &callid, CALLID_LEN);
		randomizer(RAND_TAG, (char *) &fromtag, TAG_LEN);
		len = snprintf(buf, SIPLEN, "OPTIONS sip:%s SIP/2.0\r\n"
			"Via: SIP/2.0/%s %s:%d;branch=z9hG4bK.%d;rport\r\n"
			"From: <smap@%s:%d>;tag=%s\r\n"
			"To: <%s>\r\n"
			"Call-ID: %s@%s\r\n"
			"CSeq: %d OPTIONS\r\n"
			"Contact: <sip:smap@%s:%d>\r\n"
			"Content-Length: 0\r\n"
			"Max-Forwards: 70\r\n"
			"User-Agent: smap %s\r\n"
			"Accept: text/plain\r\n\r\n",

			config.sip_domain,		/* SIP domain */
			TRANSPORT_STRING(proto),/* Via protocol */
			config.localip,			/* Via host */
			localport,				/* Via port */
			branch,					/* Via branch */
			config.sip_domain,		/* From domain */
			localport,				/* From port */
			fromtag,				/* From tag */
			config.sip_domain,		/* To domain */
			callid,					/* Call-ID local part */
			config.localip,			/* Call-ID hostpart */
			randomizer(RAND_CSEQ, NULL, 0),	/* CSeq */
			config.localip,			/* Contact hostpart */
			localport,				/* Contact port */
			SMAP_VERSION			/* smap version */
			);
		}
		break;
	case SIP_PRACK:
		/* FIXME foo */
		{

		char callid[CALLID_LEN];
		char fromtag[TAG_LEN];
		*ident = randomizer(RAND_CALLID, (char *) &callid, CALLID_LEN);
		randomizer(RAND_TAG, (char *) &fromtag, TAG_LEN);

		len = snprintf(buf, SIPLEN, "PRACK sip:%s SIP/2.0\r\n"
			"Via: SIP/2.0/%s %s:%d;branch=z9hG4bK.%d;rport\r\n"
			"From: <sip:smap@%s:%d>;tag=%s\r\n"
			"To: <sip:%s>\r\n"
			"Call-ID: %s@%s\r\n"
			"CSeq: %d PRACK\r\n"
			"Contact: <sip:smap@%s:%d>\r\n"
			"Content-Length: 0\r\n"
			"Max-Forwards: 70\r\n"
			"User-Agent: smap %s\r\n"
			"Accept: text/plain\r\n\r\n",

			config.sip_domain,		/* SIP domain */
			TRANSPORT_STRING(proto),/* Via protocol */
			config.localip,			/* Via host */
			localport,				/* Via port */
			branch,					/* Via branch */
			config.sip_domain,		/* From domain */
			localport,				/* From port */
			fromtag,				/* From tag */
			config.sip_domain,		/* To domain */
			callid,					/* Call-ID local part */
			config.localip,			/* Call-ID hostpart */
			randomizer(RAND_CSEQ, NULL, 0),	/* CSeq */
			config.localip,			/* Contact hostpart */
			localport,				/* Contact port */
			SMAP_VERSION			/* smap version */
			);
		}
		break;
	case SIP_INVITE:
		/* SIP INVITE to a hopefully non-existing extention named
		 * smap@<SIP domain>
		 */
		{

		char callid[CALLID_LEN];
		char fromtag[TAG_LEN];
		*ident = randomizer(RAND_CALLID, (char *) &callid, CALLID_LEN);
		randomizer(RAND_TAG, (char *) &fromtag, TAG_LEN);

		len = snprintf(buf, SIPLEN, "INVITE sip:smap@%s SIP/2.0\r\n"
			"Via: SIP/2.0/%s %s:%d;branch=z9hG4bK.%d;rport\r\n"
			"From: <sip:smap@%s:%d>;tag=%s\r\n"
			"To: <sip:smap@%s>\r\n"
			"Call-ID: %s@%s\r\n"
			"CSeq: %d INVITE\r\n"
			"Contact: <sip:smap@%s:%d>\r\n"
			"Content-Length: 0\r\n"
			"Max-Forwards: 70\r\n"
			"User-Agent: smap %s\r\n"
			"Accept: text/plain\r\n\r\n",

			config.sip_domain,		/* SIP domain */
			TRANSPORT_STRING(proto),/* Via protocol */
			config.localip,			/* Via host */
			localport,				/* Via port */
			branch,					/* Via branch */
			config.sip_domain,		/* From domain */
			localport,				/* From port */
			fromtag,				/* From tag */
			config.sip_domain,		/* To domain */
			callid,					/* Call-ID local part */
			config.localip,			/* Call-ID hostpart */
			randomizer(RAND_CSEQ, NULL, 0),	/* CSeq */
			config.localip,			/* Contact hostpart */
			localport,				/* Contact port */
			SMAP_VERSION			/* smap version */
			);
		}
		break;
	case SIP_PING:
		/* SIP PING
	 	 *
		 * this generates a valid PING request
		 * based on draft-fwmiller-ping-03
		 * response should be 200 OK
	 	 */
		{

		char callid[CALLID_LEN];
		char fromtag[TAG_LEN];
		*ident = randomizer(RAND_CALLID, (char *) &callid, CALLID_LEN);
		randomizer(RAND_TAG, (char *) &fromtag, TAG_LEN);
		len = snprintf(buf, SIPLEN, "PING sip:%s SIP/2.0\r\n"
			"Via: SIP/2.0/%s %s:%d;branch=z9hG4bK.%d;rport;alias\r\n"
			"From: <sip:smap@%s:%d>;tag=%s\r\n"
			"To: <sip:%s>\r\n"
			"Call-ID: %s@%s\r\n"
			"CSeq: %d PING\r\n"
			"Contact: <sip:smap@%s:%d>\r\n"
			"Content-Length: 0\r\n"
			"Max-Forwards: 70\r\n"
			"User-Agent: smap %s\r\n"
			"Accept: text/plain\r\n\r\n",

			config.sip_domain,		/* SIP domain */
			TRANSPORT_STRING(proto),/* Via protocol */
			config.localip,			/* Via host */
			localport,				/* Via port */
			branch,					/* Via branch */
			config.localip,			/* From IP */
			localport,				/* From port */
			fromtag,				/* From tag */
			config.sip_domain,		/* To hostpart */
			callid,					/* Call-ID local part */
			config.localip,			/* Call-ID hostpart */
			randomizer(RAND_CSEQ, NULL, 0),	/* CSeq */
			config.localip,			/* Contact hostpart */
			localport,				/* Contact port */
			SMAP_VERSION			/* smap version */
			);
		}
		break;
	case SIP_TMP:
		/* temporary scratchpad */
		{

		char callid[CALLID_LEN];
		char fromtag[TAG_LEN];
		*ident = randomizer(RAND_CALLID, (char *) &callid, CALLID_LEN);
		randomizer(RAND_TAG, (char *) &fromtag, TAG_LEN);

		len = snprintf(buf, SIPLEN, "INVITE sip:smap@%s SIP/2.0\r\n"
			"Via: SIP/2.0/%s %s:%d;branch=z9hG4bK.%d;rport\r\n"
			"From: sip:smap@%s:%d;tag=%s\r\n"
			"To: sip:smap@%s\r\n"
			"Call-ID: %s@%s\r\n"
			"CSeq: %d INVITE\r\n"
			"Contact: sip:smap@%s:%d\r\n"
			"Content-Length: 0\r\n"
			"Max-Forwards: 70\r\n"
			"User-Agent: smap %s\r\n"
			"Accept: text/plain\r\n\r\n",


			config.sip_domain,		/* SIP domain */
			TRANSPORT_STRING(proto),/* Via protocol */
			config.localip,			/* Via host */
			localport,				/* Via port */
			branch,					/* Via branch */
			config.sip_domain,		/* From domain */
			localport,				/* From port */
			fromtag,				/* From tag */
			config.sip_domain,		/* To domain */
			callid,					/* Call-ID local part */
			config.localip,			/* Call-ID hostpart */
			randomizer(RAND_CSEQ, NULL, 0),	/* CSeq */
			config.localip,			/* Contact hostpart */
			localport,				/* Contact port */
			SMAP_VERSION			/* smap version */
			);
		}
		break;

	default:
		fprintf(stderr, "invalid request type for sip_generate()\n");
		len = 0;
		break; /* make gcc happy */
	}

	/* simple sanity check */
	if ((len <= 128) || (len >= SIPLEN))
		return 0;
	return 1;
}

#endif /* _REQUEST_C */
