/******************************************************************************
 *
 * test_allow.c - part of smap
 * 
 * $Id: test_allow.c,v 1.17 2007-05-24 19:34:53 hscholz Exp $
 *****************************************************************************/
#ifndef _TEST_ALLOW_C
#define _TEST_ALLOW_C

#include "config.h"

/******************************************************************************
 *
 * Allow header fingerprint meta database
 *
 * this list contains all different types of Allow header combinations
 * found in live-traffic.
 *
 * How to add more?
 * - extend allow_headers[], don't forget new int (third param)
 * - add third param to fingerprint database where appropriate
 *
 *****************************************************************************/

#define ALLOW_LEN		256
#define ALLOW_DESCR		256
struct allow_header_t {
	char header[ALLOW_LEN];
	unsigned int class;
};

struct allow_header_t allow_headers[] = {
	/* AVM Fritz! Box series Router 14.04.06 (May 18 2006) */
	{"Allow: INVITE,ACK,OPTIONS,CANCEL,BYE,UPDATE,PRACK,INFO,SUBSCRIBE,NOTIFY,REFER,MESSAGE", 1},
	/* AVM Fritz! Box series Router 29.04.02 (Jan 25 2006) */
	{"Allow: INVITE, ACK, OPTIONS, CANCEL, BYE, UPDATE, PRACK, INFO, SUBSCRIBE, NOTIFY, REFER, MESSAGE", 2},
	/* AVM Fritz! Box series Router ??.04.01 (Jan 25 2006) */
	{"Allow: INVITE, ACK, OPTIONS, CANCEL, BYE, UPDATE, PRACK, INFO, SUBSCRIBE, NOTIFY, REFER, MESSAGE", 3},
	/* Siemens SX541 1.67 */
	{"Allow:INVITE,ACK,OPTIONS,BYE,CANCEL,REGISTER,INFO,PRACK,REFER", 49},
	/* DeTeWe Opencom 31/0.1 */
	{"Allow: INVITE, ACK, CANCEL, BYE", 50},
	/* DeTeWe Opencom 31/0.1 second version */
	{"Allow: INVITE, ACK, CANCEL, BYE, INFO", 51},
	/* DrayTek UA-1.2.1 */
	{"Allow: INVITE, ACK, CANCEL, OPTIONS, BYE, INFO, REFER, NOTIFY", 52},
	/* LANCOM 1724 VoIP (Annex B) / 6.14.0007 / 29.06.200? */
	{"Allow: REGISTER, INVITE, ACK, CANCEL, BYE, REFER, NOTIFY", 53},
	/* Thomson Speed Touch 780 */
	{"Allow: INVITE, ACK, BYE, REFER, NOTIFY, CANCEL, INFO, OPTIONS", 54}, 
	/* TRACER 3.95 ... unknown type */
	{"Allow: INVITE, ACK, CANCEL, BYE, OPTIONS, REGISTER, SUBSCRIBE, NOTIFY, INFO, REFER", 55},
	/* Snom 200 3.56z */
	{"Allow: INVITE, ACK, CANCEL, BYE, REFER, OPTIONS, NOTIFY, SUBSCRIBE, PRACK, MESSAGE, INFO", 56},
	/* Epygi Quadro SIP User Agent */
	{"Allow: INVITE, ACK, CANCEL, BYE, OPTIONS, INFO, SUBSCRIBE, NOTIFY, REFER, MESSAGE, UPDATE", 57},
	/* Siemens C450 IP */
	{"Allow: INVITE, ACK, CANCEL, BYE, OPTIONS, INFO", 58},
	/* kphone 4.2 */
	{"Allow: INVITE, OPTIONS, ACK, BYE, MSG, CANCEL, MESSAGE, SUBSCRIBE, NOTIFY, INFO, REFER", 100},
	/* Ekiga 2.0.1 */
	{"Allow: INVITE, ACK, OPTIONS, BYE, CANCEL, REGISTER, SUBSCRIBE, NOTIFY, REFER, MESSAGE", 101},
	/* Gizmo Project 1.5.02 */
	{"Allow: INVITE, ACK, CANCEL, OPTIONS, BYE, REFER, INFO, NOTIFY, MESSAGE", 102},
	/* linphone 1.3.x */
	{"Allow: INVITE, ACK, OPTIONS, CANCEL, BYE, SUBSCRIBE, NOTIFY, MESSAGE, INFO", 103},
	/* SJPhone 1.60.299a */
	{"Allow: INVITE, ACK, CANCEL, BYE, REFER, NOTIFY", 104},
	/* Asterisk - unknown version */
	{"Allow: INVITE, ACK, CANCEL, OPTIONS, BYE, REFER", 200},
	/* Asterisk - unknown version */
	{"Allow: INVITE, ACK, CANCEL, OPTIONS, BYE, REFER, SUBSCRIBE, NOTIFY", 201},
	/* Cisco AS5400 HPX */
	{"Allow: INVITE, OPTIONS, BYE, CANCEL, ACK, PRACK, COMET, REFER, SUBSCRIBE, NOTIFY, INFO, UPDATE, REGISTER", 202},
	/* Auerswald VoIP/ISDN */
	{"Allow: INVITE, OPTIONS, ACK, BYE, CANCEL", 203},
	/* LANCOM 1724 VoIP */
	{"Allow: REGISTER, INVITE, ACK, CANCEL, BYE, REFER, NOTIFY, OPTIONS", 204},
	/* PA168V PA1688 series phones */
	{"Allow: INVITE, ACK, OPTIONS, BYE, CANCEL, REFER, NOTIFY, INFO, PRACK, UPDATE, MESSAGE", 205},
	/* Arcor Twintel D910 series */
	{"Allow: INVITE, ACK, BYE, CANCEL, OPTIONS, MESSAGE, REFER, NOTIFY, PRACK, UPDATE, INFO, SUBSCRIBE", 206},
	/* Cisco 79x0 Phone */
	{"Allow: ACK,BYE,CANCEL,INVITE,NOTIFY,OPTIONS,REFER,REGISTER,UPDATE", 207},
	{"", 0}
};

/******************************************************************************
 *
 * test_allow()
 *
 * Locate and analyze the Allow-Header inside the passed SIP message
 *
 *****************************************************************************/

int test_allow(int *res, char *buf) {

	char *p, *n;
	struct allow_header_t hdr;
	int line, end;
	FUNCTION(__FILE__, __FUNCTION__);

	p = strcasestr(buf, "\r\nAllow:");
	if (p == NULL) {
		*res = NO_RESPONSE;
		error(DEBUG, "message does not contain an Allow header");
		return 0;
	}
	p += 2; /* skip leading \r\n */
	n = strstr(p, "\r\n");
	if (n == NULL) {
		*res = NO_RESPONSE;
		error(ERR_ERROR, "cannot locate end of Allow header");
		return 0;
	}
	if (IS_LEARNING || DEBUG)
		error(ERR_NOTICE, "%s: \"%.*s\"", __FUNCTION__,
			(int) (n-p), p);
	line = 0;
	end = 0;
	hdr = allow_headers[line];
	while ((hdr.header[0] != 0) && !end) {
		if (!strncmp(hdr.header, p, (n-p))) {
			end++;	/* correct line found */
			*res = hdr.class;
		}
		hdr = allow_headers[++line];
	}
	if (!end && IS_LEARNING)
		error(ERR_NOTICE, "%s: Please add Allow: header", __FUNCTION__);
	return end ? 1 : 0;
}

#endif /* _TEST_ALLOW_C */
