/******************************************************************************
 *
 * transport_tcp.c - TCP scanner part of smap
 *
 * $Id: transport_tcp.c,v 1.5 2007-11-16 00:42:53 hscholz Exp $
 *****************************************************************************/
#ifndef _TRANSPORT_TCP_C
#define _TRANSPORT_TCP_C

#include "config.h"

/******************************************************************************
 *
 * tcp_setup_sender()
 *
 * set up socket needed for TCP transport
 *
 *****************************************************************************/

int tcp_setup_sender() {

	return 1;
}



/**** old stuff below ****/

/******************************************************************************
 *
 * tcp_test()
 *
 * conduct a single SIP test and return the received response (might be
 * a SIP message) to the calling process
 * return 0 in case of error
 *
 * Socket-Mania:
 *
 * We have to use three ways to obtain all messages:
 * - stock TCP socket
 * - RAW socket for ICMP error messages (if available)
 * 
 *****************************************************************************/
int tcp_test(unsigned int requesttype, struct in_addr dst, char *res) {

    int i, socklen, sockflags;
    char req[SIPLEN];

    /* select() */
    fd_set fdset;
    struct timeval timeout;

	/* tcpsock = regular send() socket */
	/* rawsock = raw socket */
	int tcpsock;
#ifdef RAW_SOCKET
	int rawsock;
#endif
	struct sockaddr_in sockaddr;

	struct ip		*ippkt;
	struct icmp 	*icmppkt;
	char 			*errstr = NULL;

	unsigned int	l_callid, r_callid; /* unique Call-ID */
	int	endloop;	/* receive loop */

	FUNCTION(__FILE__, __FUNCTION__);

	bzero(&req, SIPLEN);

	/* TCP send() socket */	
	tcpsock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
	if (tcpsock == -1) {
		if (DEBUG) perror("socket");
		goto error;
	}
	sockaddr.sin_family = PF_INET;
	sockaddr.sin_addr = dst;
	sockaddr.sin_port = htons(DEFAULT_SIP_PORT);
	socklen = sizeof(sockaddr);

	/* setting the socket timeout won't help on connect() but only
	 * on write()
	 */
	set_timeout(&timeout, config.timeout);
	if (setsockopt(tcpsock, SOL_SOCKET, SO_SNDTIMEO,
		(const void *) &timeout, (socklen_t) sizeof(timeout)) != 0) {
		if (DEBUG) perror("setsockopt");
		error(ERR_NOTICE, "setting timeout failed");
	}
	/* set socket to non-blocking */
	sockflags = fcntl(tcpsock, F_GETFL, 0);
	sockflags |= O_NONBLOCK;
	if (fcntl(tcpsock, F_SETFL, sockflags) != 0) {
		if (DEBUG) perror("fcntl");
	}
	
	if (connect(tcpsock, (const struct sockaddr *) &sockaddr, socklen) != 0) {
		/* it's quite usual to get EINPROGRESS so ignore it here */
		extern int errno;
		if (errno != EINPROGRESS) {
			if (DEBUG) perror("connect");
			goto error;
		}
	}

	FD_ZERO(&fdset);
	FD_SET(tcpsock, &fdset);
	set_timeout(&timeout, config.timeout);

	if (select(tcpsock + 1, NULL, &fdset, NULL, &timeout) <= 0) {
		/* select returned <= 0 and thus an error */
		goto error;
	} else {
		/* connect may be ok */
		int err, err_len;
		err_len = sizeof(err);
		if (getsockopt(tcpsock, SOL_SOCKET, SO_ERROR, &err, (socklen_t*) &err_len) == 0) {
			if (err > 0) {
				/* received some error from inside getsockopt()
				 * -> bail out */
				goto error;
			}
		}
	} /* end of select() */

#ifdef RAW_SOCKET
	/* try to obtain a raw socket for ICMP errors and TCP messages
	 * inbound from wrong source ports (hi Cisco!)
	 */
	rawsock = -1;
	if (!(config.flags & FLAG_NOROOT)) {
		rawsock = socket(PF_INET, SOCK_RAW, IPPROTO_ICMP);
		if (rawsock == -1) {
			error(ERR_NOTICE, "could not obtain RAW socket");
		} else {
			error(ERR_DEBUG, "RAW socket open");
		}
	}
#endif

    if (request_generate(requesttype, (char *)&req, ntohs(sockaddr.sin_port), TRANSPORT_TCP, &l_callid) != 1) {
		fprintf(stderr, "\nERROR: SIP message assembly failed\n");
		goto error;
	}

    if (config.debug >= 2)
		printf("DEBUG: -- start request --\n%s"
			   "\nDEBUG: -- end of request --\n", req);

    i = send(tcpsock, req, strlen(req), 0);
	if (i != strlen(req)) {
		fprintf(stderr, "\nWarning: incomplete write()!\n");
		return 0;
	}

	/* write was ok, no we enter a receive loop which may in fact loop
	 * if we receive a retransmission of a previous request.
	 * we loop through this until we either find the response to our
	 * request or get caught in the timeout
	 */

	set_timeout(&timeout, config.timeout);
	do {

		/* prepare for select() */
		FD_ZERO(&fdset);
		FD_SET(tcpsock, &fdset);
#ifdef RAW_SOCKET
		if (rawsock != -1)
			FD_SET(rawsock, &fdset);
#endif

		if (select(FD_SETSIZE, &fdset, NULL, NULL, &timeout) > 0) {

			/* try all data sources */
			if (FD_ISSET(tcpsock, &fdset)) {
				i = read(tcpsock, res, SIPLEN);
#ifdef RAW_SOCKET
			} else if (FD_ISSET(rawsock, &fdset)) {
				i = read(rawsock, res, SIPLEN);
#endif
			}

			if (i <= 0) goto error;	

			/* check for icmp error message */
			ippkt = (struct ip *) res;
			if ((ippkt->ip_v == 4) && (ippkt->ip_p == IPPROTO_ICMP)) {
				/* ICMP */
				icmppkt = (struct icmp *) (res + (ippkt->ip_hl * 4));
				if ((errstr = icmp_message_str(icmppkt)) != NULL) { 
					error(ERR_DEBUG, "ICMP error %s", errstr);
					free(errstr);
				} else {
					error(ERR_DEBUG, "unknown ICMP message received (%d/%d)",
						icmppkt->icmp_type, icmppkt->icmp_code);
				}
				goto error;
			}
			/* DEBUG: print data */
			if (DEBUG && (i > 0)) {
				fprintf(stderr, 
				"DEBUG: -- start response --\n%.*s\n"
				"DEBUG: -- end of response --\n", i, res);
			}
			/* match unique Call-ID */
			r_callid = response_getcallid(res);
			if (r_callid == 0) {
				error(ERR_DEBUG, "could not parse Call-ID in received packet");
				/* r_callid might have been invalid but just to be sure
				 * we don't wait too long we bail out on this */
				endloop = 1;
			} else if (l_callid == r_callid) {
				endloop = 1;
			} else {
				/* probably received an old response, loop again */
				error(ERR_DEBUG, "ignoring possible SIP retransmission");
				endloop = 0;
			}
		} else {
			error(ERR_DEBUG, "select() timeout\n");
			goto error;
		}

	} while (endloop != 1); /* end of receive loop */

#ifdef RAW_SOCKET
	close(rawsock);
#endif
    close(tcpsock);
	return 1;

error:
	if (DEBUG) printf("\n");
#ifdef RAW_SOCKET
	close(rawsock);
#endif
	close(tcpsock);
	return 0;
}

#endif /* _TRANSPORT_TCP_C */
