/******************************************************************************
 * 
 * transport_udp.c - UDP scanner part of smap
 *
 * $Id: transport_udp.c,v 1.10 2008-08-20 13:02:57 hscholz Exp $
 *****************************************************************************/
#ifndef _TRANSPORT_UDP_C
#define _TRANSPORT_UDP_C

#include "config.h"

/******************************************************************************
 *
 *
 * udp_setup_rawsock()
 *
 * set up raw UDP socket
 *
 *****************************************************************************/

int udp_setup_rawsock(void) {
#ifdef RAW_SOCKET
	extern int udp_rawsock;
#endif

	FUNCTION(__FILE__, __FUNCTION__);
	udp_rawsock = -1;

#ifdef RAW_SOCKET
    if (!(config.flags & FLAG_NOROOT)) {
        udp_rawsock = socket(PF_INET, SOCK_RAW, IPPROTO_ICMP);
        if (udp_rawsock == -1) {
            error(ERR_NOTICE, "could not obtain RAW socket");
        } else {
            error(ERR_DEBUG, "RAW socket open");
        }
    }
#endif
	return 1;
}

/******************************************************************************
 *
 * udp_setup_sender()
 *
 * set up socket needed to transmit UDP datagrams
 *
 *****************************************************************************/

int udp_setup_sender() {

	extern int udp_sendsock;
	extern struct sockaddr_in udp_sendsock_sockaddr;
	int sockaddrlen;

    /* is 5060 (DEFAULT_SIP_PORT) available?
     * if yes, we'll use it since some clients send responses to
     * port 5060 even when instructed to not do so
     */

    udp_sendsock_sockaddr.sin_family = PF_INET;
    udp_sendsock_sockaddr.sin_addr.s_addr = htonl(INADDR_ANY);
    udp_sendsock_sockaddr.sin_port = htons(DEFAULT_SIP_PORT);
    sockaddrlen = sizeof(udp_sendsock_sockaddr);

    udp_sendsock = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if (udp_sendsock == -1) {
        if (DEBUG) perror("socket");
        /* socket() failed, default to random port */
        if (!(config.flags & FLAG_SIPPORTBUSY)) {
            error(ERR_NOTICE, "Could not obtain local port 5060."
                " Scanning may be unreliable!");
            config.flags |= FLAG_SIPPORTBUSY;
        }
        udp_sendsock_sockaddr.sin_family = PF_INET;
        udp_sendsock_sockaddr.sin_addr.s_addr = htonl(INADDR_ANY);
        udp_sendsock_sockaddr.sin_port = htons(0);
        sockaddrlen = sizeof(udp_sendsock_sockaddr);
    } else {
        /* try to bind() port 5060 */
        if (bind(udp_sendsock, (struct sockaddr *) &udp_sendsock_sockaddr,
            sizeof(struct sockaddr)) == -1) {
            /* port 5060 in use */
            if (!(config.flags & FLAG_SIPPORTBUSY)) {
                error(ERR_NOTICE, "Could not obtain local port 5060."
                    " Scanning may be unreliable!");
                config.flags |= FLAG_SIPPORTBUSY;
            }
            udp_sendsock_sockaddr.sin_family = PF_INET;
            udp_sendsock_sockaddr.sin_addr.s_addr = htonl(INADDR_ANY);
            udp_sendsock_sockaddr.sin_port = htons(0);
            sockaddrlen = sizeof(udp_sendsock_sockaddr);
        } else {
            /* success */
            udp_sendsock_sockaddr.sin_family = PF_INET;
            udp_sendsock_sockaddr.sin_addr.s_addr = htonl(INADDR_ANY);
            udp_sendsock_sockaddr.sin_port = htons(DEFAULT_SIP_PORT);
            sockaddrlen = sizeof(udp_sendsock_sockaddr);
        }
        close(udp_sendsock);
    }
    
    /* bind() to local port in order to receive all packets
     * do not bail out if error occurs
     */
    udp_sendsock = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if (udp_sendsock == -1) {
        error(ERR_DEBUG, "could not bind() to socket");
    } else {
        if (bind(udp_sendsock, (struct sockaddr *) &udp_sendsock_sockaddr, sizeof(struct sockaddr)) == -1) {
            if (DEBUG) perror("bind");
            error(ERR_ERROR, "could not bind to port %d",
                    ntohs(udp_sendsock_sockaddr.sin_port));
            goto error;
        } else {
            error(ERR_DEBUG, "bind() successful");
        }
    }
    if (getsockname(udp_sendsock, (struct sockaddr *) &udp_sendsock_sockaddr,
                (socklen_t *) &sockaddrlen) != 0)
    if (DEBUG) perror("getsocknameudp_sendsock");

	return 1;
error:
	error(ERR_ERROR, "setting up udp_sendsock failed");
	return 0;
}

/******************************************************************************
 *
 * udp_send()
 *
 * send a prebuilt message to a destination using an existing socket
 *
 * Returns:
 *
 *   0: successful transmission
 *   1: error
 *
 *****************************************************************************/

int udp_send(task_t *task, char *msg) {
	extern struct config_t config;
	extern int udp_sendsock;
	struct sockaddr_in target;

	FUNCTION(__FILE__, __FUNCTION__);

	if (!task) return 0;

	target.sin_family = PF_INET;
	target.sin_addr = task->target;
	target.sin_port = htons(config.sip_port);

	if (sendto(udp_sendsock, msg, strlen(msg), 0,
		(struct sockaddr *) &target, sizeof(struct sockaddr)) != strlen(msg)) {
		perror("sendto");
		error(ERR_NOTICE, "Warning: incomplete sendto()");
		return 0;
	}

	return 1;
}
#endif /* _TRANSPORT_UDP_C */
