#!/bin/sh
##############################################################################
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# Copyright (C) 2016 Eric Luehrsen
#
##############################################################################
#
# This component will copy root.key back to /etc/unbound/ periodically, but
# avoid ROM flash abuse (UCI option).
#
##############################################################################

# while useful (sh)ellcheck is pedantic and noisy
# shellcheck disable=1091,2002,2004,2034,2039,2086,2094,2140,2154,2155

. /usr/lib/unbound/defaults.sh

##############################################################################

roothints_update() {
  # TODO: Might not be implemented. Unbound doesn't natively update hints.
  # Unbound philosophy is built in root hints are good for machine life.
  return 0
}

##############################################################################

rootkey_update() {
  local basekey_date rootkey_date rootkey_age filestuff
  local dnssec=$( uci_get unbound.@unbound[0].validator )
  local dnssec_ntp=$( uci_get unbound.@unbound[0].validator_ntp )
  local dnssec_age=$( uci_get unbound.@unbound[0].root_age )

  # fix empty
  [ -z "$dnssec"     ] && dnssec=0
  [ -z "$dnssec_ntp" ] && dnssec_ntp=1
  [ -z "$dnssec_age" ] && dnssec_age=9


  if [ $dnssec_age -gt 90 ] || [ $dnssec -lt 1 ] ; then
    # Feature disabled
    return 0

  elif [ "$dnssec_ntp" -gt 0 ] && [ ! -f "$UB_TIME_FILE" ] ; then
    # We don't have time yet
    return 0
  fi


  if [ -f /etc/unbound/root.key ] ; then
    basekey_date=$( date -r /etc/unbound/root.key +%s )

  else
    # No persistent storage key
    basekey_date=$( date -d 2000-01-01 +%s )
  fi


  if [ -f "$UB_RKEY_FILE" ] ; then
    # Unbound maintains it itself
    rootkey_date=$( date -r $UB_RKEY_FILE +%s )
    rootkey_age=$(( (rootkey_date - basekey_date) / 86440 ))

  elif [ -x "$UB_ANCHOR" ] ; then
    # No tmpfs key - use unbound-anchor
    rootkey_date=$( date -I +%s )
    rootkey_age=$(( (rootkey_date - basekey_date) / 86440 ))
    $UB_ANCHOR -a $UB_RKEY_FILE

  else
    # give up
    rootkey_age=0
  fi


  if [ $rootkey_age -gt $dnssec_age ] ; then
    filestuff=$( cat $UB_RKEY_FILE )


    case "$filestuff" in
      *NOERROR*)
        # Header comment for drill and dig
        logger -t unbound -s "root.key updated after $rootkey_age days"
        cp -p $UB_RKEY_FILE /etc/unbound/root.key
        ;;

      *"state=2 [  VALID  ]"*)
        # Comment inline to key for unbound-anchor
        logger -t unbound -s "root.key updated after $rootkey_age days"
        cp -p $UB_RKEY_FILE /etc/unbound/root.key
        ;;

      *)
        logger -t unbound -s "root.key still $rootkey_age days old"
        ;;
    esac
  fi
}

##############################################################################

resolv_teardown() {
  case $( cat $UB_RESOLV_CONF ) in
    *"generated by Unbound UCI"*)
      # our resolver file, reset to auto resolver file.
      rm -f $UB_RESOLV_CONF
      ln -s $UB_RESOLV_AUTO $UB_RESOLV_CONF
      ;;
  esac
}

##############################################################################

unbound_stop() {
  resolv_teardown
  roothints_update
  rootkey_update
}

##############################################################################

