/*
 * cal2p54 - CAL to p54spi EEPROM converter
 *
 * Copyright (c) 2011 Michael Buesch <mb@bu3sch.de>
 *
 * Licensed under the GNU General Public License version 2.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <stdint.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/mman.h>


/* Template data from Linux p54spi_eeprom.h:
 *
 * Copyright (C) 2003 Conexant Americas Inc. All Rights Reserved.
 * Copyright (C) 2004, 2005, 2006 Nokia Corporation
 * Copyright 2008 Johannes Berg <johannes@sipsolutions.net>
 * Copyright 2008 Christian Lamparter <chunkeey@web.de>
 *
 * based on:
 *  - cx3110x's pda.h from Nokia
 *  - cx3110-transfer.log by Johannes Berg
 */

static const uint8_t template_eeprom_pda_wrap[] = {
	/* struct eeprom_pda_wrap */
	0x47, 0x4d, 0x55, 0xaa,	/* magic */
	0x00, 0x00,		/* pad */
	0x00, 0x00,		/* eeprom_pda_data_wrap length */
	0x00, 0x00, 0x00, 0x00,	/* arm opcode */
};

static const uint8_t template_mac_address[] = {
	/* bogus MAC address */
	0x04, 0x00, 0x01, 0x01,		/* PDR_MAC_ADDRESS */
		0x00, 0x02, 0xee, 0xc0, 0xff, 0xee,
};

static const uint8_t template_bootrec_exp_if[] = {
	/* struct bootrec_exp_if */
	0x06, 0x00, 0x01, 0x10,		/* PDR_INTERFACE_LIST */
		0x00, 0x00,			/* role */
		0x0f, 0x00,			/* if_id */
		0x85, 0x00,			/* variant = Longbow RF, 2GHz */
		0x01, 0x00,			/* btm_compat */
		0x1f, 0x00,			/* top_compat */

	0x03, 0x00, 0x02, 0x10,		/* PDR_HARDWARE_PLATFORM_COMPONENT_ID */
		0x03, 0x20, 0x00, 0x43,
};

static const uint8_t template_pda_country_list[] = {
	/* struct pda_country[6] */
	0x0d, 0x00, 0x07, 0x10,		/* PDR_COUNTRY_LIST */
		0x10, 0x00, 0x00, 0x00,
		0x20, 0x00, 0x00, 0x00,
		0x30, 0x00, 0x00, 0x00,
		0x31, 0x00, 0x00, 0x00,
		0x32, 0x00, 0x00, 0x00,
		0x40, 0x00, 0x00, 0x00,
};

static const uint8_t template_pda_country[] = {
	/* struct pda_country */
	0x03, 0x00, 0x08, 0x10,		/* PDR_DEFAULT_COUNTRY */
		0x30, 0x00, 0x00, 0x00,		/* ETSI */
};

static const uint8_t template_antenna_gain[] = {
	0x03, 0x00, 0x00, 0x11,		/* PDR_ANTENNA_GAIN */
		0x08, 0x08, 0x08, 0x08,
};

static const uint8_t template_rssi_linear_approx_custom2[] = {
	0x0a, 0x00, 0xff, 0xca,		/* PDR_RSSI_LINEAR_APPROXIMATION_CUSTOMV2 */
		0x01, 0x00, 0x0a, 0x00,
		0x00, 0x00, 0x0a, 0x00,
			0x85, 0x09, 0x0a, 0x01, 0x72, 0xfe, 0x1a, 0x00, 0x00, 0x00,
};

static const uint8_t template_pa_cal_curve[] = {
	/* struct pda_custom_wrapper */
	0x10, 0x06, 0x5d, 0xb0,		/* PDR_PRISM_PA_CAL_CURVE_DATA_CUSTOM */
		0x0d, 0x00, 0xee, 0x00,		/* 13 entries, 238 bytes per entry */
		0x00, 0x00, 0x16, 0x0c,		/* no offset, 3094 total len */
			/* 2412 MHz */
			0x6c, 0x09,
				0x10, 0x01, 0x9a, 0x84,
					0xaa, 0x8a, 0xaa, 0x8a, 0xaa, 0x8a, 0xaa, 0x8a,
					0x3c, 0xb6, 0x3c, 0xb6, 0x3c, 0xb6, 0x3c, 0xb6,
					0x3c, 0xb6, 0x3c, 0xb6, 0x3c, 0xb6, 0x3c, 0xb6,
				0xf0, 0x00, 0x94, 0x6c,
					0x99, 0x82, 0x99, 0x82, 0x99, 0x82, 0x99, 0x82,
					0x2b, 0xae, 0x2b, 0xae, 0x2b, 0xae, 0x2b, 0xae,
					0x2b, 0xae, 0x2b, 0xae, 0x2b, 0xae, 0x2b, 0xae,
				0xd0, 0x00, 0xaa, 0x5a,
					0x88, 0x7a, 0x88, 0x7a, 0x88, 0x7a, 0x88, 0x7a,
					0x1a, 0xa6, 0x1a, 0xa6, 0x1a, 0xa6, 0x1a, 0xa6,
					0x1a, 0xa6, 0x1a, 0xa6, 0x1a, 0xa6, 0x1a, 0xa6,
				0xa0, 0x00, 0xf3, 0x47,
					0x6e, 0x6e, 0x6e, 0x6e, 0x6e, 0x6e, 0x6e, 0x6e,
					0x00, 0x9a, 0x00, 0x9a, 0x00, 0x9a, 0x00, 0x9a,
					0x00, 0x9a, 0x00, 0x9a, 0x00, 0x9a, 0x00, 0x9a,
				0x50, 0x00, 0x59, 0x36,
					0x43, 0x5a, 0x43, 0x5a, 0x43, 0x5a, 0x43, 0x5a,
					0xd5, 0x85, 0xd5, 0x85, 0xd5, 0x85, 0xd5, 0x85,
					0xd5, 0x85, 0xd5, 0x85, 0xd5, 0x85, 0xd5, 0x85,
				0x00, 0x00, 0xe4, 0x2d,
					0x18, 0x46, 0x18, 0x46, 0x18, 0x46, 0x18, 0x46,
					0xaa, 0x71, 0xaa, 0x71, 0xaa, 0x71, 0xaa, 0x71,
					0xaa, 0x71, 0xaa, 0x71, 0xaa, 0x71, 0xaa, 0x71,
				0x00, 0x80, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x80, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x06, 0x80, 0x80, 0x00,

			/* 2417 MHz */
			0x71, 0x09,
				0x10, 0x01, 0xb9, 0x83,
					0x7d, 0x8a, 0x7d, 0x8a, 0x7d, 0x8a, 0x7d, 0x8a,
					0x0f, 0xb6, 0x0f, 0xb6, 0x0f, 0xb6, 0x0f, 0xb6,
					0x0f, 0xb6, 0x0f, 0xb6, 0x0f, 0xb6, 0x0f, 0xb6,
				0xf0, 0x00, 0x2e, 0x6c,
					0x68, 0x82, 0x68, 0x82, 0x68, 0x82, 0x68, 0x82,
					0xfa, 0xad, 0xfa, 0xad, 0xfa, 0xad, 0xfa, 0xad,
					0xfa, 0xad, 0xfa, 0xad, 0xfa, 0xad, 0xfa, 0xad,
				0xd0, 0x00, 0x8d, 0x5a,
					0x52, 0x7a, 0x52, 0x7a, 0x52, 0x7a, 0x52, 0x7a,
					0xe4, 0xa5, 0xe4, 0xa5, 0xe4, 0xa5, 0xe4, 0xa5,
					0xe4, 0xa5, 0xe4, 0xa5, 0xe4, 0xa5, 0xe4, 0xa5,
				0xa0, 0x00, 0x0a, 0x48,
					0x32, 0x6e, 0x32, 0x6e, 0x32, 0x6e, 0x32, 0x6e,
					0xc4, 0x99, 0xc4, 0x99, 0xc4, 0x99, 0xc4, 0x99,
					0xc4, 0x99, 0xc4, 0x99, 0xc4, 0x99, 0xc4, 0x99,
				0x50, 0x00, 0x7c, 0x36,
					0xfc, 0x59, 0xfc, 0x59, 0xfc, 0x59, 0xfc, 0x59,
					0x8e, 0x85, 0x8e, 0x85, 0x8e, 0x85, 0x8e, 0x85,
					0x8e, 0x85, 0x8e, 0x85, 0x8e, 0x85, 0x8e, 0x85,
				0x00, 0x00, 0xf5, 0x2d,
					0xc6, 0x45, 0xc6, 0x45, 0xc6, 0x45, 0xc6, 0x45,
					0x58, 0x71, 0x58, 0x71, 0x58, 0x71, 0x58, 0x71,
					0x58, 0x71, 0x58, 0x71, 0x58, 0x71, 0x58, 0x71,
				0x00, 0x80, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x80, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x06, 0x80, 0x80, 0x00,

			/* 2422 MHz */
			0x76, 0x09,
				0x10, 0x01, 0xb9, 0x83,
					0x7d, 0x8a, 0x7d, 0x8a, 0x7d, 0x8a, 0x7d, 0x8a,
					0x0f, 0xb6, 0x0f, 0xb6, 0x0f, 0xb6, 0x0f, 0xb6,
					0x0f, 0xb6, 0x0f, 0xb6, 0x0f, 0xb6, 0x0f, 0xb6,
				0xf0, 0x00, 0x2e, 0x6c,
					0x68, 0x82, 0x68, 0x82, 0x68, 0x82, 0x68, 0x82,
					0xfa, 0xad, 0xfa, 0xad, 0xfa, 0xad, 0xfa, 0xad,
					0xfa, 0xad, 0xfa, 0xad, 0xfa, 0xad, 0xfa, 0xad,
				0xd0, 0x00, 0x8d, 0x5a,
					0x52, 0x7a, 0x52, 0x7a, 0x52, 0x7a, 0x52, 0x7a,
					0xe4, 0xa5, 0xe4, 0xa5, 0xe4, 0xa5, 0xe4, 0xa5,
					0xe4, 0xa5, 0xe4, 0xa5, 0xe4, 0xa5, 0xe4, 0xa5,
				0xa0, 0x00, 0x0a, 0x48,
					0x32, 0x6e, 0x32, 0x6e, 0x32, 0x6e, 0x32, 0x6e,
					0xc4, 0x99, 0xc4, 0x99, 0xc4, 0x99, 0xc4, 0x99,
					0xc4, 0x99, 0xc4, 0x99, 0xc4, 0x99, 0xc4, 0x99,
				0x50, 0x00, 0x7c, 0x36,
					0xfc, 0x59, 0xfc, 0x59, 0xfc, 0x59, 0xfc, 0x59,
					0x8e, 0x85, 0x8e, 0x85, 0x8e, 0x85, 0x8e, 0x85,
					0x8e, 0x85, 0x8e, 0x85, 0x8e, 0x85, 0x8e, 0x85,
				0x00, 0x00, 0xf5, 0x2d,
					0xc6, 0x45, 0xc6, 0x45, 0xc6, 0x45, 0xc6, 0x45,
					0x58, 0x71, 0x58, 0x71, 0x58, 0x71, 0x58, 0x71,
					0x58, 0x71, 0x58, 0x71, 0x58, 0x71, 0x58, 0x71,
				0x00, 0x80, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x80, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x06, 0x80, 0x80, 0x00,

			/* 2427 MHz */
			0x7b, 0x09,
				0x10, 0x01, 0x48, 0x83,
					0x67, 0x8a, 0x67, 0x8a, 0x67, 0x8a, 0x67, 0x8a,
					0xf9, 0xb5, 0xf9, 0xb5, 0xf9, 0xb5, 0xf9, 0xb5,
					0xf9, 0xb5, 0xf9, 0xb5, 0xf9, 0xb5, 0xf9, 0xb5,
				0xf0, 0x00, 0xfb, 0x6b,
					0x50, 0x82, 0x50, 0x82, 0x50, 0x82, 0x50, 0x82,
					0xe2, 0xad, 0xe2, 0xad, 0xe2, 0xad, 0xe2, 0xad,
					0xe2, 0xad, 0xe2, 0xad, 0xe2, 0xad, 0xe2, 0xad,
				0xd0, 0x00, 0x7e, 0x5a,
					0x38, 0x7a, 0x38, 0x7a, 0x38, 0x7a, 0x38, 0x7a,
					0xca, 0xa5, 0xca, 0xa5, 0xca, 0xa5, 0xca, 0xa5,
					0xca, 0xa5, 0xca, 0xa5, 0xca, 0xa5, 0xca, 0xa5,
				0xa0, 0x00, 0x15, 0x48,
					0x14, 0x6e, 0x14, 0x6e, 0x14, 0x6e, 0x14, 0x6e,
					0xa6, 0x99, 0xa6, 0x99, 0xa6, 0x99, 0xa6, 0x99,
					0xa6, 0x99, 0xa6, 0x99, 0xa6, 0x99, 0xa6, 0x99,
				0x50, 0x00, 0x8e, 0x36,
					0xd9, 0x59, 0xd9, 0x59, 0xd9, 0x59, 0xd9, 0x59,
					0x6b, 0x85, 0x6b, 0x85, 0x6b, 0x85, 0x6b, 0x85,
					0x6b, 0x85, 0x6b, 0x85, 0x6b, 0x85, 0x6b, 0x85,
				0x00, 0x00, 0xfe, 0x2d,
					0x9d, 0x45, 0x9d, 0x45, 0x9d, 0x45, 0x9d, 0x45,
					0x2f, 0x71, 0x2f, 0x71, 0x2f, 0x71, 0x2f, 0x71,
					0x2f, 0x71, 0x2f, 0x71, 0x2f, 0x71, 0x2f, 0x71,
				0x00, 0x80, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x80, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x06, 0x80, 0x80, 0x00,

			/* 2432 MHz */
			0x80, 0x09,
				0x10, 0x01, 0xd7, 0x82,
					0x51, 0x8a, 0x51, 0x8a, 0x51, 0x8a, 0x51, 0x8a,
					0xe3, 0xb5, 0xe3, 0xb5, 0xe3, 0xb5, 0xe3, 0xb5,
					0xe3, 0xb5, 0xe3, 0xb5, 0xe3, 0xb5, 0xe3, 0xb5,
				0xf0, 0x00, 0xc8, 0x6b,
					0x37, 0x82, 0x37, 0x82, 0x37, 0x82, 0x37, 0x82,
					0xc9, 0xad, 0xc9, 0xad, 0xc9, 0xad, 0xc9, 0xad,
					0xc9, 0xad, 0xc9, 0xad, 0xc9, 0xad, 0xc9, 0xad,
				0xd0, 0x00, 0x6f, 0x5a,
					0x1d, 0x7a, 0x1d, 0x7a, 0x1d, 0x7a, 0x1d, 0x7a,
					0xaf, 0xa5, 0xaf, 0xa5, 0xaf, 0xa5, 0xaf, 0xa5,
					0xaf, 0xa5, 0xaf, 0xa5, 0xaf, 0xa5, 0xaf, 0xa5,
				0xa0, 0x00, 0x20, 0x48,
					0xf6, 0x6d, 0xf6, 0x6d, 0xf6, 0x6d, 0xf6, 0x6d,
					0x88, 0x99, 0x88, 0x99, 0x88, 0x99, 0x88, 0x99,
					0x88, 0x99, 0x88, 0x99, 0x88, 0x99, 0x88, 0x99,
				0x50, 0x00, 0x9f, 0x36,
					0xb5, 0x59, 0xb5, 0x59, 0xb5, 0x59, 0xb5, 0x59,
					0x47, 0x85, 0x47, 0x85, 0x47, 0x85, 0x47, 0x85,
					0x47, 0x85, 0x47, 0x85, 0x47, 0x85, 0x47, 0x85,
				0x00, 0x00, 0x06, 0x2e,
					0x74, 0x45, 0x74, 0x45, 0x74, 0x45, 0x74, 0x45,
					0x06, 0x71, 0x06, 0x71, 0x06, 0x71, 0x06, 0x71,
					0x06, 0x71, 0x06, 0x71, 0x06, 0x71, 0x06, 0x71,
				0x00, 0x80, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x80, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x06, 0x80, 0x80, 0x00,

			/* 2437 MHz */
			0x85, 0x09,
				0x10, 0x01, 0x67, 0x82,
					0x3a, 0x8a, 0x3a, 0x8a, 0x3a, 0x8a, 0x3a, 0x8a,
					0xcc, 0xb5, 0xcc, 0xb5, 0xcc, 0xb5, 0xcc, 0xb5,
					0xcc, 0xb5, 0xcc, 0xb5, 0xcc, 0xb5, 0xcc, 0xb5,
				0xf0, 0x00, 0x95, 0x6b,
					0x1f, 0x82, 0x1f, 0x82, 0x1f, 0x82, 0x1f, 0x82,
					0xb1, 0xad, 0xb1, 0xad, 0xb1, 0xad, 0xb1, 0xad,
					0xb1, 0xad, 0xb1, 0xad, 0xb1, 0xad, 0xb1, 0xad,
				0xd0, 0x00, 0x61, 0x5a,
					0x02, 0x7a, 0x02, 0x7a, 0x02, 0x7a, 0x02, 0x7a,
					0x94, 0xa5, 0x94, 0xa5, 0x94, 0xa5, 0x94, 0xa5,
					0x94, 0xa5, 0x94, 0xa5, 0x94, 0xa5, 0x94, 0xa5,
				0xa0, 0x00, 0x2c, 0x48,
					0xd8, 0x6d, 0xd8, 0x6d, 0xd8, 0x6d, 0xd8, 0x6d,
					0x6a, 0x99, 0x6a, 0x99, 0x6a, 0x99, 0x6a, 0x99,
					0x6a, 0x99, 0x6a, 0x99, 0x6a, 0x99, 0x6a, 0x99,
				0x50, 0x00, 0xb1, 0x36,
					0x92, 0x59, 0x92, 0x59, 0x92, 0x59, 0x92, 0x59,
					0x24, 0x85, 0x24, 0x85, 0x24, 0x85, 0x24, 0x85,
					0x24, 0x85, 0x24, 0x85, 0x24, 0x85, 0x24, 0x85,
				0x00, 0x00, 0x0f, 0x2e,
					0x4b, 0x45, 0x4b, 0x45, 0x4b, 0x45, 0x4b, 0x45,
					0xdd, 0x70, 0xdd, 0x70, 0xdd, 0x70, 0xdd, 0x70,
					0xdd, 0x70, 0xdd, 0x70, 0xdd, 0x70, 0xdd, 0x70,
				0x00, 0x80, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x80, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x06, 0x80, 0x80, 0x00,

			/* 2442 MHz */
			0x8a, 0x09,
				0x10, 0x01, 0xf6, 0x81,
					0x24, 0x8a, 0x24, 0x8a, 0x24, 0x8a, 0x24, 0x8a,
					0xb6, 0xb5, 0xb6, 0xb5, 0xb6, 0xb5, 0xb6, 0xb5,
					0xb6, 0xb5, 0xb6, 0xb5, 0xb6, 0xb5, 0xb6, 0xb5,
				0xf0, 0x00, 0x62, 0x6b,
					0x06, 0x82, 0x06, 0x82, 0x06, 0x82, 0x06, 0x82,
					0x98, 0xad, 0x98, 0xad, 0x98, 0xad, 0x98, 0xad,
					0x98, 0xad, 0x98, 0xad, 0x98, 0xad, 0x98, 0xad,
				0xd0, 0x00, 0x52, 0x5a,
					0xe7, 0x79, 0xe7, 0x79, 0xe7, 0x79, 0xe7, 0x79,
					0x79, 0xa5, 0x79, 0xa5, 0x79, 0xa5, 0x79, 0xa5,
					0x79, 0xa5, 0x79, 0xa5, 0x79, 0xa5, 0x79, 0xa5,
				0xa0, 0x00, 0x37, 0x48,
					0xba, 0x6d, 0xba, 0x6d, 0xba, 0x6d, 0xba, 0x6d,
					0x4c, 0x99, 0x4c, 0x99, 0x4c, 0x99, 0x4c, 0x99,
					0x4c, 0x99, 0x4c, 0x99, 0x4c, 0x99, 0x4c, 0x99,
				0x50, 0x00, 0xc2, 0x36,
					0x6e, 0x59, 0x6e, 0x59, 0x6e, 0x59, 0x6e, 0x59,
					0x00, 0x85, 0x00, 0x85, 0x00, 0x85, 0x00, 0x85,
					0x00, 0x85, 0x00, 0x85, 0x00, 0x85, 0x00, 0x85,
				0x00, 0x00, 0x17, 0x2e,
					0x22, 0x45, 0x22, 0x45, 0x22, 0x45, 0x22, 0x45,
					0xb4, 0x70, 0xb4, 0x70, 0xb4, 0x70, 0xb4, 0x70,
					0xb4, 0x70, 0xb4, 0x70, 0xb4, 0x70, 0xb4, 0x70,
				0x00, 0x80, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x80, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x06, 0x80, 0x80, 0x00,

			/* 2447 MHz */
			0x8f, 0x09,
				0x10, 0x01, 0x75, 0x83,
					0x61, 0x8a, 0x61, 0x8a, 0x61, 0x8a, 0x61, 0x8a,
					0xf3, 0xb5, 0xf3, 0xb5, 0xf3, 0xb5, 0xf3, 0xb5,
					0xf3, 0xb5, 0xf3, 0xb5, 0xf3, 0xb5, 0xf3, 0xb5,
				0xf0, 0x00, 0x4b, 0x6c,
					0x3f, 0x82, 0x3f, 0x82, 0x3f, 0x82, 0x3f, 0x82,
					0xd1, 0xad, 0xd1, 0xad, 0xd1, 0xad, 0xd1, 0xad,
					0xd1, 0xad, 0xd1, 0xad, 0xd1, 0xad, 0xd1, 0xad,
				0xd0, 0x00, 0xda, 0x5a,
					0x1c, 0x7a, 0x1c, 0x7a, 0x1c, 0x7a, 0x1c, 0x7a,
					0xae, 0xa5, 0xae, 0xa5, 0xae, 0xa5, 0xae, 0xa5,
					0xae, 0xa5, 0xae, 0xa5, 0xae, 0xa5, 0xae, 0xa5,
				0xa0, 0x00, 0x6d, 0x48,
					0xe9, 0x6d, 0xe9, 0x6d, 0xe9, 0x6d, 0xe9, 0x6d,
					0x7b, 0x99, 0x7b, 0x99, 0x7b, 0x99, 0x7b, 0x99,
					0x7b, 0x99, 0x7b, 0x99, 0x7b, 0x99, 0x7b, 0x99,
				0x50, 0x00, 0xc6, 0x36,
					0x92, 0x59, 0x92, 0x59, 0x92, 0x59, 0x92, 0x59,
					0x24, 0x85, 0x24, 0x85, 0x24, 0x85, 0x24, 0x85,
					0x24, 0x85, 0x24, 0x85, 0x24, 0x85, 0x24, 0x85,
				0x00, 0x00, 0x15, 0x2e,
					0x3c, 0x45, 0x3c, 0x45, 0x3c, 0x45, 0x3c, 0x45,
					0xce, 0x70, 0xce, 0x70, 0xce, 0x70, 0xce, 0x70,
					0xce, 0x70, 0xce, 0x70, 0xce, 0x70, 0xce, 0x70,
				0x00, 0x80, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x80, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x06, 0x80, 0x80, 0x00,

			/* 2452 MHz */
			0x94, 0x09,
				0x10, 0x01, 0xf4, 0x84,
					0x9e, 0x8a, 0x9e, 0x8a, 0x9e, 0x8a, 0x9e, 0x8a,
					0x30, 0xb6, 0x30, 0xb6, 0x30, 0xb6, 0x30, 0xb6,
					0x30, 0xb6, 0x30, 0xb6, 0x30, 0xb6, 0x30, 0xb6,
				0xf0, 0x00, 0x34, 0x6d,
					0x77, 0x82, 0x77, 0x82, 0x77, 0x82, 0x77, 0x82,
					0x09, 0xae, 0x09, 0xae, 0x09, 0xae, 0x09, 0xae,
					0x09, 0xae, 0x09, 0xae, 0x09, 0xae, 0x09, 0xae,
				0xd0, 0x00, 0x62, 0x5b,
					0x50, 0x7a, 0x50, 0x7a, 0x50, 0x7a, 0x50, 0x7a,
					0xe2, 0xa5, 0xe2, 0xa5, 0xe2, 0xa5, 0xe2, 0xa5,
					0xe2, 0xa5, 0xe2, 0xa5, 0xe2, 0xa5, 0xe2, 0xa5,
				0xa0, 0x00, 0xa2, 0x48,
					0x17, 0x6e, 0x17, 0x6e, 0x17, 0x6e, 0x17, 0x6e,
					0xa9, 0x99, 0xa9, 0x99, 0xa9, 0x99, 0xa9, 0x99,
					0xa9, 0x99, 0xa9, 0x99, 0xa9, 0x99, 0xa9, 0x99,
				0x50, 0x00, 0xc9, 0x36,
					0xb7, 0x59, 0xb7, 0x59, 0xb7, 0x59, 0xb7, 0x59,
					0x49, 0x85, 0x49, 0x85, 0x49, 0x85, 0x49, 0x85,
					0x49, 0x85, 0x49, 0x85, 0x49, 0x85, 0x49, 0x85,
				0x00, 0x00, 0x12, 0x2e,
					0x57, 0x45, 0x57, 0x45, 0x57, 0x45, 0x57, 0x45,
					0xe9, 0x70, 0xe9, 0x70, 0xe9, 0x70, 0xe9, 0x70,
					0xe9, 0x70, 0xe9, 0x70, 0xe9, 0x70, 0xe9, 0x70,
				0x00, 0x80, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x80, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x06, 0x80, 0x80, 0x00,

			/* 2452 MHz */
			0x99, 0x09,
				0x10, 0x01, 0x74, 0x86,
					0xdb, 0x8a, 0xdb, 0x8a, 0xdb, 0x8a, 0xdb, 0x8a,
					0x6d, 0xb6, 0x6d, 0xb6, 0x6d, 0xb6, 0x6d, 0xb6,
					0x6d, 0xb6, 0x6d, 0xb6, 0x6d, 0xb6, 0x6d, 0xb6,
				0xf0, 0x00, 0x1e, 0x6e,
					0xb0, 0x82, 0xb0, 0x82, 0xb0, 0x82, 0xb0, 0x82,
					0x42, 0xae, 0x42, 0xae, 0x42, 0xae, 0x42, 0xae,
					0x42, 0xae, 0x42, 0xae, 0x42, 0xae, 0x42, 0xae,
				0xd0, 0x00, 0xeb, 0x5b,
					0x85, 0x7a, 0x85, 0x7a, 0x85, 0x7a, 0x85, 0x7a,
					0x17, 0xa6, 0x17, 0xa6, 0x17, 0xa6, 0x17, 0xa6,
					0x17, 0xa6, 0x17, 0xa6, 0x17, 0xa6, 0x17, 0xa6,
				0xa0, 0x00, 0xd8, 0x48,
					0x46, 0x6e, 0x46, 0x6e, 0x46, 0x6e, 0x46, 0x6e,
					0xd8, 0x99, 0xd8, 0x99, 0xd8, 0x99, 0xd8, 0x99,
					0xd8, 0x99, 0xd8, 0x99, 0xd8, 0x99, 0xd8, 0x99,
				0x50, 0x00, 0xcd, 0x36,
					0xdb, 0x59, 0xdb, 0x59, 0xdb, 0x59, 0xdb, 0x59,
					0x6d, 0x85, 0x6d, 0x85, 0x6d, 0x85, 0x6d, 0x85,
					0x6d, 0x85, 0x6d, 0x85, 0x6d, 0x85, 0x6d, 0x85,
				0x00, 0x00, 0x10, 0x2e,
					0x71, 0x45, 0x71, 0x45, 0x71, 0x45, 0x71, 0x45,
					0x03, 0x71, 0x03, 0x71, 0x03, 0x71, 0x03, 0x71,
					0x03, 0x71, 0x03, 0x71, 0x03, 0x71, 0x03, 0x71,
				0x00, 0x80, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x80, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x06, 0x80, 0x80, 0x00,

			/* 2557 MHz */
			0x9e, 0x09,
				0x10, 0x01, 0xf3, 0x87,
					0x17, 0x8b, 0x17, 0x8b, 0x17, 0x8b, 0x17, 0x8b,
					0xa9, 0xb6, 0xa9, 0xb6, 0xa9, 0xb6, 0xa9, 0xb6,
					0xa9, 0xb6, 0xa9, 0xb6, 0xa9, 0xb6, 0xa9, 0xb6,
				0xf0, 0x00, 0x07, 0x6f,
					0xe9, 0x82, 0xe9, 0x82, 0xe9, 0x82, 0xe9, 0x82,
					0x7b, 0xae, 0x7b, 0xae, 0x7b, 0xae, 0x7b, 0xae,
					0x7b, 0xae, 0x7b, 0xae, 0x7b, 0xae, 0x7b, 0xae,
				0xd0, 0x00, 0x73, 0x5c,
					0xba, 0x7a, 0xba, 0x7a, 0xba, 0x7a, 0xba, 0x7a,
					0x4c, 0xa6, 0x4c, 0xa6, 0x4c, 0xa6, 0x4c, 0xa6,
					0x4c, 0xa6, 0x4c, 0xa6, 0x4c, 0xa6, 0x4c, 0xa6,
				0xa0, 0x00, 0x0d, 0x49,
					0x74, 0x6e, 0x74, 0x6e, 0x74, 0x6e, 0x74, 0x6e,
					0x06, 0x9a, 0x06, 0x9a, 0x06, 0x9a, 0x06, 0x9a,
					0x06, 0x9a, 0x06, 0x9a, 0x06, 0x9a, 0x06, 0x9a,
				0x50, 0x00, 0xd1, 0x36,
					0xff, 0x59, 0xff, 0x59, 0xff, 0x59, 0xff, 0x59,
					0x91, 0x85, 0x91, 0x85, 0x91, 0x85, 0x91, 0x85,
					0x91, 0x85, 0x91, 0x85, 0x91, 0x85, 0x91, 0x85,
				0x00, 0x00, 0x0e, 0x2e,
					0x8b, 0x45, 0x8b, 0x45, 0x8b, 0x45, 0x8b, 0x45,
					0x1d, 0x71, 0x1d, 0x71, 0x1d, 0x71, 0x1d, 0x71,
					0x1d, 0x71, 0x1d, 0x71, 0x1d, 0x71, 0x1d, 0x71,
				0x00, 0x80, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x80, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x06, 0x80, 0x80, 0x00,

			/* 2562 MHz */
			0xa3, 0x09,
				0x10, 0x01, 0x72, 0x89,
					0x54, 0x8b, 0x54, 0x8b, 0x54, 0x8b, 0x54, 0x8b,
					0xe6, 0xb6, 0xe6, 0xb6, 0xe6, 0xb6, 0xe6, 0xb6,
					0xe6, 0xb6, 0xe6, 0xb6, 0xe6, 0xb6, 0xe6, 0xb6,
				0xf0, 0x00, 0xf0, 0x6f,
					0x21, 0x83, 0x21, 0x83, 0x21, 0x83, 0x21, 0x83,
					0xb3, 0xae, 0xb3, 0xae, 0xb3, 0xae, 0xb3, 0xae,
					0xb3, 0xae, 0xb3, 0xae, 0xb3, 0xae, 0xb3, 0xae,
				0xd0, 0x00, 0xfb, 0x5c,
					0xee, 0x7a, 0xee, 0x7a, 0xee, 0x7a, 0xee, 0x7a,
					0x80, 0xa6, 0x80, 0xa6, 0x80, 0xa6, 0x80, 0xa6,
					0x80, 0xa6, 0x80, 0xa6, 0x80, 0xa6, 0x80, 0xa6,
				0xa0, 0x00, 0x43, 0x49,
					0xa3, 0x6e, 0xa3, 0x6e, 0xa3, 0x6e, 0xa3, 0x6e,
					0x35, 0x9a, 0x35, 0x9a, 0x35, 0x9a, 0x35, 0x9a,
					0x35, 0x9a, 0x35, 0x9a, 0x35, 0x9a, 0x35, 0x9a,
				0x50, 0x00, 0xd4, 0x36,
					0x24, 0x5a, 0x24, 0x5a, 0x24, 0x5a, 0x24, 0x5a,
					0xb6, 0x85, 0xb6, 0x85, 0xb6, 0x85, 0xb6, 0x85,
					0xb6, 0x85, 0xb6, 0x85, 0xb6, 0x85, 0xb6, 0x85,
				0x00, 0x00, 0x0b, 0x2e,
					0xa6, 0x45, 0xa6, 0x45, 0xa6, 0x45, 0xa6, 0x45,
					0x38, 0x71, 0x38, 0x71, 0x38, 0x71, 0x38, 0x71,
					0x38, 0x71, 0x38, 0x71, 0x38, 0x71, 0x38, 0x71,
				0x00, 0x80, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x80, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x06, 0x80, 0x80, 0x00,

			/* 2572 MHz */
			0xa8, 0x09,
				0x10, 0x01, 0xf1, 0x8a,
					0x91, 0x8b, 0x91, 0x8b, 0x91, 0x8b, 0x91, 0x8b,
					0x23, 0xb7, 0x23, 0xb7, 0x23, 0xb7, 0x23, 0xb7,
					0x23, 0xb7, 0x23, 0xb7, 0x23, 0xb7, 0x23, 0xb7,
				0xf0, 0x00, 0xd9, 0x70,
					0x5a, 0x83, 0x5a, 0x83, 0x5a, 0x83, 0x5a, 0x83,
					0xec, 0xae, 0xec, 0xae, 0xec, 0xae, 0xec, 0xae,
					0xec, 0xae, 0xec, 0xae, 0xec, 0xae, 0xec, 0xae,
				0xd0, 0x00, 0x83, 0x5d,
					0x23, 0x7b, 0x23, 0x7b, 0x23, 0x7b, 0x23, 0x7b,
					0xb5, 0xa6, 0xb5, 0xa6, 0xb5, 0xa6, 0xb5, 0xa6,
					0xb5, 0xa6, 0xb5, 0xa6, 0xb5, 0xa6, 0xb5, 0xa6,
				0xa0, 0x00, 0x78, 0x49,
					0xd1, 0x6e, 0xd1, 0x6e, 0xd1, 0x6e, 0xd1, 0x6e,
					0x63, 0x9a, 0x63, 0x9a, 0x63, 0x9a, 0x63, 0x9a,
					0x63, 0x9a, 0x63, 0x9a, 0x63, 0x9a, 0x63, 0x9a,
				0x50, 0x00, 0xd8, 0x36,
					0x48, 0x5a, 0x48, 0x5a, 0x48, 0x5a, 0x48, 0x5a,
					0xda, 0x85, 0xda, 0x85, 0xda, 0x85, 0xda, 0x85,
					0xda, 0x85, 0xda, 0x85, 0xda, 0x85, 0xda, 0x85,
				0x00, 0x00, 0x09, 0x2e,
					0xc0, 0x45, 0xc0, 0x45, 0xc0, 0x45, 0xc0, 0x45,
					0x52, 0x71, 0x52, 0x71, 0x52, 0x71, 0x52, 0x71,
					0x52, 0x71, 0x52, 0x71, 0x52, 0x71, 0x52, 0x71,
				0x00, 0x80, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x80, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x06, 0x80, 0x80, 0x00,
};

/*
 * Not really sure if this is actually the power_limit database,
 * it looks a bit "related" to PDR_PRISM_ZIF_TX_IQ_CALIBRATION
 */
static const uint8_t template_pa_cal_output_power_limits[] = {
	/* struct pda_custom_wrapper */
	0xae, 0x00, 0xef, 0xbe,      /* PDR_PRISM_PA_CAL_OUTPUT_POWER_LIMITS_CUSTOM */
		0x0d, 0x00, 0x1a, 0x00,		/* 13 entries, 26 bytes per entry */
		0x00, 0x00, 0x52, 0x01,		/* no offset, 338 bytes total */

			/* 2412 MHz */
			0x6c, 0x09,
				0x10, 0x01, 0x10, 0x01, 0x10, 0x01, 0x10, 0x01,
				0xe0, 0x00, 0xe0, 0x00, 0xe0, 0x00, 0xe0, 0x00,
				0xd0, 0x00, 0xd0, 0x00, 0xd0, 0x00, 0xd0, 0x00,

			/* 2417 MHz */
			0x71, 0x09,
				0x10, 0x01, 0x10, 0x01, 0x10, 0x01, 0x10, 0x01,
				0xf0, 0x00, 0xf0, 0x00, 0xf0, 0x00, 0xf0, 0x00,
				0xd0, 0x00, 0xd0, 0x00, 0xd0, 0x00, 0xd0, 0x00,

			/* 2422 MHz */
			0x76, 0x09,
				0x10, 0x01, 0x10, 0x01, 0x10, 0x01, 0x10, 0x01,
				0xf0, 0x00, 0xf0, 0x00, 0xf0, 0x00, 0xf0, 0x00,
				0xd0, 0x00, 0xd0, 0x00, 0xd0, 0x00, 0xd0, 0x00,

			/* 2427 MHz */
			0x7b, 0x09,
				0x10, 0x01, 0x10, 0x01, 0x10, 0x01, 0x10, 0x01,
				0xf0, 0x00, 0xf0, 0x00, 0xf0, 0x00, 0xf0, 0x00,
				0xd0, 0x00, 0xd0, 0x00, 0xd0, 0x00, 0xd0, 0x00,

			/* 2432 MHz */
			0x80, 0x09,
				0x10, 0x01, 0x10, 0x01, 0x10, 0x01, 0x10, 0x01,
				0xf0, 0x00, 0xf0, 0x00, 0xf0, 0x00, 0xf0, 0x00,
				0xd0, 0x00, 0xd0, 0x00, 0xd0, 0x00, 0xd0, 0x00,

			/* 2437 MHz */
			0x85, 0x09,
				0x10, 0x01, 0x10, 0x01, 0x10, 0x01, 0x10, 0x01,
				0xf0, 0x00, 0xf0, 0x00, 0xf0, 0x00, 0xf0, 0x00,
				0xd0, 0x00, 0xd0, 0x00, 0xd0, 0x00, 0xd0, 0x00,

			/* 2442 MHz */
			0x8a, 0x09,
				0x10, 0x01, 0x10, 0x01, 0x10, 0x01, 0x10, 0x01,
				0xf0, 0x00, 0xf0, 0x00, 0xf0, 0x00, 0xf0, 0x00,
				0xd0, 0x00, 0xd0, 0x00, 0xd0, 0x00, 0xd0, 0x00,

			/* 2447 MHz */
			0x8f, 0x09,
				0x10, 0x01, 0x10, 0x01, 0x10, 0x01, 0x10, 0x01,
				0xf0, 0x00, 0xf0, 0x00, 0xf0, 0x00, 0xf0, 0x00,
				0xd0, 0x00, 0xd0, 0x00, 0xd0, 0x00, 0xd0, 0x00,

			/* 2452 MHz */
			0x94, 0x09,
				0x10, 0x01, 0x10, 0x01, 0x10, 0x01, 0x10, 0x01,
				0xf0, 0x00, 0xf0, 0x00, 0xf0, 0x00, 0xf0, 0x00,
				0xd0, 0x00, 0xd0, 0x00, 0xd0, 0x00, 0xd0, 0x00,

			/* 2457 MHz */
			0x99, 0x09,
				0x10, 0x01, 0x10, 0x01, 0x10, 0x01, 0x10, 0x01,
				0xf0, 0x00, 0xf0, 0x00, 0xf0, 0x00, 0xf0, 0x00,
				0xd0, 0x00, 0xd0, 0x00, 0xd0, 0x00, 0xd0, 0x00,

			/* 2462 MHz */
			0x9e, 0x09,
				0x10, 0x01, 0x10, 0x01, 0x10, 0x01, 0x10, 0x01,
				0xf0, 0x00, 0xf0, 0x00, 0xf0, 0x00, 0xf0, 0x00,
				0xd0, 0x00, 0xd0, 0x00, 0xd0, 0x00, 0xd0, 0x00,

			/* 2467 MHz */
			0xa3, 0x09,
				0x10, 0x01, 0x10, 0x01, 0x10, 0x01, 0x10, 0x01,
				0xf0, 0x00, 0xf0, 0x00, 0xf0, 0x00, 0xf0, 0x00,
				0xd0, 0x00, 0xd0, 0x00, 0xd0, 0x00, 0xd0, 0x00,

			/* 2472 MHz */
			0xa8, 0x09,
				0x10, 0x01, 0x10, 0x01, 0x10, 0x01, 0x10, 0x01,
				0xf0, 0x00, 0xf0, 0x00, 0xf0, 0x00, 0xf0, 0x00,
				0xd0, 0x00, 0xd0, 0x00, 0xd0, 0x00, 0xd0, 0x00,
};

static const uint8_t template_pda_iq_autocal[] = {
	/* struct pda_iq_autocal_entry[13] */
	0x42, 0x00, 0x06, 0x19,		/* PDR_PRISM_ZIF_TX_IQ_CALIBRATION */
		/* 2412 MHz */
		0x6c, 0x09, 0x26, 0x00, 0xf8, 0xff, 0xf7, 0xff, 0xff, 0x00,
		/* 2417 MHz */
		0x71, 0x09, 0x26, 0x00, 0xf8, 0xff, 0xf7, 0xff, 0xff, 0x00,
		/* 2422 MHz */
		0x76, 0x09, 0x26, 0x00, 0xf8, 0xff, 0xf7, 0xff, 0xff, 0x00,
		/* 2427 MHz */
		0x7b, 0x09, 0x26, 0x00, 0xf8, 0xff, 0xf7, 0xff, 0xff, 0x00,
		/* 2432 MHz */
		0x80, 0x09, 0x25, 0x00, 0xf7, 0xff, 0xf7, 0xff, 0xff, 0x00,
		/* 2437 MHz */
		0x85, 0x09, 0x25, 0x00, 0xf7, 0xff, 0xf7, 0xff, 0xff, 0x00,
		/* 2442 MHz */
		0x8a, 0x09, 0x25, 0x00, 0xf7, 0xff, 0xf7, 0xff, 0xff, 0x00,
		/* 2447 MHz */
		0x8f, 0x09, 0x25, 0x00, 0xf7, 0xff, 0xf7, 0xff, 0xff, 0x00,
		/* 2452 MHz */
		0x94, 0x09, 0x25, 0x00, 0xf7, 0xff, 0xf7, 0xff, 0xff, 0x00,
		/* 2457 MHz */
		0x99, 0x09, 0x25, 0x00, 0xf5, 0xff, 0xf9, 0xff, 0x00, 0x01,
		/* 2462 MHz */
		0x9e, 0x09, 0x25, 0x00, 0xf5, 0xff, 0xf9, 0xff, 0x00, 0x01,
		/* 2467 MHz */
		0xa3, 0x09, 0x25, 0x00, 0xf5, 0xff, 0xf9, 0xff, 0x00, 0x01,
		/* 2472 MHz */
		0xa8, 0x09, 0x25, 0x00, 0xf5, 0xff, 0xf9, 0xff, 0x00, 0x01,
};

static const uint8_t template_pdr_end[] = {
	0x02, 0x00, 0x00, 0x00,		/* PDR_END */
};

/*	linux/lib/crc-ccitt.c
 *
 * This mysterious table is just the CRC of each possible byte. It can be
 * computed using the standard bit-at-a-time methods. The polynomial can
 * be seen in entry 128, 0x8408. This corresponds to x^0 + x^5 + x^12.
 * Add the implicit x^16, and you have the standard CRC-CCITT.
 */
static uint16_t const crc_ccitt_table[256] = {
	0x0000, 0x1189, 0x2312, 0x329b, 0x4624, 0x57ad, 0x6536, 0x74bf,
	0x8c48, 0x9dc1, 0xaf5a, 0xbed3, 0xca6c, 0xdbe5, 0xe97e, 0xf8f7,
	0x1081, 0x0108, 0x3393, 0x221a, 0x56a5, 0x472c, 0x75b7, 0x643e,
	0x9cc9, 0x8d40, 0xbfdb, 0xae52, 0xdaed, 0xcb64, 0xf9ff, 0xe876,
	0x2102, 0x308b, 0x0210, 0x1399, 0x6726, 0x76af, 0x4434, 0x55bd,
	0xad4a, 0xbcc3, 0x8e58, 0x9fd1, 0xeb6e, 0xfae7, 0xc87c, 0xd9f5,
	0x3183, 0x200a, 0x1291, 0x0318, 0x77a7, 0x662e, 0x54b5, 0x453c,
	0xbdcb, 0xac42, 0x9ed9, 0x8f50, 0xfbef, 0xea66, 0xd8fd, 0xc974,
	0x4204, 0x538d, 0x6116, 0x709f, 0x0420, 0x15a9, 0x2732, 0x36bb,
	0xce4c, 0xdfc5, 0xed5e, 0xfcd7, 0x8868, 0x99e1, 0xab7a, 0xbaf3,
	0x5285, 0x430c, 0x7197, 0x601e, 0x14a1, 0x0528, 0x37b3, 0x263a,
	0xdecd, 0xcf44, 0xfddf, 0xec56, 0x98e9, 0x8960, 0xbbfb, 0xaa72,
	0x6306, 0x728f, 0x4014, 0x519d, 0x2522, 0x34ab, 0x0630, 0x17b9,
	0xef4e, 0xfec7, 0xcc5c, 0xddd5, 0xa96a, 0xb8e3, 0x8a78, 0x9bf1,
	0x7387, 0x620e, 0x5095, 0x411c, 0x35a3, 0x242a, 0x16b1, 0x0738,
	0xffcf, 0xee46, 0xdcdd, 0xcd54, 0xb9eb, 0xa862, 0x9af9, 0x8b70,
	0x8408, 0x9581, 0xa71a, 0xb693, 0xc22c, 0xd3a5, 0xe13e, 0xf0b7,
	0x0840, 0x19c9, 0x2b52, 0x3adb, 0x4e64, 0x5fed, 0x6d76, 0x7cff,
	0x9489, 0x8500, 0xb79b, 0xa612, 0xd2ad, 0xc324, 0xf1bf, 0xe036,
	0x18c1, 0x0948, 0x3bd3, 0x2a5a, 0x5ee5, 0x4f6c, 0x7df7, 0x6c7e,
	0xa50a, 0xb483, 0x8618, 0x9791, 0xe32e, 0xf2a7, 0xc03c, 0xd1b5,
	0x2942, 0x38cb, 0x0a50, 0x1bd9, 0x6f66, 0x7eef, 0x4c74, 0x5dfd,
	0xb58b, 0xa402, 0x9699, 0x8710, 0xf3af, 0xe226, 0xd0bd, 0xc134,
	0x39c3, 0x284a, 0x1ad1, 0x0b58, 0x7fe7, 0x6e6e, 0x5cf5, 0x4d7c,
	0xc60c, 0xd785, 0xe51e, 0xf497, 0x8028, 0x91a1, 0xa33a, 0xb2b3,
	0x4a44, 0x5bcd, 0x6956, 0x78df, 0x0c60, 0x1de9, 0x2f72, 0x3efb,
	0xd68d, 0xc704, 0xf59f, 0xe416, 0x90a9, 0x8120, 0xb3bb, 0xa232,
	0x5ac5, 0x4b4c, 0x79d7, 0x685e, 0x1ce1, 0x0d68, 0x3ff3, 0x2e7a,
	0xe70e, 0xf687, 0xc41c, 0xd595, 0xa12a, 0xb0a3, 0x8238, 0x93b1,
	0x6b46, 0x7acf, 0x4854, 0x59dd, 0x2d62, 0x3ceb, 0x0e70, 0x1ff9,
	0xf78f, 0xe606, 0xd49d, 0xc514, 0xb1ab, 0xa022, 0x92b9, 0x8330,
	0x7bc7, 0x6a4e, 0x58d5, 0x495c, 0x3de3, 0x2c6a, 0x1ef1, 0x0f78
};

static inline uint16_t crc_ccitt_byte(uint16_t crc, const uint8_t c)
{
	return (crc >> 8) ^ crc_ccitt_table[(crc ^ c) & 0xff];
}

static uint16_t crc_ccitt(uint16_t crc, uint8_t const *buffer, size_t len)
{
	while (len--)
		crc = crc_ccitt_byte(crc, *buffer++);
	return crc;
}

static int generate_pa_cal_curve(uint8_t *buffer,
				 const uint8_t *txgen_image,
				 size_t txgen_image_size)
{
	memcpy(buffer, template_pa_cal_curve, sizeof(template_pa_cal_curve));

	//TODO

	return 0;
}

static int generate_iq_autocal(uint8_t *buffer,
			       const uint8_t *iqalign_image,
			       size_t iqalign_image_size)
{
	unsigned int expected_size;
	uint8_t *out;
	const uint8_t *in;
	unsigned int i;

	expected_size = 13 * 8 + 4;
	if (iqalign_image_size != expected_size) {
		fprintf(stderr, "ERROR: Unknown wlan-iq-align image size. "
			"Got %u, but expected %u\n",
			(unsigned int)iqalign_image_size, expected_size);
		return -EINVAL;
	}

	in = iqalign_image;
	out = buffer;

	memcpy(out, template_pda_iq_autocal, sizeof(template_pda_iq_autocal));

	in += 4;	/* Skip zero padding */
	out += 4;	/* Skip header */
	for (i = 0; i < 13; i++) {
		out += 2;	/* Skip little endian 16bit Mhz number */
		memcpy(out, in, 8);
		out += 8;	/* Next element */
		in += 8;	/* Next element */
	}

	return 0;
}

static int generate_eeprom(FILE *outstream,
			   const uint8_t *txgen_image,
			   size_t txgen_image_size,
			   const uint8_t *iqalign_image,
			   size_t iqalign_image_size)
{
	int err, res = 0;
	uint16_t crc = ~0;
	uint8_t crc_buf[2];
	uint8_t gen_pda_iq_autocal[sizeof(template_pda_iq_autocal)];
	uint8_t gen_pa_cal_curve[sizeof(template_pa_cal_curve)];

	err = generate_iq_autocal(gen_pda_iq_autocal,
				  iqalign_image, iqalign_image_size);
	if (err)
		return err;
	err = generate_pa_cal_curve(gen_pa_cal_curve,
				    txgen_image, txgen_image_size);
	if (err)
		return err;

#define writebuf(buffer, do_crc) do {						\
		if (do_crc)							\
			crc = crc_ccitt(crc, buffer, sizeof(buffer));		\
		res = fwrite(buffer, sizeof(buffer), 1, outstream);		\
		if (res != 1) {							\
			fprintf(stderr, "Failed to write output data\n");	\
			return -EIO;						\
		}								\
	} while (0)

	writebuf(template_eeprom_pda_wrap, 0);
	writebuf(template_mac_address, 1);
	writebuf(template_bootrec_exp_if, 1);
	writebuf(template_pda_country_list, 1);
	writebuf(template_pda_country, 1);
	writebuf(template_antenna_gain, 1);
	writebuf(template_rssi_linear_approx_custom2, 1);
	writebuf(gen_pa_cal_curve, 1);
	writebuf(template_pa_cal_output_power_limits, 1);
	writebuf(gen_pda_iq_autocal, 1);
	writebuf(template_pdr_end, 1);

	crc = ~crc;
	crc_buf[0] = crc & 0xFF;
	crc_buf[1] = (crc >> 8) & 0xFF;
	res = fwrite(crc_buf, sizeof(crc_buf), 1, outstream);
	if (res != 1) {
		fprintf(stderr, "Failed to write CRC\n");
		return -EIO;
	}

	return 0;
}

static void * map_file(const char *filepath, int readonly,
		       uint64_t *filelen)
{
	int fd;
	off_t len;
	void *data;

	fd = open(filepath, readonly ? O_RDONLY : O_RDWR);
	if (fd < 0) {
		fprintf(stderr, "Failed to open file %s: %s\n",
			filepath, strerror(errno));
		return NULL;
	}
	len = lseek(fd, 0, SEEK_END);
	if (len < 0 || lseek(fd, 0, SEEK_SET)) {
		fprintf(stderr, "Failed to calculate file length of %s: %s\n",
			filepath, strerror(errno));
		close(fd);
		return NULL;
	}

	data = mmap(NULL, len,
		    readonly ? PROT_READ : (PROT_READ | PROT_WRITE),
		    readonly ? MAP_PRIVATE : 0,
		    fd, 0);
	close(fd);
	if (data == MAP_FAILED) {
		fprintf(stderr, "Failed to MMAP file %s: %s\n",
			filepath, strerror(errno));
		return NULL;
	}
	madvise(data, len, MADV_SEQUENTIAL);

	*filelen = len;

	return data;
}

static void unmap_file(void *mapping, uint64_t len)
{
	if (mapping)
		munmap(mapping, len);
}

static void usage(FILE *fd)
{
	fprintf(fd, "cal2p54 - CAL to p54spi EEPROM converter\n");
	fprintf(fd, "\n");
	fprintf(fd, "Usage: cal2p54 WLAN-TX-GEN2-IMAGE WLAN-IQ-ALIGN-IMAGE\n");
}

int main(int argc, char **argv)
{
	int err, ret = 0;
	uint8_t *txgen_image = NULL;
	uint64_t txgen_image_len = 0;
	uint8_t *iqalign_image = NULL;
	uint64_t iqalign_image_len = 0;

	const char *txgen_filepath;
	const char *iqalign_filepath;

	if (argc != 3) {
		usage(stderr);
		return 1;
	}
	txgen_filepath = argv[1];
	iqalign_filepath = argv[2];

	txgen_image = map_file(txgen_filepath, 1, &txgen_image_len);
	iqalign_image = map_file(iqalign_filepath, 1, &iqalign_image_len);
	if (!txgen_image || !iqalign_image) {
		ret = 1;
		goto out;
	}

	err = generate_eeprom(stdout, txgen_image, txgen_image_len,
			      iqalign_image, iqalign_image_len);
	if (err) {
		ret = 1;
		goto out;
	}

out:
	unmap_file(txgen_image, txgen_image_len);
	unmap_file(iqalign_image, iqalign_image_len);

	return ret;
}
